import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { ProBadge } from "@/components/ui/pro-badge";
import { MapPin, Phone, Mail, Globe, Star } from "lucide-react";

interface Business {
  id: number;
  name: string;
  description: string;
  category: string;
  location: string;
  contactEmail?: string;
  contactPhone?: string;
  website?: string;
  rating: number;
  reviewCount: number;
  imageUrl?: string;
  isVerified: boolean;
}

export function RecommendedBusinesses() {
  const { data: businesses, isLoading } = useQuery<Business[]>({
    queryKey: ["/api/businesses/featured-homepage"],
  });

  if (isLoading) {
    return (
      <div className="mt-12 animate-fade-in-up" style={{animationDelay: '0.4s'}}>
        <h3 className="text-2xl font-bold text-white text-center mb-8">Recommended Businesses</h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          {[...Array(3)].map((_, i) => (
            <Card key={i} className="animate-pulse bg-white/10 backdrop-blur-md border-white/20">
              <CardHeader>
                <div className="h-6 bg-white/20 rounded w-3/4"></div>
                <div className="h-4 bg-white/20 rounded w-1/2"></div>
              </CardHeader>
              <CardContent>
                <div className="space-y-2">
                  <div className="h-4 bg-white/20 rounded"></div>
                  <div className="h-4 bg-white/20 rounded w-5/6"></div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  }

  if (!businesses || businesses.length === 0) {
    return null;
  }

  return (
    <div className="mt-12 animate-fade-in-up" style={{animationDelay: '0.4s'}}>
      <h3 className="text-2xl font-bold text-white text-center mb-8">Recommended Businesses</h3>
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        {businesses.map((business) => (
          <Card 
            key={business.id} 
            className="bg-white/10 backdrop-blur-md border-white/20 hover:bg-white/20 transition-all duration-300 hover:shadow-xl hover:scale-105"
            data-testid={`card-featured-business-${business.id}`}
          >
            <CardHeader>
              <div className="flex justify-between items-start">
                <div className="flex-1">
                  <CardTitle className="text-white flex items-center gap-2 text-lg">
                    {business.name}
                    <ProBadge />
                  </CardTitle>
                  <div className="flex items-center gap-1 text-white/80 text-sm">
                    <MapPin className="h-3 w-3" />
                    {business.location}
                  </div>
                </div>
              </div>
            </CardHeader>
            
            <CardContent>
              <p className="text-white/90 text-sm mb-4 line-clamp-2">
                {business.description}
              </p>
              
              <div className="space-y-2">
                {business.contactPhone && (
                  <div className="flex items-center gap-2 text-sm text-white/80">
                    <Phone className="h-3 w-3" />
                    <a 
                      href={`tel:${business.contactPhone}`} 
                      className="hover:text-white transition-colors"
                    >
                      {business.contactPhone}
                    </a>
                  </div>
                )}
                
                {business.contactEmail && (
                  <div className="flex items-center gap-2 text-sm text-white/80">
                    <Mail className="h-3 w-3" />
                    <a 
                      href={`mailto:${business.contactEmail}`} 
                      className="hover:text-white transition-colors"
                    >
                      Contact
                    </a>
                  </div>
                )}
                
                {business.website && (
                  <div className="flex items-center gap-2 text-sm text-white/80">
                    <Globe className="h-3 w-3" />
                    <a 
                      href={business.website.startsWith('http') ? business.website : `https://${business.website}`}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="hover:text-white transition-colors"
                    >
                      Visit Website
                    </a>
                  </div>
                )}
              </div>
              
              {business.rating > 0 && (
                <div className="flex items-center gap-1 mt-3 pt-3 border-t border-white/20">
                  <Star className="h-3 w-3 fill-yellow-400 text-yellow-400" />
                  <span className="text-sm font-medium text-white">{business.rating.toFixed(1)}</span>
                  <span className="text-sm text-white/70">({business.reviewCount} reviews)</span>
                </div>
              )}
            </CardContent>
          </Card>
        ))}
      </div>
    </div>
  );
}