import { useSortable } from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import { Card, CardContent, CardHeader } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Switch } from "@/components/ui/switch";
import { PageSection } from "@shared/schema";
import { GripVertical, Settings, Trash2, Copy, Eye, EyeOff } from "lucide-react";

interface SortableSectionProps {
  section: PageSection;
  onEdit: (section: PageSection) => void;
  onDelete: (sectionId: string) => void;
  onDuplicate: (section: PageSection) => void;
  onToggleVisibility: (sectionId: string, isVisible: boolean) => void;
}

export function SortableSection({
  section,
  onEdit,
  onDelete,
  onDuplicate,
  onToggleVisibility
}: SortableSectionProps) {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging
  } = useSortable({ id: section.id });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.5 : 1
  };

  const getSectionIcon = (type: string) => {
    switch (type) {
      case "hero":
        return "🎯";
      case "community":
        return "👥";
      case "about":
        return "ℹ️";
      case "features":
        return "⭐";
      case "contact":
        return "📞";
      case "business":
        return "🏢";
      default:
        return "📝";
    }
  };

  const getSectionDescription = (section: PageSection) => {
    switch (section.type) {
      case "hero":
        return section.content.heading || "Hero section";
      case "community":
        return section.content.heading || "Community section";
      case "about":
        return section.content.heading || "About section";
      case "custom":
        return "Custom HTML content";
      default:
        return section.content.description || "Content section";
    }
  };

  return (
    <Card
      ref={setNodeRef}
      style={style}
      className={`transition-all duration-200 ${
        isDragging ? "shadow-lg" : "shadow-sm"
      } ${!section.isVisible ? "opacity-60" : ""}`}
    >
      <CardHeader className="pb-3">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <div
              {...attributes}
              {...listeners}
              className="cursor-grab active:cursor-grabbing p-1 rounded hover:bg-gray-100"
            >
              <GripVertical className="w-4 h-4 text-gray-400" />
            </div>
            <div className="flex items-center space-x-2">
              <span className="text-lg">{getSectionIcon(section.type)}</span>
              <div>
                <h3 className="font-medium text-gray-900">{section.title}</h3>
                <p className="text-sm text-gray-500 truncate max-w-xs">
                  {getSectionDescription(section)}
                </p>
              </div>
            </div>
          </div>
          <div className="flex items-center space-x-2">
            <div className="flex items-center space-x-2">
              {section.isVisible ? (
                <Eye className="w-4 h-4 text-gray-400" />
              ) : (
                <EyeOff className="w-4 h-4 text-gray-400" />
              )}
              <Switch
                checked={section.isVisible}
                onCheckedChange={(checked) => onToggleVisibility(section.id, checked)}
              />
            </div>
            <Button
              variant="ghost"
              size="sm"
              onClick={() => onEdit(section)}
              className="h-8 w-8 p-0"
            >
              <Settings className="w-4 h-4" />
            </Button>
            <Button
              variant="ghost"
              size="sm"
              onClick={() => onDuplicate(section)}
              className="h-8 w-8 p-0"
            >
              <Copy className="w-4 h-4" />
            </Button>
            <Button
              variant="ghost"
              size="sm"
              onClick={() => onDelete(section.id)}
              className="h-8 w-8 p-0 text-red-500 hover:text-red-700"
            >
              <Trash2 className="w-4 h-4" />
            </Button>
          </div>
        </div>
      </CardHeader>
      <CardContent className="pt-0">
        <div className="text-xs text-gray-500 space-y-1">
          <div>Type: {section.type}</div>
          <div>Order: {section.order}</div>
          {section.styles.background && (
            <div className="flex items-center space-x-2">
              <span>Background:</span>
              <div
                className="w-4 h-4 rounded border border-gray-300"
                style={{ background: section.styles.background }}
              />
            </div>
          )}
        </div>
      </CardContent>
    </Card>
  );
}