import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { CheckCircle, XCircle, Clock, MapPin, Phone, Mail, Globe } from "lucide-react";
import { AdminLayout } from "@/components/admin-layout";

export default function AdminBusinessApprovals() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [rejectionReason, setRejectionReason] = useState("");
  const [selectedBusinessId, setSelectedBusinessId] = useState<number | null>(null);
  const [showRejectDialog, setShowRejectDialog] = useState(false);

  // Get pending businesses
  const { data: pendingBusinesses = [], isLoading } = useQuery({
    queryKey: ['/api/admin/businesses/pending'],
  });

  // Approve business mutation
  const approveMutation = useMutation({
    mutationFn: async (businessId: number) => {
      await apiRequest(`/api/admin/businesses/${businessId}/approve`, "POST", {});
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/admin/businesses/pending'] });
      toast({
        title: "Business approved",
        description: "The business has been approved and published to the directory.",
      });
    },
    onError: (error) => {
      toast({
        title: "Approval failed",
        description: "Failed to approve the business. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Reject business mutation
  const rejectMutation = useMutation({
    mutationFn: async ({ businessId, reason }: { businessId: number; reason: string }) => {
      await apiRequest(`/api/admin/businesses/${businessId}/reject`, "POST", { reason });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['/api/admin/businesses/pending'] });
      setShowRejectDialog(false);
      setRejectionReason("");
      setSelectedBusinessId(null);
      toast({
        title: "Business rejected",
        description: "The business submission has been rejected.",
      });
    },
    onError: (error) => {
      toast({
        title: "Rejection failed",
        description: "Failed to reject the business. Please try again.",
        variant: "destructive",
      });
    },
  });

  const handleApprove = (businessId: number) => {
    approveMutation.mutate(businessId);
  };

  const handleReject = (businessId: number) => {
    setSelectedBusinessId(businessId);
    setShowRejectDialog(true);
  };

  const confirmReject = () => {
    if (selectedBusinessId && rejectionReason.trim()) {
      rejectMutation.mutate({ 
        businessId: selectedBusinessId, 
        reason: rejectionReason.trim() 
      });
    }
  };

  if (isLoading) {
    return (
      <AdminLayout title="Business Approvals" description="Review and approve business submissions for the directory">
        <div className="text-center">
          <Clock className="mx-auto h-12 w-12 text-gray-400" />
          <h3 className="mt-2 text-sm font-medium text-gray-900">Loading pending submissions...</h3>
        </div>
      </AdminLayout>
    );
  }

  return (
    <AdminLayout title="Business Approvals" description="Review and approve business submissions for the directory">
      <div>

        {/* Statistics */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Pending Reviews</CardTitle>
              <Clock className="h-4 w-4 text-muted-foreground" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{pendingBusinesses.length}</div>
              <p className="text-xs text-muted-foreground">
                Awaiting admin approval
              </p>
            </CardContent>
          </Card>
          
          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Processing</CardTitle>
              <CheckCircle className="h-4 w-4 text-green-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {approveMutation.isPending ? "1" : "0"}
              </div>
              <p className="text-xs text-muted-foreground">
                Currently being processed
              </p>
            </CardContent>
          </Card>

          <Card>
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">Actions</CardTitle>
              <XCircle className="h-4 w-4 text-red-600" />
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">
                {rejectMutation.isPending ? "1" : "0"}
              </div>
              <p className="text-xs text-muted-foreground">
                Actions in progress
              </p>
            </CardContent>
          </Card>
        </div>

        {/* Pending Businesses List */}
        {pendingBusinesses.length === 0 ? (
          <Card>
            <CardContent className="text-center py-12">
              <CheckCircle className="mx-auto h-12 w-12 text-green-500 mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">All caught up!</h3>
              <p className="text-gray-500">There are no pending business submissions to review.</p>
            </CardContent>
          </Card>
        ) : (
          <div className="space-y-6">
            {pendingBusinesses.map((business: any) => (
              <Card key={business.id} className="overflow-hidden">
                <CardHeader>
                  <div className="flex justify-between items-start">
                    <div>
                      <CardTitle className="text-xl">{business.name}</CardTitle>
                      <CardDescription className="flex items-center gap-1 mt-1">
                        <MapPin className="h-3 w-3" />
                        {business.location}
                      </CardDescription>
                    </div>
                    <div className="flex items-center gap-2">
                      <Badge variant="outline" className="bg-yellow-50 text-yellow-700 border-yellow-200">
                        <Clock className="h-3 w-3 mr-1" />
                        Pending
                      </Badge>
                      <Badge variant="outline">
                        {business.category}
                      </Badge>
                    </div>
                  </div>
                </CardHeader>
                
                <CardContent className="space-y-4">
                  <div>
                    <h4 className="font-medium text-gray-900 mb-2">Business Description</h4>
                    <p className="text-gray-600 text-sm leading-relaxed">
                      {business.description}
                    </p>
                  </div>

                  {/* Contact Information */}
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                      <h4 className="font-medium text-gray-900 mb-2">Contact Information</h4>
                      <div className="space-y-2 text-sm">
                        {business.contactEmail && (
                          <div className="flex items-center gap-2 text-gray-600">
                            <Mail className="h-3 w-3" />
                            {business.contactEmail}
                          </div>
                        )}
                        {business.contactPhone && (
                          <div className="flex items-center gap-2 text-gray-600">
                            <Phone className="h-3 w-3" />
                            {business.contactPhone}
                          </div>
                        )}
                        {business.website && (
                          <div className="flex items-center gap-2 text-gray-600">
                            <Globe className="h-3 w-3" />
                            <a 
                              href={business.website} 
                              target="_blank" 
                              rel="noopener noreferrer"
                              className="text-blue-600 hover:underline"
                            >
                              {business.website}
                            </a>
                          </div>
                        )}
                      </div>
                    </div>

                    <div>
                      <h4 className="font-medium text-gray-900 mb-2">Submission Details</h4>
                      <div className="space-y-1 text-sm text-gray-600">
                        <p>Submitted: {new Date(business.createdAt).toLocaleDateString()}</p>
                        <p>Time: {new Date(business.createdAt).toLocaleTimeString()}</p>
                        {business.whatsappNumber && (
                          <p>WhatsApp: {business.whatsappNumber}</p>
                        )}
                      </div>
                    </div>
                  </div>

                  {/* Action Buttons */}
                  <div className="flex gap-3 pt-4 border-t">
                    <Button
                      onClick={() => handleApprove(business.id)}
                      disabled={approveMutation.isPending}
                      className="bg-green-600 hover:bg-green-700 text-white"
                      data-testid={`button-approve-${business.id}`}
                    >
                      <CheckCircle className="h-4 w-4 mr-2" />
                      {approveMutation.isPending ? "Approving..." : "Approve"}
                    </Button>
                    
                    <Button
                      variant="destructive"
                      onClick={() => handleReject(business.id)}
                      disabled={rejectMutation.isPending}
                      data-testid={`button-reject-${business.id}`}
                    >
                      <XCircle className="h-4 w-4 mr-2" />
                      Reject
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}

        {/* Rejection Dialog */}
        <Dialog open={showRejectDialog} onOpenChange={setShowRejectDialog}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Reject Business Submission</DialogTitle>
              <DialogDescription>
                Please provide a reason for rejecting this business submission. This will help improve future submissions.
              </DialogDescription>
            </DialogHeader>
            
            <div className="space-y-4">
              <Textarea
                placeholder="Reason for rejection (e.g., incomplete information, inappropriate content, duplicate listing...)"
                value={rejectionReason}
                onChange={(e) => setRejectionReason(e.target.value)}
                className="min-h-[100px]"
                data-testid="textarea-rejection-reason"
              />
              
              <div className="flex justify-end gap-3">
                <Button
                  variant="outline"
                  onClick={() => {
                    setShowRejectDialog(false);
                    setRejectionReason("");
                    setSelectedBusinessId(null);
                  }}
                  data-testid="button-cancel-reject"
                >
                  Cancel
                </Button>
                <Button
                  variant="destructive"
                  onClick={confirmReject}
                  disabled={!rejectionReason.trim() || rejectMutation.isPending}
                  data-testid="button-confirm-reject"
                >
                  {rejectMutation.isPending ? "Rejecting..." : "Confirm Rejection"}
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      </div>
    </AdminLayout>
  );
}