import { useState, useEffect } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage, FormDescription } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { useToast } from "@/hooks/use-toast";
import { Loader2, TestTube, Info, ArrowLeft } from "lucide-react";
import { Switch } from "@/components/ui/switch";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { Link } from "wouter";

const customerEmailSchema = z.object({
  fromEmail: z.string().email("Please enter a valid from email address"),
  fromName: z.string().min(1, "From name is required"),
  replyToEmail: z.string().email("Please enter a valid reply-to email address").optional().or(z.literal("")),
  subject: z.string().min(5, "Subject must be at least 5 characters"),
  emailTemplate: z.string().min(50, "Email template must be at least 50 characters"),
  isActive: z.boolean().optional(),
});

type CustomerEmailFormData = z.infer<typeof customerEmailSchema>;

export default function AdminCustomerEmailPage() {
  const [isLoading, setIsLoading] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [showPreview, setShowPreview] = useState(false);
  const [previewContent, setPreviewContent] = useState("");
  const { toast } = useToast();

  const form = useForm<CustomerEmailFormData>({
    resolver: zodResolver(customerEmailSchema),
    defaultValues: {
      fromEmail: "",
      fromName: "LoveMedia Foundation",
      replyToEmail: "info@lovemedia.foundation",
      subject: "Thank You for Your Data Deal Order - {{orderNumber}}",
      emailTemplate: "",
      isActive: true,
    },
  });

  const loadSettings = async () => {
    setIsLoading(true);
    try {
      const response = await fetch("/api/admin/data-deals/customer-email-settings");
      if (response.ok) {
        const settings = await response.json();
        if (settings) {
          form.reset({
            fromEmail: settings.fromEmail || "",
            fromName: settings.fromName || "LoveMedia Foundation",
            replyToEmail: settings.replyToEmail || "info@lovemedia.foundation",
            subject: settings.subject || "Thank You for Your Data Deal Order - {{orderNumber}}",
            emailTemplate: settings.emailTemplate || "",
            isActive: settings.isActive !== false,
          });
        }
      } else {
        console.error("Failed to load customer email settings");
      }
    } catch (error) {
      console.error("Error loading customer email settings:", error);
      toast({
        title: "Error",
        description: "Failed to load customer email settings",
        variant: "destructive",
      });
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    loadSettings();
  }, []);

  const onSubmit = async (data: CustomerEmailFormData) => {
    setIsSaving(true);
    try {
      const response = await fetch("/api/admin/data-deals/customer-email-settings", {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(data),
      });

      if (response.ok) {
        toast({
          title: "Success",
          description: "Customer email settings saved successfully",
        });
      } else {
        const errorData = await response.json();
        throw new Error(errorData.message || "Failed to save settings");
      }
    } catch (error) {
      console.error("Error saving customer email settings:", error);
      toast({
        title: "Error",
        description: error instanceof Error ? error.message : "Failed to save customer email settings",
        variant: "destructive",
      });
    } finally {
      setIsSaving(false);
    }
  };

  const generatePreview = () => {
    const formData = form.getValues();
    const sampleData = {
      orderNumber: "LMF-2025-001",
      customerName: "John Smith",
      customerEmail: "john.smith@example.com", 
      customerMobile: "0823456789",
      packageName: "50GB Premium Data Package",
      totalPrice: "599",
      installationAddress: "123 Main Street, Cape Town, 8001",
      donationAmount: "200"
    };

    let content = formData.emailTemplate;
    Object.entries(sampleData).forEach(([key, value]) => {
      const regex = new RegExp(`{{${key}}}`, 'g');
      content = content.replace(regex, value);
    });

    setPreviewContent(content);
    setShowPreview(true);
  };

  const availableVariables = [
    "orderNumber", "orderDate", "customerName", "customerEmail", "customerMobile", 
    "customerIdNumber", "packageName", "basePrice", "premiumPrice", "totalPrice", 
    "installationAddress", "deliveryAddress", "donationAmount"
  ];

  return (
    <div className="container mx-auto py-6 space-y-6">
      <div className="flex items-center gap-4 mb-6">
        <Link href="/admin">
          <Button variant="outline" size="sm">
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back to Admin
          </Button>
        </Link>
        <div>
          <h1 className="text-2xl font-bold">Customer Confirmation Email Settings</h1>
          <p className="text-muted-foreground">Configure email templates for customer order confirmations</p>
        </div>
      </div>

      <Alert>
        <Info className="h-4 w-4" />
        <AlertDescription>
          These settings control the thank you emails sent to customers after they place data deal orders. 
          The email is sent automatically after the order is processed.
        </AlertDescription>
      </Alert>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <div className="lg:col-span-2">
          <Card>
            <CardHeader>
              <CardTitle>Email Configuration</CardTitle>
              <CardDescription>
                Configure the email settings for customer confirmation emails
              </CardDescription>
            </CardHeader>
            <CardContent>
              {isLoading ? (
                <div className="flex items-center justify-center py-8">
                  <Loader2 className="h-6 w-6 animate-spin" />
                  <span className="ml-2">Loading settings...</span>
                </div>
              ) : (
                <Form {...form}>
                  <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <FormField
                        control={form.control}
                        name="fromEmail"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>From Email</FormLabel>
                            <FormControl>
                              <Input placeholder="noreply@lovemedia.foundation" {...field} />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="fromName"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>From Name</FormLabel>
                            <FormControl>
                              <Input placeholder="LoveMedia Foundation" {...field} />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <FormField
                        control={form.control}
                        name="replyToEmail"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Reply-To Email</FormLabel>
                            <FormControl>
                              <Input placeholder="info@lovemedia.foundation" {...field} />
                            </FormControl>
                            <FormDescription>
                              Email address where customer replies will be sent
                            </FormDescription>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="isActive"
                        render={({ field }) => (
                          <FormItem className="flex flex-row items-center justify-between rounded-lg border p-3 shadow-sm">
                            <div className="space-y-0.5">
                              <FormLabel>Active</FormLabel>
                              <FormDescription>
                                Enable customer confirmation emails
                              </FormDescription>
                            </div>
                            <FormControl>
                              <Switch
                                checked={field.value}
                                onCheckedChange={field.onChange}
                              />
                            </FormControl>
                          </FormItem>
                        )}
                      />
                    </div>

                    <FormField
                      control={form.control}
                      name="subject"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Email Subject</FormLabel>
                          <FormControl>
                            <Input 
                              placeholder="Thank You for Your Data Deal Order - {{orderNumber}}" 
                              {...field} 
                            />
                          </FormControl>
                          <FormDescription>
                            Use variables like {"{{orderNumber}}"} and {"{{customerName}}"}
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <FormField
                      control={form.control}
                      name="emailTemplate"
                      render={({ field }) => (
                        <FormItem>
                          <FormLabel>Email Template</FormLabel>
                          <FormControl>
                            <Textarea
                              placeholder="Enter your HTML email template here..."
                              className="min-h-[300px] font-mono text-sm"
                              {...field}
                            />
                          </FormControl>
                          <FormDescription>
                            HTML email template. Use variables like {"{{customerName}}"}, {"{{orderNumber}}"}, etc.
                          </FormDescription>
                          <FormMessage />
                        </FormItem>
                      )}
                    />

                    <div className="flex gap-3">
                      <Button type="submit" disabled={isSaving}>
                        {isSaving && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                        Save Settings
                      </Button>
                      
                      <Button type="button" variant="outline" onClick={generatePreview}>
                        <TestTube className="mr-2 h-4 w-4" />
                        Preview
                      </Button>
                    </div>
                  </form>
                </Form>
              )}
            </CardContent>
          </Card>
        </div>

        <div className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Available Variables</CardTitle>
              <CardDescription>
                Use these variables in your subject line and email template
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-2">
                {availableVariables.map((variable) => (
                  <Badge key={variable} variant="secondary" className="mr-2 mb-2">
                    {"{{" + variable + "}}"}
                  </Badge>
                ))}
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Email Flow</CardTitle>
            </CardHeader>
            <CardContent className="space-y-3 text-sm">
              <div className="flex items-start gap-3">
                <div className="w-6 h-6 rounded-full bg-blue-100 text-blue-600 flex items-center justify-center text-xs font-semibold">1</div>
                <div>
                  <p className="font-medium">Order Submitted</p>
                  <p className="text-muted-foreground">Customer submits data deal order</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <div className="w-6 h-6 rounded-full bg-blue-100 text-blue-600 flex items-center justify-center text-xs font-semibold">2</div>
                <div>
                  <p className="font-medium">Order to OpenWeb</p>
                  <p className="text-muted-foreground">Order details sent to wow@openweb.email</p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <div className="w-6 h-6 rounded-full bg-green-100 text-green-600 flex items-center justify-center text-xs font-semibold">3</div>
                <div>
                  <p className="font-medium">Customer Email</p>
                  <p className="text-muted-foreground">Thank you email sent to customer</p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>

      {showPreview && (
        <Card>
          <CardHeader>
            <CardTitle>Email Preview</CardTitle>
            <CardDescription>
              This is how the email will look with sample data
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="border rounded-lg p-4 bg-gray-50">
              <div 
                dangerouslySetInnerHTML={{ __html: previewContent }}
                className="prose prose-sm max-w-none"
              />
            </div>
            <div className="mt-4">
              <Button variant="outline" onClick={() => setShowPreview(false)}>
                Close Preview
              </Button>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}