import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useAuth } from "@/hooks/use-auth";
import { queryClient } from "@/lib/queryClient";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import { useToast } from "@/hooks/use-toast";
import { Link } from "wouter";
import { apiRequest } from "@/lib/queryClient";
import { 
  Users, FileText, MessageSquare, Settings, BarChart3, 
  Mail, Heart, LogOut, Crown, Building, DollarSign, Edit, Search, 
  Image, Star, CheckSquare, Globe, Code, Megaphone, ChevronDown, ChevronRight 
} from "lucide-react";

interface MenuCategory {
  id: string;
  title: string;
  icon: React.ReactNode;
  items: {
    id: string;
    title: string;
    icon: React.ReactNode;
    path?: string;
    isInternal?: boolean;
  }[];
}

const menuCategories: MenuCategory[] = [
  {
    id: "overview",
    title: "Dashboard",
    icon: <BarChart3 className="w-4 h-4" />,
    items: [
      { id: "stats", title: "Overview", icon: <BarChart3 className="w-4 h-4" />, isInternal: true }
    ]
  },
  {
    id: "business",
    title: "Business Management",
    icon: <Building className="w-4 h-4" />,
    items: [
      { id: "approvals", title: "Business Approvals", icon: <CheckSquare className="w-4 h-4" />, path: "/admin/business-approvals" },
      { id: "featured", title: "Featured Businesses", icon: <Star className="w-4 h-4" />, path: "/admin/featured-businesses" },
      { id: "images", title: "Business Images", icon: <Image className="w-4 h-4" />, path: "/admin/business-images" }
    ]
  },
  {
    id: "content",
    title: "Content & Website",
    icon: <FileText className="w-4 h-4" />,
    items: [
      { id: "content-mgmt", title: "Content Management", icon: <FileText className="w-4 h-4" />, isInternal: true },
      { id: "website-editor", title: "Website Editor", icon: <Edit className="w-4 h-4" />, path: "/admin/website-editor" },
      { id: "seo", title: "SEO Management", icon: <Search className="w-4 h-4" />, path: "/admin/seo-management" }
    ]
  },
  {
    id: "communication",
    title: "Communication",
    icon: <Mail className="w-4 h-4" />,
    items: [
      { id: "newsletter", title: "Newsletter", icon: <Megaphone className="w-4 h-4" />, isInternal: true },
      { id: "data-deals-email", title: "Data Deals Email", icon: <Mail className="w-4 h-4" />, path: "/admin/data-deals/email-settings" },
      { id: "customer-email", title: "Customer Email", icon: <Heart className="w-4 h-4" />, path: "/admin/data-deals/customer-email-settings" },
      { id: "header-tags", title: "Header Tags", icon: <Code className="w-4 h-4" />, path: "/admin/header-tags" }
    ]
  },
  {
    id: "community",
    title: "Users & Community",
    icon: <Users className="w-4 h-4" />,
    items: [
      { id: "users", title: "User Management", icon: <Users className="w-4 h-4" />, isInternal: true },
      { id: "submissions", title: "Contact Submissions", icon: <MessageSquare className="w-4 h-4" />, isInternal: true }
    ]
  },
  {
    id: "settings",
    title: "System Settings",
    icon: <Settings className="w-4 h-4" />,
    items: [
      { id: "general", title: "General Settings", icon: <Settings className="w-4 h-4" />, isInternal: true }
    ]
  }
];

export default function AdminDashboard() {
  const { user, logoutMutation } = useAuth();
  const { toast } = useToast();
  const [activeView, setActiveView] = useState("stats");
  const [expandedCategories, setExpandedCategories] = useState<string[]>(["overview"]);

  // Fetch admin stats
  const { data: stats, isLoading: statsLoading } = useQuery({
    queryKey: ["/api/admin/stats"],
  });

  const toggleCategory = (categoryId: string) => {
    setExpandedCategories(prev => 
      prev.includes(categoryId) 
        ? prev.filter(id => id !== categoryId)
        : [...prev, categoryId]
    );
  };

  const handleMenuClick = (item: MenuCategory['items'][0]) => {
    if (item.isInternal) {
      setActiveView(item.id);
    }
    // External paths will be handled by Link components
  };

  const renderStatsView = () => (
    <div className="space-y-6">
      <div>
        <h2 className="text-3xl font-bold tracking-tight text-gray-900 dark:text-white">
          Admin Dashboard
        </h2>
        <p className="text-gray-600 dark:text-gray-400">
          Manage your community, content, and user engagement.
        </p>
      </div>

      {statsLoading ? (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {[...Array(8)].map((_, i) => (
            <Card key={i} className="animate-pulse">
              <CardContent className="p-6">
                <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded mb-2"></div>
                <div className="h-8 bg-gray-200 dark:bg-gray-700 rounded"></div>
              </CardContent>
            </Card>
          ))}
        </div>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Total Users</p>
                  <p className="text-3xl font-bold text-gray-900 dark:text-white">
                    {stats?.totalUsers || 0}
                  </p>
                </div>
                <Users className="w-8 h-8 text-blue-500" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Active Users</p>
                  <p className="text-3xl font-bold text-gray-900 dark:text-white">
                    {stats?.activeUsers || 0}
                  </p>
                </div>
                <Users className="w-8 h-8 text-green-500" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Total Businesses</p>
                  <p className="text-3xl font-bold text-gray-900 dark:text-white">
                    {stats?.totalBusinesses || 0}
                  </p>
                </div>
                <Building className="w-8 h-8 text-purple-500" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Published Content</p>
                  <p className="text-3xl font-bold text-gray-900 dark:text-white">
                    {stats?.publishedContent || 0}
                  </p>
                </div>
                <FileText className="w-8 h-8 text-orange-500" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">WhatsApp Groups</p>
                  <p className="text-3xl font-bold text-gray-900 dark:text-white">
                    {stats?.whatsappGroups || 0}
                  </p>
                </div>
                <MessageSquare className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Total Members</p>
                  <p className="text-3xl font-bold text-gray-900 dark:text-white">
                    {stats?.totalMembers || 0}
                  </p>
                </div>
                <Users className="w-8 h-8 text-pink-500" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Pending Submissions</p>
                  <p className="text-3xl font-bold text-gray-900 dark:text-white">
                    {stats?.pendingSubmissions || 0}
                  </p>
                </div>
                <Mail className="w-8 h-8 text-red-500" />
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Revenue Model</p>
                  <p className="text-xl font-bold text-gray-900 dark:text-white">
                    FREE Platform
                  </p>
                </div>
                <Heart className="w-8 h-8 text-red-500" />
              </div>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Quick Actions */}
      <div className="mt-8">
        <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">Quick Actions</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          <Link href="/admin/business-approvals">
            <Card className="hover:shadow-md transition-shadow cursor-pointer">
              <CardContent className="p-4 text-center">
                <CheckSquare className="w-8 h-8 mx-auto text-blue-500 mb-2" />
                <h4 className="font-medium text-gray-900 dark:text-white">Review Business Approvals</h4>
                <p className="text-sm text-gray-500">Pending submissions waiting</p>
              </CardContent>
            </Card>
          </Link>

          <Link href="/admin/featured-businesses">
            <Card className="hover:shadow-md transition-shadow cursor-pointer">
              <CardContent className="p-4 text-center">
                <Star className="w-8 h-8 mx-auto text-yellow-500 mb-2" />
                <h4 className="font-medium text-gray-900 dark:text-white">Manage Featured</h4>
                <p className="text-sm text-gray-500">Promote businesses</p>
              </CardContent>
            </Card>
          </Link>

          <div onClick={() => setActiveView("content-mgmt")} className="cursor-pointer">
            <Card className="hover:shadow-md transition-shadow">
              <CardContent className="p-4 text-center">
                <FileText className="w-8 h-8 mx-auto text-green-500 mb-2" />
                <h4 className="font-medium text-gray-900 dark:text-white">Create Content</h4>
                <p className="text-sm text-gray-500">News & updates</p>
              </CardContent>
            </Card>
          </div>

          <div onClick={() => setActiveView("users")} className="cursor-pointer">
            <Card className="hover:shadow-md transition-shadow">
              <CardContent className="p-4 text-center">
                <Users className="w-8 h-8 mx-auto text-purple-500 mb-2" />
                <h4 className="font-medium text-gray-900 dark:text-white">Manage Users</h4>
                <p className="text-sm text-gray-500">Community members</p>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );

  const renderPlaceholderView = (title: string, description: string) => (
    <div className="space-y-6">
      <div>
        <h2 className="text-3xl font-bold tracking-tight text-gray-900 dark:text-white">
          {title}
        </h2>
        <p className="text-gray-600 dark:text-gray-400">
          {description}
        </p>
      </div>
      <Card>
        <CardContent className="p-8 text-center">
          <Settings className="w-16 h-16 mx-auto text-gray-400 mb-4" />
          <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
            Feature Coming Soon
          </h3>
          <p className="text-gray-500">
            This section will be fully implemented with the existing functionality.
          </p>
        </CardContent>
      </Card>
    </div>
  );

  const renderMainContent = () => {
    switch (activeView) {
      case "stats":
        return renderStatsView();
      case "content-mgmt":
        return renderPlaceholderView("Content Management", "Create and manage community content");
      case "users":
        return renderPlaceholderView("User Management", "Manage community members");
      case "newsletter":
        return renderPlaceholderView("Newsletter", "Manage email campaigns and subscribers");
      case "submissions":
        return renderPlaceholderView("Contact Submissions", "Review community inquiries");
      case "general":
        return renderPlaceholderView("General Settings", "Configure system settings");
      default:
        return renderStatsView();
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 dark:bg-gray-900 flex">
      {/* Sidebar */}
      <div className="w-64 bg-white dark:bg-gray-800 border-r border-gray-200 dark:border-gray-700 flex flex-col">
        {/* Header */}
        <div className="p-4 border-b border-gray-200 dark:border-gray-700">
          <Link href="/" className="flex items-center space-x-3">
            <div className="w-8 h-8 bg-gradient-to-br from-red-500 to-pink-500 rounded-full flex items-center justify-center">
              <Heart className="w-4 h-4 text-white" />
            </div>
            <div>
              <h1 className="text-sm font-bold text-gray-900 dark:text-white">LoveMedia Foundation</h1>
              <p className="text-xs text-gray-500 dark:text-gray-400">Admin Dashboard</p>
            </div>
          </Link>
        </div>

        {/* Navigation */}
        <ScrollArea className="flex-1 p-4">
          <nav className="space-y-2">
            {menuCategories.map((category) => (
              <div key={category.id}>
                <button
                  onClick={() => toggleCategory(category.id)}
                  className="w-full flex items-center justify-between p-2 text-sm font-medium text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md transition-colors"
                >
                  <div className="flex items-center space-x-2">
                    {category.icon}
                    <span>{category.title}</span>
                  </div>
                  {expandedCategories.includes(category.id) ? (
                    <ChevronDown className="w-4 h-4" />
                  ) : (
                    <ChevronRight className="w-4 h-4" />
                  )}
                </button>

                {expandedCategories.includes(category.id) && (
                  <div className="ml-4 mt-1 space-y-1">
                    {category.items.map((item) => (
                      item.path ? (
                        <Link key={item.id} href={item.path}>
                          <div className="flex items-center space-x-2 p-2 text-sm text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md transition-colors cursor-pointer">
                            {item.icon}
                            <span>{item.title}</span>
                          </div>
                        </Link>
                      ) : (
                        <button
                          key={item.id}
                          onClick={() => handleMenuClick(item)}
                          className={`w-full flex items-center space-x-2 p-2 text-sm transition-colors rounded-md ${
                            activeView === item.id
                              ? "bg-blue-100 dark:bg-blue-900 text-blue-700 dark:text-blue-300"
                              : "text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-gray-700"
                          }`}
                        >
                          {item.icon}
                          <span>{item.title}</span>
                        </button>
                      )
                    ))}
                  </div>
                )}
              </div>
            ))}
          </nav>
        </ScrollArea>

        {/* User Info & Logout */}
        <div className="p-4 border-t border-gray-200 dark:border-gray-700">
          <div className="flex items-center space-x-3 mb-3">
            <div className="w-8 h-8 bg-gradient-to-br from-blue-500 to-purple-500 rounded-full flex items-center justify-center">
              <Crown className="w-4 h-4 text-white" />
            </div>
            <div className="flex-1 min-w-0">
              <p className="text-sm font-medium text-gray-900 dark:text-white truncate">
                {user?.fullName || user?.username}
              </p>
              <p className="text-xs text-gray-500 dark:text-gray-400">Administrator</p>
            </div>
          </div>
          <Button
            variant="outline"
            size="sm"
            onClick={() => logoutMutation.mutate()}
            disabled={logoutMutation.isPending}
            className="w-full"
          >
            <LogOut className="w-4 h-4 mr-2" />
            Logout
          </Button>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 overflow-auto">
        <div className="p-8">
          {renderMainContent()}
        </div>
      </div>
    </div>
  );
}