import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useAuth } from "@/hooks/use-auth";
import { queryClient } from "@/lib/queryClient";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { 
  Users, 
  FileText, 
  MessageSquare, 
  BarChart3, 
  Plus, 
  Crown,
  Heart,
  LogOut,
  Settings,
  Mail,
  Building,
  DollarSign,
  Edit,
  Search,
  Image,
  Star
} from "lucide-react";
import { AdminMassMail } from "@/components/admin-mass-mail";
import { SystemSettings, InsertSystemSettings } from "@shared/schema";
import { apiRequest } from "@/lib/queryClient";
import { Link } from "wouter";
import { useToast } from "@/hooks/use-toast";

export default function AdminDashboard() {
  const { user, logoutMutation } = useAuth();
  const { toast } = useToast();
  const [activeTab, setActiveTab] = useState("overview");
  const [editingSettings, setEditingSettings] = useState(false);
  const [settingsForm, setSettingsForm] = useState<Partial<SystemSettings>>({});

  // Fetch admin stats
  const { data: stats, isLoading: statsLoading } = useQuery({
    queryKey: ["/api/admin/stats"],
  });

  // Fetch users
  const { data: users, isLoading: usersLoading } = useQuery({
    queryKey: ["/api/admin/users"],
  });

  // Fetch content
  const { data: content, isLoading: contentLoading } = useQuery({
    queryKey: ["/api/content"],
  });

  // Fetch contact submissions
  const { data: submissions, isLoading: submissionsLoading } = useQuery({
    queryKey: ["/api/admin/contact-submissions"],
  });

  // Fetch system settings
  const { data: systemSettings, isLoading: settingsLoading } = useQuery<SystemSettings>({
    queryKey: ["/api/admin/system-settings"],
  });

  // Update system settings mutation
  const updateSettingsMutation = useMutation({
    mutationFn: async (updates: Partial<SystemSettings>) => {
      const response = await apiRequest("PUT", "/api/admin/system-settings", updates);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/system-settings"] });
      toast({
        title: "Settings updated",
        description: "System settings have been updated successfully.",
      });
      setEditingSettings(false);
      setSettingsForm({});
    },
    onError: (error) => {
      toast({
        title: "Error updating settings",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleEditSettings = () => {
    if (systemSettings) {
      setSettingsForm(systemSettings);
      setEditingSettings(true);
    }
  };

  const handleSaveSettings = () => {
    updateSettingsMutation.mutate(settingsForm);
  };

  const handleCancelEdit = () => {
    setEditingSettings(false);
    setSettingsForm({});
  };

  // Create content mutation
  const createContentMutation = useMutation({
    mutationFn: async (contentData: any) => {
      const res = await fetch("/api/content", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify(contentData),
      });
      if (!res.ok) throw new Error("Failed to create content");
      return res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/content"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/stats"] });
      toast({ title: "Content created successfully" });
    },
    onError: () => {
      toast({ title: "Failed to create content", variant: "destructive" });
    },
  });

  // Update content mutation
  const updateContentMutation = useMutation({
    mutationFn: async ({ id, updates }: { id: number; updates: any }) => {
      const res = await fetch(`/api/content/${id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify(updates),
      });
      if (!res.ok) throw new Error("Failed to update content");
      return res.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/content"] });
      toast({ title: "Content updated successfully" });
    },
    onError: () => {
      toast({ title: "Failed to update content", variant: "destructive" });
    },
  });

  const handleCreateContent = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const formData = new FormData(e.currentTarget);
    
    const contentData = {
      title: formData.get("title") as string,
      body: formData.get("body") as string,
      type: formData.get("type") as string,
      status: formData.get("status") as string,
      priority: formData.get("priority") as string,
    };

    createContentMutation.mutate(contentData);
    e.currentTarget.reset();
  };

  const handlePublishContent = (id: number) => {
    updateContentMutation.mutate({
      id,
      updates: { status: "published", publishedAt: new Date().toISOString() },
    });
  };

  return (
    <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
      {/* Header */}
      <header className="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center space-x-4">
              <Link href="/" className="flex items-center space-x-3">
                <div className="w-10 h-10 bg-gradient-to-br from-red-500 to-pink-500 rounded-full flex items-center justify-center">
                  <Heart className="w-5 h-5 text-white" />
                </div>
                <div>
                  <h1 className="text-lg font-bold text-gray-900 dark:text-white">LoveMedia Foundation</h1>
                  <p className="text-xs text-gray-500 dark:text-gray-400">Admin Dashboard</p>
                </div>
              </Link>
            </div>

            <div className="flex items-center space-x-4">
              <Link href="/admin/website-editor">
                <Button variant="outline" size="sm">
                  <Edit className="w-4 h-4 mr-2" />
                  Website Editor
                </Button>
              </Link>
              <Link href="/admin/header-tags">
                <Button variant="outline" size="sm">
                  <Settings className="w-4 h-4 mr-2" />
                  Header Tags
                </Button>
              </Link>
              <Link href="/admin/seo-management">
                <Button variant="outline" size="sm">
                  <Search className="w-4 h-4 mr-2" />
                  SEO Management
                </Button>
              </Link>
              <Link href="/admin/data-deals/email-settings">
                <Button variant="outline" size="sm">
                  <Mail className="w-4 h-4 mr-2" />
                  Data Deals Email
                </Button>
              </Link>
              <Link href="/admin/data-deals/customer-email-settings">
                <Button variant="outline" size="sm">
                  <Heart className="w-4 h-4 mr-2" />
                  Customer Email
                </Button>
              </Link>
              <Link href="/admin/business-images">
                <Button variant="outline" size="sm">
                  <Image className="w-4 h-4 mr-2" />
                  Business Images
                </Button>
              </Link>
              <Link href="/admin/featured-businesses">
                <Button variant="outline" size="sm">
                  <Star className="w-4 h-4 mr-2" />
                  Featured Businesses
                </Button>
              </Link>
              <div className="flex items-center space-x-2">
                <Crown className="w-4 h-4 text-yellow-500" />
                <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                  {user?.fullName}
                </span>
              </div>
              <Button
                variant="outline"
                size="sm"
                onClick={() => logoutMutation.mutate()}
                disabled={logoutMutation.isPending}
              >
                <LogOut className="w-4 h-4 mr-2" />
                Logout
              </Button>
            </div>
          </div>
        </div>
      </header>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
          <TabsList className="grid w-full grid-cols-6">
            <TabsTrigger value="overview" className="flex items-center gap-2">
              <BarChart3 className="w-4 h-4" />
              Overview
            </TabsTrigger>
            <TabsTrigger value="content" className="flex items-center gap-2">
              <FileText className="w-4 h-4" />
              Content
            </TabsTrigger>
            <TabsTrigger value="users" className="flex items-center gap-2">
              <Users className="w-4 h-4" />
              Users
            </TabsTrigger>
            <TabsTrigger value="newsletter" className="flex items-center gap-2">
              <Mail className="w-4 h-4" />
              Newsletter
            </TabsTrigger>
            <TabsTrigger value="submissions" className="flex items-center gap-2">
              <MessageSquare className="w-4 h-4" />
              Submissions
            </TabsTrigger>
            <TabsTrigger value="settings" className="flex items-center gap-2">
              <Settings className="w-4 h-4" />
              Settings
            </TabsTrigger>
          </TabsList>

          {/* Overview Tab */}
          <TabsContent value="overview" className="space-y-6">
            <div>
              <h2 className="text-3xl font-bold tracking-tight text-gray-900 dark:text-white">
                Admin Dashboard
              </h2>
              <p className="text-gray-600 dark:text-gray-400">
                Manage your community, content, and user engagement.
              </p>
            </div>

            {statsLoading ? (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                {[...Array(8)].map((_, i) => (
                  <Card key={i} className="animate-pulse">
                    <CardContent className="p-6">
                      <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded mb-2"></div>
                      <div className="h-8 bg-gray-200 dark:bg-gray-700 rounded"></div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                <Card>
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Total Users</p>
                        <p className="text-3xl font-bold text-gray-900 dark:text-white">
                          {stats?.totalUsers || 0}
                        </p>
                      </div>
                      <Users className="w-8 h-8 text-blue-500" />
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Active Users</p>
                        <p className="text-3xl font-bold text-gray-900 dark:text-white">
                          {stats?.activeUsers || 0}
                        </p>
                      </div>
                      <Users className="w-8 h-8 text-green-500" />
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Businesses</p>
                        <p className="text-3xl font-bold text-gray-900 dark:text-white">
                          {stats?.totalBusinesses || 0}
                        </p>
                      </div>
                      <Building className="w-8 h-8 text-purple-500" />
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Published Content</p>
                        <p className="text-3xl font-bold text-gray-900 dark:text-white">
                          {stats?.publishedContent || 0}
                        </p>
                      </div>
                      <FileText className="w-8 h-8 text-orange-500" />
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">WhatsApp Groups</p>
                        <p className="text-3xl font-bold text-gray-900 dark:text-white">
                          {stats?.whatsappGroups || 0}
                        </p>
                      </div>
                      <MessageSquare className="w-8 h-8 text-green-600" />
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Total Members</p>
                        <p className="text-3xl font-bold text-gray-900 dark:text-white">
                          {stats?.totalMembers || 0}
                        </p>
                      </div>
                      <Users className="w-8 h-8 text-pink-500" />
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Pending Submissions</p>
                        <p className="text-3xl font-bold text-gray-900 dark:text-white">
                          {stats?.pendingSubmissions || 0}
                        </p>
                      </div>
                      <Mail className="w-8 h-8 text-red-500" />
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="text-sm font-medium text-gray-600 dark:text-gray-400">Revenue Model</p>
                        <p className="text-xl font-bold text-gray-900 dark:text-white">
                          Wholesale + 5%
                        </p>
                      </div>
                      <DollarSign className="w-8 h-8 text-yellow-500" />
                    </div>
                  </CardContent>
                </Card>
              </div>
            )}
          </TabsContent>

          {/* Content Management Tab */}
          <TabsContent value="content" className="space-y-6">
            <div className="flex justify-between items-center">
              <div>
                <h2 className="text-2xl font-bold text-gray-900 dark:text-white">Content Management</h2>
                <p className="text-gray-600 dark:text-gray-400">Create and manage community content</p>
              </div>
            </div>

            <div className="grid lg:grid-cols-2 gap-6">
              {/* Create Content Form */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Plus className="w-5 h-5" />
                    Create New Content
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <form onSubmit={handleCreateContent} className="space-y-4">
                    <div>
                      <Label htmlFor="title">Title</Label>
                      <Input
                        id="title"
                        name="title"
                        required
                        placeholder="Content title"
                        disabled={createContentMutation.isPending}
                      />
                    </div>

                    <div>
                      <Label htmlFor="type">Type</Label>
                      <Select name="type" required>
                        <SelectTrigger>
                          <SelectValue placeholder="Select content type" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="news">News</SelectItem>
                          <SelectItem value="alert">Alert</SelectItem>
                          <SelectItem value="update">Update</SelectItem>
                          <SelectItem value="event">Event</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="priority">Priority</Label>
                      <Select name="priority" defaultValue="normal">
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="low">Low</SelectItem>
                          <SelectItem value="normal">Normal</SelectItem>
                          <SelectItem value="high">High</SelectItem>
                          <SelectItem value="urgent">Urgent</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="status">Status</Label>
                      <Select name="status" defaultValue="draft">
                        <SelectTrigger>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="draft">Draft</SelectItem>
                          <SelectItem value="published">Published</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>

                    <div>
                      <Label htmlFor="body">Content</Label>
                      <Textarea
                        id="body"
                        name="body"
                        required
                        placeholder="Content body"
                        rows={6}
                        disabled={createContentMutation.isPending}
                      />
                    </div>

                    <Button
                      type="submit"
                      className="w-full"
                      disabled={createContentMutation.isPending}
                    >
                      {createContentMutation.isPending ? "Creating..." : "Create Content"}
                    </Button>
                  </form>
                </CardContent>
              </Card>

              {/* Content List */}
              <Card>
                <CardHeader>
                  <CardTitle>Recent Content</CardTitle>
                </CardHeader>
                <CardContent>
                  {contentLoading ? (
                    <div className="space-y-4">
                      {[...Array(5)].map((_, i) => (
                        <div key={i} className="animate-pulse">
                          <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded mb-2"></div>
                          <div className="h-3 bg-gray-200 dark:bg-gray-700 rounded w-3/4"></div>
                        </div>
                      ))}
                    </div>
                  ) : (
                    <div className="space-y-4 max-h-96 overflow-y-auto">
                      {content?.slice(0, 10).map((item: any) => (
                        <div key={item.id} className="p-4 border border-gray-200 dark:border-gray-700 rounded-lg">
                          <div className="flex justify-between items-start mb-2">
                            <h4 className="font-medium text-gray-900 dark:text-white">{item.title}</h4>
                            <div className="flex gap-2">
                              <Badge variant={item.status === 'published' ? 'default' : 'secondary'}>
                                {item.status}
                              </Badge>
                              <Badge variant={item.priority === 'urgent' ? 'destructive' : 'outline'}>
                                {item.priority}
                              </Badge>
                            </div>
                          </div>
                          <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">
                            {item.body.substring(0, 100)}...
                          </p>
                          <div className="flex justify-between items-center">
                            <span className="text-xs text-gray-500">
                              {new Date(item.createdAt).toLocaleDateString()}
                            </span>
                            {item.status === 'draft' && (
                              <Button
                                size="sm"
                                variant="outline"
                                onClick={() => handlePublishContent(item.id)}
                                disabled={updateContentMutation.isPending}
                              >
                                Publish
                              </Button>
                            )}
                          </div>
                        </div>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            </div>
          </TabsContent>

          {/* Users Tab */}
          <TabsContent value="users" className="space-y-6">
            <div>
              <h2 className="text-2xl font-bold text-gray-900 dark:text-white">User Management</h2>
              <p className="text-gray-600 dark:text-gray-400">Manage community members</p>
            </div>

            <Card>
              <CardContent className="p-6">
                {usersLoading ? (
                  <div className="space-y-4">
                    {[...Array(10)].map((_, i) => (
                      <div key={i} className="flex items-center space-x-4 animate-pulse">
                        <div className="w-10 h-10 bg-gray-200 dark:bg-gray-700 rounded-full"></div>
                        <div className="flex-1">
                          <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded mb-2"></div>
                          <div className="h-3 bg-gray-200 dark:bg-gray-700 rounded w-1/2"></div>
                        </div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="space-y-4">
                    {users?.map((user: any) => (
                      <div key={user.id} className="flex items-center justify-between p-4 border border-gray-200 dark:border-gray-700 rounded-lg">
                        <div className="flex items-center space-x-4">
                          <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-purple-500 rounded-full flex items-center justify-center">
                            <span className="text-white font-medium">
                              {user.fullName?.charAt(0) || user.username.charAt(0)}
                            </span>
                          </div>
                          <div>
                            <p className="font-medium text-gray-900 dark:text-white">{user.fullName}</p>
                            <p className="text-sm text-gray-600 dark:text-gray-400">{user.email}</p>
                            <p className="text-xs text-gray-500">{user.location}</p>
                          </div>
                        </div>
                        <div className="flex items-center space-x-2">
                          <Badge variant={user.role === 'admin' ? 'default' : 'secondary'}>
                            {user.role}
                          </Badge>
                          <Badge variant={user.isActive ? 'default' : 'destructive'}>
                            {user.isActive ? 'Active' : 'Inactive'}
                          </Badge>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Contact Submissions Tab */}
          <TabsContent value="submissions" className="space-y-6">
            <div>
              <h2 className="text-2xl font-bold text-gray-900 dark:text-white">Contact Submissions</h2>
              <p className="text-gray-600 dark:text-gray-400">Manage community inquiries and feedback</p>
            </div>

            <Card>
              <CardContent className="p-6">
                {submissionsLoading ? (
                  <div className="space-y-4">
                    {[...Array(5)].map((_, i) => (
                      <div key={i} className="animate-pulse p-4 border border-gray-200 dark:border-gray-700 rounded-lg">
                        <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded mb-2"></div>
                        <div className="h-3 bg-gray-200 dark:bg-gray-700 rounded w-3/4 mb-2"></div>
                        <div className="h-16 bg-gray-200 dark:bg-gray-700 rounded"></div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <div className="space-y-4">
                    {submissions?.map((submission: any) => (
                      <div key={submission.id} className="p-4 border border-gray-200 dark:border-gray-700 rounded-lg">
                        <div className="flex justify-between items-start mb-2">
                          <div>
                            <h4 className="font-medium text-gray-900 dark:text-white">{submission.name}</h4>
                            <p className="text-sm text-gray-600 dark:text-gray-400">{submission.email}</p>
                          </div>
                          <Badge variant={submission.status === 'pending' ? 'destructive' : 'default'}>
                            {submission.status}
                          </Badge>
                        </div>
                        <p className="font-medium text-gray-900 dark:text-white mb-2">{submission.subject}</p>
                        <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">{submission.message}</p>
                        <p className="text-xs text-gray-500">
                          {new Date(submission.createdAt).toLocaleDateString()}
                        </p>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          {/* Newsletter Tab */}
          <TabsContent value="newsletter" className="space-y-6">
            <AdminMassMail />
          </TabsContent>

          {/* Settings Tab */}
          <TabsContent value="settings" className="space-y-6">
            <div>
              <h2 className="text-2xl font-bold text-gray-900 dark:text-white">Settings</h2>
              <p className="text-gray-600 dark:text-gray-400">Configure system settings and preferences</p>
            </div>

            <Card>
              <CardHeader>
                <div className="flex justify-between items-start">
                  <div>
                    <CardTitle>System Configuration</CardTitle>
                    <CardDescription>
                      Basic system settings and configuration options
                    </CardDescription>
                  </div>
                  {!editingSettings && (
                    <Button variant="outline" size="sm" onClick={handleEditSettings}>
                      <Edit className="w-4 h-4 mr-2" />
                      Edit
                    </Button>
                  )}
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                {settingsLoading ? (
                  <div className="text-gray-600 dark:text-gray-400">Loading settings...</div>
                ) : editingSettings ? (
                  <div className="space-y-4">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="organization">Organization</Label>
                        <Input
                          id="organization"
                          value={settingsForm.organization || ''}
                          onChange={(e) => setSettingsForm(prev => ({ ...prev, organization: e.target.value }))}
                          placeholder="Organization name"
                        />
                      </div>
                      <div>
                        <Label htmlFor="location">Location</Label>
                        <Input
                          id="location"
                          value={settingsForm.location || ''}
                          onChange={(e) => setSettingsForm(prev => ({ ...prev, location: e.target.value }))}
                          placeholder="Location"
                        />
                      </div>
                      <div>
                        <Label htmlFor="adminFee">Admin Fee</Label>
                        <Input
                          id="adminFee"
                          value={settingsForm.adminFee || ''}
                          onChange={(e) => setSettingsForm(prev => ({ ...prev, adminFee: e.target.value }))}
                          placeholder="Admin fee structure"
                        />
                      </div>
                      <div>
                        <Label htmlFor="businessModel">Business Model</Label>
                        <Input
                          id="businessModel"
                          value={settingsForm.businessModel || ''}
                          onChange={(e) => setSettingsForm(prev => ({ ...prev, businessModel: e.target.value }))}
                          placeholder="Business model"
                        />
                      </div>
                      <div>
                        <Label htmlFor="contactEmail">Contact Email</Label>
                        <Input
                          id="contactEmail"
                          type="email"
                          value={settingsForm.contactEmail || ''}
                          onChange={(e) => setSettingsForm(prev => ({ ...prev, contactEmail: e.target.value }))}
                          placeholder="Contact email"
                        />
                      </div>
                      <div>
                        <Label htmlFor="contactPhone">Contact Phone</Label>
                        <Input
                          id="contactPhone"
                          value={settingsForm.contactPhone || ''}
                          onChange={(e) => setSettingsForm(prev => ({ ...prev, contactPhone: e.target.value }))}
                          placeholder="Contact phone number"
                        />
                      </div>
                      <div className="md:col-span-2">
                        <Label htmlFor="website">Website</Label>
                        <Input
                          id="website"
                          value={settingsForm.website || ''}
                          onChange={(e) => setSettingsForm(prev => ({ ...prev, website: e.target.value }))}
                          placeholder="Website URL"
                        />
                      </div>
                    </div>
                    <div className="flex gap-2">
                      <Button onClick={handleSaveSettings} disabled={updateSettingsMutation.isPending}>
                        {updateSettingsMutation.isPending ? "Saving..." : "Save Changes"}
                      </Button>
                      <Button variant="outline" onClick={handleCancelEdit}>
                        Cancel
                      </Button>
                    </div>
                  </div>
                ) : (
                  <div className="text-sm text-gray-600 dark:text-gray-400">
                    <p><strong>Organization:</strong> {systemSettings?.organization || "LoveMedia Foundation NPC"}</p>
                    <p><strong>Location:</strong> {systemSettings?.location || "Ballito, North Coast, KwaZulu-Natal"}</p>
                    <p><strong>Admin Fee:</strong> {systemSettings?.adminFee || "5% on wholesale prices"}</p>
                    <p><strong>Business Model:</strong> {systemSettings?.businessModel || "Non-profit with transparent pricing"}</p>
                    {(systemSettings?.contactEmail || "info@lovemedia.foundation") && (
                      <p><strong>Contact Email:</strong> {systemSettings?.contactEmail || "info@lovemedia.foundation"}</p>
                    )}
                    {systemSettings?.contactPhone && (
                      <p><strong>Contact Phone:</strong> {systemSettings.contactPhone}</p>
                    )}
                    {(systemSettings?.website || "https://lovemedia.foundation") && (
                      <p><strong>Website:</strong> {systemSettings?.website || "https://lovemedia.foundation"}</p>
                    )}
                  </div>
                )}
                
                <Separator />
                <div className="space-y-2">
                  <h4 className="font-medium">Quick Actions</h4>
                  <div className="flex gap-2">
                    <Button variant="outline" size="sm">Export Users</Button>
                    <Button variant="outline" size="sm">Backup Data</Button>
                    <Button variant="outline" size="sm">System Reports</Button>
                  </div>
                </div>
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}
