import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Plus, Edit2, Trash2, Save, X } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { HeaderTag, InsertHeaderTag } from "@shared/schema";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { AdminLayout } from "@/components/admin-layout";

export default function AdminHeaderTags() {
  const { toast } = useToast();
  const [isAddingTag, setIsAddingTag] = useState(false);
  const [editingTag, setEditingTag] = useState<HeaderTag | null>(null);
  const [newTag, setNewTag] = useState<Partial<InsertHeaderTag>>({
    name: "",
    type: "script",
    code: "",
    priority: 1,
    isActive: true,
  });

  const { data: headerTags, isLoading } = useQuery<HeaderTag[]>({
    queryKey: ["/api/admin/header-tags"],
    queryFn: async () => {
      const response = await fetch("/api/admin/header-tags");
      if (!response.ok) throw new Error("Failed to fetch header tags");
      return response.json();
    },
  });

  const createHeaderTagMutation = useMutation({
    mutationFn: async (tagData: InsertHeaderTag) => {
      const response = await apiRequest("POST", "/api/admin/header-tags", tagData);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/header-tags"] });
      toast({
        title: "Header tag created",
        description: "The header tag has been created successfully.",
      });
      setIsAddingTag(false);
      setNewTag({
        name: "",
        type: "script",
        code: "",
        priority: 1,
        isActive: true,
      });
    },
    onError: (error) => {
      toast({
        title: "Error creating header tag",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const updateHeaderTagMutation = useMutation({
    mutationFn: async ({ id, ...updates }: Partial<HeaderTag> & { id: number }) => {
      const response = await apiRequest("PUT", `/api/admin/header-tags/${id}`, updates);
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/header-tags"] });
      toast({
        title: "Header tag updated",
        description: "The header tag has been updated successfully.",
      });
      setEditingTag(null);
    },
    onError: (error) => {
      toast({
        title: "Error updating header tag",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const deleteHeaderTagMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/admin/header-tags/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/header-tags"] });
      toast({
        title: "Header tag deleted",
        description: "The header tag has been deleted successfully.",
      });
    },
    onError: (error) => {
      toast({
        title: "Error deleting header tag",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleCreateTag = () => {
    if (!newTag.name || !newTag.code) {
      toast({
        title: "Missing required fields",
        description: "Please fill in all required fields.",
        variant: "destructive",
      });
      return;
    }
    createHeaderTagMutation.mutate(newTag as InsertHeaderTag);
  };

  const handleUpdateTag = () => {
    if (!editingTag) return;
    // Exclude timestamp fields - they should be handled by the database
    const { id, createdAt, updatedAt, createdBy, ...updates } = editingTag;
    updateHeaderTagMutation.mutate({ id, ...updates });
  };

  const handleDeleteTag = (id: number) => {
    if (confirm("Are you sure you want to delete this header tag?")) {
      deleteHeaderTagMutation.mutate(id);
    }
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-pink-600"></div>
      </div>
    );
  }

  return (
    <AdminLayout title="Header Tags Management" description="Manage tracking codes and scripts that appear in the website header">
      <div className="space-y-6">
        <div className="flex justify-between items-center">
          <Button
            onClick={() => setIsAddingTag(true)}
            className="bg-pink-600 hover:bg-pink-700"
          >
            <Plus className="w-4 h-4 mr-2" />
            Add Header Tag
          </Button>
        </div>

      {/* Add New Header Tag Form */}
      {isAddingTag && (
        <Card className="mb-6">
          <CardHeader>
            <CardTitle>Add New Header Tag</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label htmlFor="name">Name</Label>
                <Input
                  id="name"
                  value={newTag.name}
                  onChange={(e) => setNewTag({ ...newTag, name: e.target.value })}
                  placeholder="e.g., Google Tag Manager"
                />
              </div>
              <div>
                <Label htmlFor="type">Type</Label>
                <select
                  id="type"
                  value={newTag.type}
                  onChange={(e) => setNewTag({ ...newTag, type: e.target.value })}
                  className="w-full p-2 border rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100"
                >
                  <option value="script">Script (JavaScript, tracking codes)</option>
                  <option value="noscript">NoScript (fallback content)</option>
                  <option value="meta">Meta (SEO, social media tags)</option>
                  <option value="link">Link (stylesheets, icons, prefetch)</option>
                  <option value="style">Style (inline CSS)</option>
                </select>
              </div>
            </div>
            <div>
              <Label htmlFor="code">Code</Label>
              <Textarea
                id="code"
                value={newTag.code}
                onChange={(e) => setNewTag({ ...newTag, code: e.target.value })}
                placeholder="Paste your tracking code here..."
                rows={8}
              />
            </div>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label htmlFor="priority">Priority</Label>
                <Input
                  id="priority"
                  type="number"
                  value={newTag.priority}
                  onChange={(e) => setNewTag({ ...newTag, priority: parseInt(e.target.value) || 1 })}
                  min="1"
                  max="100"
                />
              </div>
              <div className="flex items-center space-x-2">
                <Switch
                  id="isActive"
                  checked={newTag.isActive}
                  onCheckedChange={(checked) => setNewTag({ ...newTag, isActive: checked })}
                />
                <Label htmlFor="isActive">Active</Label>
              </div>
            </div>
            <div className="flex gap-2">
              <Button onClick={handleCreateTag} className="bg-green-600 hover:bg-green-700">
                <Save className="w-4 h-4 mr-2" />
                Save
              </Button>
              <Button
                onClick={() => setIsAddingTag(false)}
                variant="outline"
              >
                <X className="w-4 h-4 mr-2" />
                Cancel
              </Button>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Header Tags List */}
      <div className="space-y-4">
        {headerTags?.map((tag) => (
          <Card key={tag.id}>
            <CardContent className="p-6">
              {editingTag?.id === tag.id ? (
                <div className="space-y-4">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="edit-name">Name</Label>
                      <Input
                        id="edit-name"
                        value={editingTag.name}
                        onChange={(e) => setEditingTag({ ...editingTag, name: e.target.value })}
                      />
                    </div>
                    <div>
                      <Label htmlFor="edit-type">Type</Label>
                      <select
                        id="edit-type"
                        value={editingTag.type}
                        onChange={(e) => setEditingTag({ ...editingTag, type: e.target.value })}
                        className="w-full p-2 border rounded-md bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100"
                      >
                        <option value="script">Script (JavaScript, tracking codes)</option>
                        <option value="noscript">NoScript (fallback content)</option>
                        <option value="meta">Meta (SEO, social media tags)</option>
                        <option value="link">Link (stylesheets, icons, prefetch)</option>
                        <option value="style">Style (inline CSS)</option>
                      </select>
                    </div>
                  </div>
                  <div>
                    <Label htmlFor="edit-code">Code</Label>
                    <Textarea
                      id="edit-code"
                      value={editingTag.code}
                      onChange={(e) => setEditingTag({ ...editingTag, code: e.target.value })}
                      rows={8}
                    />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <Label htmlFor="edit-priority">Priority</Label>
                      <Input
                        id="edit-priority"
                        type="number"
                        value={editingTag.priority}
                        onChange={(e) => setEditingTag({ ...editingTag, priority: parseInt(e.target.value) || 1 })}
                        min="1"
                        max="100"
                      />
                    </div>
                    <div className="flex items-center space-x-2">
                      <Switch
                        id="edit-isActive"
                        checked={editingTag.isActive}
                        onCheckedChange={(checked) => setEditingTag({ ...editingTag, isActive: checked })}
                      />
                      <Label htmlFor="edit-isActive">Active</Label>
                    </div>
                  </div>
                  <div className="flex gap-2">
                    <Button onClick={handleUpdateTag} className="bg-green-600 hover:bg-green-700">
                      <Save className="w-4 h-4 mr-2" />
                      Save
                    </Button>
                    <Button
                      onClick={() => setEditingTag(null)}
                      variant="outline"
                    >
                      <X className="w-4 h-4 mr-2" />
                      Cancel
                    </Button>
                  </div>
                </div>
              ) : (
                <div>
                  <div className="flex justify-between items-start">
                    <div>
                      <h3 className="text-lg font-semibold">{tag.name}</h3>
                      <p className="text-sm text-gray-600 mb-2">
                        Type: {tag.type} | Priority: {tag.priority} | 
                        Status: {tag.isActive ? "Active" : "Inactive"}
                      </p>
                      <pre className="bg-gray-100 dark:bg-gray-800 p-2 rounded text-xs overflow-x-auto text-gray-800 dark:text-gray-200">
                        {tag.code.substring(0, 200)}
                        {tag.code.length > 200 && "..."}
                      </pre>
                    </div>
                    <div className="flex gap-2">
                      <Button
                        onClick={() => setEditingTag(tag)}
                        variant="outline"
                        size="sm"
                      >
                        <Edit2 className="w-4 h-4" />
                      </Button>
                      <Button
                        onClick={() => handleDeleteTag(tag.id)}
                        variant="outline"
                        size="sm"
                        className="text-red-600 hover:text-red-700"
                      >
                        <Trash2 className="w-4 h-4" />
                      </Button>
                    </div>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>
        ))}
      </div>

        {headerTags?.length === 0 && (
          <Card>
            <CardContent className="text-center py-8">
              <p className="text-gray-500">No header tags found. Add your first tracking code to get started.</p>
            </CardContent>
          </Card>
        )}
      </div>
    </AdminLayout>
  );
}