import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Textarea } from "@/components/ui/textarea";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Search, MapPin, Phone, Mail, Globe, Star, Crown, Sparkles, TrendingUp, Plus, ChevronLeft, ChevronRight } from "lucide-react";
import { SEOHead } from "@/components/SEOHead";
import Navigation from "@/components/navigation";
import Footer from "@/components/footer";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

const businessListingSchema = z.object({
  name: z.string().min(2, "Business name must be at least 2 characters"),
  description: z.string().min(10, "Description must be at least 10 characters"),
  category: z.string().min(1, "Please select a category"),
  location: z.string().min(5, "Please provide a detailed location"),
  contactEmail: z.string().email("Please enter a valid email address").optional().or(z.literal("")),
  contactPhone: z.string().min(10, "Please enter a valid phone number").optional().or(z.literal("")),
  website: z.string().url("Please enter a valid website URL").optional().or(z.literal("")),
  whatsappNumber: z.string().optional(),
});

type BusinessListingForm = z.infer<typeof businessListingSchema>;

// Business categories for better organization
const businessCategories = [
  "Agriculture", "Car & Automotive", "Education Training Lessons", "Engineering", 
  "Events & entertainment", "Financial & Insurance Services", "Home Building & Trade", 
  "Hospitality", "IT Computer & Technology", "Marketing Business, and Legal", 
  "Medical Wellness & Beauty", "Public Service & Government", "Security Systems & Protection", 
  "Telecommunication", "Trade & Industry", "Transport"
];

// South African provinces for location selection
const southAfricanProvinces = [
  "Eastern Cape", "Free State", "Gauteng", "KwaZulu-Natal", "Limpopo", 
  "Mpumalanga", "Northern Cape", "North West", "Western Cape"
];

export default function BusinessDirectory() {
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedCategory, setSelectedCategory] = useState<string>("all");
  const [selectedLocation, setSelectedLocation] = useState<string>("all");
  const [currentPage, setCurrentPage] = useState(1);
  const [showListingDialog, setShowListingDialog] = useState(false);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { toast } = useToast();

  const itemsPerPage = 10;

  const form = useForm<BusinessListingForm>({
    resolver: zodResolver(businessListingSchema),
    defaultValues: {
      name: "",
      description: "",
      category: "",
      location: "",
      contactEmail: "",
      contactPhone: "",
      website: "",
      whatsappNumber: "",
    },
  });

  const { data: businessesData, isLoading: businessesLoading } = useQuery({
    queryKey: ['/api/businesses', searchTerm, selectedCategory, selectedLocation, currentPage],
    queryFn: () => {
      const params = new URLSearchParams({
        search: searchTerm,
        category: selectedCategory,
        location: selectedLocation,
        page: currentPage.toString(),
        limit: itemsPerPage.toString()
      });
      return fetch(`/api/businesses?${params}`, { credentials: 'include' })
        .then(res => res.json());
    },
  });

  const businesses = businessesData?.businesses || [];
  const totalCount = businessesData?.total || 0;
  const totalPages = Math.ceil(totalCount / itemsPerPage);

  // Reset to page 1 when filters change
  const resetPagination = () => {
    setCurrentPage(1);
  };

  // Update filter handlers to reset pagination
  const handleCategoryChange = (value: string) => {
    setSelectedCategory(value);
    resetPagination();
  };

  const handleLocationChange = (value: string) => {
    setSelectedLocation(value);
    resetPagination();
  };

  const handleSearchChange = (value: string) => {
    setSearchTerm(value);
    resetPagination();
  };

  // Use predefined categories and locations for the form
  const categories = businessCategories;
  const locations = southAfricanProvinces;

  const handleSubmitBusiness = async (data: BusinessListingForm) => {
    try {
      setIsSubmitting(true);
      
      await apiRequest("/api/businesses/submit", "POST", data);

      toast({
        title: "Business submitted successfully!",
        description: "Your business listing has been submitted for admin review. You'll be notified once it's approved.",
      });

      form.reset();
      setShowListingDialog(false);
    } catch (error) {
      toast({
        title: "Submission failed",
        description: "There was an error submitting your business listing. Please try again.",
        variant: "destructive",
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  const filteredBusinesses = businesses.filter((business: any) => {
    const matchesSearch = !searchTerm || 
      business.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      business.description?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      business.location.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesCategory = selectedCategory === "all" || business.category === selectedCategory;
    const matchesLocation = selectedLocation === "all" || business.location.toLowerCase().includes(selectedLocation.toLowerCase());
    
    return matchesSearch && matchesCategory && matchesLocation;
  });

  // Sort businesses: featured first, then top placement, then regular
  const sortedBusinesses = filteredBusinesses.sort((a: any, b: any) => {
    if (a.topPlacement && !b.topPlacement) return -1;
    if (!a.topPlacement && b.topPlacement) return 1;
    if (a.isFeatured && !b.isFeatured) return -1;
    if (!a.isFeatured && b.isFeatured) return 1;
    return 0;
  });

  return (
    <>
      <SEOHead 
        title="Business Directory - LoveMedia Foundation"
        description="Discover thousands of businesses across South Africa. Find local services, compare options, and support your community through our comprehensive business directory."
        keywords="business directory, South Africa businesses, local services, companies, professional services"
      />
      
      <Navigation />
      
      <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-purple-50 pt-16">
        {/* Hero Section */}
        <div className="bg-gradient-to-r from-blue-600 to-purple-600 text-white py-16">
          <div className="container mx-auto px-4">
            <div className="text-center max-w-4xl mx-auto">
              <h1 className="text-4xl md:text-6xl font-bold mb-6">
                South Africa's Premier Business Directory
              </h1>
              <p className="text-xl md:text-2xl mb-8 opacity-90">
                Discover over 15,000 verified businesses across 16 categories. 
                Find the services you need and support local enterprises.
              </p>
              
              <Dialog open={showListingDialog} onOpenChange={setShowListingDialog}>
                <DialogTrigger asChild>
                  <Button size="lg" className="bg-white text-blue-600 hover:bg-gray-100 text-lg px-8 py-3">
                    <Plus className="mr-2 h-5 w-5" />
                    List Your Business - FREE
                  </Button>
                </DialogTrigger>
                <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
                  <DialogHeader>
                    <DialogTitle>List Your Business - FREE</DialogTitle>
                    <DialogDescription>
                      Submit your business for free! All listings are reviewed by our admin team to ensure quality. Your business will appear in our directory once approved.
                    </DialogDescription>
                  </DialogHeader>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                    {paymentOptions.map((option) => (
                      <Card 
                        key={option.id}
                        className={`cursor-pointer transition-all ${
                          selectedPaymentOption === option.id 
                            ? 'ring-2 ring-blue-500 bg-blue-50 dark:bg-blue-900/20 border-blue-500 dark:border-blue-400' 
                            : 'hover:shadow-md hover:border-gray-300'
                        }`}
                        onClick={() => setSelectedPaymentOption(option.id)}
                      >
                        <CardHeader>
                          <div className="flex justify-between items-center">
                            <CardTitle className={`flex items-center gap-2 ${selectedPaymentOption === option.id ? 'dark:text-white' : ''}`}>
                              {option.id === 'featured' && <Crown className="h-5 w-5 text-yellow-500" />}
                              {option.id === 'highlight' && <Sparkles className="h-5 w-5 text-purple-500" />}
                              {option.id === 'top_placement' && <TrendingUp className="h-5 w-5 text-green-500" />}
                              {option.name}
                            </CardTitle>
                            <Badge variant="secondary">${option.price}</Badge>
                          </div>
                          <CardDescription className={selectedPaymentOption === option.id ? 'dark:text-blue-100' : ''}>
                            {option.description}
                          </CardDescription>
                        </CardHeader>
                        <CardContent>
                          <ul className="space-y-2 text-sm">
                            {option.features.map((feature, index) => (
                              <li key={index} className={`flex items-center gap-2 ${selectedPaymentOption === option.id ? 'dark:text-blue-50' : ''}`}>
                                <div className="w-1.5 h-1.5 bg-blue-500 rounded-full" />
                                {feature}
                              </li>
                            ))}
                          </ul>
                          <p className={`text-xs mt-2 ${selectedPaymentOption === option.id ? 'text-gray-600 dark:text-blue-200' : 'text-gray-500'}`}>
                            {option.duration}
                          </p>
                        </CardContent>
                      </Card>
                    ))}
                  </div>

                  <Form {...form}>
                    <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <FormField
                          control={form.control}
                          name="name"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Business Name *</FormLabel>
                              <FormControl>
                                <Input placeholder="Your Business Name" {...field} data-testid="input-business-name" />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                        
                        <FormField
                          control={form.control}
                          name="category"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Category *</FormLabel>
                              <Select onValueChange={field.onChange} defaultValue={field.value}>
                                <FormControl>
                                  <SelectTrigger data-testid="select-category">
                                    <SelectValue placeholder="Select a category" />
                                  </SelectTrigger>
                                </FormControl>
                                <SelectContent>
                                  {categories.map((category: any) => (
                                    <SelectItem key={category.id} value={category.name}>
                                      {category.name}
                                    </SelectItem>
                                  ))}
                                </SelectContent>
                              </Select>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <FormField
                        control={form.control}
                        name="description"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Business Description *</FormLabel>
                            <FormControl>
                              <Textarea 
                                placeholder="Describe your business, services, and what makes you unique..."
                                className="min-h-[100px]"
                                {...field}
                                data-testid="textarea-description"
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <FormField
                        control={form.control}
                        name="location"
                        render={({ field }) => (
                          <FormItem>
                            <FormLabel>Full Address *</FormLabel>
                            <FormControl>
                              <Input 
                                placeholder="Street address, City, Province, Postal Code"
                                {...field}
                                data-testid="input-location"
                              />
                            </FormControl>
                            <FormMessage />
                          </FormItem>
                        )}
                      />

                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <FormField
                          control={form.control}
                          name="contactEmail"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Email Address</FormLabel>
                              <FormControl>
                                <Input 
                                  type="email"
                                  placeholder="business@example.com"
                                  {...field}
                                  data-testid="input-email"
                                />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />

                        <FormField
                          control={form.control}
                          name="contactPhone"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Phone Number</FormLabel>
                              <FormControl>
                                <Input 
                                  placeholder="+27 XX XXX XXXX"
                                  {...field}
                                  data-testid="input-phone"
                                />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <FormField
                          control={form.control}
                          name="website"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>Website URL</FormLabel>
                              <FormControl>
                                <Input 
                                  placeholder="https://www.yourwebsite.com"
                                  {...field}
                                  data-testid="input-website"
                                />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />

                        <FormField
                          control={form.control}
                          name="whatsappNumber"
                          render={({ field }) => (
                            <FormItem>
                              <FormLabel>WhatsApp Number</FormLabel>
                              <FormControl>
                                <Input 
                                  placeholder="+27 XX XXX XXXX"
                                  {...field}
                                  data-testid="input-whatsapp"
                                />
                              </FormControl>
                              <FormMessage />
                            </FormItem>
                          )}
                        />
                      </div>

                      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                        <h4 className="font-semibold text-yellow-800 mb-2">💝 Supporting Our Community</h4>
                        <p className="text-yellow-700 text-sm">
                          All business listing fees are processed as donations to LoveMedia Foundation NPC, 
                          supporting South African communities while promoting your business.
                        </p>
                      </div>

                      <div className="flex gap-4 pt-4">
                        <Button 
                          type="button" 
                          variant="outline" 
                          onClick={() => setShowListingDialog(false)}
                          data-testid="button-cancel"
                        >
                          Cancel
                        </Button>
                        <Button 
                          type="submit"
                          className="flex-1"
                          data-testid="button-submit-listing"
                        >
                          Proceed to Payment - ${paymentOptions.find(opt => opt.id === selectedPaymentOption)?.price}
                        </Button>
                      </div>
                    </form>
                  </Form>
                </DialogContent>
              </Dialog>
            </div>
          </div>
        </div>

        {/* Search and Filters */}
        <div className="container mx-auto px-4 py-8">
          <div className="bg-white rounded-xl shadow-lg p-6 mb-8">
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div className="md:col-span-2">
                <div className="relative">
                  <Search className="absolute left-3 top-3 h-5 w-5 text-gray-400" />
                  <Input
                    placeholder="Search businesses, services, or locations..."
                    value={searchTerm}
                    onChange={(e) => handleSearchChange(e.target.value)}
                    className="pl-10"
                    data-testid="input-search"
                  />
                </div>
              </div>
              
              <Select value={selectedCategory} onValueChange={handleCategoryChange}>
                <SelectTrigger data-testid="select-category-filter">
                  <SelectValue placeholder="All Categories" />
                </SelectTrigger>
                <SelectContent className="bg-white border border-gray-200 shadow-lg">
                  <SelectItem value="all" className="text-gray-900 hover:bg-gray-100">All Categories</SelectItem>
                  {categories.map((category: any, index: number) => (
                    <SelectItem key={category.id || `category-${index}`} value={category.name} className="text-gray-900 hover:bg-gray-100">
                      {category.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>

              <Select value={selectedLocation} onValueChange={handleLocationChange}>
                <SelectTrigger data-testid="select-location-filter">
                  <SelectValue placeholder="All Locations" />
                </SelectTrigger>
                <SelectContent className="bg-white border border-gray-200 shadow-lg">
                  <SelectItem value="all" className="text-gray-900 hover:bg-gray-100">All Locations</SelectItem>
                  {locations.map((location: string, index: number) => (
                    <SelectItem key={`location-${index}`} value={location} className="text-gray-900 hover:bg-gray-100">
                      {location}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>

          {/* Results */}
          <div className="flex justify-between items-center mb-6">
            <h2 className="text-2xl font-bold">
              {businessesLoading ? 'Loading...' : 
               searchTerm || selectedCategory !== 'all' || selectedLocation !== 'all' 
                 ? `${totalCount} Businesses Found` 
                 : 'Featured Businesses'}
            </h2>
            <div className="text-sm text-gray-600">
              Showing results for: {selectedCategory !== 'all' ? selectedCategory : 'All Categories'}
              {selectedLocation !== 'all' && ` in ${selectedLocation}`}
              {totalPages > 1 && ` (Page ${currentPage} of ${totalPages})`}
            </div>
          </div>

          {businessesLoading ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {[...Array(9)].map((_, i) => (
                <Card key={i} className="animate-pulse">
                  <CardHeader>
                    <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                    <div className="h-3 bg-gray-200 rounded w-1/2"></div>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-2">
                      <div className="h-3 bg-gray-200 rounded"></div>
                      <div className="h-3 bg-gray-200 rounded w-5/6"></div>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
              {sortedBusinesses.map((business: any) => (
                <Card 
                  key={business.id} 
                  className={`hover:shadow-lg transition-all duration-200 ${
                    business.highlightColor ? `border-l-4 border-l-${business.highlightColor}-500` : ''
                  } ${business.topPlacement ? 'ring-2 ring-green-200 bg-green-50' : ''}`}
                  data-testid={`card-business-${business.id}`}
                >
                  <CardHeader>
                    <div className="flex justify-between items-start">
                      <div className="flex-1">
                        <CardTitle className="flex items-center gap-2">
                          {business.name}
                          {business.topPlacement && <TrendingUp className="h-4 w-4 text-green-500" />}
                          {business.isFeatured && <Crown className="h-4 w-4 text-yellow-500" />}
                          {business.isVerified && <Badge variant="secondary" className="text-xs">Verified</Badge>}
                        </CardTitle>
                        <CardDescription className="flex items-center gap-1">
                          <MapPin className="h-3 w-3" />
                          {business.location}
                        </CardDescription>
                      </div>
                      <Badge variant="outline">{business.category}</Badge>
                    </div>
                  </CardHeader>
                  
                  <CardContent>
                    <p className="text-sm text-gray-600 mb-4 line-clamp-3">
                      {business.description}
                    </p>
                    
                    {business.slogan && (
                      <p className="text-sm font-medium text-blue-600 italic mb-3">
                        "{business.slogan}"
                      </p>
                    )}
                    
                    <div className="space-y-2">
                      {business.contactPhone && (
                        <div className="flex items-center gap-2 text-sm">
                          <Phone className="h-4 w-4 text-gray-400" />
                          <a href={`tel:${business.contactPhone}`} className="hover:text-blue-600">
                            {business.contactPhone}
                          </a>
                        </div>
                      )}
                      
                      {business.contactEmail && (
                        <div className="flex items-center gap-2 text-sm">
                          <Mail className="h-4 w-4 text-gray-400" />
                          <a href={`mailto:${business.contactEmail}`} className="hover:text-blue-600">
                            {business.contactEmail}
                          </a>
                        </div>
                      )}
                      
                      {business.website && (
                        <div className="flex items-center gap-2 text-sm">
                          <Globe className="h-4 w-4 text-gray-400" />
                          <a 
                            href={business.website.startsWith('http') ? business.website : `https://${business.website}`}
                            target="_blank"
                            rel="noopener noreferrer"
                            className="hover:text-blue-600"
                          >
                            Visit Website
                          </a>
                        </div>
                      )}
                    </div>
                    
                    {business.rating > 0 && (
                      <div className="flex items-center gap-1 mt-3 pt-3 border-t">
                        <Star className="h-4 w-4 fill-yellow-400 text-yellow-400" />
                        <span className="text-sm font-medium">{business.rating.toFixed(1)}</span>
                        <span className="text-sm text-gray-500">({business.reviewCount} reviews)</span>
                      </div>
                    )}
                  </CardContent>
                </Card>
              ))}
            </div>
          )}

          {/* Pagination */}
          {totalPages > 1 && (
            <div className="flex justify-center items-center space-x-2 mt-8">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setCurrentPage(prev => Math.max(prev - 1, 1))}
                disabled={currentPage === 1}
                className="flex items-center gap-1"
              >
                <ChevronLeft className="w-4 h-4" />
                Previous
              </Button>
              
              <div className="flex items-center space-x-1">
                {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                  let pageNum;
                  if (totalPages <= 5) {
                    pageNum = i + 1;
                  } else if (currentPage <= 3) {
                    pageNum = i + 1;
                  } else if (currentPage >= totalPages - 2) {
                    pageNum = totalPages - 4 + i;
                  } else {
                    pageNum = currentPage - 2 + i;
                  }
                  
                  return (
                    <Button
                      key={pageNum}
                      variant={currentPage === pageNum ? "default" : "outline"}
                      size="sm"
                      onClick={() => setCurrentPage(pageNum)}
                      className="w-10 h-10"
                    >
                      {pageNum}
                    </Button>
                  );
                })}
              </div>
              
              <Button
                variant="outline"
                size="sm"
                onClick={() => setCurrentPage(prev => Math.min(prev + 1, totalPages))}
                disabled={currentPage === totalPages}
                className="flex items-center gap-1"
              >
                Next
                <ChevronRight className="w-4 h-4" />
              </Button>
            </div>
          )}

          {!businessesLoading && businesses.length === 0 && (
            <div className="text-center py-12">
              <div className="text-gray-400 mb-4">
                <Search className="h-16 w-16 mx-auto" />
              </div>
              <h3 className="text-xl font-semibold mb-2">No businesses found</h3>
              <p className="text-gray-600 mb-4">
                Try adjusting your search criteria or browse all categories.
              </p>
              <Button onClick={() => {
                setSearchTerm("");
                setSelectedCategory("all");
                setSelectedLocation("all");
                setCurrentPage(1);
              }}>
                Clear Filters
              </Button>
            </div>
          )}
        </div>
      </div>
      
      <Footer />
    </>
  );
}