import { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { CheckCircle, Heart, Wifi, Zap, Trophy, Shield } from 'lucide-react';
import { SEOHead, seoData } from '@/components/SEOHead';
import { DataDealsSignupForm } from '@/components/data-deals-signup-form';
import Navigation from '@/components/navigation';

// Pricing data from OpenWeb
const pricingPlans = [
  {
    id: '100gb-anytime',
    name: '100GB Anytime',
    price: 299,
    network: 'Telkom',
    speed: '4G Compatible',
    data: '100GB Anytime Data',
    fup: null,
    features: [
      '100GB Anytime Data',
      '4G Compatible',
      'Use your own 4G/5G router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'BUDGET',
    popular: false
  },
  {
    id: 'uncapped-2mbps',
    name: 'Uncapped 2Mbps',
    price: 337,
    network: '4G',
    speed: 'Up to 2Mbps',
    data: 'Uncapped',
    fup: '100GB then 1Mbps',
    features: [
      'Uncapped up to 2Mbps',
      'FUP: 100GB then 1Mbps',
      'Use your own 4G/5G router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'STARTER',
    popular: true
  },
  {
    id: 'wildcard-lite',
    name: 'Uncapped Wildcard Lite',
    price: 337,
    network: '4G / 5G',
    speed: '1-8Mbps',
    data: 'Uncapped',
    fup: '100GB then 1Mbps',
    features: [
      'Uncapped between 1Mbps and 8Mbps',
      'Speed varies based on network load',
      'FUP: 100GB then 1Mbps',
      'Use your own 4G/5G router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'FLEXIBLE',
    popular: false
  },
  {
    id: 'uncapped-10mbps-basic',
    name: 'Uncapped 10Mbps Basic',
    price: 387,
    network: '4G/5G',
    speed: 'Up to 10Mbps',
    data: 'Uncapped',
    fup: '100GB then 4Mbps, 150GB+ then 2Mbps',
    features: [
      'Uncapped up to 10Mbps Basic',
      'FUP: 100GB then 4Mbps, 150GB+ then 2Mbps',
      'Use your own 4G/5G router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'BASIC',
    popular: false
  },
  {
    id: 'full-speed-lite',
    name: 'Uncapped Full Speed Lite',
    price: 397,
    network: '4G / 5G',
    speed: 'Full Speed (5G)',
    data: 'Uncapped',
    fup: '100GB then 1Mbps',
    features: [
      'Uncapped Full Speed (5G)',
      'FUP: 100GB then 1Mbps',
      'Use your own 4G/5G router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'SPEED',
    popular: false
  },
  {
    id: 'uncapped-wildcard',
    name: 'Uncapped Wildcard',
    price: 427,
    network: '4G / 5G',
    speed: '2-20Mbps',
    data: 'Uncapped',
    fup: '150GB then 1Mbps',
    features: [
      'Uncapped between 2Mbps and 20Mbps',
      'Speed varies based on network load',
      'FUP: 150GB then 1Mbps',
      'Use your own 4G/5G router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'ADAPTIVE',
    popular: true
  },
  {
    id: 'uncapped-5mbps',
    name: 'Uncapped 5Mbps',
    price: 427,
    network: '4G',
    speed: 'Up to 5Mbps',
    data: 'Uncapped',
    fup: '120GB then Shared Pool',
    features: [
      'Uncapped up to 5Mbps',
      'FUP: 120GB then Shared Pool',
      'Shared Pool: 1Mbps to 10Mbps speeds',
      'Use your own 4G/5G router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'RELIABLE',
    popular: false
  },
  {
    id: 'uncapped-10mbps',
    name: 'Uncapped 10Mbps',
    price: 487,
    network: '4G / 5G',
    speed: 'Up to 10Mbps',
    data: 'Uncapped',
    fup: '200GB then Shared Pool',
    features: [
      'Uncapped up to 10Mbps',
      'FUP: 200GB then Shared Pool',
      'Shared Pool: 1Mbps to 10Mbps speeds',
      'Use your own LTE router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'BALANCED',
    popular: true
  },
  {
    id: 'uncapped-20mbps',
    name: 'Uncapped 20Mbps',
    price: 587,
    network: '4G / 5G',
    speed: 'Up to 20Mbps',
    data: 'Uncapped',
    fup: '300GB then Shared Pool',
    features: [
      'Uncapped up to 20Mbps',
      'AirFibre Technology',
      'FUP: 300GB then Shared Pool',
      'Uncapped LTE Anytime - works 24/7',
      'Use your own LTE router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'BUSINESS',
    popular: false
  },
  {
    id: 'uncapped-50mbps',
    name: 'Uncapped 50Mbps',
    price: 787,
    network: '4G / 5G',
    speed: 'Up to 50Mbps',
    data: 'Uncapped',
    fup: '350GB then Shared Pool',
    features: [
      'Uncapped up to 50Mbps',
      'FUP: 350GB then Shared Pool',
      'Complimentary Upgrade to Premium',
      'High Network Priority',
      'Uncapped 5G Anytime - works 24/7',
      'Use your own 4G/5G router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'PREMIUM',
    popular: false
  },
  {
    id: 'full-speed-pro',
    name: 'Uncapped Full Speed Pro',
    price: 987,
    network: '4G / 5G',
    speed: 'Full Speed',
    data: 'Uncapped',
    fup: '400GB then Shared Pool',
    features: [
      'Uncapped Full Speed',
      'FUP: 400GB then Shared Pool',
      'Complimentary Upgrade to Premium',
      'High Network Priority',
      'Uncapped 5G Anytime - works 24/7',
      'Use your own 4G/5G router',
      'Free Delivery',
      'Plug and Play',
      'Month to Month',
      'Network selection at OpenWeb\'s discretion'
    ],
    badge: 'ULTRA',
    popular: false
  }
];

const faqs = [
  {
    question: "What is a Shared Pool?",
    answer: "Instead of applying a blanket throttle to your account once you have reached the usage limit, we will place you in a Shared Pool with other users who have reached their limit. This means that you will contend with the bandwidth available between other users who have been placed in the pool. This is a much better approach than blank throttling your account as you will still have access to higher speeds when the Shared Pool is not busy. On average, users in the shared pool can get anything from 1Mbps to 10Mbps speeds."
  },
  {
    question: "Can I use the SIM card in my phone?",
    answer: "For the best performance, we would suggest you use the SIM in a router, however, this SIM will work in a phone (coverage dependent)."
  },
  {
    question: "Are speeds guaranteed?",
    answer: "Due to the nature of Wireless (LTE) technology, speeds are reliant on many factors. It is therefore that OpenWeb cannot guarantee your speed. If you are experiencing slow speeds, kindly move the router around your house. A different spot may be closer to the tower you are using and improve speed and stability."
  },
  {
    question: "I have used up my data, can I topup?",
    answer: "You most definitely can! Simply drop our OpenWeb LTE Service Champions a mail on lte@openweb.email to get you loaded up. Alternatively, login to our client panel at https://home.openweb.co.za and topup instantly with a credit card."
  },
  {
    question: "Are there any contracts?",
    answer: "Unless otherwise stated when you signup (it will clearly be advertised), all plans are month to month with a calendar months cancellation notice requirement."
  },
  {
    question: "What happens when I move? Can I use my LTE-A service at another location?",
    answer: "The great thing about the LTE-A Mobile Wifi product is that you can literally use it anywhere in South Africa where there is coverage. It is a truly mobile service."
  },
  {
    question: "What speed will I get on the LTE-A Mobile Wifi service?",
    answer: "Speed is reliant on your coverage and which router you use. If your coverage is great, you will get great speeds. On Average users get between around 10Mbps and 50Mbps (on Capped plans). For Uncapped plans, speed is limited to a maximum of the speed you have purchased."
  }
];

export function DataDealsPage() {
  const [selectedPlan, setSelectedPlan] = useState<string | null>(null);
  const [showSignupForm, setShowSignupForm] = useState(false);

  const handleSignup = (planId: string) => {
    setSelectedPlan(planId);
    setShowSignupForm(true);
  };

  return (
    <>
      <SEOHead {...seoData.dataDeals} />
      <Navigation />

      <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-purple-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
        {/* Hero Section */}
        <div className="pt-24 pb-12">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center">
              <div className="flex items-center justify-center mb-6">
                <div className="flex items-center space-x-4">
                  <div className="w-12 h-12 bg-gradient-to-br from-red-500 to-pink-500 rounded-full flex items-center justify-center">
                    <Heart className="w-6 h-6 text-white" />
                  </div>
                  <span className="text-2xl font-bold text-gray-600 dark:text-gray-400">×</span>
                  <div className="w-12 h-12 bg-gradient-to-br from-blue-500 to-purple-500 rounded-full flex items-center justify-center">
                    <Wifi className="w-6 h-6 text-white" />
                  </div>
                </div>
              </div>
              <h1 className="text-4xl md:text-6xl font-bold mb-6">
                <span className="bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
                  4G/5G Data Deals
                </span>
              </h1>
              <p className="text-xl md:text-2xl text-gray-600 dark:text-gray-300 mb-8 max-w-4xl mx-auto">
                Exclusive data packages powered by OpenWeb.co.za with{' '}
                <span className="font-bold text-red-600 dark:text-red-400">R200 donated</span> to 
                LoveMedia Foundation for each purchase
              </p>
              
              {/* Key Benefits */}
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6 max-w-5xl mx-auto mb-12">
                <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-lg border border-gray-100 dark:border-gray-700">
                  <Trophy className="w-8 h-8 text-yellow-500 mx-auto mb-3" />
                  <h3 className="font-semibold text-gray-900 dark:text-white mb-2">Priority Protocols</h3>
                  <p className="text-sm text-gray-600 dark:text-gray-400">Streaming, Gaming & more get priority bandwidth</p>
                </div>
                <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-lg border border-gray-100 dark:border-gray-700">
                  <Heart className="w-8 h-8 text-red-500 mx-auto mb-3" />
                  <h3 className="font-semibold text-gray-900 dark:text-white mb-2">R200 Donation</h3>
                  <p className="text-sm text-gray-600 dark:text-gray-400">Each purchase supports our foundation</p>
                </div>
                <div className="bg-white dark:bg-gray-800 p-6 rounded-xl shadow-lg border border-gray-100 dark:border-gray-700">
                  <Shield className="w-8 h-8 text-blue-500 mx-auto mb-3" />
                  <h3 className="font-semibold text-gray-900 dark:text-white mb-2">Exclusive Discounts</h3>
                  <p className="text-sm text-gray-600 dark:text-gray-400">Special pricing for LoveMedia members</p>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Pricing Plans */}
        <div className="py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                Choose Your Perfect Data Plan
              </h2>
              <p className="text-lg text-gray-600 dark:text-gray-300 max-w-2xl mx-auto">
                All plans include priority streaming & gaming protocols, plus R200 donation to LoveMedia Foundation
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
              {pricingPlans.map((plan) => (
                <Card key={plan.id} className={`relative transition-all duration-300 hover:scale-105 hover:shadow-xl ${
                  plan.popular ? 'ring-2 ring-blue-500 dark:ring-blue-400' : ''
                }`}>
                  {plan.popular && (
                    <div className="absolute -top-3 left-1/2 transform -translate-x-1/2">
                      <Badge className="bg-gradient-to-r from-blue-500 to-purple-500 text-white px-3 py-1">
                        POPULAR
                      </Badge>
                    </div>
                  )}
                  <CardHeader className="text-center">
                    <div className="flex items-center justify-between mb-2">
                      <Badge variant="outline" className="text-xs">
                        {plan.badge}
                      </Badge>
                      <Badge variant="secondary" className="text-xs">
                        {plan.network}
                      </Badge>
                    </div>
                    <CardTitle className="text-lg font-bold text-gray-900 dark:text-white">
                      {plan.name}
                    </CardTitle>
                    <div className="mt-4">
                      <div className="flex items-baseline justify-center">
                        <span className="text-3xl font-bold text-blue-600 dark:text-blue-400">
                          R{plan.price}
                        </span>
                        <span className="text-sm text-gray-500 dark:text-gray-400 ml-1">
                          /month
                        </span>
                      </div>
                      <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                        {plan.speed}
                      </p>
                    </div>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-3 mb-6">
                      {plan.features.map((feature, index) => (
                        <div key={index} className="flex items-start">
                          <CheckCircle className="w-4 h-4 text-green-500 mt-0.5 mr-2 flex-shrink-0" />
                          <span className="text-sm text-gray-600 dark:text-gray-400">
                            {feature}
                          </span>
                        </div>
                      ))}
                    </div>
                    
                    {/* Donation Notice */}
                    <div className="bg-gradient-to-r from-red-50 to-pink-50 dark:from-red-900/20 dark:to-pink-900/20 p-3 rounded-lg mb-4 border border-red-100 dark:border-red-800">
                      <div className="flex items-center">
                        <Heart className="w-4 h-4 text-red-500 mr-2 flex-shrink-0" />
                        <p className="text-xs text-red-700 dark:text-red-300 font-medium">
                          R200 donated to LoveMedia Foundation + amount to TAFTA
                        </p>
                      </div>
                    </div>

                    <Dialog open={showSignupForm && selectedPlan === plan.id} onOpenChange={(open) => {
                      if (!open) {
                        setShowSignupForm(false);
                        setSelectedPlan(null);
                      }
                    }}>
                      <DialogTrigger asChild>
                        <Button 
                          className="w-full bg-gradient-to-r from-blue-500 to-purple-500 hover:from-blue-600 hover:to-purple-600 text-white"
                          onClick={() => handleSignup(plan.id)}
                        >
                          Order Now
                        </Button>
                      </DialogTrigger>
                      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto" aria-describedby="order-form-description">
                        <DialogHeader>
                          <DialogTitle>Complete Your Order - {plan.name}</DialogTitle>
                        </DialogHeader>
                        <div id="order-form-description" className="sr-only">
                          Complete your data package order with LoveMedia Foundation partnership benefits
                        </div>
                        <DataDealsSignupForm 
                          selectedPackage={plan.id}
                          onClose={() => {
                            setShowSignupForm(false);
                            setSelectedPlan(null);
                          }}
                        />
                      </DialogContent>
                    </Dialog>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </div>

        {/* Features Section */}
        <div className="py-16 bg-white dark:bg-gray-800">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">
                Why Choose Our Data Deals?
              </h2>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
              <div className="text-center">
                <div className="w-16 h-16 bg-gradient-to-br from-blue-500 to-purple-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Zap className="w-8 h-8 text-white" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                  Superfast Speeds
                </h3>
                <p className="text-gray-600 dark:text-gray-400 text-sm">
                  Average speeds of 10-50Mbps, coverage and router dependent. Quality over quantity!
                </p>
              </div>
              
              <div className="text-center">
                <div className="w-16 h-16 bg-gradient-to-br from-green-500 to-blue-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Shield className="w-8 h-8 text-white" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                  Network Selection
                </h3>
                <p className="text-gray-600 dark:text-gray-400 text-sm">
                  OpenWeb selects the best network for your location to ensure optimal performance.
                </p>
              </div>
              
              <div className="text-center">
                <div className="w-16 h-16 bg-gradient-to-br from-purple-500 to-pink-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Trophy className="w-8 h-8 text-white" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                  Month to Month
                </h3>
                <p className="text-gray-600 dark:text-gray-400 text-sm">
                  No long-term contracts. Use your plans month to month with a calendar months cancellation notice required.
                </p>
              </div>
              
              <div className="text-center">
                <div className="w-16 h-16 bg-gradient-to-br from-red-500 to-orange-500 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Heart className="w-8 h-8 text-white" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                  All Inclusive
                </h3>
                <p className="text-gray-600 dark:text-gray-400 text-sm">
                  No hidden costs or weird line rentals. The price you see is what you pay.
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* FAQ Section */}
        <div className="py-16">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">
                Frequently Asked Questions
              </h2>
              <p className="text-lg text-gray-600 dark:text-gray-300">
                Everything you need to know about our data deals
              </p>
            </div>

            <Tabs defaultValue="general" className="w-full">
              <TabsList className="grid w-full grid-cols-2">
                <TabsTrigger value="general">General</TabsTrigger>
                <TabsTrigger value="technical">Technical</TabsTrigger>
              </TabsList>
              
              <TabsContent value="general" className="mt-8">
                <div className="space-y-6">
                  {faqs.slice(0, 4).map((faq, index) => (
                    <Card key={index}>
                      <CardHeader>
                        <CardTitle className="text-lg text-gray-900 dark:text-white">
                          {faq.question}
                        </CardTitle>
                      </CardHeader>
                      <CardContent>
                        <p className="text-gray-600 dark:text-gray-400">
                          {faq.answer}
                        </p>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </TabsContent>
              
              <TabsContent value="technical" className="mt-8">
                <div className="space-y-6">
                  {faqs.slice(4).map((faq, index) => (
                    <Card key={index}>
                      <CardHeader>
                        <CardTitle className="text-lg text-gray-900 dark:text-white">
                          {faq.question}
                        </CardTitle>
                      </CardHeader>
                      <CardContent>
                        <p className="text-gray-600 dark:text-gray-400">
                          {faq.answer}
                        </p>
                      </CardContent>
                    </Card>
                  ))}
                </div>
              </TabsContent>
            </Tabs>
          </div>
        </div>

        {/* CTA Section */}
        <div className="py-16 bg-gradient-to-r from-blue-600 to-purple-600">
          <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
            <h2 className="text-3xl md:text-4xl font-bold text-white mb-6">
              Ready to Get Connected?
            </h2>
            <p className="text-xl text-blue-100 mb-8 max-w-2xl mx-auto">
              Join thousands of satisfied customers and support the LoveMedia Foundation 
              with every purchase. Your data plan makes a difference!
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Dialog open={showSignupForm && !selectedPlan} onOpenChange={(open) => {
                if (!open) {
                  setShowSignupForm(false);
                  setSelectedPlan(null);
                }
              }}>
                <DialogTrigger asChild>
                  <Button 
                    size="lg" 
                    className="bg-white text-blue-600 hover:bg-gray-100"
                    onClick={() => setShowSignupForm(true)}
                  >
                    Start Your Order
                  </Button>
                </DialogTrigger>
                <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto" aria-describedby="general-order-description">
                  <DialogHeader>
                    <DialogTitle>Complete Your Data Deal Order</DialogTitle>
                  </DialogHeader>
                  <div id="general-order-description" className="sr-only">
                    Order your data package with exclusive LoveMedia Foundation member benefits
                  </div>
                  <DataDealsSignupForm 
                    onClose={() => {
                      setShowSignupForm(false);
                      setSelectedPlan(null);
                    }}
                  />
                </DialogContent>
              </Dialog>
              <Button 
                variant="outline" 
                size="lg" 
                className="border-white text-white hover:bg-white hover:text-blue-600"
                onClick={() => window.open('https://tafta.org.za/', '_blank')}
              >
                Learn About TAFTA
              </Button>
            </div>
          </div>
        </div>
      </div>
    </>
  );
}