import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { DndContext, DragEndEvent, closestCenter } from "@dnd-kit/core";
import { arrayMove, SortableContext, verticalListSortingStrategy } from "@dnd-kit/sortable";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { ColorPicker } from "@/components/ui/color-picker";
import { toast } from "@/hooks/use-toast";
import { PageSection } from "@shared/schema";
import { SortableSection } from "@/components/website-editor/sortable-section";
import { SectionPreview } from "@/components/website-editor/section-preview";
import { Save, Eye, Plus, GripVertical, Settings, Trash2, Copy } from "lucide-react";

interface PageLayout {
  id: number;
  pageName: string;
  sections: PageSection[];
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

const defaultSections: PageSection[] = [
  {
    id: "hero",
    type: "hero",
    title: "Hero Section",
    content: {
      heading: "Connecting South Africa",
      subheading: "Join South Africa's Largest WhatsApp Community",
      description: "Your trusted source for local news, community alerts, and discounted products from sponsors that save members thousands of rands monthly",
      buttonText: "Learn More",
      buttonLink: "#about",
      backgroundImage: "",
      stats: [
        { number: "6,000+", label: "Community Members" },
        { number: "5", label: "WhatsApp Groups" },
        { number: "24/7", label: "Community Support" }
      ]
    },
    styles: {
      background: "linear-gradient(135deg, #ef4444 0%, #ec4899 50%, #f97316 100%)",
      textColor: "#ffffff",
      padding: "80px 20px",
      margin: "0"
    },
    isVisible: true,
    order: 1
  },
  {
    id: "community",
    type: "community",
    title: "Community Section",
    content: {
      heading: "Join South Africa's Largest WhatsApp Community",
      description: "Connect with over 6,000 neighbors across South Africa and access exclusive discounts that save thousands monthly",
      groups: [
        {
          name: "LoveBallito Discussions",
          members: "2,500+",
          description: "General community discussions, local news, and daily interactions for Ballito residents",
          url: "https://chat.whatsapp.com/invite/ballito-discussions"
        },
        {
          name: "LoveBallito Alerts",
          members: "1,800+",
          description: "Emergency alerts, safety notifications, and urgent community announcements",
          url: "https://chat.whatsapp.com/invite/ballito-alerts"
        }
      ]
    },
    styles: {
      background: "linear-gradient(135deg, #f9fafb 0%, #e5e7eb 100%)",
      textColor: "#1f2937",
      padding: "60px 20px",
      margin: "0"
    },
    isVisible: true,
    order: 2
  },
  {
    id: "about",
    type: "about",
    title: "About Section",
    content: {
      heading: "About LoveMedia Foundation",
      description: "LoveMedia Foundation is a non-profit organization dedicated to connecting communities across South Africa through digital platforms, local news, and exclusive member benefits.",
      features: [
        "Community WhatsApp Groups",
        "Local News & Updates",
        "Exclusive Discounts",
        "24/7 Support"
      ],
      image: "https://images.unsplash.com/photo-1578662996442-48f60103fc96?ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D&auto=format&fit=crop&w=1200&h=400"
    },
    styles: {
      background: "#ffffff",
      textColor: "#1f2937",
      padding: "60px 20px",
      margin: "0"
    },
    isVisible: true,
    order: 3
  }
];

export default function WebsiteEditor() {
  const queryClient = useQueryClient();
  const [activeSection, setActiveSection] = useState<PageSection | null>(null);
  const [previewMode, setPreviewMode] = useState(false);
  const [editingSection, setEditingSection] = useState<PageSection | null>(null);

  // Fetch page layout
  const { data: pageLayout, isLoading } = useQuery<PageLayout>({
    queryKey: ["/api/admin/page-layout/home"],
    queryFn: async () => {
      const response = await fetch("/api/admin/page-layout/home");
      if (!response.ok) {
        if (response.status === 404) {
          return {
            id: 0,
            pageName: "home",
            sections: defaultSections,
            isActive: true,
            createdAt: new Date().toISOString(),
            updatedAt: new Date().toISOString()
          };
        }
        throw new Error("Failed to fetch page layout");
      }
      return response.json();
    },
  });

  // Save page layout mutation
  const saveLayoutMutation = useMutation({
    mutationFn: async (layout: { pageName: string; sections: PageSection[] }) => {
      const response = await fetch("/api/admin/page-layout", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(layout),
      });
      if (!response.ok) throw new Error("Failed to save layout");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/page-layout/home"] });
      toast({
        title: "Layout saved successfully",
        description: "Your website changes have been saved.",
      });
    },
    onError: (error) => {
      toast({
        title: "Error saving layout",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;
    if (active.id !== over?.id && pageLayout) {
      const oldIndex = pageLayout.sections.findIndex((section) => section.id === active.id);
      const newIndex = pageLayout.sections.findIndex((section) => section.id === over?.id);
      const newSections = arrayMove(pageLayout.sections, oldIndex, newIndex).map((section, index) => ({
        ...section,
        order: index + 1
      }));
      
      queryClient.setQueryData(["/api/admin/page-layout/home"], {
        ...pageLayout,
        sections: newSections
      });
    }
  };

  const updateSection = (updatedSection: PageSection) => {
    if (!pageLayout) return;
    
    const newSections = pageLayout.sections.map((section) =>
      section.id === updatedSection.id ? updatedSection : section
    );
    
    queryClient.setQueryData(["/api/admin/page-layout/home"], {
      ...pageLayout,
      sections: newSections
    });
  };

  const deleteSection = (sectionId: string) => {
    if (!pageLayout) return;
    
    const newSections = pageLayout.sections.filter((section) => section.id !== sectionId);
    
    queryClient.setQueryData(["/api/admin/page-layout/home"], {
      ...pageLayout,
      sections: newSections
    });
  };

  const duplicateSection = (section: PageSection) => {
    if (!pageLayout) return;
    
    const newSection: PageSection = {
      ...section,
      id: `${section.id}_copy_${Date.now()}`,
      title: `${section.title} (Copy)`,
      order: pageLayout.sections.length + 1
    };
    
    queryClient.setQueryData(["/api/admin/page-layout/home"], {
      ...pageLayout,
      sections: [...pageLayout.sections, newSection]
    });
  };

  const addNewSection = () => {
    if (!pageLayout) return;
    
    const newSection: PageSection = {
      id: `custom_${Date.now()}`,
      type: "custom",
      title: "New Section",
      content: {
        heading: "New Section",
        description: "Add your content here",
        html: "<p>Custom HTML content</p>"
      },
      styles: {
        background: "#ffffff",
        textColor: "#1f2937",
        padding: "60px 20px",
        margin: "0"
      },
      isVisible: true,
      order: pageLayout.sections.length + 1
    };
    
    queryClient.setQueryData(["/api/admin/page-layout/home"], {
      ...pageLayout,
      sections: [...pageLayout.sections, newSection]
    });
  };

  const handleSave = () => {
    if (pageLayout) {
      saveLayoutMutation.mutate({
        pageName: pageLayout.pageName,
        sections: pageLayout.sections
      });
    }
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-lg">Loading website editor...</div>
      </div>
    );
  }

  if (!pageLayout) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-lg">Error loading website editor</div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="sticky top-0 z-50 bg-white border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex items-center justify-between h-16">
            <div className="flex items-center space-x-4">
              <h1 className="text-xl font-semibold text-gray-900">Website Editor</h1>
              <span className="text-sm text-gray-500">Home Page</span>
            </div>
            <div className="flex items-center space-x-4">
              <Button
                variant="outline"
                onClick={() => setPreviewMode(!previewMode)}
                className="flex items-center space-x-2"
              >
                <Eye className="w-4 h-4" />
                <span>{previewMode ? "Edit" : "Preview"}</span>
              </Button>
              <Button
                onClick={handleSave}
                disabled={saveLayoutMutation.isPending}
                className="flex items-center space-x-2"
              >
                <Save className="w-4 h-4" />
                <span>{saveLayoutMutation.isPending ? "Saving..." : "Save Changes"}</span>
              </Button>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {previewMode ? (
          <div className="space-y-0">
            {pageLayout.sections
              .filter((section) => section.isVisible)
              .sort((a, b) => a.order - b.order)
              .map((section) => (
                <SectionPreview key={section.id} section={section} />
              ))}
          </div>
        ) : (
          <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
            {/* Sections Panel */}
            <div className="lg:col-span-3">
              <div className="mb-6">
                <Button
                  onClick={addNewSection}
                  className="flex items-center space-x-2"
                >
                  <Plus className="w-4 h-4" />
                  <span>Add New Section</span>
                </Button>
              </div>

              <DndContext
                collisionDetection={closestCenter}
                onDragEnd={handleDragEnd}
              >
                <SortableContext
                  items={pageLayout.sections.map((section) => section.id)}
                  strategy={verticalListSortingStrategy}
                >
                  <div className="space-y-4">
                    {pageLayout.sections
                      .sort((a, b) => a.order - b.order)
                      .map((section) => (
                        <SortableSection
                          key={section.id}
                          section={section}
                          onEdit={setEditingSection}
                          onDelete={deleteSection}
                          onDuplicate={duplicateSection}
                          onToggleVisibility={(id, isVisible) => {
                            const updatedSection = pageLayout.sections.find(s => s.id === id);
                            if (updatedSection) {
                              updateSection({ ...updatedSection, isVisible });
                            }
                          }}
                        />
                      ))}
                  </div>
                </SortableContext>
              </DndContext>
            </div>

            {/* Properties Panel */}
            <div className="lg:col-span-1">
              <Card>
                <CardHeader>
                  <CardTitle>Section Properties</CardTitle>
                </CardHeader>
                <CardContent>
                  {editingSection ? (
                    <SectionEditor
                      section={editingSection}
                      onUpdate={updateSection}
                      onClose={() => setEditingSection(null)}
                    />
                  ) : (
                    <p className="text-sm text-gray-500">
                      Select a section to edit its properties
                    </p>
                  )}
                </CardContent>
              </Card>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}

// Section Editor Component
function SectionEditor({
  section,
  onUpdate,
  onClose
}: {
  section: PageSection;
  onUpdate: (section: PageSection) => void;
  onClose: () => void;
}) {
  const [editedSection, setEditedSection] = useState<PageSection>(section);

  const handleSave = () => {
    onUpdate(editedSection);
    onClose();
  };

  return (
    <div className="space-y-4">
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-medium">Edit Section</h3>
        <Button variant="ghost" size="sm" onClick={onClose}>
          ×
        </Button>
      </div>

      <Tabs defaultValue="content" className="w-full">
        <TabsList className="grid w-full grid-cols-2">
          <TabsTrigger value="content">Content</TabsTrigger>
          <TabsTrigger value="styles">Styles</TabsTrigger>
        </TabsList>

        <TabsContent value="content" className="space-y-4">
          <div>
            <Label htmlFor="title">Section Title</Label>
            <Input
              id="title"
              value={editedSection.title}
              onChange={(e) =>
                setEditedSection({ ...editedSection, title: e.target.value })
              }
            />
          </div>

          {editedSection.type === "hero" && (
            <div className="space-y-4">
              <div>
                <Label htmlFor="heading">Heading</Label>
                <Input
                  id="heading"
                  value={editedSection.content.heading || ""}
                  onChange={(e) =>
                    setEditedSection({
                      ...editedSection,
                      content: { ...editedSection.content, heading: e.target.value }
                    })
                  }
                />
              </div>
              <div>
                <Label htmlFor="subheading">Subheading</Label>
                <Input
                  id="subheading"
                  value={editedSection.content.subheading || ""}
                  onChange={(e) =>
                    setEditedSection({
                      ...editedSection,
                      content: { ...editedSection.content, subheading: e.target.value }
                    })
                  }
                />
              </div>
              <div>
                <Label htmlFor="description">Description</Label>
                <Textarea
                  id="description"
                  value={editedSection.content.description || ""}
                  onChange={(e) =>
                    setEditedSection({
                      ...editedSection,
                      content: { ...editedSection.content, description: e.target.value }
                    })
                  }
                />
              </div>
            </div>
          )}

          {editedSection.type === "custom" && (
            <div>
              <Label htmlFor="html">HTML Content</Label>
              <Textarea
                id="html"
                value={editedSection.content.html || ""}
                onChange={(e) =>
                  setEditedSection({
                    ...editedSection,
                    content: { ...editedSection.content, html: e.target.value }
                  })
                }
                rows={10}
              />
            </div>
          )}
        </TabsContent>

        <TabsContent value="styles" className="space-y-4">
          <div>
            <Label htmlFor="background">Background</Label>
            <ColorPicker
              value={editedSection.styles.background || "#ffffff"}
              onChange={(color) =>
                setEditedSection({
                  ...editedSection,
                  styles: { ...editedSection.styles, background: color }
                })
              }
            />
          </div>

          <div>
            <Label htmlFor="textColor">Text Color</Label>
            <ColorPicker
              value={editedSection.styles.textColor || "#000000"}
              onChange={(color) =>
                setEditedSection({
                  ...editedSection,
                  styles: { ...editedSection.styles, textColor: color }
                })
              }
            />
          </div>

          <div>
            <Label htmlFor="padding">Padding</Label>
            <Input
              id="padding"
              value={editedSection.styles.padding || ""}
              onChange={(e) =>
                setEditedSection({
                  ...editedSection,
                  styles: { ...editedSection.styles, padding: e.target.value }
                })
              }
              placeholder="e.g., 60px 20px"
            />
          </div>

          <div>
            <Label htmlFor="customCSS">Custom CSS</Label>
            <Textarea
              id="customCSS"
              value={editedSection.styles.customCSS || ""}
              onChange={(e) =>
                setEditedSection({
                  ...editedSection,
                  styles: { ...editedSection.styles, customCSS: e.target.value }
                })
              }
              placeholder="Enter custom CSS rules"
              rows={5}
            />
          </div>
        </TabsContent>
      </Tabs>

      <div className="flex space-x-2">
        <Button onClick={handleSave} className="flex-1">
          Save Changes
        </Button>
        <Button variant="outline" onClick={onClose}>
          Cancel
        </Button>
      </div>
    </div>
  );
}