import { readFileSync } from 'fs';
import { db } from '../server/db.ts';
import { businesses, businessCategories } from '../shared/schema.ts';

async function importBusinesses() {
  console.log('Starting business import...');
  
  // Read processed businesses data
  const businessesData = JSON.parse(readFileSync('processed_businesses.json', 'utf8'));
  
  console.log(`Found ${businessesData.length} businesses to import`);
  
  // Get unique categories
  const categories = [...new Set(businessesData.map(b => b.category))];
  console.log(`Found ${categories.length} unique categories:`, categories);
  
  // Import categories first
  console.log('Importing categories...');
  const categoryIcons = {
    'Agriculture': 'Sprout',
    'Car & Automotive': 'Car',
    'Education & Training': 'GraduationCap',
    'Engineering': 'Cog',
    'Events & Entertainment': 'Calendar',
    'Financial & Insurance': 'DollarSign',
    'Home Building & Trade': 'Home',
    'Hospitality': 'Coffee',
    'IT & Technology': 'Monitor',
    'Marketing & Legal': 'Briefcase',
    'Medical & Beauty': 'Heart',
    'Public Service': 'Building',
    'Security & Protection': 'Shield', 
    'Telecommunication': 'Phone',
    'Trade & Industry': 'Factory',
    'Transport': 'Truck'
  };
  
  for (let i = 0; i < categories.length; i++) {
    const category = categories[i];
    try {
      await db.insert(businessCategories).values({
        name: category,
        description: `${category} businesses and services`,
        iconName: categoryIcons[category] || 'Building2',
        displayOrder: i + 1,
        isActive: true
      }).onConflictDoNothing();
      console.log(`✓ Imported category: ${category}`);
    } catch (error) {
      console.error(`Error importing category ${category}:`, error.message);
    }
  }
  
  // Import businesses in batches of 100
  const batchSize = 100;
  let imported = 0;
  let skipped = 0;
  
  for (let i = 0; i < businessesData.length; i += batchSize) {
    const batch = businessesData.slice(i, i + batchSize);
    
    console.log(`Processing batch ${Math.floor(i / batchSize) + 1}/${Math.ceil(businessesData.length / batchSize)}`);
    
    const businessesToInsert = batch.map(business => ({
      storeId: business.store_id?.toString() || null,
      name: business.name,
      description: business.description || `${business.name} - ${business.category} services`,
      category: business.category,
      location: business.location || 'South Africa',
      contactEmail: business.contact_email || null,
      contactPhone: business.contact_phone || null,
      website: business.website || null,
      whatsappNumber: null,
      contactPerson: business.contact_person || null,
      slogan: business.slogan || null,
      latitude: business.latitude?.toString() || null,
      longitude: business.longitude?.toString() || null,
      imageUrl: business.image_url || null,
      originalImageUrl: business.original_image_url || null,
      isVerified: business.is_verified || false,
      rating: business.rating || 0,
      reviewCount: business.review_count || 0,
      isFeatured: business.is_featured || false,
      featuredUntil: business.featured_until ? new Date(business.featured_until) : null,
      highlightColor: business.highlight_color || null,
      topPlacement: false,
      topPlacementUntil: null,
      createdBy: null
    }));
    
    try {
      await db.insert(businesses).values(businessesToInsert);
      imported += batch.length;
      console.log(`✓ Imported batch of ${batch.length} businesses (Total: ${imported})`);
    } catch (error) {
      console.error(`Error importing batch:`, error.message);
      skipped += batch.length;
      
      // Try importing one by one to identify problematic records
      for (const business of businessesToInsert) {
        try {
          await db.insert(businesses).values(business);
          imported++;
        } catch (err) {
          console.error(`Failed to import business: ${business.name} - ${err.message}`);
          skipped++;
        }
      }
    }
  }
  
  console.log(`\n✅ Import completed!`);
  console.log(`✓ Imported: ${imported} businesses`);
  console.log(`✗ Skipped: ${skipped} businesses`);
  console.log(`📊 Total categories: ${categories.length}`);
  
  // Show category distribution
  console.log('\n📈 Business distribution by category:');
  const distribution = {};
  businessesData.forEach(b => {
    distribution[b.category] = (distribution[b.category] || 0) + 1;
  });
  
  Object.entries(distribution)
    .sort(([,a], [,b]) => b - a)
    .forEach(([category, count]) => {
      console.log(`  ${category}: ${count} businesses`);
    });
}

// Run import
importBusinesses().catch(console.error);