import XLSX from 'xlsx';
import { readFileSync, writeFileSync, existsSync, mkdirSync } from 'fs';
import { join } from 'path';
import https from 'https';
import http from 'http';
import { URL } from 'url';

// Create images directory if it doesn't exist
if (!existsSync('public/images')) {
  mkdirSync('public/images', { recursive: true });
}

// Function to download image from URL and save locally
async function downloadImage(url, filename) {
  if (!url || url.trim() === '') return null;
  
  return new Promise((resolve, reject) => {
    try {
      const parsedUrl = new URL(url);
      const protocol = parsedUrl.protocol === 'https:' ? https : http;
      const filepath = join('public/images', filename);
      
      // Skip if file already exists
      if (existsSync(filepath)) {
        console.log(`Image already exists: ${filename}`);
        resolve(`/images/${filename}`);
        return;
      }
      
      const request = protocol.get(url, (response) => {
        if (response.statusCode === 200) {
          const chunks = [];
          response.on('data', chunk => chunks.push(chunk));
          response.on('end', () => {
            try {
              writeFileSync(filepath, Buffer.concat(chunks));
              console.log(`Downloaded image: ${filename}`);
              resolve(`/images/${filename}`);
            } catch (error) {
              console.error(`Error saving image ${filename}:`, error.message);
              resolve(null);
            }
          });
        } else {
          console.error(`Failed to download ${url}: HTTP ${response.statusCode}`);
          resolve(null);
        }
      });
      
      request.on('error', (error) => {
        console.error(`Error downloading ${url}:`, error.message);
        resolve(null);
      });
      
      request.setTimeout(10000, () => {
        request.destroy();
        console.error(`Timeout downloading ${url}`);
        resolve(null);
      });
    } catch (error) {
      console.error(`Invalid URL ${url}:`, error.message);
      resolve(null);
    }
  });
}

// Function to clean and format text
function cleanText(text) {
  if (!text) return '';
  return String(text).trim().replace(/\s+/g, ' ');
}

// Function to extract phone numbers from text
function extractPhone(text) {
  if (!text) return '';
  const phoneRegex = /(\+?27|0)[1-9]\d{8}|\d{3}[-\s]?\d{3}[-\s]?\d{4}/g;
  const matches = String(text).match(phoneRegex);
  return matches ? matches[0] : '';
}

// Function to extract email from text
function extractEmail(text) {
  if (!text) return '';
  const emailRegex = /[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}/g;
  const matches = String(text).match(emailRegex);
  return matches ? matches[0] : '';
}

// Function to extract website from text
function extractWebsite(text) {
  if (!text) return '';
  const websiteRegex = /(https?:\/\/[^\s]+|www\.[^\s]+|[a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/g;
  const matches = String(text).match(websiteRegex);
  if (matches) {
    let website = matches[0];
    if (!website.startsWith('http')) {
      website = 'https://' + website;
    }
    return website;
  }
  return '';
}

// Process Excel file
async function processExcelFile(filepath, category) {
  console.log(`Processing file: ${filepath} for category: ${category}`);
  
  const workbook = XLSX.readFile(filepath);
  const sheetName = workbook.SheetNames[0];
  const worksheet = workbook.Sheets[sheetName];
  const data = XLSX.utils.sheet_to_json(worksheet);
  
  const businesses = [];
  
  for (let i = 0; i < data.length; i++) {
    const row = data[i];
    console.log(`Processing business ${i + 1}/${data.length} in ${category}`);
    
    // Extract business information from different possible column names
    const name = cleanText(
      row['Business Name'] || 
      row['Name'] || 
      row['Company Name'] || 
      row['Title'] ||
      ''
    );
    
    if (!name) continue; // Skip if no business name
    
    const description = cleanText(
      row['Description'] || 
      row['About'] || 
      row['Services'] ||
      row['Details'] ||
      ''
    );
    
    const address = cleanText(
      row['Address'] || 
      row['Location'] || 
      row['Street Address'] ||
      row['Physical Address'] ||
      ''
    );
    
    const contact_phone = extractPhone(
      row['Phone'] || 
      row['Contact Number'] || 
      row['Mobile'] ||
      row['Cell'] ||
      ''
    );
    
    const contact_email = extractEmail(
      row['Email'] || 
      row['Email Address'] || 
      row['Contact Email'] ||
      ''
    );
    
    const website = extractWebsite(
      row['Website'] || 
      row['Web'] || 
      row['URL'] ||
      ''
    );
    
    // Handle image URL
    const imageUrl = row['Image'] || row['Photo'] || row['Logo'] || '';
    let localImagePath = null;
    
    if (imageUrl && imageUrl.trim() !== '') {
      const imageExtension = imageUrl.includes('.jpg') ? '.jpg' : 
                           imageUrl.includes('.png') ? '.png' : 
                           imageUrl.includes('.jpeg') ? '.jpeg' : '.jpg';
      const sanitizedName = name.replace(/[^a-zA-Z0-9]/g, '_').toLowerCase();
      const filename = `${sanitizedName}_${Date.now()}${imageExtension}`;
      
      try {
        localImagePath = await downloadImage(imageUrl, filename);
      } catch (error) {
        console.error(`Failed to download image for ${name}:`, error.message);
      }
    }
    
    const business = {
      name: name,
      description: description || `${name} - ${category} services`,
      category: category,
      location: address || 'South Africa',
      contact_email: contact_email,
      contact_phone: contact_phone,
      website: website,
      image_url: localImagePath,
      is_verified: false,
      rating: 0,
      review_count: 0,
      is_featured: false,
      featured_until: null,
      highlight_color: null,
      created_at: new Date().toISOString()
    };
    
    businesses.push(business);
  }
  
  console.log(`Processed ${businesses.length} businesses from ${category}`);
  return businesses;
}

// Main processing function
async function processAllFiles() {
  const categories = [
    { file: 'yep-co-za_Agriculture_1753460631863.xlsx', category: 'Agriculture' },
    { file: 'yep-co-za_Car_&_Automotive_1753460631864.xlsx', category: 'Car & Automotive' },
    { file: 'yep-co-za_Education_Training_Lessons_1753460631864.xlsx', category: 'Education & Training' },
    { file: 'yep-co-za_Engineering_1753460631864.xlsx', category: 'Engineering' },
    { file: 'yep-co-za_Events_&_entertainment_1753460631865.xlsx', category: 'Events & Entertainment' },
    { file: 'yep-co-za_Financial_&_Insurance_Services_1753460631865.xlsx', category: 'Financial & Insurance' },
    { file: 'yep-co-za_Home_Building_&_Trade_1753460631865.xlsx', category: 'Home Building & Trade' },
    { file: 'yep-co-za_Hospitality_1753460631866.xlsx', category: 'Hospitality' },
    { file: 'yep-co-za_IT_Computer_&_Technology_1753460631866.xlsx', category: 'IT & Technology' },
    { file: 'yep-co-za_Marketing_Business,_and_Legal_1753460631860.xlsx', category: 'Marketing & Legal' },
    { file: 'yep-co-za_Medical_Wellness_&_Beauty_1753460631861.xlsx', category: 'Medical & Beauty' },
    { file: 'yep-co-za_Public_Service_&_Government_1753460631861.xlsx', category: 'Public Service' },
    { file: 'yep-co-za_Security_Systems_&_Protection_1753460631862.xlsx', category: 'Security & Protection' },
    { file: 'yep-co-za_Telecommunication_1753460631862.xlsx', category: 'Telecommunication' },
    { file: 'yep-co-za_Trade_&_Industry_1753460631862.xlsx', category: 'Trade & Industry' },
    { file: 'yep-co-za_Transport_1753460631863.xlsx', category: 'Transport' }
  ];
  
  const allBusinesses = [];
  
  for (const { file, category } of categories) {
    const filepath = join('attached_assets', file);
    
    if (existsSync(filepath)) {
      try {
        const businesses = await processExcelFile(filepath, category);
        allBusinesses.push(...businesses);
      } catch (error) {
        console.error(`Error processing ${file}:`, error.message);
      }
    } else {
      console.log(`File not found: ${filepath}`);
    }
  }
  
  // Save all businesses to JSON file
  writeFileSync('processed_businesses.json', JSON.stringify(allBusinesses, null, 2));
  console.log(`Total businesses processed: ${allBusinesses.length}`);
  console.log('Data saved to processed_businesses.json');
  
  return allBusinesses;
}

// Run the processing
processAllFiles().catch(console.error);