#!/usr/bin/env node

import xlsx from 'xlsx';
import fs from 'fs';
import path from 'path';

console.log('=== COMPREHENSIVE EXCEL FILE ANALYSIS ===\n');

async function analyzeExcelFiles() {
  const attachedAssetsDir = path.join(process.cwd(), '..', 'attached_assets');
  
  try {
    // Get all Excel files with the correct timestamp
    const excelFiles = fs.readdirSync(attachedAssetsDir)
      .filter(file => file.endsWith('.xlsx') && file.includes('1753526977'));
    
    console.log(`Found ${excelFiles.length} Excel files to analyze:\n`);
    
    for (let i = 0; i < excelFiles.length; i++) {
      const filename = excelFiles[i];
      const filepath = path.join(attachedAssetsDir, filename);
      
      console.log(`\n${'='.repeat(80)}`);
      console.log(`FILE ${i + 1}/${excelFiles.length}: ${filename}`);
      console.log(`${'='.repeat(80)}`);
      
      try {
        // Read the Excel file
        const workbook = xlsx.readFile(filepath);
        const sheetNames = workbook.SheetNames;
        
        console.log(`Sheet Names: ${sheetNames.join(', ')}`);
        
        // Analyze the first sheet
        const firstSheetName = sheetNames[0];
        const worksheet = workbook.Sheets[firstSheetName];
        const data = xlsx.utils.sheet_to_json(worksheet);
        
        console.log(`Total rows: ${data.length}`);
        
        if (data.length > 0) {
          const firstRow = data[0];
          const columnNames = Object.keys(firstRow);
          
          console.log(`\nColumn Names (${columnNames.length} total):`);
          columnNames.forEach((col, index) => {
            console.log(`  ${index + 1}. "${col}"`);
          });
          
          // Look for potential image URL columns
          console.log('\n--- SEARCHING FOR IMAGE URL COLUMNS ---');
          const imageUrlPatterns = [
            /image.*url/i,
            /imageurl/i,
            /img.*url/i,
            /photo.*url/i,
            /picture.*url/i,
            /logo.*url/i,
            /^image$/i,
            /^photo$/i,
            /^picture$/i,
            /^logo$/i,
            /url.*image/i,
            /url.*photo/i
          ];
          
          const possibleImageColumns = [];
          columnNames.forEach(col => {
            const matches = imageUrlPatterns.some(pattern => pattern.test(col));
            if (matches) {
              possibleImageColumns.push(col);
            }
          });
          
          if (possibleImageColumns.length > 0) {
            console.log(`Found ${possibleImageColumns.length} potential image URL columns:`);
            possibleImageColumns.forEach(col => {
              const sampleValue = firstRow[col];
              console.log(`  - "${col}": "${sampleValue}"`);
            });
          } else {
            console.log('No obvious image URL columns found');
          }
          
          // Look for potential business name columns
          console.log('\n--- SEARCHING FOR BUSINESS NAME COLUMNS ---');
          const namePatterns = [
            /^name$/i,
            /business.*name/i,
            /company.*name/i,
            /^business$/i,
            /^company$/i,
            /entity.*name/i,
            /organisation.*name/i,
            /organization.*name/i
          ];
          
          const possibleNameColumns = [];
          columnNames.forEach(col => {
            const matches = namePatterns.some(pattern => pattern.test(col));
            if (matches) {
              possibleNameColumns.push(col);
            }
          });
          
          if (possibleNameColumns.length > 0) {
            console.log(`Found ${possibleNameColumns.length} potential name columns:`);
            possibleNameColumns.forEach(col => {
              const sampleValue = firstRow[col];
              console.log(`  - "${col}": "${sampleValue}"`);
            });
          } else {
            console.log('No obvious business name columns found');
          }
          
          // Show first 3 rows as samples
          console.log('\n--- SAMPLE DATA (First 3 rows) ---');
          const sampleRows = data.slice(0, 3);
          sampleRows.forEach((row, index) => {
            console.log(`\nRow ${index + 1}:`);
            Object.entries(row).forEach(([key, value]) => {
              // Only show first 100 characters of each value
              const displayValue = String(value).length > 100 
                ? String(value).substring(0, 100) + '...' 
                : String(value);
              console.log(`  ${key}: "${displayValue}"`);
            });
          });
          
          // Check if any values look like URLs
          console.log('\n--- SCANNING FOR URL-LIKE VALUES ---');
          const urlPattern = /https?:\/\/[^\s]+/;
          let foundUrls = 0;
          
          sampleRows.forEach((row, rowIndex) => {
            Object.entries(row).forEach(([key, value]) => {
              if (urlPattern.test(String(value))) {
                console.log(`  Row ${rowIndex + 1}, Column "${key}": "${value}"`);
                foundUrls++;
              }
            });
          });
          
          if (foundUrls === 0) {
            console.log('  No URL-like values found in sample data');
          }
          
        } else {
          console.log('No data rows found in the sheet');
        }
        
      } catch (error) {
        console.error(`Error processing ${filename}:`, error.message);
      }
    }
    
    console.log(`\n${'='.repeat(80)}`);
    console.log('ANALYSIS COMPLETE');
    console.log(`${'='.repeat(80)}`);
    
  } catch (error) {
    console.error('Error during analysis:', error.message);
  }
}

// Run the analysis
analyzeExcelFiles().catch(console.error);