import { 
  users, businesses, businessCategories, content, whatsappGroups, contactSubmissions, products,
  emailNewsletters, smtpConfig, emailCampaigns, pageLayouts, headerTags, seoSettings, systemSettings, dataDealsOrders, dataDealsEmailSettings, dataDealsCustomerEmailSettings,
  type User, type InsertUser, type Business, type InsertBusiness,
  type Content, type InsertContent, type WhatsAppGroup, type ContactSubmission,
  type InsertContact, type Product, type InsertProduct, type EmailNewsletter,
  type InsertEmailNewsletter, type SmtpConfig, type InsertSmtpConfig,
  type EmailCampaign, type InsertEmailCampaign, type PageLayout, type InsertPageLayout,
  type HeaderTag, type InsertHeaderTag, type SeoSettings, type InsertSeoSettings,
  type SystemSettings, type InsertSystemSettings, type DataDealsOrder, type InsertDataDealsOrder,
  type DataDealsEmailSettings, type InsertDataDealsEmailSettings
} from "@shared/schema";

// Need to add the customer email settings types manually due to import complexity
type DataDealsCustomerEmailSettings = typeof dataDealsCustomerEmailSettings.$inferSelect;
type InsertDataDealsCustomerEmailSettings = typeof dataDealsCustomerEmailSettings.$inferInsert;
import { db } from "./db";
import { eq, desc, and, like, sql, asc, inArray } from "drizzle-orm";
import session from "express-session";
import connectPg from "connect-pg-simple";
import { pool } from "./db";

const PostgresSessionStore = connectPg(session);

export interface IStorage {
  // User management
  getUser(id: number): Promise<User | undefined>;
  getUserByUsername(username: string): Promise<User | undefined>;
  getUserByEmail(email: string): Promise<User | undefined>;
  createUser(user: InsertUser): Promise<User>;
  updateUser(id: number, updates: Partial<User>): Promise<User | undefined>;
  getAllUsers(): Promise<User[]>;

  // Business directory
  getBusinesses(category?: string, location?: string, search?: string, page?: number, limit?: number): Promise<{businesses: Business[], total: number}>;
  getBusiness(id: number): Promise<Business | undefined>;
  createBusiness(business: InsertBusiness & { createdBy: number | null }): Promise<Business>;
  updateBusiness(id: number, updates: Partial<Business>): Promise<Business | undefined>;
  deleteBusiness(id: number): Promise<boolean>;
  getBusinessCategories(): Promise<any[]>;
  getBusinessLocations(): Promise<string[]>;
  
  // Business approval workflow
  getPendingBusinesses(): Promise<Business[]>;
  submitBusiness(businessData: any): Promise<Business>;
  approveBusiness(businessId: number, adminId: number): Promise<Business>;
  rejectBusiness(businessId: number, adminId: number, reason: string): Promise<Business>;
  
  // Featured businesses for homepage
  getFeaturedBusinessesForHomepage(): Promise<Business[]>;
  updateBusinessFeaturedStatus(id: number, featured: boolean): Promise<Business | undefined>;
  getAllFeaturedBusinesses(): Promise<Business[]>;

  // Content management
  getContent(type?: string, status?: string): Promise<Content[]>;
  getContentById(id: number): Promise<Content | undefined>;
  createContent(content: InsertContent & { authorId: number }): Promise<Content>;
  updateContent(id: number, updates: Partial<Content>): Promise<Content | undefined>;
  deleteContent(id: number): Promise<boolean>;
  getPublishedContent(type?: string): Promise<Content[]>;

  // WhatsApp groups
  getWhatsAppGroups(): Promise<WhatsAppGroup[]>;
  getActiveWhatsAppGroups(): Promise<WhatsAppGroup[]>;
  updateWhatsAppGroup(id: number, updates: Partial<WhatsAppGroup>): Promise<WhatsAppGroup | undefined>;

  // Contact submissions
  createContactSubmission(submission: InsertContact): Promise<ContactSubmission>;
  getContactSubmissions(status?: string): Promise<ContactSubmission[]>;
  updateContactSubmission(id: number, updates: Partial<ContactSubmission>): Promise<ContactSubmission | undefined>;

  // Products and pricing
  getProducts(category?: string): Promise<Product[]>;
  getProduct(id: number): Promise<Product | undefined>;
  createProduct(product: InsertProduct): Promise<Product>;
  updateProduct(id: number, updates: Partial<Product>): Promise<Product | undefined>;
  deleteProduct(id: number): Promise<boolean>;

  // Newsletter management
  subscribeToNewsletter(email: string): Promise<EmailNewsletter>;
  unsubscribeFromNewsletter(email: string): Promise<boolean>;
  getNewsletterSubscribers(activeOnly?: boolean): Promise<EmailNewsletter[]>;
  getNewsletterSubscription(email: string): Promise<EmailNewsletter | undefined>;

  // SMTP configuration
  getSmtpConfig(): Promise<SmtpConfig | undefined>;
  updateSmtpConfig(config: InsertSmtpConfig): Promise<SmtpConfig>;
  testSmtpConnection(config: SmtpConfig): Promise<boolean>;

  // Email campaigns
  createEmailCampaign(campaign: InsertEmailCampaign & { sentBy: number }): Promise<EmailCampaign>;
  updateEmailCampaign(id: number, updates: Partial<EmailCampaign>): Promise<EmailCampaign | undefined>;
  getEmailCampaigns(): Promise<EmailCampaign[]>;
  getEmailCampaign(id: number): Promise<EmailCampaign | undefined>;

  // Page layouts for website editor
  getPageLayout(pageName: string): Promise<PageLayout | undefined>;
  createPageLayout(layout: InsertPageLayout & { createdBy: number }): Promise<PageLayout>;
  updatePageLayout(pageName: string, updates: Partial<PageLayout>): Promise<PageLayout | undefined>;
  deletePageLayout(pageName: string): Promise<boolean>;
  getAllPageLayouts(): Promise<PageLayout[]>;

  // Header tags for tracking codes
  getHeaderTags(): Promise<HeaderTag[]>;
  getActiveHeaderTags(): Promise<HeaderTag[]>;
  getHeaderTag(id: number): Promise<HeaderTag | undefined>;
  createHeaderTag(tag: InsertHeaderTag & { createdBy: number }): Promise<HeaderTag>;
  updateHeaderTag(id: number, updates: Partial<HeaderTag>): Promise<HeaderTag | undefined>;
  deleteHeaderTag(id: number): Promise<boolean>;

  // SEO settings for pages
  getSeoSettings(): Promise<SeoSettings[]>;
  getSeoSettingsByPage(pageName: string): Promise<SeoSettings | undefined>;
  getSeoSettingsById(id: number): Promise<SeoSettings | undefined>;
  createSeoSettings(settings: InsertSeoSettings & { createdBy: number }): Promise<SeoSettings>;
  updateSeoSettings(id: number, updates: Partial<SeoSettings>): Promise<SeoSettings | undefined>;
  deleteSeoSettings(id: number): Promise<boolean>;

  // System settings
  getSystemSettings(): Promise<SystemSettings | undefined>;
  updateSystemSettings(updates: Partial<SystemSettings> & { updatedBy: number }): Promise<SystemSettings>;
  initializeSystemSettings(): Promise<SystemSettings>;

  // Data deals orders
  createDataDealsOrder(order: InsertDataDealsOrder): Promise<DataDealsOrder>;
  getDataDealsOrders(status?: string): Promise<DataDealsOrder[]>;
  getDataDealsOrder(id: number): Promise<DataDealsOrder | undefined>;
  updateDataDealsOrder(id: number, updates: Partial<DataDealsOrder>): Promise<DataDealsOrder | undefined>;
  markOrderAsSentToOpenWeb(id: number, orderNumber: string): Promise<DataDealsOrder | undefined>;

  // Data deals email settings
  getDataDealsEmailSettings(): Promise<DataDealsEmailSettings | undefined>;
  createDataDealsEmailSettings(settings: InsertDataDealsEmailSettings & { createdBy: number }): Promise<DataDealsEmailSettings>;
  updateDataDealsEmailSettings(id: number, updates: Partial<DataDealsEmailSettings>): Promise<DataDealsEmailSettings | undefined>;
  initializeDataDealsEmailSettings(userId: number): Promise<DataDealsEmailSettings>;

  // Customer confirmation email settings for data deals
  getDataDealsCustomerEmailSettings(): Promise<DataDealsCustomerEmailSettings | undefined>;
  createDataDealsCustomerEmailSettings(settings: InsertDataDealsCustomerEmailSettings & { createdBy: number }): Promise<DataDealsCustomerEmailSettings>;
  updateDataDealsCustomerEmailSettings(id: number, updates: Partial<DataDealsCustomerEmailSettings>): Promise<DataDealsCustomerEmailSettings | undefined>;
  initializeDataDealsCustomerEmailSettings(userId: number): Promise<DataDealsCustomerEmailSettings>;

  // Session store
  sessionStore: session.Store;
}

export class DatabaseStorage implements IStorage {
  sessionStore: session.Store;

  constructor() {
    this.sessionStore = new PostgresSessionStore({ 
      pool, 
      createTableIfMissing: true 
    });
  }

  // User management
  async getUser(id: number): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user || undefined;
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.username, username));
    return user || undefined;
  }

  async getUserByEmail(email: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.email, email));
    return user || undefined;
  }

  async createUser(insertUser: InsertUser): Promise<User> {
    const [user] = await db.insert(users).values(insertUser).returning();
    return user;
  }

  async updateUser(id: number, updates: Partial<User>): Promise<User | undefined> {
    const [user] = await db.update(users).set(updates).where(eq(users.id, id)).returning();
    return user || undefined;
  }

  async getAllUsers(): Promise<User[]> {
    return await db.select().from(users).orderBy(desc(users.createdAt));
  }

  // Business directory - only show approved businesses
  async getBusinesses(category?: string, location?: string, search?: string, page?: number, limit?: number): Promise<{businesses: Business[], total: number}> {
    let query = db.select().from(businesses);
    let countQuery = db.select({ count: sql`count(*)` }).from(businesses);
    
    const conditions = [eq(businesses.status, 'approved')]; // Only show approved businesses
    if (category && category !== 'all') conditions.push(eq(businesses.category, category));
    if (location && location !== 'all') {
      // Match businesses whose location contains the selected province
      conditions.push(sql`${businesses.location} ILIKE ${`%${location}%`}`);
    }
    if (search) {
      conditions.push(
        sql`(${businesses.name} ILIKE ${`%${search}%`} 
        OR ${businesses.description} ILIKE ${`%${search}%`}
        OR ${businesses.location} ILIKE ${`%${search}%`})`
      );
    }
    
    if (conditions.length > 0) {
      query = query.where(and(...conditions));
      countQuery = countQuery.where(and(...conditions));
      
      // Get total count
      const [{ count }] = await countQuery;
      const total = parseInt(count as string);
      
      // Apply pagination
      const currentPage = page || 1;
      const itemsPerPage = limit || 10;
      const offset = (currentPage - 1) * itemsPerPage;
      
      const businessResults = await query
        .orderBy(desc(businesses.createdAt))
        .limit(itemsPerPage)
        .offset(offset);
      
      return { businesses: businessResults, total };
    } else {
      // If no filters applied, return 10 random businesses
      const businessResults = await query.orderBy(sql`RANDOM()`).limit(10);
      return { businesses: businessResults, total: businessResults.length };
    }
  }

  async getBusiness(id: number): Promise<Business | undefined> {
    const [business] = await db.select().from(businesses).where(eq(businesses.id, id));
    return business || undefined;
  }

  async createBusiness(business: InsertBusiness & { createdBy: number | null }): Promise<Business> {
    const [newBusiness] = await db.insert(businesses).values(business).returning();
    return newBusiness;
  }

  async getBusinessCategories(): Promise<any[]> {
    const categories = await db.select().from(businessCategories).orderBy(businessCategories.displayOrder);
    return categories;
  }

  // Featured businesses for homepage with rotation logic
  async getFeaturedBusinessesForHomepage(): Promise<Business[]> {
    // Get 3 featured businesses with lowest view counts for fair rotation
    const featuredBusinesses = await db
      .select()
      .from(businesses)
      .where(eq(businesses.featuredOnHomepage, true))
      .orderBy(asc(businesses.homepageViewCount), sql`RANDOM()`)
      .limit(3);
    
    // Increment view count for selected businesses
    if (featuredBusinesses.length > 0) {
      const businessIds = featuredBusinesses.map(b => b.id);
      await db
        .update(businesses)
        .set({ 
          homepageViewCount: sql`${businesses.homepageViewCount} + 1`
        })
        .where(inArray(businesses.id, businessIds));
    }
    
    return featuredBusinesses;
  }

  async updateBusinessFeaturedStatus(id: number, featured: boolean): Promise<Business | undefined> {
    const [business] = await db
      .update(businesses)
      .set({ 
        featuredOnHomepage: featured,
        // Reset view count when featuring/unfeaturing
        homepageViewCount: 0
      })
      .where(eq(businesses.id, id))
      .returning();
    return business || undefined;
  }

  // Business approval workflow
  async getPendingBusinesses(): Promise<Business[]> {
    return await db.select().from(businesses).where(eq(businesses.status, 'pending')).orderBy(desc(businesses.createdAt));
  }

  async submitBusiness(businessData: any): Promise<Business> {
    const [business] = await db.insert(businesses)
      .values({
        ...businessData,
        status: 'pending',
        isVerified: false
      })
      .returning();
    return business;
  }

  async approveBusiness(businessId: number, adminId: number): Promise<Business> {
    const [business] = await db.update(businesses)
      .set({ 
        status: 'approved', 
        approvedAt: new Date(),
        approvedBy: adminId 
      })
      .where(eq(businesses.id, businessId))
      .returning();
    return business;
  }

  async rejectBusiness(businessId: number, adminId: number, reason: string): Promise<Business> {
    const [business] = await db.update(businesses)
      .set({ 
        status: 'rejected', 
        approvedBy: adminId,
        rejectionReason: reason 
      })
      .where(eq(businesses.id, businessId))
      .returning();
    return business;
  }

  async getAllFeaturedBusinesses(): Promise<Business[]> {
    return await db
      .select()
      .from(businesses)
      .where(eq(businesses.featuredOnHomepage, true))
      .orderBy(desc(businesses.createdAt));
  }

  async getBusinessLocations(): Promise<string[]> {
    // Return South African provinces instead of individual addresses
    const provinces = [
      "Western Cape",
      "Gauteng", 
      "KwaZulu-Natal",
      "Eastern Cape",
      "Free State",
      "Limpopo",
      "Mpumalanga",
      "Northern Cape",
      "North West"
    ];
    return provinces.sort();
  }

  async updateBusiness(id: number, updates: Partial<Business>): Promise<Business | undefined> {
    const [business] = await db.update(businesses).set(updates).where(eq(businesses.id, id)).returning();
    return business || undefined;
  }

  async deleteBusiness(id: number): Promise<boolean> {
    const result = await db.delete(businesses).where(eq(businesses.id, id));
    return result.rowCount ? result.rowCount > 0 : false;
  }

  // Content management
  async getContent(type?: string, status?: string): Promise<Content[]> {
    let query = db.select().from(content);
    
    const conditions = [];
    if (type) conditions.push(eq(content.type, type));
    if (status) conditions.push(eq(content.status, status));
    
    if (conditions.length > 0) {
      query = query.where(and(...conditions));
    }
    
    return await query.orderBy(desc(content.createdAt));
  }

  async getContentById(id: number): Promise<Content | undefined> {
    const [contentItem] = await db.select().from(content).where(eq(content.id, id));
    return contentItem || undefined;
  }

  async createContent(contentData: InsertContent & { authorId: number }): Promise<Content> {
    const [newContent] = await db.insert(content).values({
      ...contentData,
      updatedAt: new Date(),
    }).returning();
    return newContent;
  }

  async updateContent(id: number, updates: Partial<Content>): Promise<Content | undefined> {
    const [contentItem] = await db.update(content).set({
      ...updates,
      updatedAt: new Date(),
    }).where(eq(content.id, id)).returning();
    return contentItem || undefined;
  }

  async deleteContent(id: number): Promise<boolean> {
    const result = await db.delete(content).where(eq(content.id, id));
    return result.rowCount ? result.rowCount > 0 : false;
  }

  async getPublishedContent(type?: string): Promise<Content[]> {
    let query = db.select().from(content).where(eq(content.status, 'published'));
    
    if (type) {
      query = query.where(and(eq(content.status, 'published'), eq(content.type, type)));
    }
    
    return await query.orderBy(desc(content.publishedAt));
  }

  // WhatsApp groups
  async getWhatsAppGroups(): Promise<WhatsAppGroup[]> {
    return await db.select().from(whatsappGroups).orderBy(desc(whatsappGroups.memberCount));
  }

  async getActiveWhatsAppGroups(): Promise<WhatsAppGroup[]> {
    return await db.select()
      .from(whatsappGroups)
      .where(eq(whatsappGroups.isActive, true))
      .orderBy(desc(whatsappGroups.memberCount));
  }

  async updateWhatsAppGroup(id: number, updates: Partial<WhatsAppGroup>): Promise<WhatsAppGroup | undefined> {
    const [group] = await db.update(whatsappGroups).set(updates).where(eq(whatsappGroups.id, id)).returning();
    return group || undefined;
  }

  // Contact submissions
  async createContactSubmission(submission: InsertContact): Promise<ContactSubmission> {
    const [newSubmission] = await db.insert(contactSubmissions).values(submission).returning();
    return newSubmission;
  }

  async getContactSubmissions(status?: string): Promise<ContactSubmission[]> {
    let query = db.select().from(contactSubmissions);
    
    if (status) {
      query = query.where(eq(contactSubmissions.status, status));
    }
    
    return await query.orderBy(desc(contactSubmissions.createdAt));
  }

  async updateContactSubmission(id: number, updates: Partial<ContactSubmission>): Promise<ContactSubmission | undefined> {
    const [submission] = await db.update(contactSubmissions).set(updates).where(eq(contactSubmissions.id, id)).returning();
    return submission || undefined;
  }

  // Products and pricing
  async getProducts(category?: string): Promise<Product[]> {
    let query = db.select().from(products).where(eq(products.isActive, true));
    
    if (category) {
      query = query.where(and(eq(products.isActive, true), eq(products.category, category)));
    }
    
    return await query.orderBy(desc(products.createdAt));
  }

  async getProduct(id: number): Promise<Product | undefined> {
    const [product] = await db.select().from(products).where(eq(products.id, id));
    return product || undefined;
  }

  async createProduct(product: InsertProduct): Promise<Product> {
    // Calculate final price: wholesale + admin fee
    const finalPrice = product.wholesalePrice 
      ? product.wholesalePrice + Math.round(product.wholesalePrice * (product.adminFeePercentage || 5) / 100)
      : 0;

    const [newProduct] = await db.insert(products).values({
      ...product,
      finalPrice,
      updatedAt: new Date(),
    }).returning();
    return newProduct;
  }

  async updateProduct(id: number, updates: Partial<Product>): Promise<Product | undefined> {
    // Recalculate final price if wholesale price or admin fee changes
    if (updates.wholesalePrice || updates.adminFeePercentage) {
      const currentProduct = await this.getProduct(id);
      if (currentProduct) {
        const wholesalePrice = updates.wholesalePrice ?? currentProduct.wholesalePrice;
        const adminFeePercentage = updates.adminFeePercentage ?? currentProduct.adminFeePercentage;
        
        if (wholesalePrice) {
          updates.finalPrice = wholesalePrice + Math.round(wholesalePrice * adminFeePercentage / 100);
        }
      }
    }

    const [product] = await db.update(products).set({
      ...updates,
      updatedAt: new Date(),
    }).where(eq(products.id, id)).returning();
    return product || undefined;
  }

  async deleteProduct(id: number): Promise<boolean> {
    const result = await db.delete(products).where(eq(products.id, id));
    return result.rowCount ? result.rowCount > 0 : false;
  }

  // Newsletter management
  async subscribeToNewsletter(email: string): Promise<EmailNewsletter> {
    // Check if already subscribed
    const existing = await this.getNewsletterSubscription(email);
    if (existing) {
      if (!existing.isActive) {
        // Reactivate if previously unsubscribed
        const [reactivated] = await db.update(emailNewsletters)
          .set({ isActive: true, unsubscribedAt: null })
          .where(eq(emailNewsletters.email, email))
          .returning();
        return reactivated;
      }
      return existing;
    }

    const [subscription] = await db.insert(emailNewsletters)
      .values({ email })
      .returning();
    return subscription;
  }

  async unsubscribeFromNewsletter(email: string): Promise<boolean> {
    const result = await db.update(emailNewsletters)
      .set({ isActive: false, unsubscribedAt: new Date() })
      .where(eq(emailNewsletters.email, email));
    return result.rowCount ? result.rowCount > 0 : false;
  }

  async getNewsletterSubscribers(activeOnly: boolean = true): Promise<EmailNewsletter[]> {
    let query = db.select().from(emailNewsletters);
    
    if (activeOnly) {
      query = query.where(eq(emailNewsletters.isActive, true));
    }
    
    return await query.orderBy(desc(emailNewsletters.createdAt));
  }

  async getNewsletterSubscription(email: string): Promise<EmailNewsletter | undefined> {
    const [subscription] = await db.select().from(emailNewsletters)
      .where(eq(emailNewsletters.email, email));
    return subscription || undefined;
  }

  // SMTP configuration
  async getSmtpConfig(): Promise<SmtpConfig | undefined> {
    const [config] = await db.select().from(smtpConfig)
      .where(eq(smtpConfig.isActive, true))
      .orderBy(desc(smtpConfig.createdAt));
    return config || undefined;
  }

  async updateSmtpConfig(config: InsertSmtpConfig): Promise<SmtpConfig> {
    // Deactivate existing configurations
    await db.update(smtpConfig).set({ isActive: false });
    
    // Insert new configuration
    const [newConfig] = await db.insert(smtpConfig)
      .values({ ...config, updatedAt: new Date() })
      .returning();
    return newConfig;
  }

  async testSmtpConnection(config: SmtpConfig): Promise<boolean> {
    // This will be implemented with nodemailer in the routes
    return true;
  }

  // Email campaigns
  async createEmailCampaign(campaign: InsertEmailCampaign & { sentBy: number }): Promise<EmailCampaign> {
    const [newCampaign] = await db.insert(emailCampaigns)
      .values(campaign)
      .returning();
    return newCampaign;
  }

  async updateEmailCampaign(id: number, updates: Partial<EmailCampaign>): Promise<EmailCampaign | undefined> {
    const [campaign] = await db.update(emailCampaigns)
      .set(updates)
      .where(eq(emailCampaigns.id, id))
      .returning();
    return campaign || undefined;
  }

  async getEmailCampaigns(): Promise<EmailCampaign[]> {
    return await db.select().from(emailCampaigns)
      .orderBy(desc(emailCampaigns.createdAt));
  }

  async getEmailCampaign(id: number): Promise<EmailCampaign | undefined> {
    const [campaign] = await db.select().from(emailCampaigns)
      .where(eq(emailCampaigns.id, id));
    return campaign || undefined;
  }

  // Page layouts for website editor
  async getPageLayout(pageName: string): Promise<PageLayout | undefined> {
    const [layout] = await db.select().from(pageLayouts).where(eq(pageLayouts.pageName, pageName));
    return layout || undefined;
  }

  async createPageLayout(layout: InsertPageLayout & { createdBy: number }): Promise<PageLayout> {
    const [newLayout] = await db.insert(pageLayouts).values(layout).returning();
    return newLayout;
  }

  async updatePageLayout(pageName: string, updates: Partial<PageLayout>): Promise<PageLayout | undefined> {
    const [updatedLayout] = await db
      .update(pageLayouts)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(pageLayouts.pageName, pageName))
      .returning();
    return updatedLayout || undefined;
  }

  async deletePageLayout(pageName: string): Promise<boolean> {
    const result = await db.delete(pageLayouts).where(eq(pageLayouts.pageName, pageName));
    return result.rowCount > 0;
  }

  async getAllPageLayouts(): Promise<PageLayout[]> {
    return await db.select().from(pageLayouts).orderBy(pageLayouts.createdAt);
  }

  // Header tags for tracking codes
  async getHeaderTags(): Promise<HeaderTag[]> {
    return await db.select().from(headerTags).orderBy(headerTags.priority);
  }

  async getActiveHeaderTags(): Promise<HeaderTag[]> {
    return await db.select().from(headerTags)
      .where(eq(headerTags.isActive, true))
      .orderBy(headerTags.priority);
  }

  async getHeaderTag(id: number): Promise<HeaderTag | undefined> {
    const [tag] = await db.select().from(headerTags).where(eq(headerTags.id, id));
    return tag || undefined;
  }

  async createHeaderTag(tag: InsertHeaderTag & { createdBy: number }): Promise<HeaderTag> {
    const [newTag] = await db.insert(headerTags).values(tag).returning();
    return newTag;
  }

  async updateHeaderTag(id: number, updates: Partial<HeaderTag>): Promise<HeaderTag | undefined> {
    const [updatedTag] = await db.update(headerTags)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(headerTags.id, id))
      .returning();
    return updatedTag || undefined;
  }

  async deleteHeaderTag(id: number): Promise<boolean> {
    const result = await db.delete(headerTags).where(eq(headerTags.id, id));
    return result.rowCount > 0;
  }

  // SEO settings for pages
  async getSeoSettings(): Promise<SeoSettings[]> {
    return await db.select().from(seoSettings).orderBy(seoSettings.pageName);
  }

  async getSeoSettingsByPage(pageName: string): Promise<SeoSettings | undefined> {
    const [settings] = await db.select().from(seoSettings).where(eq(seoSettings.pageName, pageName));
    return settings || undefined;
  }

  async getSeoSettingsById(id: number): Promise<SeoSettings | undefined> {
    const [settings] = await db.select().from(seoSettings).where(eq(seoSettings.id, id));
    return settings || undefined;
  }

  async createSeoSettings(settings: InsertSeoSettings & { createdBy: number }): Promise<SeoSettings> {
    const [newSettings] = await db.insert(seoSettings).values(settings).returning();
    return newSettings;
  }

  async updateSeoSettings(id: number, updates: Partial<SeoSettings>): Promise<SeoSettings | undefined> {
    const [updatedSettings] = await db.update(seoSettings)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(seoSettings.id, id))
      .returning();
    return updatedSettings || undefined;
  }

  async deleteSeoSettings(id: number): Promise<boolean> {
    const result = await db.delete(seoSettings).where(eq(seoSettings.id, id));
    return result.rowCount > 0;
  }

  // System settings methods
  async getSystemSettings(): Promise<SystemSettings | undefined> {
    const [settings] = await db.select().from(systemSettings).limit(1);
    if (!settings) {
      // If no settings exist, create default ones
      return this.initializeSystemSettings();
    }
    return settings;
  }

  async updateSystemSettings(updates: Partial<SystemSettings> & { updatedBy: number }): Promise<SystemSettings> {
    const existingSettings = await this.getSystemSettings();
    
    if (existingSettings) {
      const [updatedSettings] = await db.update(systemSettings)
        .set({ ...updates, updatedAt: new Date() })
        .where(eq(systemSettings.id, existingSettings.id))
        .returning();
      return updatedSettings;
    } else {
      // Create new settings if none exist
      const [newSettings] = await db.insert(systemSettings)
        .values({ ...updates, updatedAt: new Date() })
        .returning();
      return newSettings;
    }
  }

  async initializeSystemSettings(): Promise<SystemSettings> {
    const [settings] = await db.insert(systemSettings)
      .values({
        organization: "LoveMedia Foundation NPC",
        location: "Ballito, North Coast, KwaZulu-Natal",
        adminFee: "5% on wholesale prices",
        businessModel: "Non-profit with transparent pricing",
        contactEmail: "info@lovemedia.foundation",
        website: "https://lovemedia.foundation",
      })
      .returning();
    return settings;
  }

  // Data deals orders methods
  async createDataDealsOrder(order: InsertDataDealsOrder): Promise<DataDealsOrder> {
    // Generate order number
    const orderNumber = `LM-${Date.now()}-${Math.random().toString(36).substr(2, 4).toUpperCase()}`;
    
    const [newOrder] = await db.insert(dataDealsOrders)
      .values({
        ...order,
        orderNumber,
        status: 'pending',
        donationAmount: 20000 // R200 in cents
      })
      .returning();
    return newOrder;
  }

  async getDataDealsOrders(status?: string): Promise<DataDealsOrder[]> {
    let query = db.select().from(dataDealsOrders);
    
    if (status) {
      query = query.where(eq(dataDealsOrders.status, status));
    }
    
    return await query.orderBy(desc(dataDealsOrders.createdAt));
  }

  async getDataDealsOrder(id: number): Promise<DataDealsOrder | undefined> {
    const [order] = await db.select().from(dataDealsOrders)
      .where(eq(dataDealsOrders.id, id));
    return order || undefined;
  }

  async updateDataDealsOrder(id: number, updates: Partial<DataDealsOrder>): Promise<DataDealsOrder | undefined> {
    const [updatedOrder] = await db.update(dataDealsOrders)
      .set(updates)
      .where(eq(dataDealsOrders.id, id))
      .returning();
    return updatedOrder || undefined;
  }

  async markOrderAsSentToOpenWeb(id: number, orderNumber: string): Promise<DataDealsOrder | undefined> {
    const [updatedOrder] = await db.update(dataDealsOrders)
      .set({ 
        sentToOpenWeb: true,
        status: 'sent',
        processedAt: new Date(),
        orderNumber 
      })
      .where(eq(dataDealsOrders.id, id))
      .returning();
    return updatedOrder || undefined;
  }

  // Data deals email settings methods
  async getDataDealsEmailSettings(): Promise<DataDealsEmailSettings | undefined> {
    try {
      console.log('[DEBUG] Getting data deals email settings...');
      
      const [settings] = await db.select().from(dataDealsEmailSettings)
        .where(eq(dataDealsEmailSettings.isActive, true))
        .orderBy(desc(dataDealsEmailSettings.createdAt))
        .limit(1);
      
      console.log('[DEBUG] Retrieved settings:', settings ? 'Found settings' : 'No settings found');
      
      if (!settings) {
        console.log('[DEBUG] No settings found, attempting to create default settings...');
        // If no settings exist, create default ones with admin user
        const adminUser = await this.getUserByEmail('ceo@openweb.co.za');
        if (adminUser) {
          console.log('[DEBUG] Admin user found, initializing default settings...');
          return this.initializeDataDealsEmailSettings(adminUser.id);
        } else {
          console.warn('[WARN] No admin user found for email settings initialization');
        }
      }
      
      return settings || undefined;
    } catch (error) {
      console.error('[ERROR] Failed to get data deals email settings:', error);
      throw error;
    }
  }

  async createDataDealsEmailSettings(settings: InsertDataDealsEmailSettings & { createdBy: number }): Promise<DataDealsEmailSettings> {
    try {
      console.log('[DEBUG] Creating new data deals email settings...');
      
      // Deactivate existing settings
      console.log('[DEBUG] Deactivating existing settings...');
      await db.update(dataDealsEmailSettings)
        .set({ isActive: false, updatedAt: new Date() });
      
      console.log('[DEBUG] Inserting new settings:', {
        toEmail: settings.toEmail,
        fromEmail: settings.fromEmail,
        createdBy: settings.createdBy
      });
      
      const [newSettings] = await db.insert(dataDealsEmailSettings)
        .values({ ...settings, isActive: true })
        .returning();
      
      console.log('[DEBUG] Successfully created new email settings with ID:', newSettings.id);
      return newSettings;
    } catch (error) {
      console.error('[ERROR] Failed to create data deals email settings:', error);
      throw error;
    }
  }

  async updateDataDealsEmailSettings(id: number, updates: Partial<DataDealsEmailSettings>): Promise<DataDealsEmailSettings | undefined> {
    try {
      console.log('[DEBUG] Updating data deals email settings ID:', id);
      console.log('[DEBUG] Update data:', {
        toEmail: updates.toEmail,
        fromEmail: updates.fromEmail,
        subject: updates.subject
      });
      
      const [updatedSettings] = await db.update(dataDealsEmailSettings)
        .set({ ...updates, updatedAt: new Date() })
        .where(eq(dataDealsEmailSettings.id, id))
        .returning();
      
      console.log('[DEBUG] Successfully updated email settings:', updatedSettings ? 'Updated' : 'No rows affected');
      return updatedSettings || undefined;
    } catch (error) {
      console.error('[ERROR] Failed to update data deals email settings:', error);
      throw error;
    }
  }

  async initializeDataDealsEmailSettings(userId: number): Promise<DataDealsEmailSettings> {
    const defaultSettings = {
      toEmail: "wow@openweb.email",
      ccEmail: "info@lovemedia.foundation",
      fromEmail: "orders@lovemedia.foundation",
      fromName: "LoveMedia Foundation",
      replyToEmail: "{{customerEmail}}",
      subject: "New Data Deal Order - {{orderNumber}} - LoveMedia Foundation Partnership",
      emailTemplate: `<h2>New LoveMedia Foundation Data Deal Order</h2>
<p><strong>Order Number:</strong> {{orderNumber}}</p>
<p><strong>Order Date:</strong> {{orderDate}}</p>

<h3>Customer Information</h3>
<ul>
  <li><strong>Name:</strong> {{customerName}}</li>
  <li><strong>Email:</strong> {{customerEmail}}</li>
  <li><strong>Mobile:</strong> {{customerMobile}}</li>
  <li><strong>ID/Passport:</strong> {{customerIdNumber}}</li>
</ul>

<h3>Installation & Delivery</h3>
<ul>
  <li><strong>Installation Address:</strong><br>{{installationAddress}}</li>
  <li><strong>Delivery Address:</strong><br>{{deliveryAddress}}</li>
</ul>

<h3>Package Details</h3>
<ul>
  <li><strong>Package:</strong> {{packageName}}</li>
  <li><strong>Base Price:</strong> R{{basePrice}}</li>
  {{#if upgradeToPremium}}<li><strong>Premium Upgrade:</strong> R{{premiumPrice}}</li>{{/if}}
  <li><strong>Total Monthly Price:</strong> R{{totalPrice}}</li>
</ul>

<h3>Payment Information</h3>
<ul>
  <li><strong>Payment Method:</strong> {{paymentMethodText}}</li>
  <li><strong>Bank:</strong> {{bank}}</li>
  <li><strong>Account Number:</strong> {{accountNumber}}</li>
  <li><strong>Account Type:</strong> {{accountType}}</li>
  <li><strong>Branch Code:</strong> {{branchCode}}</li>
  <li><strong>Electronic Mandate Signature:</strong> {{mandateSignature}}</li>
</ul>

<h3>LoveMedia Foundation Partnership</h3>
<p><strong>Donation to LoveMedia Foundation:</strong> R{{donationAmount}}</p>
<p>This order includes priority streaming & gaming protocols and exclusive discounts for LoveMedia Foundation members.</p>

<hr>
<p><em>This order was submitted through the LoveMedia Foundation platform and includes our partnership agreement for R{{donationAmount}} donation per order.</em></p>
<p><strong>For questions contact:</strong> info@lovemedia.foundation</p>`,
      createdBy: userId,
      isActive: true
    };

    const [newSettings] = await db.insert(dataDealsEmailSettings)
      .values(defaultSettings)
      .returning();
    return newSettings;
  }

  // Customer confirmation email settings methods
  async getDataDealsCustomerEmailSettings(): Promise<DataDealsCustomerEmailSettings | undefined> {
    try {
      console.log('[DEBUG] Getting customer email settings...');
      
      const [settings] = await db.select().from(dataDealsCustomerEmailSettings)
        .where(eq(dataDealsCustomerEmailSettings.isActive, true))
        .orderBy(desc(dataDealsCustomerEmailSettings.createdAt))
        .limit(1);
      
      console.log('[DEBUG] Retrieved customer settings:', settings ? 'Found settings' : 'No settings found');
      
      if (!settings) {
        console.log('[DEBUG] No customer settings found, attempting to create default settings...');
        // If no settings exist, create default ones with admin user
        const adminUser = await this.getUserByEmail('ceo@openweb.co.za');
        if (adminUser) {
          console.log('[DEBUG] Admin user found, initializing default customer settings...');
          return this.initializeDataDealsCustomerEmailSettings(adminUser.id);
        } else {
          console.warn('[WARN] No admin user found for customer email settings initialization');
        }
      }
      
      return settings || undefined;
    } catch (error) {
      console.error('[ERROR] Failed to get customer email settings:', error);
      throw error;
    }
  }

  async createDataDealsCustomerEmailSettings(settings: InsertDataDealsCustomerEmailSettings & { createdBy: number }): Promise<DataDealsCustomerEmailSettings> {
    try {
      console.log('[DEBUG] Creating new customer email settings...');
      
      // Deactivate existing settings
      console.log('[DEBUG] Deactivating existing customer settings...');
      await db.update(dataDealsCustomerEmailSettings)
        .set({ isActive: false, updatedAt: new Date() });
      
      console.log('[DEBUG] Inserting new customer settings:', {
        fromEmail: settings.fromEmail,
        subject: settings.subject,
        createdBy: settings.createdBy
      });
      
      const [newSettings] = await db.insert(dataDealsCustomerEmailSettings)
        .values({ ...settings, isActive: true })
        .returning();
      
      console.log('[DEBUG] Successfully created new customer email settings with ID:', newSettings.id);
      return newSettings;
    } catch (error) {
      console.error('[ERROR] Failed to create customer email settings:', error);
      throw error;
    }
  }

  async updateDataDealsCustomerEmailSettings(id: number, updates: Partial<DataDealsCustomerEmailSettings>): Promise<DataDealsCustomerEmailSettings | undefined> {
    try {
      console.log('[DEBUG] Updating customer email settings ID:', id);
      console.log('[DEBUG] Customer update data:', {
        fromEmail: updates.fromEmail,
        subject: updates.subject
      });
      
      const [updatedSettings] = await db.update(dataDealsCustomerEmailSettings)
        .set({ ...updates, updatedAt: new Date() })
        .where(eq(dataDealsCustomerEmailSettings.id, id))
        .returning();
      
      console.log('[DEBUG] Successfully updated customer email settings:', updatedSettings ? 'Updated' : 'No rows affected');
      return updatedSettings || undefined;
    } catch (error) {
      console.error('[ERROR] Failed to update customer email settings:', error);
      throw error;
    }
  }

  async initializeDataDealsCustomerEmailSettings(userId: number): Promise<DataDealsCustomerEmailSettings> {
    const defaultSettings = {
      fromEmail: "noreply@lovemedia.foundation",
      fromName: "LoveMedia Foundation",
      replyToEmail: "info@lovemedia.foundation",
      subject: "Thank You for Your Data Deal Order - {{orderNumber}}",
      emailTemplate: `<div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; color: #333;">
  <div style="text-align: center; margin-bottom: 30px;">
    <h1 style="color: #e11d48; margin-bottom: 10px;">Thank You for Your Order!</h1>
    <p style="font-size: 18px; color: #666;">Order Number: <strong>{{orderNumber}}</strong></p>
  </div>

  <div style="background: linear-gradient(135deg, #3b82f6, #8b5cf6); color: white; padding: 20px; border-radius: 10px; margin-bottom: 30px;">
    <h2 style="margin: 0 0 10px 0;">Hi {{customerName}}!</h2>
    <p style="margin: 0; font-size: 16px;">Thank you for choosing our exclusive 4G/5G data deals through the LoveMedia Foundation partnership.</p>
  </div>

  <div style="background: #f8fafc; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
    <h3 style="color: #374151; margin-top: 0;">What Happens Next?</h3>
    <ul style="color: #6b7280; line-height: 1.6;">
      <li><strong>Order Processing:</strong> Your order is being processed by our partners at OpenWeb.co.za</li>
      <li><strong>Client Ambassador Contact:</strong> An OpenWeb Client Ambassador will reach out to you within 24-48 hours to complete your order setup</li>
      <li><strong>Installation:</strong> They will arrange delivery and installation at your specified address</li>
      <li><strong>Support:</strong> You'll receive all the technical details and support information you need</li>
    </ul>
  </div>

  <div style="background: #fef2f2; border-left: 4px solid #ef4444; padding: 15px; margin-bottom: 20px;">
    <h4 style="color: #dc2626; margin: 0 0 10px 0;">💝 Your Impact</h4>
    <p style="margin: 0; color: #7f1d1d;">With your order, <strong>R200 will be donated</strong> to LoveMedia Foundation to support South African communities. Thank you for making a difference!</p>
  </div>

  <div style="border: 1px solid #e5e7eb; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
    <h3 style="color: #374151; margin-top: 0;">Your Order Details</h3>
    <table style="width: 100%; border-collapse: collapse;">
      <tr>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;"><strong>Package:</strong></td>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;">{{packageName}}</td>
      </tr>
      <tr>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;"><strong>Monthly Cost:</strong></td>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;">R{{totalPrice}}</td>
      </tr>
      <tr>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;"><strong>Installation Address:</strong></td>
        <td style="padding: 8px 0; border-bottom: 1px solid #f3f4f6;">{{installationAddress}}</td>
      </tr>
      <tr>
        <td style="padding: 8px 0;"><strong>Contact Mobile:</strong></td>
        <td style="padding: 8px 0;">{{customerMobile}}</td>
      </tr>
    </table>
  </div>

  <div style="text-align: center; margin: 30px 0;">
    <p style="color: #6b7280; margin-bottom: 15px;">Need help or have questions?</p>
    <p style="margin: 5px 0;"><strong>Email:</strong> <a href="mailto:info@lovemedia.foundation" style="color: #3b82f6;">info@lovemedia.foundation</a></p>
    <p style="margin: 5px 0;"><strong>WhatsApp:</strong> Join our community groups for support</p>
  </div>

  <div style="text-align: center; padding: 20px; background: #f9fafb; border-radius: 8px; margin-top: 30px;">
    <p style="color: #6b7280; margin: 0; font-size: 14px;">Thank you for supporting LoveMedia Foundation and choosing premium data solutions!</p>
  </div>
</div>`,
      createdBy: userId,
      isActive: true
    };

    const [newSettings] = await db.insert(dataDealsCustomerEmailSettings)
      .values(defaultSettings)
      .returning();
    return newSettings;
  }
}

export const storage = new DatabaseStorage();
