import { useState, useEffect } from "react";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent } from "@/components/ui/card";
import { Separator } from "@/components/ui/separator";
import { Package, Sparkles, Wrench, Bug, X } from "lucide-react";
import { VERSION, CHANGELOG, ChangelogEntry } from "@shared/version";

interface ChangelogModalProps {
  isOpen: boolean;
  onClose: () => void;
  version?: string;
}

const getTypeBadgeColor = (type: ChangelogEntry["type"]) => {
  switch (type) {
    case "major":
      return "bg-yellow-100 text-yellow-800 border-yellow-200";
    case "minor":
      return "bg-blue-100 text-blue-800 border-blue-200";
    case "patch":
      return "bg-green-100 text-green-800 border-green-200";
    default:
      return "bg-gray-100 text-gray-800 border-gray-200";
  }
};

const getTypeIcon = (type: ChangelogEntry["type"]) => {
  switch (type) {
    case "major":
      return <Package className="w-4 h-4" />;
    case "minor":
      return <Sparkles className="w-4 h-4" />;
    case "patch":
      return <Bug className="w-4 h-4" />;
    default:
      return <Wrench className="w-4 h-4" />;
  }
};

export default function ChangelogModal({ isOpen, onClose, version = VERSION }: ChangelogModalProps) {
  const latestEntry = CHANGELOG.find(entry => entry.version === version);

  if (!latestEntry) {
    return null;
  }

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="sm:max-w-[700px] max-h-[80vh] overflow-y-auto">
        <DialogHeader>
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <div className="w-10 h-10 bg-primary rounded-lg flex items-center justify-center">
                <Package className="text-white" size={20} />
              </div>
              <div>
                <DialogTitle className="text-xl font-bold">
                  What's New in Masterfile Enterprise Edition v{version}
                </DialogTitle>
                <DialogDescription>
                  Welcome to the latest version! Here's what we've improved.
                </DialogDescription>
              </div>
            </div>
            <Button
              variant="ghost"
              size="sm"
              onClick={onClose}
              className="h-8 w-8 p-0"
            >
              <X className="w-4 h-4" />
            </Button>
          </div>
        </DialogHeader>

        <div className="space-y-6">
          <Card className="border-2 border-primary/20 bg-gradient-to-r from-blue-50 to-indigo-50">
            <CardContent className="p-6">
              <div className="flex items-center gap-3 mb-4">
                <Badge className={`${getTypeBadgeColor(latestEntry.type)} flex items-center gap-1`}>
                  {getTypeIcon(latestEntry.type)}
                  v{latestEntry.version} - {latestEntry.type}
                </Badge>
                <span className="text-sm text-gray-600">{latestEntry.date}</span>
              </div>

              {latestEntry.features.length > 0 && (
                <div className="mb-6">
                  <h4 className="font-semibold text-green-800 mb-3 flex items-center gap-2">
                    <Sparkles className="w-4 h-4" />
                    New Features
                  </h4>
                  <ul className="space-y-2">
                    {latestEntry.features.map((feature, index) => (
                      <li key={index} className="flex items-start gap-2">
                        <div className="w-2 h-2 bg-green-500 rounded-full mt-2 flex-shrink-0" />
                        <span className="text-sm text-gray-700">{feature}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              )}

              {latestEntry.improvements.length > 0 && (
                <div className="mb-6">
                  <h4 className="font-semibold text-blue-800 mb-3 flex items-center gap-2">
                    <Wrench className="w-4 h-4" />
                    Improvements
                  </h4>
                  <ul className="space-y-2">
                    {latestEntry.improvements.map((improvement, index) => (
                      <li key={index} className="flex items-start gap-2">
                        <div className="w-2 h-2 bg-blue-500 rounded-full mt-2 flex-shrink-0" />
                        <span className="text-sm text-gray-700">{improvement}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              )}

              {latestEntry.fixes.length > 0 && (
                <div>
                  <h4 className="font-semibold text-red-800 mb-3 flex items-center gap-2">
                    <Bug className="w-4 h-4" />
                    Bug Fixes
                  </h4>
                  <ul className="space-y-2">
                    {latestEntry.fixes.map((fix, index) => (
                      <li key={index} className="flex items-start gap-2">
                        <div className="w-2 h-2 bg-red-500 rounded-full mt-2 flex-shrink-0" />
                        <span className="text-sm text-gray-700">{fix}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              )}
            </CardContent>
          </Card>

          <Separator />

          <div className="flex items-center justify-between pt-4">
            <p className="text-sm text-gray-600">
              Thank you for using Masterfile! We hope you enjoy the new features.
            </p>
            <Button onClick={onClose} className="px-6">
              Got it!
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}