import { Link, useLocation } from "wouter";
import { Database, ChartLine, IdCard, Users, Search, FileText, User, Radio, Wifi, UserCog, LogOut, Package, Bell, Key, Activity } from "lucide-react";
import { useAuth } from "@/hooks/use-auth";
import { Button } from "@/components/ui/button";
import { VERSION } from "@shared/version";

export default function Sidebar() {
  const [location] = useLocation();
  const { user, logoutMutation } = useAuth();

  const getNavItems = () => {
    const baseItems = [
      { path: "/", label: "Dashboard", icon: ChartLine },
      { path: "/telkom", label: "Telkom Clients", icon: IdCard },
      { path: "/telkom-smart", label: "Telkom Smart", icon: Radio },
      { path: "/mtn-mobile", label: "MTN Mobile (GSM)", icon: Radio },
      { path: "/mtn-fixed", label: "MTN Fixed", icon: Wifi },
      { path: "/axxess", label: "Axxess (MTN Fixed)", icon: Users },
      { path: "/axxess-telkom", label: "Axxess (Telkom)", icon: Users },
      { path: "/axxess-fibre", label: "Axxess (Fibre Lines)", icon: Wifi },
      { path: "/openserve-fibre", label: "OpenServe (Fibre Lines)", icon: Wifi },
      { path: "/search", label: "Search & Filter", icon: Search },
    ];

    // Add admin-only items
    if (user?.role === "admin") {
      baseItems.push(
        { path: "/reports", label: "Reports", icon: FileText },
        { path: "/changelog", label: "Changelog", icon: Package },
        { path: "/system-users", label: "System Users", icon: UserCog },
        { path: "/staff-notifications", label: "Staff Notifications", icon: Bell },
        { path: "/api-keys", label: "API Keys", icon: Key },
        { path: "/action-log", label: "Action Log", icon: Activity }
      );
    }

    return baseItems;
  };

  const navItems = getNavItems();

  const handleLogout = () => {
    logoutMutation.mutate();
  };

  return (
    <div className="w-64 bg-white shadow-lg border-r border-gray-200 flex flex-col">
      {/* Logo and Title */}
      <div className="p-6 border-b border-gray-200">
        <div className="flex items-center space-x-3">
          <div className="w-12 h-12 rounded-lg flex items-center justify-center overflow-hidden">
            <img 
              src="/ow-emblem.jpg" 
              alt="OpenWeb Logo" 
              className="w-full h-full object-contain"
            />
          </div>
          <div>
            <h1 className="text-lg font-bold text-gray-900">Masterfile {VERSION}</h1>
            <p className="text-xs text-gray-500">author: Keoma Wright</p>
          </div>
        </div>
      </div>

      {/* Navigation Menu */}
      <nav className="flex-1 p-4">
        <ul className="space-y-2">
          {navItems.map((item) => {
            const Icon = item.icon;
            const isActive = location === item.path;
            return (
              <li key={item.path}>
                <Link href={item.path}>
                  <div
                    className={`flex items-center space-x-3 px-3 py-2 rounded-lg transition-colors cursor-pointer ${
                      isActive
                        ? "bg-primary text-white font-medium"
                        : "text-gray-700 hover:bg-gray-100"
                    }`}
                  >
                    <Icon size={20} />
                    <span>{item.label}</span>
                  </div>
                </Link>
              </li>
            );
          })}
        </ul>
      </nav>

      {/* User Info */}
      <div className="p-4 border-t border-gray-200">
        <div className="flex items-center space-x-3 mb-3">
          <div className="w-8 h-8 bg-blue-600 rounded-full flex items-center justify-center">
            <User className="text-white" size={16} />
          </div>
          <div className="flex-1">
            <p className="text-sm font-medium text-gray-900">{user?.name || "Admin User"}</p>
            <p className="text-xs text-gray-500">{user?.username || "admin@example.com"}</p>
          </div>
        </div>
        <Button 
          variant="outline" 
          size="sm" 
          className="w-full mb-3"
          onClick={handleLogout}
          disabled={logoutMutation.isPending}
        >
          <LogOut className="h-4 w-4 mr-2" />
          {logoutMutation.isPending ? "Logging out..." : "Logout"}
        </Button>
        
        {/* Version Display */}
        <div className="text-center">
          <p className="text-xs text-gray-500 mb-1">Masterfile</p>
          <p className="text-xs font-medium text-gray-700">Version {VERSION}</p>
        </div>
      </div>
    </div>
  );
}
