import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Building, Star, Layers, Edit, Trash2, Eye } from "lucide-react";
import RoleGuard from "@/components/role-guard";
import { useAuth } from "@/hooks/use-auth";
import type { TelkomClient } from "@shared/schema";

interface TelkomClientCardProps {
  client: TelkomClient;
  onEdit: (client: TelkomClient) => void;
  onDelete: (clientId: number) => void;
  onEditSim: (sim: any, simType: "primary" | "secondary") => void;
  onView: (client: TelkomClient) => void;
}

export default function TelkomClientCard({ client, onEdit, onDelete, onEditSim, onView }: TelkomClientCardProps) {
  const { user } = useAuth();
  const getStatusColor = (status: string) => {
    switch (status.toLowerCase()) {
      case "active":
        return "bg-green-600 text-white border-0";
      case "suspended":
        return "bg-red-600 text-white border-0";
      case "cancelled":
        return "bg-gray-700 text-white border-0";
      case "debit order unpaid":
        return "bg-orange-600 text-white border-0";
      case "pending":
        return "bg-yellow-600 text-white border-0";
      default:
        return "bg-gray-600 text-white border-0";
    }
  };

  return (
    <Card className="shadow-sm border mb-4 border-gray-200">
      <CardContent className="p-6">
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center space-x-4">
            <div className="w-12 h-12 rounded-lg flex items-center justify-center bg-primary">
              <Building className="text-white" size={20} />
            </div>
            <div>
              <div className="flex items-center space-x-2">
                <h4 className="text-lg font-bold text-gray-900">{client.name}</h4>
              </div>
              <p className="text-gray-600">{client.contactInfo || client.email}</p>
              <Badge className="bg-blue-100 text-blue-800 hover:bg-blue-100">
                Telkom Category
              </Badge>
            </div>
          </div>
          <div className="flex items-center space-x-2">
            <RoleGuard requiredRole="admin">
              <Button
                variant="ghost"
                size="sm"
                onClick={() => onEdit(client)}
                className="p-2 text-gray-400 hover:text-gray-600"
              >
                <Edit size={16} />
              </Button>
            </RoleGuard>
            <RoleGuard requiredRole="admin">
              <Button
                variant="ghost"
                size="sm"
                onClick={() => onDelete(client.id)}
                className="p-2 text-gray-400 hover:text-error"
              >
                <Trash2 size={16} />
              </Button>
            </RoleGuard>
            <RoleGuard requiredRole="staff" fallback={null}>
              <Button
                variant="ghost"
                size="sm"
                onClick={() => onView(client)}
                className="p-2 text-blue-400 hover:text-blue-600"
              >
                <Eye size={16} />
              </Button>
            </RoleGuard>
          </div>
        </div>

        {/* Primary SIM */}
        <div className="mb-4">
          <div className="bg-gradient-to-r from-yellow-50 to-amber-50 border-2 border-yellow-300 rounded-lg p-1 mb-3">
            <h5 className="text-sm font-bold text-yellow-800 flex items-center justify-center py-2">
              <Star className="text-yellow-600 mr-2 fill-current" size={18} />
              PRIMARY SIM
              <Star className="text-yellow-600 ml-2 fill-current" size={18} />
            </h5>
          </div>
          <div className={`rounded-lg p-5 border-2 shadow-md ${client.isReseller ? 'bg-gradient-to-r from-blue-50 to-indigo-50 border-blue-400' : 'bg-gradient-to-r from-green-50 to-emerald-50 border-green-400'}`}>
            <div className="flex justify-between items-start mb-3">
              <div className={`grid grid-cols-1 md:grid-cols-2 gap-6 flex-1 ${client.fup ? 'lg:grid-cols-7' : 'lg:grid-cols-6'}`}>
                <div className="min-w-0">
                  <p className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-1">Client Name</p>
                  <div className="flex items-center space-x-2">
                    <p className="text-sm font-bold text-gray-900">{client.name}</p>
                    {client.isReseller && (
                      <Badge className="bg-blue-600 text-white text-xs px-2 py-1 hover:bg-blue-600">
                        Reseller
                      </Badge>
                    )}
                  </div>
                </div>
                <div className="min-w-0">
                  <p className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-1">SIM Number</p>
                  <p className="text-sm font-bold text-gray-900 break-all">{client.primarySim.simNumber}</p>
                </div>
                <div className="min-w-0">
                  <p className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-1">Account Number</p>
                  <p className="text-sm font-bold text-gray-900">{client.accountNumber || ''}</p>
                </div>
                <div className="min-w-0">
                  <p className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-1">Status</p>
                  <Badge className={`${getStatusColor(client.primarySim.status)} text-white font-medium`}>
                    {client.primarySim.status}
                  </Badge>
                </div>
                <div className="min-w-0">
                  <p className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-1">Status 2</p>
                  <Badge className={`${client.status2 ? getStatusColor(client.status2) : 'bg-gray-400'} text-white font-medium`}>
                    {client.status2 || 'N/A'}
                  </Badge>
                </div>
                <div className="min-w-0">
                  <p className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-1">Data Plan</p>
                  <p className="text-sm font-bold text-gray-900 break-words whitespace-normal">{client.primarySim.dataPlan}</p>
                </div>
                {/* FUP field - only show if configured */}
                {client.fup && (
                  <div className="min-w-0">
                    <p className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-1">FUP</p>
                    <div className="flex items-center space-x-1">
                      <p className="text-sm font-bold text-gray-900">{client.fup}</p>
                      <span className="text-xs text-gray-500">GB</span>
                    </div>
                  </div>
                )}
              </div>
              <div className="ml-4 flex-shrink-0 flex items-center space-x-2">
                <RoleGuard requiredRole="admin">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => onEditSim(client, "primary")}
                  >
                    <Edit className="h-4 w-4 mr-1" />
                    Edit
                  </Button>
                </RoleGuard>
                <RoleGuard requiredRole="staff" fallback={null}>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => onView(client)}
                  >
                    <Eye className="h-4 w-4 mr-1" />
                    View
                  </Button>
                </RoleGuard>
              </div>
            </div>
          </div>
        </div>

        {/* Secondary SIMs Grid */}
        <div>
          <h5 className="text-sm font-semibold text-gray-700 mb-3 flex items-center">
            <Layers className="text-gray-500 mr-2" size={16} />
            Secondary SIMs ({client.secondarySims.length} connected)
          </h5>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
            {client.secondarySims.map((sim, index) => (
              <div
                key={`secondary-${index}-${sim.simNumber || index}`}
                className={`rounded-lg p-3 border hover:shadow-sm transition-shadow ${
                  sim.isReseller 
                    ? 'bg-blue-100 border-blue-300' 
                    : 'bg-gray-50 border-gray-200'
                }`}
              >
                <div className="flex items-center justify-between mb-2">
                  <span className="text-xs font-medium text-gray-500">Secondary #{index + 1}</span>
                  <div className="flex items-center space-x-1">
                    <div className="flex flex-col space-y-1">
                      <Badge className={`${getStatusColor(sim.status)} font-medium text-xs`}>
                        {sim.status}
                      </Badge>
                      {sim.status2 && (
                        <Badge className={`${getStatusColor(sim.status2)} font-medium text-xs`}>
                          {sim.status2}
                        </Badge>
                      )}
                    </div>
                    <RoleGuard requiredRole="admin">
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => onEditSim(sim, "secondary")}
                        className="h-6 w-6 p-0"
                      >
                        <Edit className="h-3 w-3" />
                      </Button>
                    </RoleGuard>
                    <RoleGuard requiredRole="staff" fallback={null}>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => onView(client)}
                        className="h-6 w-6 p-0"
                      >
                        <Eye className="h-3 w-3" />
                      </Button>
                    </RoleGuard>
                  </div>
                </div>
                <div className="mb-1">
                  <span className="text-xs font-medium text-gray-500">Client: </span>
                  <div className="flex items-center space-x-1">
                    <span className="text-xs text-gray-900 font-medium">{sim.name}</span>
                    {sim.isReseller && (
                      <Badge className="bg-blue-600 text-white text-xs px-1 py-0.5 hover:bg-blue-600">
                        Reseller
                      </Badge>
                    )}
                  </div>
                </div>
                <p className="text-sm font-bold text-gray-900 mb-1 break-all">{sim.simNumber}</p>
                <div className="mb-1">
                  <span className="text-xs font-medium text-gray-500">Phone: </span>
                  <span className="text-xs text-gray-900 font-medium">{sim.phoneNumber}</span>
                </div>
                <p className="text-xs text-gray-500 mt-1 line-clamp-2">{sim.dataPlan}</p>
              </div>
            ))}
          </div>
        </div>
      </CardContent>
    </Card>
  );
}
