import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
} from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent } from "@/components/ui/card";
import { Progress } from "@/components/ui/progress";
import { TrendingUp, Clock, HardDrive, Trophy } from "lucide-react";
import { formatBytes } from "@/lib/utils";

interface Top15ModalProps {
  isOpen: boolean;
  onClose: () => void;
  data: {
    success: boolean;
    year: number;
    month: number;
    monthName: string;
    users: {
      username: string;
      name: string;
      msisdn: string;
      totalBytes: number;
      connectedTimeMinutes: number;
    }[];
    count: number;
  } | null;
  isLoading: boolean;
  isBottomModal?: boolean;
  serviceType?: 'MTN Mobile' | 'MTN Fixed';
}

export default function Top15Modal({ isOpen, onClose, data, isLoading, isBottomModal = false, serviceType = 'MTN Mobile' }: Top15ModalProps) {
  if (!data?.users || data.users.length === 0) {
    return (
      <Dialog open={isOpen} onOpenChange={onClose}>
        <DialogContent className="sm:max-w-[900px] max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <Trophy className={`w-5 h-5 ${isBottomModal ? 'text-red-600' : 'text-amber-600'}`} />
              {isBottomModal ? `Bottom 30 ${serviceType} Users` : `Top 15 ${serviceType} Users`}
            </DialogTitle>
            <DialogDescription>
              {isLoading 
                ? `Loading ${isBottomModal ? 'bottom 30' : 'top'} users...` 
                : "No usage data available for this month"
              }
            </DialogDescription>
          </DialogHeader>
          
          {isLoading ? (
            <div className="flex items-center justify-center p-8">
              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
            </div>
          ) : (
            <div className="text-center py-8">
              <p className="text-gray-500">No usage data available for this month</p>
            </div>
          )}
        </DialogContent>
      </Dialog>
    );
  }

  const maxUsage = data.users.length > 0 ? data.users[0].totalBytes : 0;

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="sm:max-w-[900px] max-h-[80vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Trophy className={`w-5 h-5 ${isBottomModal ? 'text-red-600' : 'text-amber-600'}`} />
            {isBottomModal ? 'Bottom 30' : 'Top 15'} {serviceType} Users - {data.monthName} {data.year}
          </DialogTitle>
          <DialogDescription>
            {isBottomModal 
              ? 'Lowest data usage for the current month'
              : 'Highest data usage for the current month'
            } • {data.count} users shown
          </DialogDescription>
        </DialogHeader>
        
        <div className="space-y-4">
          {data.users.map((user, index) => {
            const rankColor = index === 0 ? 'text-amber-600' : 
                            index === 1 ? 'text-gray-500' : 
                            index === 2 ? 'text-amber-700' : 'text-gray-400';
            
            const progressPercentage = maxUsage > 0 ? (user.totalBytes / maxUsage) * 100 : 0;
            
            return (
              <Card key={user.username} className="hover:shadow-md transition-shadow">
                <CardContent className="p-4">
                  <div className="flex items-center justify-between mb-3">
                    <div className="flex items-center gap-3">
                      <div className={`font-bold text-lg ${rankColor} min-w-[30px] text-center`}>
                        #{index + 1}
                      </div>
                      <div>
                        <h4 className="font-semibold text-gray-900">{user.name}</h4>
                        <p className="text-sm text-gray-600">{user.msisdn || 'No MSISDN'}</p>
                      </div>
                    </div>
                    
                    <div className="text-right">
                      <div className="flex items-center gap-1 text-sm text-gray-600 mb-1">
                        <TrendingUp className="w-4 h-4" />
                        <span>Account: {user.username}</span>
                      </div>
                      <Badge variant="secondary" className="text-xs">
                        {serviceType}
                      </Badge>
                    </div>
                  </div>
                  
                  <div className="space-y-2">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-2">
                        <HardDrive className="w-4 h-4 text-blue-600" />
                        <span className="text-sm font-medium">Data Usage</span>
                      </div>
                      <span className="font-semibold text-blue-600">
                        {formatBytes(user.totalBytes)}
                      </span>
                    </div>
                    
                    <Progress 
                      value={progressPercentage} 
                      className="h-2"
                    />
                    
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-2">
                        <Clock className="w-4 h-4 text-green-600" />
                        <span className="text-sm font-medium">Connected Time</span>
                      </div>
                      <span className="text-sm text-gray-600">
                        {Math.floor(user.connectedTimeMinutes / 60)}h {user.connectedTimeMinutes % 60}m
                      </span>
                    </div>
                  </div>
                </CardContent>
              </Card>
            );
          })}
          
          {data.users.length === 0 && (
            <div className="text-center py-8">
              <p className="text-gray-500">No usage data available for this month</p>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
}