import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { Calendar, Star, Zap, Bug, Package } from "lucide-react";
import { CHANGELOG, VERSION, type ChangelogEntry } from "@shared/version";

const getTypeIcon = (type: ChangelogEntry["type"]) => {
  switch (type) {
    case "major":
      return <Star className="w-5 h-5 text-yellow-500" />;
    case "minor":
      return <Zap className="w-5 h-5 text-blue-500" />;
    case "patch":
      return <Bug className="w-5 h-5 text-green-500" />;
    default:
      return <Package className="w-5 h-5 text-gray-500" />;
  }
};

const getTypeBadgeColor = (type: ChangelogEntry["type"]) => {
  switch (type) {
    case "major":
      return "bg-yellow-100 text-yellow-800 border-yellow-200";
    case "minor":
      return "bg-blue-100 text-blue-800 border-blue-200";
    case "patch":
      return "bg-green-100 text-green-800 border-green-200";
    default:
      return "bg-gray-100 text-gray-800 border-gray-200";
  }
};

export default function Changelog() {
  return (
    <div className="p-6 max-w-4xl mx-auto">
      <div className="mb-8">
        <div className="flex items-center gap-3 mb-4">
          <div className="w-12 h-12 bg-primary rounded-lg flex items-center justify-center">
            <Package className="text-white" size={24} />
          </div>
          <div>
            <h1 className="text-3xl font-bold text-gray-900">Changelog</h1>
            <p className="text-gray-600">Track all changes and improvements to Masterfile</p>
          </div>
        </div>
        
        <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-6">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-xl font-semibold text-blue-900 mb-2">Current Version</h2>
              <p className="text-blue-700">You are running Masterfile version {VERSION}</p>
            </div>
            <Badge className="text-lg px-4 py-2 bg-blue-600 hover:bg-blue-700">
              v{VERSION}
            </Badge>
          </div>
        </div>
      </div>

      <div className="relative">
        <div className="absolute top-0 right-2 text-xs text-gray-400 bg-white px-2 py-1 rounded-md border shadow-sm z-10">
          Scroll to view all versions ↕
        </div>
        <div className="max-h-[calc(100vh-300px)] overflow-y-auto space-y-8 pr-4 custom-scrollbar">
        {CHANGELOG.map((entry, index) => (
          <Card key={entry.version} className="relative overflow-hidden">
            {index === 0 && (
              <div className="absolute top-0 right-0 bg-gradient-to-l from-blue-500 to-blue-600 text-white px-4 py-1 rounded-bl-lg">
                <span className="text-sm font-medium">Latest</span>
              </div>
            )}
            
            <CardHeader className="pb-4">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  {getTypeIcon(entry.type)}
                  <div>
                    <CardTitle className="text-2xl">Version {entry.version}</CardTitle>
                    <div className="flex items-center gap-2 mt-1">
                      <Calendar className="w-4 h-4 text-gray-500" />
                      <span className="text-sm text-gray-600">{entry.date}</span>
                    </div>
                  </div>
                </div>
                <Badge className={getTypeBadgeColor(entry.type)}>
                  {entry.type.charAt(0).toUpperCase() + entry.type.slice(1)} Release
                </Badge>
              </div>
            </CardHeader>

            <CardContent className="space-y-6">
              {entry.features.length > 0 && (
                <div>
                  <h3 className="flex items-center gap-2 font-semibold text-green-700 mb-3">
                    <Star className="w-4 h-4" />
                    New Features
                  </h3>
                  <ul className="space-y-2">
                    {entry.features.map((feature, i) => (
                      <li key={i} className="flex items-start gap-2">
                        <div className="w-2 h-2 bg-green-500 rounded-full mt-2 flex-shrink-0"></div>
                        <span className="text-gray-700">{feature}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              )}

              {entry.improvements.length > 0 && (
                <div>
                  <Separator className="my-4" />
                  <h3 className="flex items-center gap-2 font-semibold text-blue-700 mb-3">
                    <Zap className="w-4 h-4" />
                    Improvements
                  </h3>
                  <ul className="space-y-2">
                    {entry.improvements.map((improvement, i) => (
                      <li key={i} className="flex items-start gap-2">
                        <div className="w-2 h-2 bg-blue-500 rounded-full mt-2 flex-shrink-0"></div>
                        <span className="text-gray-700">{improvement}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              )}

              {entry.fixes.length > 0 && (
                <div>
                  <Separator className="my-4" />
                  <h3 className="flex items-center gap-2 font-semibold text-red-700 mb-3">
                    <Bug className="w-4 h-4" />
                    Bug Fixes
                  </h3>
                  <ul className="space-y-2">
                    {entry.fixes.map((fix, i) => (
                      <li key={i} className="flex items-start gap-2">
                        <div className="w-2 h-2 bg-red-500 rounded-full mt-2 flex-shrink-0"></div>
                        <span className="text-gray-700">{fix}</span>
                      </li>
                    ))}
                  </ul>
                </div>
              )}
            </CardContent>
          </Card>
        ))}
        </div>
      </div>

      <div className="mt-12 p-6 bg-gray-50 rounded-lg border border-gray-200 text-center">
        <h3 className="font-semibold text-gray-900 mb-2">Stay Updated</h3>
        <p className="text-gray-600">
          This changelog is automatically updated with each release. 
          Check back regularly to see what's new in Masterfile.
        </p>
      </div>
    </div>
  );
}