#!/usr/bin/env python3
"""
Script to regenerate protocol usage data for existing records
"""
import sys
sys.path.append('.')

from database import DatabaseManager
from protocol_analyzer import ProtocolAnalyzer

def regenerate_protocol_data():
    """Regenerate protocol usage data for all usage records"""
    db = DatabaseManager()
    protocol_analyzer = ProtocolAnalyzer(db)
    
    # Get all usage records that need protocol data regenerated
    with db.get_connection() as conn:
        import pandas as pd
        usage_records = pd.read_sql_query("""
            SELECT id, total_usage_gb 
            FROM usage_records 
            WHERE id NOT IN (SELECT DISTINCT usage_record_id FROM protocol_usage)
        """, conn)
    
    print(f"Found {len(usage_records)} usage records needing protocol data")
    
    for _, record in usage_records.iterrows():
        usage_record_id = int(record['id'])
        total_usage_gb = float(record['total_usage_gb'])
        
        print(f"Generating protocol data for record {usage_record_id} ({total_usage_gb} GB)")
        protocol_analyzer.generate_protocol_usage(usage_record_id, total_usage_gb)
    
    print("Protocol data regeneration complete")

if __name__ == "__main__":
    regenerate_protocol_data()