import { Switch, Route, useLocation } from "wouter";
import { queryClient } from "./lib/queryClient";
import { QueryClientProvider } from "@tanstack/react-query";
import { Toaster } from "@/components/ui/toaster";
import { TooltipProvider } from "@/components/ui/tooltip";
import { AuthProvider, useAuth } from "@/hooks/use-auth";
import Dashboard from "@/pages/dashboard";
import Analytics from "@/pages/analytics";
import VideoLibrary from "@/pages/video-library";
import ApiPage from "@/pages/api";
import ApiServices from "@/pages/api-services";
import SystemUsers from "@/pages/system-users";
import Orders from "@/pages/orders";
import AdminPoints from "@/pages/admin-points";
import SecurityDashboard from "@/pages/security";
import SystemOperations from "@/pages/system-operations";
import Login from "@/pages/login";
import Landing from "@/pages/landing";
import About from "@/pages/about";
import Changelog from "@/pages/changelog";
import RedditManagement from "@/pages/reddit-management";
import SubredditManager from "@/pages/subreddit-manager";
import RedditModeration from "@/pages/reddit-moderation";
import ModerationLog from "@/pages/moderation-log";
import Modmail from "@/pages/modmail";
import ChannelManager from "@/pages/channel-manager";
import PackageManager from "@/pages/package-manager";
import RedditHomeFeed from "@/pages/reddit-home-feed";
import HeaderCodesPage from "@/pages/header-codes";
import AiConfigurationPage from "@/pages/ai-configuration";
import Settings from "@/pages/settings";
import HeaderCodesInjector from "@/components/header-codes-injector";
import NotFound from "@/pages/not-found";
import { useEffect } from "react";

function Router() {
  const { isAuthenticated, isLoading } = useAuth();
  const [location, setLocation] = useLocation();

  useEffect(() => {
    if (!isLoading) {
      if (!isAuthenticated && !["/login", "/", "/about", "/changelog"].includes(location)) {
        setLocation("/");
      } else if (isAuthenticated && ["/login", "/"].includes(location)) {
        // Default to reddit moderation platform for AI-powered Reddit moderation
        setLocation("/reddit-moderation");
      }
    }
  }, [isAuthenticated, isLoading, location, setLocation]);

  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="text-lg">Authenticating...</div>
      </div>
    );
  }

  return (
    <Switch>
      <Route path="/login">
        {!isAuthenticated ? <Login /> : <Dashboard />}
      </Route>
      <Route path="/home">
        {isAuthenticated ? <RedditHomeFeed /> : <Login />}
      </Route>
      <Route path="/dashboard">
        {isAuthenticated ? <Dashboard /> : <Login />}
      </Route>
      <Route path="/analytics">
        {isAuthenticated ? <Analytics /> : <Login />}
      </Route>
      <Route path="/video-library">
        {isAuthenticated ? <VideoLibrary /> : <Login />}
      </Route>
      <Route path="/api">
        {isAuthenticated ? <ApiPage /> : <Login />}
      </Route>
      <Route path="/api-services">
        {isAuthenticated ? <ApiServices /> : <Login />}
      </Route>
      <Route path="/system-users">
        {isAuthenticated ? <SystemUsers /> : <Login />}
      </Route>
      <Route path="/orders">
        {isAuthenticated ? <Orders /> : <Login />}
      </Route>
      <Route path="/admin/points">
        {isAuthenticated ? <AdminPoints /> : <Login />}
      </Route>
      <Route path="/admin/security">
        {isAuthenticated ? <SecurityDashboard /> : <Login />}
      </Route>
      <Route path="/package-manager">
        {isAuthenticated ? <PackageManager /> : <Login />}
      </Route>
      <Route path="/admin/header-codes">
        {isAuthenticated ? <HeaderCodesPage /> : <Login />}
      </Route>
      <Route path="/ai-configuration">
        {isAuthenticated ? <AiConfigurationPage /> : <Login />}
      </Route>
      <Route path="/settings">
        {isAuthenticated ? <Settings /> : <Login />}
      </Route>
      <Route path="/system-operations">
        {isAuthenticated ? <SystemOperations /> : <Login />}
      </Route>
      <Route path="/reddit-management">
        {isAuthenticated ? <RedditManagement /> : <Login />}
      </Route>
      <Route path="/subreddit-manager">
        {isAuthenticated ? <SubredditManager /> : <Login />}
      </Route>
      <Route path="/reddit-moderation">
        {isAuthenticated ? <RedditModeration /> : <Login />}
      </Route>
      <Route path="/moderation-log">
        {isAuthenticated ? <ModerationLog /> : <Login />}
      </Route>
      <Route path="/modmail">
        {isAuthenticated ? <Modmail /> : <Login />}
      </Route>
      <Route path="/channel-manager">
        {isAuthenticated ? <ChannelManager /> : <Login />}
      </Route>
      <Route path="/about" component={About} />
      <Route path="/changelog" component={Changelog} />
      <Route path="/">
        {isAuthenticated ? <Dashboard /> : <Landing />}
      </Route>
      <Route component={NotFound} />
    </Switch>
  );
}

function App() {
  return (
    <QueryClientProvider client={queryClient}>
      <AuthProvider>
        <TooltipProvider>
          <div className="min-h-screen bg-background text-foreground">
            <HeaderCodesInjector />
            <Toaster />
            <Router />
          </div>
        </TooltipProvider>
      </AuthProvider>
    </QueryClientProvider>
  );
}

export default App;
