import { useState } from "react";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { AlertTriangle, Mail, X } from "lucide-react";
import { useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { useAuth } from "@/hooks/use-auth";

export default function AnonymousWarningBanner() {
  const { user } = useAuth();
  const [showEmailDialog, setShowEmailDialog] = useState(false);
  const [email, setEmail] = useState("");
  const [dismissed, setDismissed] = useState(false);
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const updateEmailMutation = useMutation({
    mutationFn: async (email: string) => {
      const response = await apiRequest("/api/update-email", "PATCH", { email });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to update email");
      }
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Email Updated",
        description: "Your email has been added for notifications and password recovery.",
        variant: "default"
      });
      queryClient.invalidateQueries({ queryKey: ["/api/me"] });
      setShowEmailDialog(false);
      setDismissed(true);
    },
    onError: (error: any) => {
      toast({
        title: "Update Failed",
        description: error.message,
        variant: "destructive"
      });
    }
  });

  const handleUpdateEmail = () => {
    if (!email || !email.includes("@")) {
      toast({
        title: "Invalid Email",
        description: "Please enter a valid email address",
        variant: "destructive"
      });
      return;
    }
    updateEmailMutation.mutate(email);
  };

  // Don't show banner if user is not anonymous, has email, or banner was dismissed
  if (!user || user.accountType !== "anonymous" || user.email || dismissed) {
    return null;
  }

  return (
    <>
      <Alert className="border-red-200 bg-red-50 dark:bg-red-900/20 mb-4">
        <AlertTriangle className="h-4 w-4 text-red-600" />
        <AlertDescription className="flex items-center justify-between text-red-800 dark:text-red-200">
          <div className="flex items-center gap-2">
            <span className="font-medium">Anonymous Account Warning:</span>
            <span>Add your email for notifications and password recovery</span>
          </div>
          <div className="flex items-center gap-2">
            <Button
              size="sm"
              variant="outline"
              className="border-red-300 text-red-700 hover:bg-red-100 dark:border-red-700 dark:text-red-300"
              onClick={() => setShowEmailDialog(true)}
            >
              <Mail className="h-3 w-3 mr-1" />
              Add Email
            </Button>
            <Button
              size="sm"
              variant="ghost"
              className="text-red-600 hover:bg-red-100 dark:text-red-400"
              onClick={() => setDismissed(true)}
            >
              <X className="h-3 w-3" />
            </Button>
          </div>
        </AlertDescription>
      </Alert>

      <Dialog open={showEmailDialog} onOpenChange={setShowEmailDialog}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Add Email Address</DialogTitle>
            <DialogDescription>
              Adding an email enables notifications about channel changes and password recovery for your account.
            </DialogDescription>
          </DialogHeader>
          
          <div className="space-y-4">
            <div>
              <Label htmlFor="email">Email Address</Label>
              <Input
                id="email"
                type="email"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
                placeholder="your@email.com"
              />
            </div>
            
            <div className="flex gap-2">
              <Button 
                onClick={handleUpdateEmail}
                disabled={updateEmailMutation.isPending}
                className="flex-1"
              >
                {updateEmailMutation.isPending ? "Updating..." : "Update Email"}
              </Button>
              <Button 
                variant="outline" 
                onClick={() => setShowEmailDialog(false)}
                className="flex-1"
              >
                Skip
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </>
  );
}