import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { getQueryFn, apiRequest } from "@/lib/queryClient";
import { formatNumber } from "@/lib/utils";
import { Youtube, Users, Eye, TrendingUp, Trash2 } from "lucide-react";
import { useAuth } from "@/hooks/use-auth";

interface Channel {
  id: number;
  channelId: string;
  channelName: string;
  totalViews: number;
  subscribers: number;
  agScore: number;
  isActive: boolean;
}

interface ChannelDropdownProps {
  selectedChannelId?: number;
  onChannelChange: (channelId: number) => void;
}

export function ChannelDropdown({ selectedChannelId, onChannelChange }: ChannelDropdownProps) {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const { user } = useAuth();
  const [removeDialogOpen, setRemoveDialogOpen] = useState(false);
  const [channelToRemove, setChannelToRemove] = useState<Channel | null>(null);

  const { data: channels = [], isLoading } = useQuery({
    queryKey: ["/api/channels"],
    queryFn: getQueryFn({ on401: "throw" }),
  });

  const { mutate: setActiveChannel } = useMutation({
    mutationFn: async (channelId: number) => {
      const response = await apiRequest("/api/set-active-channel", "POST", { channelId });
      return response;
    },
    onSuccess: (data) => {
      toast({
        title: "Channel Selected",
        description: `Now managing ${data.channel.channelName}`,
        variant: "default",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      onChannelChange(data.channel.id);
    },
    onError: (error: any) => {
      toast({
        title: "Failed to Switch Channel",
        description: error.message || "Unable to switch channels",
        variant: "destructive",
      });
    },
  });

  const { mutate: removeChannel, isPending: isRemoving } = useMutation({
    mutationFn: async (channelId: number) => {
      const isAdmin = user?.role === 'admin';
      const endpoint = isAdmin ? `/api/admin/remove-channel/${channelId}` : `/api/remove-channel/${channelId}`;
      const response = await apiRequest(endpoint, "DELETE");
      return response;
    },
    onSuccess: (data) => {
      const isAdmin = user?.role === 'admin';
      toast({
        title: "Channel Removed",
        description: isAdmin 
          ? `${channelToRemove?.channelName} has been completely removed from the system`
          : `${channelToRemove?.channelName} has been removed from your profile`,
        variant: "default",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/channels"] });
      queryClient.invalidateQueries({ queryKey: ["/api/channel"] });
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      setRemoveDialogOpen(false);
      setChannelToRemove(null);
      
      // If removed channel was active, clear selection
      if (channelToRemove?.id === selectedChannelId) {
        onChannelChange(0);
      }
    },
    onError: (error: any) => {
      toast({
        title: "Failed to Remove Channel",
        description: error.message || "Unable to remove channel",
        variant: "destructive",
      });
    },
  });

  const handleChannelSelect = (channelId: string) => {
    const id = parseInt(channelId);
    setActiveChannel(id);
  };

  const selectedChannel = (channels as Channel[]).find((c: Channel) => c.id === selectedChannelId);

  if (isLoading) {
    return (
      <Card className="mb-6">
        <CardContent className="p-4">
          <div className="animate-pulse flex space-x-4">
            <div className="rounded-full bg-gray-300 h-10 w-10"></div>
            <div className="flex-1 space-y-2 py-1">
              <div className="h-4 bg-gray-300 rounded w-3/4"></div>
              <div className="h-3 bg-gray-300 rounded w-1/2"></div>
            </div>
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card className="mb-6 bg-white border border-gray-200">
      <CardHeader className="pb-3">
        <CardTitle className="flex items-center space-x-2">
          <Youtube className="w-5 h-5 text-red-600" />
          <span className="text-gray-900">Channel Management</span>
        </CardTitle>
      </CardHeader>
      <CardContent className="pt-0">
        {(channels as Channel[]).length > 0 ? (
          <div className="space-y-4">
            <div className="flex items-center space-x-4">
              <div className="flex-1">
                <Select
                  value={selectedChannelId?.toString()}
                  onValueChange={handleChannelSelect}
                >
                  <SelectTrigger className="w-full">
                    <SelectValue placeholder="Select a channel to manage" />
                  </SelectTrigger>
                  <SelectContent>
                    {(channels as Channel[]).map((channel: Channel) => (
                      <SelectItem key={channel.id} value={channel.id.toString()}>
                        <div className="flex items-center justify-between w-full">
                          <div className="flex items-center space-x-3">
                            <div className="w-8 h-8 bg-red-100 rounded-full flex items-center justify-center">
                              <Youtube className="w-4 h-4 text-red-600" />
                            </div>
                            <div>
                              <div className="font-medium text-gray-900">{channel.channelName}</div>
                              <div className="text-xs text-gray-500">
                                {formatNumber(channel.subscribers)} subscribers
                              </div>
                            </div>
                          </div>
                          <Button
                            variant="ghost"
                            size="sm"
                            className="h-6 w-6 p-0 hover:bg-red-100 hover:text-red-600"
                            onClick={(e) => {
                              e.preventDefault();
                              e.stopPropagation();
                              setChannelToRemove(channel);
                              setRemoveDialogOpen(true);
                            }}
                          >
                            <Trash2 className="w-3 h-3" />
                          </Button>
                        </div>
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
            
            {selectedChannel && (
              <div className="bg-gray-50 rounded-lg p-4 border">
                <div className="flex items-center justify-between mb-3">
                  <h3 className="font-semibold text-gray-900">{selectedChannel.channelName}</h3>
                  <Badge variant="secondary" className="bg-green-100 text-green-800">
                    Active
                  </Badge>
                </div>
                <div className="grid grid-cols-3 gap-4">
                  <div className="text-center">
                    <div className="flex items-center justify-center mb-1">
                      <Eye className="w-4 h-4 text-blue-600 mr-1" />
                    </div>
                    <div className="text-sm font-medium text-gray-900">
                      {formatNumber(selectedChannel.totalViews)}
                    </div>
                    <div className="text-xs text-gray-500">Total Views</div>
                  </div>
                  <div className="text-center">
                    <div className="flex items-center justify-center mb-1">
                      <Users className="w-4 h-4 text-green-600 mr-1" />
                    </div>
                    <div className="text-sm font-medium text-gray-900">
                      {formatNumber(selectedChannel.subscribers)}
                    </div>
                    <div className="text-xs text-gray-500">Subscribers</div>
                  </div>
                  <div className="text-center">
                    <div className="flex items-center justify-center mb-1">
                      <TrendingUp className="w-4 h-4 text-purple-600 mr-1" />
                    </div>
                    <div className="text-sm font-medium text-gray-900">
                      {selectedChannel.agScore.toFixed(1)}
                    </div>
                    <div className="text-xs text-gray-500">AG Score</div>
                  </div>
                </div>
              </div>
            )}
          </div>
        ) : (
          <div className="text-center py-8">
            <Youtube className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No Channels Added</h3>
            <p className="text-gray-500 mb-4">
              Add your first YouTube channel to start managing and optimizing content
            </p>
          </div>
        )}
      </CardContent>

      {/* Channel Removal Confirmation Dialog */}
      <Dialog open={removeDialogOpen} onOpenChange={setRemoveDialogOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Remove Channel</DialogTitle>
            <DialogDescription>
              {user?.role === 'admin' ? (
                <>
                  <strong>Admin Action:</strong> This will completely remove "{channelToRemove?.channelName}" from the entire system, including all associated data, orders, and remove it from all user profiles. This action cannot be undone.
                </>
              ) : (
                <>
                  This will remove "{channelToRemove?.channelName}" from your profile only. The channel will remain available for other users and you can re-add it later if needed.
                </>
              )}
            </DialogDescription>
          </DialogHeader>
          <div className="flex space-x-3 mt-4">
            <Button
              variant="outline"
              onClick={() => {
                setRemoveDialogOpen(false);
                setChannelToRemove(null);
              }}
              className="flex-1"
            >
              Cancel
            </Button>
            <Button
              variant="destructive"
              onClick={() => {
                if (channelToRemove) {
                  removeChannel(channelToRemove.id);
                }
              }}
              disabled={isRemoving}
              className="flex-1"
            >
              {isRemoving ? (
                <div className="flex items-center space-x-2">
                  <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                  <span>Removing...</span>
                </div>
              ) : (
                user?.role === 'admin' ? 'Delete from System' : 'Remove from Profile'
              )}
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </Card>
  );
}