import { useState, useEffect } from "react";
import { Link, useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { useAuth } from "@/hooks/use-auth";
import { useToast } from "@/hooks/use-toast";
import { useMobile } from "@/hooks/use-mobile";
import { 
  Youtube, 
  LayoutDashboard, 
  Video, 
  BarChart3, 
  Package, 
  Settings, 
  Users, 
  Coins, 
  AlignJustify, 
  Zap,
  Shield,
  LogOut,
  Link as LinkIcon,
  ChevronDown,
  Globe,
  Activity,
  MessageSquare,
  Mail,
  Menu,
  X,
  Code
} from "lucide-react";
import InsufficientPointsModal from "./InsufficientPointsModal";
import { ApiKeyManager } from "./api-key-manager";

interface DashboardLayoutProps {
  children: React.ReactNode;
}

interface PointsData {
  remainingPoints: number;
  totalPoints: number;
  usedPoints: number;
  date: string;
}

interface ChannelData {
  id: number;
  userId: number;
  channelId: string;
  channelName: string;
  channelHandle: string;
  customUrl: string;
  lastReplenishTime?: string;
}

export function DashboardLayout({ children }: DashboardLayoutProps) {
  const [location, setLocation] = useLocation();
  const { user, logout } = useAuth();
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const isMobile = useMobile();
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [showTopupModal, setShowTopupModal] = useState(false);
  const [selectedPlatform, setSelectedPlatform] = useState<string>(
    localStorage.getItem('selectedPlatform') || "youtube"
  );

  // Prevent cross-platform contamination: Block Reddit pages when on YouTube platform
  useEffect(() => {
    const isRedditPage = location.includes('reddit') || location.includes('moderation') || location.includes('modmail') || location.includes('subreddit');
    if (selectedPlatform === 'youtube' && isRedditPage) {
      console.log('🔄 Redirecting to YouTube dashboard - YouTube platform cannot access Reddit pages');
      setLocation('/dashboard');
    }
  }, [location, selectedPlatform, setLocation]);

  // Close sidebar when route changes on mobile
  useEffect(() => {
    if (isMobile) {
      setSidebarOpen(false);
    }
  }, [location, isMobile]);

  // Update localStorage when platform changes
  const handlePlatformChange = (platform: string) => {
    setSelectedPlatform(platform);
    localStorage.setItem('selectedPlatform', platform);
    // Force page refresh to update dashboard content
    window.location.reload();
  };

  // Fetch points data
  const { data: pointsData } = useQuery<PointsData>({
    queryKey: ["/api/points"],
    refetchInterval: 30000, // Refresh every 30 seconds
  });

  // Fetch channel data for replenish timing
  const { data: channelData } = useQuery<ChannelData>({
    queryKey: ["/api/channel"],
    refetchInterval: 60000, // Refresh every minute
  });

  // Logout handler
  const handleLogout = () => {
    logout();
    toast({
      title: "Logged out successfully",
      description: "You have been logged out of your account",
      variant: "default"
    });
  };

  // YouTube-specific navigation items
  const youtubeNavigationItems = [
    {
      href: "/dashboard",
      label: "Dashboard",
      icon: <LayoutDashboard className="h-5 w-5" />,
      active: location === "/dashboard"
    },
    {
      href: "/video-library",
      label: "Video Library",
      icon: <Video className="h-5 w-5" />,
      active: location === "/video-library"
    },
    {
      href: "/analytics",
      label: "Analytics",
      icon: <BarChart3 className="h-5 w-5" />,
      active: location === "/analytics"
    }
  ];

  // Reddit-specific navigation items
  const redditNavigationItems = [
    {
      href: "/",
      label: "Home Feed",
      icon: <MessageSquare className="h-5 w-5" />,
      active: location === "/" || location === "/home"
    },
    {
      href: "/dashboard",
      label: "Dashboard",
      icon: <LayoutDashboard className="h-5 w-5" />,
      active: location === "/dashboard"
    },
    {
      href: "/reddit-management",
      label: "Reddit Management",
      icon: <Activity className="h-5 w-5" />,
      active: location === "/reddit-management"
    },
    {
      href: "/subreddit-manager",
      label: "Subreddit Manager",
      icon: <Globe className="h-5 w-5" />,
      active: location === "/subreddit-manager"
    },
    {
      href: "/reddit-moderation",
      label: "AI Moderation",
      icon: <Shield className="h-5 w-5" />,
      active: location === "/reddit-moderation"
    },
    {
      href: "/moderation-log",
      label: "Moderation Log",
      icon: <Shield className="h-5 w-5" />,
      active: location === "/moderation-log"
    },
    {
      href: "/modmail",
      label: "Modmail",
      icon: <Mail className="h-5 w-5" />,
      active: location === "/modmail"
    }
  ];

  // Get platform-specific base navigation
  const baseNavigationItems = selectedPlatform === "reddit" ? redditNavigationItems : youtubeNavigationItems;

  // YouTube admin navigation items
  const youtubeAdminItems = [
    {
      href: "/orders",
      label: "Orders",
      icon: <Package className="h-5 w-5" />,
      active: location === "/orders"
    },
    {
      href: "/api-services",
      label: "API Services",
      icon: <Code className="h-5 w-5" />,
      active: location === "/api-services"
    },
    {
      href: "/system-users",
      label: "System Users",
      icon: <Users className="h-5 w-5" />,
      active: location === "/system-users"
    },
    {
      href: "/system-operations",
      label: "System Operations",
      icon: <Activity className="h-5 w-5" />,
      active: location === "/system-operations"
    },
    {
      href: "/admin/points",
      label: "Point Management",
      icon: <Coins className="h-5 w-5" />,
      active: location === "/admin/points"
    },
    {
      href: "/package-manager",
      label: "Package Manager",
      icon: <Package className="h-5 w-5" />,
      active: location === "/package-manager"
    },
    {
      href: "/admin/security",
      label: "Algorithm Tools",
      icon: <Zap className="h-5 w-5" />,
      active: location === "/admin/security"
    },
    {
      href: "/admin/header-codes",
      label: "Header Codes",
      icon: <Code className="h-5 w-5" />,
      active: location === "/admin/header-codes"
    },
    {
      href: "/ai-configuration",
      label: "AI Configuration",
      icon: <Settings className="h-5 w-5" />,
      active: location === "/ai-configuration"
    },
    {
      href: "/settings",
      label: "Settings",
      icon: <Settings className="h-5 w-5" />,
      active: location === "/settings"
    }
  ];

  // Reddit admin navigation items
  const redditAdminItems = [
    {
      href: "/api-services",
      label: "API Services",
      icon: <Code className="h-5 w-5" />,
      active: location === "/api-services"
    },
    {
      href: "/admin/points",
      label: "Point Management",
      icon: <Coins className="h-5 w-5" />,
      active: location === "/admin/points"
    },
    {
      href: "/settings",
      label: "Settings",
      icon: <Settings className="h-5 w-5" />,
      active: location === "/settings"
    }
  ];

  // Get platform-specific admin navigation
  const adminNavigationItems = selectedPlatform === "reddit" ? redditAdminItems : youtubeAdminItems;

  // Combine navigation items based on user role
  const navigationItems = user?.role === "admin" 
    ? [...baseNavigationItems, ...adminNavigationItems]
    : baseNavigationItems;

  const pointsPercentage = pointsData ? (pointsData.remainingPoints / pointsData.totalPoints) * 100 : 0;
  
  // Calculate next replenish time based on channel's last replenish time
  const getNextReplenishTime = () => {
    if (!channelData?.lastReplenishTime) {
      return "Available now";
    }
    
    const lastReplenish = new Date(channelData.lastReplenishTime).getTime();
    const now = Date.now();
    const cooldownPeriod = 24 * 60 * 60 * 1000; // 24 hours
    const timeLeft = (lastReplenish + cooldownPeriod) - now;
    
    if (timeLeft <= 0) {
      return "Available now";
    }
    
    const hours = Math.floor(timeLeft / (1000 * 60 * 60));
    const minutes = Math.floor((timeLeft % (1000 * 60 * 60)) / (1000 * 60));
    return `${hours}h ${minutes}m`;
  };
  
  const nextReplenish = getNextReplenishTime();

  // Sidebar content component that can be reused
  const SidebarContent = ({ onLinkClick }: { onLinkClick?: () => void }) => (
    <div className="flex flex-col h-full overflow-hidden">
      {/* Header */}
      <div className="p-4 lg:p-6 border-b border-gray-200 dark:border-gray-700 flex-shrink-0">
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center space-x-2 lg:space-x-3">
            {selectedPlatform === "reddit" ? (
              <Globe className="w-6 h-6 lg:w-8 lg:h-8 text-orange-600" />
            ) : (
              <Youtube className="w-6 h-6 lg:w-8 lg:h-8 text-red-600" />
            )}
            <div>
              <h1 className="text-lg lg:text-xl font-bold text-gray-900 dark:text-white">TubeBoost</h1>
              <p className="text-xs lg:text-sm text-gray-600 dark:text-gray-400">
                {selectedPlatform === "reddit" ? "Reddit Growth Management" : "Multi-Platform Growth Management"}
              </p>
            </div>
          </div>
          <div className="flex items-center space-x-2">
            <Button
              variant="ghost"
              size="sm"
              onClick={() => window.open("https://tubeboost.cc", "_blank")}
              className="text-gray-500 hover:text-gray-700"
            >
              <LinkIcon className="h-4 w-4" />
            </Button>
            {isMobile && (
              <Button
                variant="ghost"
                size="sm"
                onClick={onLinkClick}
                className="text-gray-500 hover:text-gray-700"
              >
                <X className="h-4 w-4" />
              </Button>
            )}
          </div>
        </div>

        {/* Platform Selector */}
        <div className="space-y-2">
          <label className="text-sm font-medium text-gray-700 dark:text-gray-300">Platform</label>
          <Select value={selectedPlatform} onValueChange={handlePlatformChange}>
            <SelectTrigger className="w-full">
              <div className="flex items-center space-x-2">
                {selectedPlatform === "youtube" ? (
                  <Youtube className="w-4 h-4 text-red-600" />
                ) : (
                  <Globe className="w-4 h-4 text-orange-600" />
                )}
                <SelectValue />
              </div>
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="youtube">
                <div className="flex items-center space-x-2">
                  <Youtube className="w-4 h-4 text-red-600" />
                  <span>YouTube</span>
                </div>
              </SelectItem>
              <SelectItem value="reddit">
                <div className="flex items-center space-x-2">
                  <Globe className="w-4 h-4 text-orange-600" />
                  <span>Reddit</span>
                  <Badge variant="secondary" className="ml-2 text-xs">Beta</Badge>
                </div>
              </SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      {/* Navigation */}
      <nav className="flex-1 p-3 lg:p-4 space-y-1 lg:space-y-2 overflow-y-auto">
        {navigationItems.map((item) => (
          <Link key={item.href} href={item.href}>
            <Button
              variant={item.active ? "default" : "ghost"}
              className={`w-full justify-start text-left text-sm lg:text-base ${
                item.active 
                  ? "bg-blue-600 text-white hover:bg-blue-700" 
                  : "text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700"
              }`}
              onClick={onLinkClick}
            >
              {item.icon}
              <span className="ml-2 lg:ml-3">{item.label}</span>
            </Button>
          </Link>
        ))}
      </nav>

      {/* Points Balance */}
      <div className="p-3 lg:p-4 border-t border-gray-200 dark:border-gray-700 flex-shrink-0">
        <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-3 lg:p-4">
          <div className="flex items-center space-x-2 mb-3">
            <Zap className="h-4 w-4 lg:h-5 lg:w-5 text-blue-600" />
            <span className="font-medium text-sm lg:text-base text-gray-900 dark:text-white">Points Balance</span>
          </div>
          
          {/* Daily Points (Used First) */}
          <div className="mb-3">
            <div className="flex items-center justify-between mb-1">
              <span className="text-xs lg:text-sm text-gray-600 dark:text-gray-400">Daily Points</span>
              <span className="text-sm lg:text-base font-semibold text-blue-600">
                {pointsData?.remainingPoints?.toFixed(1) || "0.0"}
              </span>
            </div>
            <Progress 
              value={pointsPercentage} 
              className="h-1.5 mb-1"
            />
            <div className="text-xs text-gray-500 dark:text-gray-400">
              Next replenish: {nextReplenish}
            </div>
          </div>

          {/* Purchased Points (Used After Daily) */}
          {pointsData?.purchasedPoints > 0 && (
            <>
              <div className="border-t border-gray-200 dark:border-gray-600 pt-3">
                <div className="flex items-center justify-between mb-1">
                  <div className="flex items-center space-x-1">
                    <span className="text-xs lg:text-sm text-gray-600 dark:text-gray-400">Purchased Points</span>
                    <div className="w-2 h-2 bg-yellow-500 rounded-full"></div>
                  </div>
                  <span className="text-sm lg:text-base font-semibold text-yellow-600">
                    {pointsData.purchasedPoints.toLocaleString()}
                  </span>
                </div>
                <div className="text-xs text-yellow-600 dark:text-yellow-400 font-medium">
                  ✓ Never Expires
                </div>
              </div>
            </>
          )}

          {/* Total Available */}
          <div className="border-t border-gray-200 dark:border-gray-600 pt-3 mt-3">
            <div className="flex items-center justify-between mb-3">
              <span className="text-sm lg:text-base font-medium text-gray-900 dark:text-white">Total Available</span>
              <span className="text-lg lg:text-xl font-bold text-gray-900 dark:text-white">
                {((pointsData?.remainingPoints || 0) + (pointsData?.purchasedPoints || 0)).toFixed(1)}
              </span>
            </div>
            
            {/* Prominent Topup Points Button */}
            <Button
              size="sm"
              className="w-full bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white font-medium py-2 px-4 rounded-lg shadow-md hover:shadow-lg transition-all duration-200 transform hover:scale-105"
              onClick={() => setShowTopupModal(true)}
            >
              <Zap className="h-4 w-4 mr-2" />
              Topup Points
            </Button>
          </div>
        </div>
      </div>



      {/* User Section */}
      <div className="p-3 lg:p-4 border-t border-gray-200 dark:border-gray-700">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-2 lg:space-x-3 min-w-0 flex-1">
            <div className="w-6 h-6 lg:w-8 lg:h-8 bg-blue-600 rounded-full flex items-center justify-center flex-shrink-0">
              <span className="text-white text-xs lg:text-sm font-medium">
                {user?.email?.charAt(0).toUpperCase() || "U"}
              </span>
            </div>
            <div className="flex-1 min-w-0">
              <p className="text-xs lg:text-sm font-medium text-gray-900 dark:text-white truncate">
                {user?.email || "User"}
              </p>
              <p className="text-xs text-gray-500 dark:text-gray-400">
                Registered User
              </p>
            </div>
          </div>
          <Button
            variant="ghost"
            size="sm"
            onClick={() => handleLogout()}
            className="text-gray-500 hover:text-gray-700 flex-shrink-0"
          >
            <LogOut className="h-4 w-4" />
          </Button>
        </div>
      </div>
    </div>
  );

  return (
    <div className="flex h-screen bg-gray-50 dark:bg-gray-900 overflow-hidden">
      {/* Desktop Sidebar */}
      {!isMobile && (
        <div className="w-80 bg-white dark:bg-gray-800 border-r border-gray-200 dark:border-gray-700 flex flex-col overflow-hidden">
          <SidebarContent />
        </div>
      )}

      {/* Mobile Header and Sidebar */}
      {isMobile && (
        <>
          {/* Mobile Header */}
          <div className="fixed top-0 left-0 right-0 z-50 bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700 px-4 py-3 flex items-center justify-between shadow-sm">
            <div className="flex items-center space-x-3">
              {selectedPlatform === "reddit" ? (
                <Globe className="w-6 h-6 text-orange-600" />
              ) : (
                <Youtube className="w-6 h-6 text-red-600" />
              )}
              <div>
                <h1 className="text-lg font-bold text-gray-900 dark:text-white">TubeBoost</h1>
                <p className="text-xs text-gray-600 dark:text-gray-400">
                  {selectedPlatform === "reddit" ? "Reddit Management" : "YouTube Management"}
                </p>
              </div>
            </div>
            <div className="flex items-center space-x-2">
              {/* Points Display */}
              <div className="text-right mr-2">
                <p className="text-xs text-gray-500 dark:text-gray-400">Total Points</p>
                <p className="text-sm font-bold text-blue-600">
                  {((pointsData?.remainingPoints || 0) + (pointsData?.purchasedPoints || 0)).toFixed(1)}
                </p>
                {pointsData?.purchasedPoints > 0 && (
                  <div className="flex items-center space-x-1">
                    <div className="w-1.5 h-1.5 bg-yellow-500 rounded-full"></div>
                    <span className="text-xs text-yellow-600">{pointsData.purchasedPoints}</span>
                  </div>
                )}
              </div>
              <Sheet open={sidebarOpen} onOpenChange={setSidebarOpen}>
                <SheetTrigger asChild>
                  <Button variant="outline" size="sm" className="p-2">
                    <Menu className="h-5 w-5" />
                  </Button>
                </SheetTrigger>
                <SheetContent side="left" className="p-0 w-80 max-w-[90vw]">
                  <SidebarContent onLinkClick={() => setSidebarOpen(false)} />
                </SheetContent>
              </Sheet>
            </div>
          </div>
        </>
      )}

      {/* Main Content */}
      <div className={`flex-1 flex flex-col overflow-hidden ${isMobile ? 'pt-20' : ''}`}>
        <div className="flex-1 overflow-y-auto">
          {children}
        </div>
      </div>

      {/* Topup Points Modal */}
      <InsufficientPointsModal
        isOpen={showTopupModal}
        onClose={() => setShowTopupModal(false)}
        currentPoints={((pointsData?.remainingPoints || 0) + (pointsData?.purchasedPoints || 0))}
        requiredPoints={0}
        actionDescription="Purchase additional points"
        onPurchaseSuccess={() => {
          setShowTopupModal(false);
          queryClient.invalidateQueries({ queryKey: ["/api/points"] });
          toast({
            title: "Points purchased successfully",
            description: "Your points balance has been updated",
            variant: "default"
          });
        }}
      />
    </div>
  );
}

export default DashboardLayout;