import { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Youtube, MessageSquare, Shield, Settings } from 'lucide-react';
import { Link } from 'wouter';
import { DashboardLayout } from '@/components/dashboard-layout';
import { RedditAnalyticsDashboard } from '@/components/reddit-analytics-dashboard';
import { RedditConnection } from '@/components/reddit-connection';
import { ChannelSelector } from '@/components/channel-selector';
import { useQuery } from '@tanstack/react-query';

type Platform = 'youtube' | 'reddit';

interface PlatformDashboardProps {
  youtubeContent: React.ReactNode;
}

export function PlatformDashboard({ youtubeContent }: PlatformDashboardProps) {
  const [selectedPlatform, setSelectedPlatform] = useState<Platform>(() => {
    return (localStorage.getItem('selectedPlatform') as Platform) || 'youtube';
  });

  useEffect(() => {
    localStorage.setItem('selectedPlatform', selectedPlatform);
  }, [selectedPlatform]);

  // Fetch Reddit connection status for conditional rendering
  const { data: redditStatus } = useQuery({
    queryKey: ['/api/reddit/status'],
    enabled: selectedPlatform === 'reddit',
  });

  // Fetch moderated subreddits for Reddit dashboard
  const { data: moderatedSubreddits = [] } = useQuery({
    queryKey: ['/api/subreddits/moderated'],
    enabled: selectedPlatform === 'reddit',
  });

  const renderYouTubeDashboard = () => {
    return (
      <DashboardLayout>
        <div className="space-y-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <Youtube className="h-8 w-8 text-red-600" />
              <div>
                <h1 className="text-3xl font-bold">YouTube Growth Platform</h1>
                <p className="text-muted-foreground">Advanced tools for YouTube channel optimization</p>
              </div>
            </div>
            <div className="flex items-center gap-2">
              <Badge variant="secondary" className="bg-red-100 text-red-700">
                YouTube Active
              </Badge>
              <PlatformSelector 
                selectedPlatform={selectedPlatform} 
                onPlatformChange={setSelectedPlatform} 
              />
            </div>
          </div>

          {/* Channel Management Section */}
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Youtube className="h-5 w-5 text-red-600" />
                Channel Management
              </CardTitle>
            </CardHeader>
            <CardContent>
              <ChannelSelector />
            </CardContent>
          </Card>

          {youtubeContent}
        </div>
      </DashboardLayout>
    );
  };

  const renderRedditDashboard = () => {
    return (
      <DashboardLayout>
        <div className="space-y-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3">
              <MessageSquare className="h-8 w-8 text-orange-600" />
              <div>
                <h1 className="text-3xl font-bold">Reddit Management Platform</h1>
                <p className="text-muted-foreground">AI-powered moderation and community management</p>
              </div>
            </div>
            <div className="flex items-center gap-2">
              <Badge variant="secondary" className="bg-orange-100 text-orange-700">
                Reddit Active
              </Badge>
              <PlatformSelector 
                selectedPlatform={selectedPlatform} 
                onPlatformChange={setSelectedPlatform} 
              />
            </div>
          </div>

          {/* Reddit Platform Content */}
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
            {/* Reddit Connection Status */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <MessageSquare className="h-5 w-5 text-orange-600" />
                  Reddit Connection
                </CardTitle>
              </CardHeader>
              <CardContent>
                <RedditConnection />
              </CardContent>
            </Card>

            {/* AI Moderation Tools */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Shield className="h-5 w-5 text-blue-600" />
                  AI Moderation
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <p className="text-sm text-muted-foreground">
                  Advanced AI-powered moderation using Azure Llama 3.2 90B
                </p>
                <div className="space-y-2">
                  <Link href="/reddit-moderation">
                    <Button className="w-full" variant="default">
                      <Shield className="h-4 w-4 mr-2" />
                      AI Reddit Moderation
                    </Button>
                  </Link>
                  <Link href="/moderation-log">
                    <Button className="w-full" variant="outline">
                      <Shield className="h-4 w-4 mr-2" />
                      Moderation Log
                    </Button>
                  </Link>
                  <Link href="/subreddit-manager">
                    <Button className="w-full" variant="outline">
                      <Settings className="h-4 w-4 mr-2" />
                      Subreddit Manager
                    </Button>
                  </Link>
                </div>
              </CardContent>
            </Card>

            {/* Quick Stats */}
            <Card>
              <CardHeader>
                <CardTitle>Platform Statistics</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  <div className="flex justify-between items-center">
                    <span className="text-sm text-muted-foreground">Moderated Subreddits</span>
                    <Badge variant="secondary">{moderatedSubreddits.length}</Badge>
                  </div>
                  <div className="flex justify-between items-center">
                    <span className="text-sm text-muted-foreground">Connection Status</span>
                    <Badge variant={redditStatus?.connected ? "default" : "destructive"}>
                      {redditStatus?.connected ? "Connected" : "Disconnected"}
                    </Badge>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Reddit Analytics Dashboard */}
          {redditStatus?.connected && (
            <Card>
              <CardHeader>
                <CardTitle>Reddit Analytics</CardTitle>
              </CardHeader>
              <CardContent>
                <RedditAnalyticsDashboard />
              </CardContent>
            </Card>
          )}
        </div>
      </DashboardLayout>
    );
  };

  if (selectedPlatform === 'reddit') {
    return renderRedditDashboard();
  }

  return renderYouTubeDashboard();
}

interface PlatformSelectorProps {
  selectedPlatform: Platform;
  onPlatformChange: (platform: Platform) => void;
}

function PlatformSelector({ selectedPlatform, onPlatformChange }: PlatformSelectorProps) {
  return (
    <Select value={selectedPlatform} onValueChange={(value: Platform) => onPlatformChange(value)}>
      <SelectTrigger className="w-48">
        <SelectValue placeholder="Select Platform" />
      </SelectTrigger>
      <SelectContent>
        <SelectItem value="youtube">
          <div className="flex items-center gap-2">
            <Youtube className="h-4 w-4 text-red-600" />
            YouTube Platform
          </div>
        </SelectItem>
        <SelectItem value="reddit">
          <div className="flex items-center gap-2">
            <MessageSquare className="h-4 w-4 text-orange-600" />
            Reddit Platform
          </div>
        </SelectItem>
      </SelectContent>
    </Select>
  );
}