import { useQuery } from "@tanstack/react-query";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { format } from "date-fns";

interface PointTransaction {
  id: number;
  transactionType: string;
  pointsAmount: number;
  description: string;
  relatedOrderType?: string;
  relatedOrderId?: number;
  balanceAfter: number;
  createdAt: string;
}

interface PointTransactionHistoryProps {
  limit?: number;
  className?: string;
}

export function PointTransactionHistory({ limit = 20, className = "" }: PointTransactionHistoryProps) {
  const { data: transactions, isLoading } = useQuery<PointTransaction[]>({
    queryKey: ["point-transactions", limit],
    queryFn: async () => {
      const response = await fetch(`/api/points/history?limit=${limit}`, {
        credentials: "include",
      });
      if (!response.ok) {
        throw new Error("Failed to fetch point transactions");
      }
      return response.json();
    },
    retry: false,
  });

  const getTransactionIcon = (type: string) => {
    switch (type) {
      case 'spend':
        return 'remove_circle';
      case 'refund':
        return 'add_circle';
      case 'daily_reset':
        return 'refresh';
      default:
        return 'history';
    }
  };

  const getTransactionColor = (type: string) => {
    switch (type) {
      case 'spend':
        return 'text-red-600 bg-red-50';
      case 'refund':
        return 'text-green-600 bg-green-50';
      case 'daily_reset':
        return 'text-blue-600 bg-blue-50';
      default:
        return 'text-gray-600 bg-gray-50';
    }
  };

  const getOrderTypeDisplay = (orderType?: string) => {
    switch (orderType) {
      case 'video_views':
        return 'Video Views';
      case 'subscribers':
        return 'Subscribers';
      case 'ag_session':
        return 'AG Session';
      case 'boost':
        return 'Video Boost';
      case 'devalue':
        return 'Devalue Session';
      case 'whatsapp':
        return 'WhatsApp Boost';
      default:
        return orderType || '';
    }
  };

  if (isLoading) {
    return (
      <Card className={className}>
        <CardHeader>
          <CardTitle className="flex items-center space-x-2">
            <span className="material-icons text-purple-600">history</span>
            <span>Point Transaction History</span>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {[...Array(5)].map((_, i) => (
              <div key={i} className="animate-pulse">
                <div className="flex items-center justify-between p-3 bg-gray-100 rounded-lg">
                  <div className="flex items-center space-x-3">
                    <div className="w-8 h-8 bg-gray-200 rounded-full"></div>
                    <div className="space-y-1">
                      <div className="w-32 h-4 bg-gray-200 rounded"></div>
                      <div className="w-24 h-3 bg-gray-200 rounded"></div>
                    </div>
                  </div>
                  <div className="w-16 h-4 bg-gray-200 rounded"></div>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    );
  }

  if (!transactions || transactions.length === 0) {
    return (
      <Card className={className}>
        <CardHeader>
          <CardTitle className="flex items-center space-x-2">
            <span className="material-icons text-purple-600">history</span>
            <span>Point Transaction History</span>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="text-center py-8">
            <span className="material-icons text-gray-400 text-4xl mb-2">history</span>
            <p className="text-gray-500">No point transactions yet</p>
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card className={className}>
      <CardHeader>
        <CardTitle className="flex items-center space-x-2">
          <span className="material-icons text-purple-600">history</span>
          <span>Point Transaction History</span>
        </CardTitle>
      </CardHeader>
      <CardContent>
        <div className="space-y-3 max-h-96 overflow-y-auto">
          {transactions.map((transaction) => (
            <div 
              key={transaction.id} 
              className="flex items-center justify-between p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors"
            >
              <div className="flex items-center space-x-3">
                <div className={`w-8 h-8 rounded-full flex items-center justify-center ${getTransactionColor(transaction.transactionType)}`}>
                  <span className="material-icons text-sm">
                    {getTransactionIcon(transaction.transactionType)}
                  </span>
                </div>
                <div>
                  <p className="font-medium text-gray-900 text-sm">
                    {transaction.description}
                  </p>
                  <div className="flex items-center space-x-2 text-xs text-gray-500">
                    <span>{format(new Date(transaction.createdAt), 'MMM d, h:mm a')}</span>
                    {transaction.relatedOrderType && (
                      <>
                        <span>•</span>
                        <Badge variant="outline" className="text-xs">
                          {getOrderTypeDisplay(transaction.relatedOrderType)}
                        </Badge>
                      </>
                    )}
                  </div>
                </div>
              </div>
              <div className="text-right">
                <div className={`font-semibold text-sm ${
                  transaction.pointsAmount > 0 ? 'text-green-600' : 'text-red-600'
                }`}>
                  {transaction.pointsAmount > 0 ? '+' : ''}{transaction.pointsAmount}
                </div>
                <div className="text-xs text-gray-500">
                  Balance: {transaction.balanceAfter}
                </div>
              </div>
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
}