import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Separator } from "@/components/ui/separator";
import { Zap, Crown, Plus, Coins, TrendingUp } from "lucide-react";
import InsufficientPointsModal from "./InsufficientPointsModal";

interface PointsData {
  remainingPoints: number;
  totalPoints: number;
  usedPoints: number;
  purchasedPoints: number;
  date: string;
}

interface PointsDisplayProps {
  showLabel?: boolean;
}

export function PointsDisplay({ showLabel = true }: PointsDisplayProps) {
  const [showBuyModal, setShowBuyModal] = useState(false);
  const { data: pointsData, isLoading, refetch } = useQuery<PointsData>({
    queryKey: ["/api/points"],
    refetchInterval: 30000, // Refresh every 30 seconds
  });

  if (isLoading) {
    return (
      <Card className="w-fit">
        <CardContent className="py-3 px-4">
          <div className="flex items-center space-x-2">
            <Coins className="w-4 h-4 text-blue-600 animate-pulse" />
            <div className="text-sm font-medium">Loading...</div>
          </div>
        </CardContent>
      </Card>
    );
  }

  if (!pointsData) {
    return null;
  }

  // If showLabel is false, return just the number for compact display
  if (!showLabel) {
    return <span>{pointsData.remainingPoints + (pointsData.purchasedPoints || 0)}</span>;
  }

  const handleBuyPoints = () => {
    setShowBuyModal(true);
  };

  const handlePurchaseSuccess = () => {
    refetch(); // Refresh points display after successful purchase
  };

  const isLowPoints = pointsData.remainingPoints < pointsData.totalPoints * 0.2; // Less than 20%
  const isCriticalPoints = pointsData.remainingPoints < pointsData.totalPoints * 0.1; // Less than 10%

  return (
    <>
      <Card className="w-fit">
        <CardContent className="py-3 px-4 space-y-2">
          {/* Purchased Points (Premium) - Display prominently at top */}
          {pointsData.purchasedPoints > 0 && (
            <>
              <div className="flex items-center justify-between space-x-2">
                <div className="flex items-center space-x-2">
                  <Crown className="h-4 w-4 text-yellow-500" />
                  <div className="text-sm">
                    <span className="font-bold text-yellow-600 dark:text-yellow-400">
                      {pointsData.purchasedPoints.toLocaleString()}
                    </span>
                    <span className="text-muted-foreground text-xs ml-1">purchased</span>
                  </div>
                </div>
                <Badge variant="outline" className="text-xs py-0 px-1.5 bg-yellow-50 text-yellow-700 border-yellow-200">
                  Never Expires
                </Badge>
              </div>
              <Separator />
            </>
          )}
          
          {/* Regular Daily Points */}
          <div className="flex items-center space-x-2">
            <Zap className={`h-4 w-4 ${
              isCriticalPoints 
                ? 'text-red-500'
                : isLowPoints
                  ? 'text-yellow-500'
                  : 'text-blue-500'
            }`} />
            <div className="text-sm">
              <span className={`font-semibold ${
                isCriticalPoints 
                  ? 'text-red-600 dark:text-red-400'
                  : isLowPoints
                    ? 'text-yellow-600 dark:text-yellow-400'
                    : 'text-blue-600 dark:text-blue-400'
              }`}>
                {pointsData.remainingPoints}
              </span>
              <span className="text-muted-foreground text-xs ml-1">daily points</span>
            </div>
            {(isLowPoints || isCriticalPoints) && (
              <Badge variant="outline" className={`text-xs py-0 px-1.5 ${
                isCriticalPoints 
                  ? 'bg-red-50 text-red-700 border-red-200'
                  : 'bg-yellow-50 text-yellow-700 border-yellow-200'
              }`}>
                {isCriticalPoints ? 'Critical' : 'Low'}
              </Badge>
            )}
          </div>
          
          {/* Buy Points Button */}
          <Button
            variant="outline"
            size="sm"
            className="w-full text-xs h-7 hover:bg-primary hover:text-primary-foreground transition-colors"
            onClick={handleBuyPoints}
          >
            <Plus className="h-3 w-3 mr-1" />
            Buy More Points
          </Button>
        </CardContent>
      </Card>

      <InsufficientPointsModal
        isOpen={showBuyModal}
        onClose={() => setShowBuyModal(false)}
        requiredPoints={1000} // Default suggestion
        currentPoints={pointsData.remainingPoints + (pointsData.purchasedPoints || 0)}
        onSuccess={handlePurchaseSuccess}
      />
    </>
  );
}