import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { useAuth } from "@/hooks/use-auth";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { useToast } from "@/hooks/use-toast";
import { Play, Pause, X, Clock, Users, Video, Zap, RefreshCw, MessageSquare } from "lucide-react";
import type { QueueItem } from "@shared/schema";

interface QueuePanelProps {
  className?: string;
}

interface QueueItemWithChannel extends QueueItem {
  channelName?: string;
  videoTitle?: string;
}

export function QueuePanel({ className }: QueuePanelProps) {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const { user } = useAuth();

  // Fetch queue items
  const { data: queueItems = [], isLoading } = useQuery<QueueItemWithChannel[]>({
    queryKey: ["/api/queue"],
    refetchInterval: 2000, // Update every 2 seconds for real-time status
  });

  // Cancel queue item mutation
  const { mutate: cancelQueueItem } = useMutation({
    mutationFn: async (queueItemId: number) => {
      const response = await apiRequest(`/api/queue/${queueItemId}`, "DELETE");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/queue"] });
      toast({
        title: "Queue Item Cancelled",
        description: "The simulation has been removed from the queue.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to cancel queue item",
        variant: "destructive"
      });
    }
  });

  // Pause/Resume queue item mutation
  const { mutate: toggleQueueItem } = useMutation({
    mutationFn: async ({ queueItemId, action }: { queueItemId: number; action: 'pause' | 'resume' }) => {
      const response = await apiRequest(`/api/queue/${queueItemId}/${action}`, "POST");
      return response.json();
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/queue"] });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to update queue item",
        variant: "destructive"
      });
    }
  });

  // Clear all stuck items mutation
  const { mutate: clearAllStuckItems, isPending: isClearingAll } = useMutation({
    mutationFn: async () => {
      const stuckItems = queueItems.filter(item => 
        item.status === 'running' || item.status === 'queued' || item.status === 'paused' || item.status === 'pending'
      );
      
      console.log(`🧹 Starting bulk clear of ${stuckItems.length} stuck items:`, stuckItems.map(item => ({ id: item.id, status: item.status, actionType: item.actionType })));
      
      if (stuckItems.length === 0) {
        console.log('🧹 No stuck items found to clear');
        return 0;
      }
      
      // Delete each stuck item individually using the existing endpoint
      const deletePromises = stuckItems.map(async (item, index) => {
        try {
          console.log(`🧹 Deleting item ${index + 1}/${stuckItems.length}: ID ${item.id} (${item.actionType})`);
          await apiRequest(`/api/queue/${item.id}`, "DELETE");
          console.log(`✅ Successfully deleted queue item ${item.id}`);
          return { success: true, itemId: item.id };
        } catch (error) {
          console.error(`❌ Failed to delete queue item ${item.id}:`, error);
          return { success: false, itemId: item.id, error };
        }
      });
      
      const results = await Promise.all(deletePromises);
      const successful = results.filter(r => r.success).length;
      const failed = results.filter(r => !r.success).length;
      
      console.log(`🧹 Bulk clear results: ${successful} successful, ${failed} failed`);
      
      if (failed > 0) {
        const failedItems = results.filter(r => !r.success);
        console.error('🧹 Failed deletions:', failedItems);
        throw new Error(`Failed to delete ${failed} items. ${successful} items were successfully deleted.`);
      }
      
      return successful;
    },
    onSuccess: (clearedCount) => {
      console.log(`🎉 Successfully cleared ${clearedCount} stuck queue items`);
      queryClient.invalidateQueries({ queryKey: ["/api/queue"] });
      toast({
        title: "Stuck Items Cleared",
        description: `Successfully removed ${clearedCount} stuck queue items`,
      });
    },
    onError: (error: any) => {
      console.error('🧹 Error during bulk clear operation:', error);
      toast({
        title: "Error Clearing Queue",
        description: error.message || "Failed to clear some stuck items",
        variant: "destructive"
      });
    }
  });

  const getActionIcon = (actionType: string) => {
    switch (actionType) {
      case 'boost':
        return <Zap className="w-4 h-4" />;
      case 'session':
        return <Video className="w-4 h-4" />;
      case 'replenish':
        return <RefreshCw className="w-4 h-4" />;
      case 'devalue':
        return <Users className="w-4 h-4" />;
      case 'whatsapp':
        return <MessageSquare className="w-4 h-4" />;
      default:
        return <Clock className="w-4 h-4" />;
    }
  };

  const getActionLabel = (actionType: string) => {
    switch (actionType) {
      case 'boost':
        return 'Video Boost';
      case 'session':
        return 'AG Session';
      case 'replenish':
        return 'AG Replenish';
      case 'devalue':
        return 'AG Devalue';
      case 'whatsapp':
        return 'WhatsApp Boost';
      default:
        return actionType;
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'queued':
        return 'bg-gray-100 text-gray-800';
      case 'running':
        return 'bg-blue-100 text-blue-800';
      case 'completed':
        return 'bg-green-100 text-green-800';
      case 'failed':
        return 'bg-red-100 text-red-800';
      case 'cancelled':
        return 'bg-orange-100 text-orange-800';
      case 'paused':
        return 'bg-yellow-100 text-yellow-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const formatTimeRemaining = (estimatedDuration: number, progress: number) => {
    if (!estimatedDuration) return "Unknown";
    const remaining = Math.ceil((estimatedDuration * (100 - progress)) / 100);
    const minutes = Math.floor(remaining / 60);
    const seconds = remaining % 60;
    return `${minutes}:${seconds.toString().padStart(2, '0')}`;
  };

  // Group queue items by channel
  const groupedItems = queueItems.reduce((acc, item) => {
    const channelKey = `${item.channelId}-${item.channelName || 'Unknown Channel'}`;
    if (!acc[channelKey]) {
      acc[channelKey] = [];
    }
    acc[channelKey].push(item);
    return acc;
  }, {} as Record<string, QueueItemWithChannel[]>);

  if (isLoading) {
    return (
      <Card className={className}>
        <CardHeader>
          <CardTitle className="flex items-center space-x-2">
            <Clock className="w-5 h-5" />
            <span>Queue Manager</span>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="text-center text-gray-500">Loading queue...</div>
        </CardContent>
      </Card>
    );
  }

  if (queueItems.length === 0) {
    return (
      <Card className={className}>
        <CardHeader>
          <CardTitle className="flex items-center space-x-2">
            <Clock className="w-5 h-5" />
            <span>Queue Manager</span>
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="text-center text-gray-500">
            <Clock className="w-8 h-8 mx-auto mb-2 opacity-50" />
            <p>No simulations in queue</p>
            <p className="text-sm">Add simulations to manage multiple channels simultaneously</p>
          </div>
        </CardContent>
      </Card>
    );
  }

  // Count stuck items
  const stuckItems = queueItems.filter(item => 
    item.status === 'running' || item.status === 'queued' || item.status === 'paused' || item.status === 'pending'
  );

  return (
    <Card className={className}>
      <CardHeader>
        <CardTitle className="flex items-center justify-between">
          <div className="flex items-center space-x-2">
            <Clock className="w-5 h-5" />
            <span>Queue Manager</span>
            <Badge variant="secondary">{queueItems.length}</Badge>
          </div>
          {stuckItems.length > 0 && (
            <Button
              onClick={() => clearAllStuckItems()}
              disabled={isClearingAll}
              variant="destructive"
              size="sm"
              className="ml-2"
            >
              {isClearingAll ? (
                <>
                  <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                  Clearing...
                </>
              ) : (
                <>
                  <X className="w-4 h-4 mr-2" />
                  Clear {stuckItems.length} Stuck Items
                </>
              )}
            </Button>
          )}
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        {Object.entries(groupedItems).map(([channelKey, items]) => {
          const channelName = channelKey.split('-').slice(1).join('-');
          
          return (
            <div key={channelKey} className="border rounded-lg p-3 space-y-3">
              <div className="flex items-center justify-between">
                <h4 className="font-medium text-sm text-gray-900">{channelName}</h4>
                <Badge variant="outline" className="text-xs">
                  {items.length} {items.length === 1 ? 'item' : 'items'}
                </Badge>
              </div>
              
              <div className="space-y-2">
                {items.map((item) => (
                  <div key={item.id} className="bg-gray-50 rounded-md p-3 space-y-2">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-2">
                        {getActionIcon(item.actionType)}
                        <span className="text-sm font-medium">
                          {getActionLabel(item.actionType)}
                        </span>
                        {item.videoTitle && (
                          <span className="text-xs text-gray-500 truncate max-w-32">
                            • {item.videoTitle}
                          </span>
                        )}
                      </div>
                      
                      <div className="flex items-center space-x-1">
                        <Badge 
                          className={`text-xs ${getStatusColor(item.status)}`}
                          variant="secondary"
                        >
                          {item.status}
                        </Badge>
                        
                        <div className="flex items-center space-x-1">
                          {item.status === 'running' && (
                            <Button
                              size="sm"
                              variant="ghost"
                              className="h-6 w-6 p-0"
                              onClick={() => toggleQueueItem({ queueItemId: item.id, action: 'pause' })}
                            >
                              <Pause className="w-3 h-3" />
                            </Button>
                          )}
                          
                          {item.status === 'paused' && (
                            <Button
                              size="sm"
                              variant="ghost"
                              className="h-6 w-6 p-0"
                              onClick={() => toggleQueueItem({ queueItemId: item.id, action: 'resume' })}
                            >
                              <Play className="w-3 h-3" />
                            </Button>
                          )}
                          
                          {['queued', 'running', 'paused', 'pending'].includes(item.status) && (
                            <Button
                              size="sm"
                              variant="ghost"
                              className="h-6 w-6 p-0 text-red-500 hover:text-red-700"
                              onClick={() => cancelQueueItem(item.id)}
                            >
                              <X className="w-3 h-3" />
                            </Button>
                          )}
                        </div>
                      </div>
                    </div>
                    
                    {item.status === 'running' && (
                      <div className="space-y-1">
                        <div className="flex justify-between text-xs text-gray-600">
                          <span>{item.currentStep || 'Processing...'}</span>
                          <span>
                            {item.estimatedDuration ? 
                              formatTimeRemaining(item.estimatedDuration, item.progress) : 
                              'Unknown'
                            }
                          </span>
                        </div>
                        <Progress value={item.progress} className="h-1" />
                      </div>
                    )}
                    
                    {item.status === 'failed' && item.currentStep && (
                      <div className="text-xs text-red-600">
                        Error: {item.currentStep}
                      </div>
                    )}
                  </div>
                ))}
              </div>
            </div>
          );
        })}
      </CardContent>
    </Card>
  );
}