import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Separator } from "@/components/ui/separator";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { RefreshCw, TrendingUp, Users, MessageSquare, Calendar, ExternalLink, Plus, Search, Shield } from "lucide-react";
import { formatNumber } from "@/lib/utils";
import { useToast } from "@/hooks/use-toast";

interface SubredditAnalytics {
  subreddit: string;
  displayName: string;
  title: string;
  description: string;
  subscribers: number;
  activeUsers: number;
  created: number;
  isNsfw: boolean;
  subredditType: string;
  language: string;
  publicDescription: string;
  weeklyGrowth?: number;
  engagementRate?: number;
  averagePostScore?: number;
  averageCommentCount?: number;
  topKeywords?: string[];
  postingFrequency?: number;
  moderatorCount?: number;
  communityHealth?: 'excellent' | 'good' | 'fair' | 'poor';
}

interface ManagedSubreddit {
  id: number;
  subredditName: string;
  displayName: string;
  createdAt: string;
  subscribers?: number;
  activeUsers?: number;
  isModerator?: boolean;
}

export function RedditAnalyticsDashboard() {
  const [selectedSubreddit, setSelectedSubreddit] = useState<string>("");
  const [refreshing, setRefreshing] = useState(false);
  const [searchDialogOpen, setSearchDialogOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState("");
  const [searchResults, setSearchResults] = useState<any[]>([]);
  const [searchLoading, setSearchLoading] = useState(false);
  const queryClient = useQueryClient();
  const { toast } = useToast();

  // Fetch managed subreddits with duplicate prevention
  const { data: managedSubreddits = [], isLoading: loadingManaged, refetch: refetchManaged } = useQuery({
    queryKey: ['/api/subreddits/managed'],
    queryFn: async () => {
      const subreddits = await apiRequest("/api/subreddits/managed", "GET") as ManagedSubreddit[];
      
      // Remove duplicates based on subredditName
      const uniqueSubreddits = subreddits.filter((subreddit, index, self) => 
        index === self.findIndex(s => s.subredditName === subreddit.subredditName)
      );
      
      return uniqueSubreddits;
    },
  });

  // Fetch analytics for selected subreddit
  const { data: analytics, isLoading: loadingAnalytics, refetch, error: analyticsError } = useQuery({
    queryKey: ["subreddit-analytics", selectedSubreddit],
    queryFn: async () => {
      if (!selectedSubreddit) return null;
      
      try {
        const response = await apiRequest(`/api/subreddits/analytics/${selectedSubreddit}`, "GET");
        return response as SubredditAnalytics;
      } catch (error) {
        console.error(`[REDDIT ANALYTICS] Error fetching analytics:`, error);
        throw error;
      }
    },
    enabled: !!selectedSubreddit,
    refetchInterval: 5 * 60 * 1000, // Refresh every 5 minutes
  });

  // Auto-select first managed subreddit if available
  useEffect(() => {
    if (managedSubreddits.length > 0 && !selectedSubreddit) {
      setSelectedSubreddit(managedSubreddits[0].subredditName);
    }
  }, [managedSubreddits, selectedSubreddit]);

  const handleRefresh = async () => {
    setRefreshing(true);
    await refetch();
    setRefreshing(false);
  };

  const searchMutation = useMutation({
    mutationFn: async (query: string) => {
      const response = await apiRequest(`/api/subreddits/search?q=${encodeURIComponent(query)}`, 'GET');
      return await response.json();
    },
    onSuccess: (data) => {
      setSearchResults(data || []);
    },
    onError: (error) => {
      console.error('Search error:', error);
      toast({
        title: "Search Failed",
        description: "Unable to search subreddits. Please try again.",
        variant: "destructive",
      });
    }
  });

  const addSubredditMutation = useMutation({
    mutationFn: async (subredditName: string) => {
      return await apiRequest('/api/subreddits/manage', 'POST', { subredditName });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["managed-subreddits"] });
      toast({
        title: "Success",
        description: "Subreddit added to your managed list!",
      });
      setSearchDialogOpen(false);
      setSearchQuery("");
      setSearchResults([]);
    },
    onError: (error: any) => {
      const message = error.message?.includes('409') 
        ? "Subreddit is already in your managed list"
        : "Failed to add subreddit. Please try again.";
      toast({
        title: "Error",
        description: message,
        variant: "destructive",
      });
    }
  });

  const handleSearch = async () => {
    if (!searchQuery.trim()) return;
    setSearchLoading(true);
    try {
      await searchMutation.mutateAsync(searchQuery.trim());
    } finally {
      setSearchLoading(false);
    }
  };

  const handleAddSubreddit = (subredditName: string) => {
    addSubredditMutation.mutate(subredditName);
  };

  const formatDate = (timestamp: number) => {
    return new Date(timestamp * 1000).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'long',
      day: 'numeric'
    });
  };

  const getHealthColor = (health?: string) => {
    switch (health) {
      case 'excellent': return 'text-green-600 bg-green-100';
      case 'good': return 'text-blue-600 bg-blue-100';
      case 'fair': return 'text-yellow-600 bg-yellow-100';
      case 'poor': return 'text-red-600 bg-red-100';
      default: return 'text-gray-600 bg-gray-100';
    }
  };

  if (loadingManaged) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingUp className="w-5 h-5" />
            Reddit Analytics
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex items-center justify-center py-8">
            <RefreshCw className="w-6 h-6 animate-spin text-gray-400" />
            <span className="ml-2 text-gray-600">Loading managed subreddits...</span>
          </div>
        </CardContent>
      </Card>
    );
  }

  // Add manual refresh for debugging
  const handleForceRefresh = async () => {
    await refetchManaged();
  };

  if (managedSubreddits.length === 0) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingUp className="w-5 h-5" />
            Reddit Analytics
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="text-center py-8">
            <MessageSquare className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No Managed Subreddits</h3>
            <p className="text-gray-600 mb-4">
              Add subreddits to your management list to view detailed analytics here.
            </p>
            <div className="flex gap-2 justify-center">
              <Button variant="outline" asChild>
                <a href="/subreddit-manager">
                  <ExternalLink className="w-4 h-4 mr-2" />
                  Manage Subreddits
                </a>
              </Button>
              <Button variant="outline" onClick={handleForceRefresh}>
                <RefreshCw className="w-4 h-4 mr-2" />
                Refresh
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <div className="space-y-6">
      {/* Subreddit Selector */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center gap-2">
              <TrendingUp className="w-5 h-5" />
              Reddit Analytics Dashboard
            </CardTitle>
            <div className="flex items-center gap-2">
              <Dialog open={searchDialogOpen} onOpenChange={setSearchDialogOpen}>
                <DialogTrigger asChild>
                  <Button variant="outline" size="sm">
                    <Plus className="w-4 h-4 mr-2" />
                    Add Subreddit to Manage
                  </Button>
                </DialogTrigger>
                <DialogContent className="sm:max-w-md">
                  <DialogHeader>
                    <DialogTitle>Add Subreddit to Manage</DialogTitle>
                  </DialogHeader>
                  <div className="space-y-4">
                    <div className="flex items-center space-x-2">
                      <Input
                        placeholder="Enter subreddit name (e.g., programming, javascript)"
                        value={searchQuery}
                        onChange={(e) => setSearchQuery(e.target.value)}
                        onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
                      />
                      <Button 
                        onClick={handleSearch}
                        disabled={searchLoading || !searchQuery.trim()}
                        size="sm"
                      >
                        <Search className={`w-4 h-4 ${searchLoading ? 'animate-spin' : ''}`} />
                      </Button>
                    </div>
                    
                    {searchResults.length > 0 && (
                      <div className="space-y-2 max-h-96 overflow-y-auto">
                        <h4 className="text-sm font-medium text-gray-700">Search Results:</h4>
                        {searchResults.map((subreddit: any) => (
                          <div 
                            key={subreddit.display_name} 
                            className="flex items-center justify-between p-3 border rounded-lg hover:bg-gray-50"
                          >
                            <div className="flex-1">
                              <div className="font-medium">{subreddit.display_name_prefixed}</div>
                              <div className="text-sm text-gray-600">{subreddit.title}</div>
                              <div className="text-xs text-gray-500">
                                {formatNumber(subreddit.subscribers)} subscribers
                              </div>
                            </div>
                            <Button
                              onClick={() => handleAddSubreddit(subreddit.display_name)}
                              disabled={addSubredditMutation.isPending}
                              size="sm"
                              variant="outline"
                            >
                              Add
                            </Button>
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                </DialogContent>
              </Dialog>
              <Button
                variant="outline"
                size="sm"
                onClick={handleRefresh}
                disabled={refreshing || !selectedSubreddit}
              >
                <RefreshCw className={`w-4 h-4 mr-2 ${refreshing ? 'animate-spin' : ''}`} />
                Refresh
              </Button>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <div className="flex items-center gap-4">
            <div className="flex-1">
              <label className="text-sm font-medium text-gray-700 mb-2 block">
                Select Subreddit
              </label>
              <Select value={selectedSubreddit} onValueChange={setSelectedSubreddit}>
                <SelectTrigger>
                  <SelectValue placeholder="Choose a managed subreddit..." />
                </SelectTrigger>
                <SelectContent>
                  {managedSubreddits.map((subreddit) => (
                    <SelectItem key={subreddit.id} value={subreddit.subredditName}>
                      <div className="flex items-center gap-2">
                        <span className="font-medium">{subreddit.displayName}</span>
                        {subreddit.isModerator && (
                          <Badge variant="default" className="text-xs bg-blue-600 hover:bg-blue-700">
                            <Shield className="w-3 h-3 mr-1" />
                            Mod
                          </Badge>
                        )}
                        <span className="text-sm text-gray-500">
                          Added {subreddit.createdAt ? new Date(subreddit.createdAt).toLocaleDateString() : 'Unknown'}
                        </span>
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Analytics Display */}
      {selectedSubreddit && (
        <>
          {loadingAnalytics ? (
            <Card>
              <CardContent className="pt-6">
                <div className="flex items-center justify-center py-8">
                  <RefreshCw className="w-6 h-6 animate-spin text-gray-400" />
                  <span className="ml-2 text-gray-600">Loading analytics data...</span>
                </div>
              </CardContent>
            </Card>
          ) : analytics ? (
            <>
              {/* Overview Stats */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                <Card>
                  <CardContent className="pt-6">
                    <div className="flex items-center">
                      <Users className="w-8 h-8 text-blue-600" />
                      <div className="ml-4">
                        <p className="text-2xl font-bold">{formatNumber(analytics.subscribers)}</p>
                        <p className="text-xs text-gray-600">Total Subscribers</p>
                      </div>
                    </div>
                  </CardContent>
                </Card>
                
                <Card>
                  <CardContent className="pt-6">
                    <div className="flex items-center">
                      <div className="w-8 h-8 bg-green-100 rounded-full flex items-center justify-center">
                        <div className="w-3 h-3 bg-green-600 rounded-full"></div>
                      </div>
                      <div className="ml-4">
                        <p className="text-2xl font-bold">{formatNumber(analytics.activeUsers)}</p>
                        <p className="text-xs text-gray-600">Active Users</p>
                      </div>
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardContent className="pt-6">
                    <div className="flex items-center">
                      <MessageSquare className="w-8 h-8 text-purple-600" />
                      <div className="ml-4">
                        <p className="text-2xl font-bold">{analytics.averagePostScore || 'N/A'}</p>
                        <p className="text-xs text-gray-600">Avg Post Score</p>
                      </div>
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardContent className="pt-6">
                    <div className="flex items-center">
                      <TrendingUp className="w-8 h-8 text-orange-600" />
                      <div className="ml-4">
                        <p className="text-2xl font-bold">
                          {analytics.engagementRate ? `${analytics.engagementRate}%` : 'N/A'}
                        </p>
                        <p className="text-xs text-gray-600">Engagement Rate</p>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              </div>

              {/* Detailed Information */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                {/* Subreddit Details */}
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <div className="flex items-center gap-2">
                        <span>{analytics.displayName}</span>
                        <Badge variant="outline" className={getHealthColor(analytics.communityHealth)}>
                          {analytics.communityHealth || 'Unknown'}
                        </Badge>
                      </div>
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <div>
                      <h4 className="font-medium text-gray-900 mb-2">About</h4>
                      <p className="text-sm text-gray-600">{analytics.description || 'No description available'}</p>
                    </div>
                    
                    <Separator />
                    
                    <div className="grid grid-cols-2 gap-4 text-sm">
                      <div>
                        <span className="text-gray-600">Type:</span>
                        <p className="font-medium capitalize">{analytics.subredditType}</p>
                      </div>
                      <div>
                        <span className="text-gray-600">Created:</span>
                        <p className="font-medium">{formatDate(analytics.created)}</p>
                      </div>
                      <div>
                        <span className="text-gray-600">Language:</span>
                        <p className="font-medium">{analytics.language || 'English'}</p>
                      </div>
                      <div>
                        <span className="text-gray-600">NSFW:</span>
                        <p className="font-medium">{analytics.isNsfw ? 'Yes' : 'No'}</p>
                      </div>
                    </div>

                    {analytics.topKeywords && analytics.topKeywords.length > 0 && (
                      <>
                        <Separator />
                        <div>
                          <h4 className="font-medium text-gray-900 mb-2">Top Keywords</h4>
                          <div className="flex flex-wrap gap-2">
                            {analytics.topKeywords.slice(0, 6).map((keyword, index) => (
                              <Badge key={index} variant="secondary" className="text-xs">
                                {keyword}
                              </Badge>
                            ))}
                          </div>
                        </div>
                      </>
                    )}
                  </CardContent>
                </Card>

                {/* Growth & Engagement */}
                <Card>
                  <CardHeader>
                    <CardTitle>Growth & Engagement Metrics</CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    {analytics.weeklyGrowth !== undefined && (
                      <div>
                        <div className="flex justify-between items-center mb-2">
                          <span className="text-sm font-medium">Weekly Growth</span>
                          <span className={`text-sm font-bold ${analytics.weeklyGrowth >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                            {analytics.weeklyGrowth >= 0 ? '+' : ''}{analytics.weeklyGrowth}%
                          </span>
                        </div>
                        <Progress value={Math.abs(analytics.weeklyGrowth)} className="h-2" />
                      </div>
                    )}

                    {analytics.engagementRate !== undefined && (
                      <div>
                        <div className="flex justify-between items-center mb-2">
                          <span className="text-sm font-medium">Engagement Rate</span>
                          <span className="text-sm font-bold text-blue-600">{analytics.engagementRate}%</span>
                        </div>
                        <Progress value={analytics.engagementRate} className="h-2" />
                      </div>
                    )}

                    <Separator />

                    <div className="grid grid-cols-1 gap-3 text-sm">
                      {analytics.postingFrequency !== undefined && (
                        <div className="flex justify-between">
                          <span className="text-gray-600">Posts per Day:</span>
                          <span className="font-medium">{analytics.postingFrequency}</span>
                        </div>
                      )}
                      
                      {analytics.averageCommentCount !== undefined && (
                        <div className="flex justify-between">
                          <span className="text-gray-600">Avg Comments:</span>
                          <span className="font-medium">{analytics.averageCommentCount}</span>
                        </div>
                      )}
                      
                      {analytics.moderatorCount !== undefined && (
                        <div className="flex justify-between">
                          <span className="text-gray-600">Moderators:</span>
                          <span className="font-medium">{analytics.moderatorCount}</span>
                        </div>
                      )}
                    </div>

                    <div className="pt-4">
                      <Button variant="outline" size="sm" className="w-full" asChild>
                        <a href={`https://reddit.com/${analytics.displayName}`} target="_blank" rel="noopener noreferrer">
                          <ExternalLink className="w-4 h-4 mr-2" />
                          View on Reddit
                        </a>
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              </div>
            </>
          ) : (
            <Card>
              <CardContent className="pt-6">
                <div className="text-center py-8">
                  <MessageSquare className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                  <h3 className="text-lg font-medium text-gray-900 mb-2">No Analytics Available</h3>
                  <p className="text-gray-600">
                    Unable to fetch analytics for this subreddit. Please try refreshing.
                  </p>
                  <Button
                    variant="outline"
                    onClick={handleRefresh}
                    className="mt-4"
                  >
                    <RefreshCw className="w-4 h-4 mr-2" />
                    Refresh Analytics
                  </Button>
                </div>
              </CardContent>
            </Card>
          )}
        </>
      )}
    </div>
  );
}