import { useState } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Globe, Link as LinkIcon, Unlink, CheckCircle } from "lucide-react";

interface RedditConnectionProps {
  className?: string;
}

interface RedditStatus {
  connected: boolean;
  message: string;
}

export function RedditConnection({ className }: RedditConnectionProps) {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Check Reddit connection status
  const { data: redditStatus, isLoading } = useQuery<RedditStatus>({
    queryKey: ["/api/reddit/status"],
    refetchInterval: 30000, // Check every 30 seconds
  });

  // Disconnect Reddit mutation
  const { mutate: disconnectReddit, isPending: isDisconnecting } = useMutation({
    mutationFn: async () => {
      return await apiRequest("/api/reddit/disconnect", "POST");
    },
    onSuccess: () => {
      toast({
        title: "Reddit Disconnected",
        description: "Your Reddit account has been successfully disconnected.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/reddit/status"] });
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: error.message || "Failed to disconnect Reddit account",
        variant: "destructive",
      });
    },
  });

  const handleConnectReddit = () => {
    // Open Reddit OAuth flow in new tab to bypass Replit Preview window restrictions
    window.open("/api/reddit/auth", "_blank", "noopener,noreferrer");
  };

  const handleDisconnectReddit = () => {
    disconnectReddit();
  };

  return (
    <Card className={className}>
      <CardHeader>
        <CardTitle className="flex items-center space-x-2">
          <Globe className="h-5 w-5 text-orange-600" />
          <span>Reddit Integration</span>
          <Badge variant="secondary" className="text-xs">Beta</Badge>
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        {isLoading ? (
          <div className="flex items-center space-x-2 text-gray-600">
            <div className="w-4 h-4 border-2 border-gray-300 border-t-blue-600 rounded-full animate-spin"></div>
            <span>Checking connection...</span>
          </div>
        ) : redditStatus?.connected ? (
          <div className="space-y-3">
            <div className="flex items-center space-x-2 text-green-600">
              <CheckCircle className="h-4 w-4" />
              <span className="font-medium">Connected to Reddit</span>
            </div>
            <p className="text-sm text-gray-600 dark:text-gray-400">
              Your Reddit account is connected and ready for content management.
            </p>
            <Button
              variant="outline"
              size="sm"
              onClick={handleDisconnectReddit}
              disabled={isDisconnecting}
              className="w-full"
            >
              <Unlink className="h-4 w-4 mr-2" />
              {isDisconnecting ? "Disconnecting..." : "Disconnect Reddit"}
            </Button>
          </div>
        ) : (
          <div className="space-y-3">
            <div className="flex items-center space-x-2 text-gray-600">
              <Globe className="h-4 w-4" />
              <span>Not connected</span>
            </div>
            <p className="text-sm text-gray-600 dark:text-gray-400">
              Connect your Reddit account to access cross-platform content management and analytics.
            </p>
            <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-md p-3 mb-3">
              <p className="text-xs text-blue-700 dark:text-blue-300">
                <strong>Note:</strong> Authentication will open in a new tab. After completing Reddit login, return to this page to see your connected status.
              </p>
            </div>
            <Button
              onClick={handleConnectReddit}
              className="w-full bg-orange-600 hover:bg-orange-700 text-white"
            >
              <LinkIcon className="h-4 w-4 mr-2" />
              Connect Reddit Account
            </Button>
            <div className="text-xs text-gray-500 dark:text-gray-400">
              <p className="font-medium mb-1">Beta Features Include:</p>
              <ul className="space-y-1 list-disc list-inside ml-2">
                <li>Account authentication</li>
                <li>Profile management</li>
                <li>Cross-platform analytics</li>
                <li>Content scheduling (coming soon)</li>
              </ul>
            </div>
          </div>
        )}
      </CardContent>
    </Card>
  );
}