/**
 * Advanced Error Logging System for TubeBoost
 * Provides comprehensive error tracking and debugging capabilities
 */

export interface ErrorLogEntry {
  id: string;
  timestamp: string;
  level: 'error' | 'warning' | 'info' | 'debug';
  message: string;
  stack?: string;
  context?: {
    userId?: string;
    platform?: string;
    page?: string;
    action?: string;
    apiEndpoint?: string;
    httpMethod?: string;
    statusCode?: number;
    userAgent?: string;
    url?: string;
    [key: string]: any;
  };
  tags?: string[];
}

class ErrorLogger {
  private logs: ErrorLogEntry[] = [];
  private maxLogs = 1000;

  /**
   * Log an error with full context
   */
  error(message: string, error?: Error, context?: any): void {
    this.log('error', message, error, context);
  }

  /**
   * Log a warning
   */
  warning(message: string, context?: any): void {
    this.log('warning', message, undefined, context);
  }

  /**
   * Log information
   */
  info(message: string, context?: any): void {
    this.log('info', message, undefined, context);
  }

  /**
   * Log debug information
   */
  debug(message: string, context?: any): void {
    this.log('debug', message, undefined, context);
  }

  /**
   * Core logging method
   */
  private log(level: ErrorLogEntry['level'], message: string, error?: Error, context?: any): void {
    const entry: ErrorLogEntry = {
      id: this.generateId(),
      timestamp: new Date().toISOString(),
      level,
      message,
      stack: error?.stack,
      context: {
        ...this.getSystemContext(),
        ...context
      },
      tags: this.generateTags(level, message, context)
    };

    this.logs.unshift(entry);
    this.trimLogs();

    // Console output for development
    if (import.meta.env.DEV) {
      this.consoleOutput(entry);
    }

    // Send critical errors to backend for monitoring
    if (level === 'error') {
      this.sendToBackend(entry);
    }
  }

  /**
   * Get system context information
   */
  private getSystemContext() {
    return {
      userAgent: navigator.userAgent,
      url: window.location.href,
      platform: localStorage.getItem('selectedPlatform') || 'youtube',
      userId: this.getCurrentUserId(),
      timestamp: Date.now(),
      viewport: `${window.innerWidth}x${window.innerHeight}`,
      timezone: Intl.DateTimeFormat().resolvedOptions().timeZone
    };
  }

  /**
   * Generate unique ID for log entry
   */
  private generateId(): string {
    return `${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
  }

  /**
   * Generate relevant tags for filtering
   */
  private generateTags(level: string, message: string, context?: any): string[] {
    const tags = [level];
    
    if (context?.apiEndpoint) tags.push('api');
    if (context?.platform) tags.push(context.platform);
    if (message.includes('fetch')) tags.push('network');
    if (message.includes('auth')) tags.push('authentication');
    if (context?.statusCode >= 400) tags.push('http-error');
    if (context?.statusCode >= 500) tags.push('server-error');
    
    return tags;
  }

  /**
   * Get current user ID from auth context
   */
  private getCurrentUserId(): string | undefined {
    try {
      // This should be replaced with actual auth context
      const userId = localStorage.getItem('userId');
      return userId || undefined;
    } catch {
      return undefined;
    }
  }

  /**
   * Console output for development
   */
  private consoleOutput(entry: ErrorLogEntry): void {
    const style = this.getConsoleStyle(entry.level);
    console.groupCollapsed(`%c[${entry.level.toUpperCase()}] ${entry.message}`, style);
    console.log('Timestamp:', entry.timestamp);
    console.log('Context:', entry.context);
    if (entry.stack) console.log('Stack:', entry.stack);
    if (entry.tags) console.log('Tags:', entry.tags);
    console.groupEnd();
  }

  /**
   * Get console styling for different log levels
   */
  private getConsoleStyle(level: string): string {
    const styles = {
      error: 'color: #ff4444; font-weight: bold',
      warning: 'color: #ffaa00; font-weight: bold',
      info: 'color: #0066cc; font-weight: bold',
      debug: 'color: #666666'
    };
    return styles[level as keyof typeof styles] || styles.debug;
  }

  /**
   * Send critical errors to backend
   */
  private async sendToBackend(entry: ErrorLogEntry): Promise<void> {
    try {
      await fetch('/api/error-logs', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(entry)
      });
    } catch (error) {
      console.warn('Failed to send error log to backend:', error);
    }
  }

  /**
   * Trim logs to prevent memory issues
   */
  private trimLogs(): void {
    if (this.logs.length > this.maxLogs) {
      this.logs = this.logs.slice(0, this.maxLogs);
    }
  }

  /**
   * Get all logs with optional filtering
   */
  getLogs(filter?: {
    level?: string;
    tag?: string;
    since?: Date;
  }): ErrorLogEntry[] {
    let filteredLogs = [...this.logs];

    if (filter?.level) {
      filteredLogs = filteredLogs.filter(log => log.level === filter.level);
    }

    if (filter?.tag) {
      filteredLogs = filteredLogs.filter(log => log.tags?.includes(filter.tag!));
    }

    if (filter?.since) {
      filteredLogs = filteredLogs.filter(log => 
        new Date(log.timestamp) >= filter.since!
      );
    }

    return filteredLogs;
  }

  /**
   * Clear all logs
   */
  clear(): void {
    this.logs = [];
  }

  /**
   * Export logs for debugging
   */
  export(): string {
    return JSON.stringify(this.logs, null, 2);
  }
}

// Global error logger instance
export const errorLogger = new ErrorLogger();

// Global error handler
window.addEventListener('error', (event) => {
  errorLogger.error('Unhandled JavaScript Error', event.error, {
    filename: event.filename,
    lineno: event.lineno,
    colno: event.colno
  });
});

// Unhandled promise rejection handler
window.addEventListener('unhandledrejection', (event) => {
  errorLogger.error('Unhandled Promise Rejection', event.reason, {
    type: 'promise-rejection'
  });
});

// API request error wrapper with parameter validation
export const logApiError = (endpoint: string, method: string, error: any, context?: any) => {
  // Check for common parameter order mistakes
  const httpMethods = ['GET', 'POST', 'PUT', 'DELETE', 'PATCH'];
  const isEndpointHttpMethod = httpMethods.includes(endpoint.toUpperCase());
  const isMethodUrl = method.startsWith('/api/') || method.startsWith('http');
  
  if (isEndpointHttpMethod && isMethodUrl) {
    errorLogger.error(`API Parameter Order Error: Arguments appear to be swapped`, error, {
      providedEndpoint: endpoint,
      providedMethod: method,
      suggestedFix: `Change apiRequest('${endpoint}', '${method}', data) to apiRequest('${method}', '${endpoint}', data)`,
      apiEndpoint: method, // Correct endpoint
      httpMethod: endpoint, // Correct method
      statusCode: error?.status || error?.response?.status,
      errorType: 'parameter-order-mismatch',
      ...context
    });
    return;
  }
  
  errorLogger.error(`API Request Failed: ${method} ${endpoint}`, error, {
    apiEndpoint: endpoint,
    httpMethod: method,
    statusCode: error?.status || error?.response?.status,
    ...context
  });
};

// Platform validation helper
export const validatePlatformApi = (endpoint: string, platform: string) => {
  const youtubeEndpoints = ['/api/youtube-stats', '/api/videos'];
  const redditEndpoints = ['/api/reddit/status', '/api/subreddits'];
  
  // Allow these endpoints on any platform
  const universalEndpoints = ['/api/channels', '/api/youtube/search'];

  // Skip validation for universal endpoints
  if (universalEndpoints.some(ep => endpoint.includes(ep))) {
    return true;
  }

  if (platform === 'reddit' && youtubeEndpoints.some(ep => endpoint.includes(ep))) {
    if (typeof errorLogger !== 'undefined') {
      errorLogger.warning(`YouTube API called on Reddit platform: ${endpoint}`, {
        platform,
        apiEndpoint: endpoint,
        action: 'platform-mismatch'
      });
    }
    return false;
  }

  if (platform === 'youtube' && redditEndpoints.some(ep => endpoint.includes(ep))) {
    if (typeof errorLogger !== 'undefined') {
      errorLogger.warning(`Reddit API called on YouTube platform: ${endpoint}`, {
        platform,
        apiEndpoint: endpoint,
        action: 'platform-mismatch'
      });
    }
    return false;
  }

  return true;
};