// Advanced device fingerprinting for security and fraud prevention

export interface DeviceFingerprint {
  userAgent: string;
  screenResolution: string;
  timezone: string;
  language: string;
  platform: string;
  webglRenderer?: string;
  canvasFingerprint?: string;
  audioFingerprint?: string;
  macAddress?: string;
  networkInterfaces?: string;
  hardwareFingerprint?: string;
  browserFingerprint?: string;
}

export class DeviceFingerprintCollector {
  private static instance: DeviceFingerprintCollector;
  private fingerprint: DeviceFingerprint | null = null;

  static getInstance(): DeviceFingerprintCollector {
    if (!this.instance) {
      this.instance = new DeviceFingerprintCollector();
    }
    return this.instance;
  }

  async collectFingerprint(): Promise<DeviceFingerprint> {
    if (this.fingerprint) {
      return this.fingerprint;
    }

    this.fingerprint = {
      userAgent: navigator.userAgent,
      screenResolution: `${screen.width}x${screen.height}x${screen.colorDepth}`,
      timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
      language: navigator.language,
      platform: navigator.platform,
      webglRenderer: await this.getWebGLRenderer(),
      canvasFingerprint: this.getCanvasFingerprint(),
      audioFingerprint: await this.getAudioFingerprint(),
      macAddress: await this.getMACAddress(),
      networkInterfaces: await this.getNetworkInterfaces(),
      hardwareFingerprint: this.getHardwareFingerprint(),
      browserFingerprint: this.getBrowserFingerprint(),
    };

    return this.fingerprint;
  }

  private getWebGLRenderer(): string | undefined {
    try {
      const canvas = document.createElement('canvas');
      const gl = (canvas.getContext('webgl') || canvas.getContext('experimental-webgl')) as WebGLRenderingContext | null;
      if (!gl) return undefined;

      const debugInfo = gl.getExtension('WEBGL_debug_renderer_info');
      if (!debugInfo) return undefined;

      const vendor = gl.getParameter(debugInfo.UNMASKED_VENDOR_WEBGL);
      const renderer = gl.getParameter(debugInfo.UNMASKED_RENDERER_WEBGL);
      
      return `${vendor} ${renderer}`;
    } catch (e) {
      return undefined;
    }
  }

  private getCanvasFingerprint(): string {
    try {
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      if (!ctx) return '';

      canvas.width = 200;
      canvas.height = 50;

      // Add text with various properties
      ctx.textBaseline = 'top';
      ctx.font = '14px Arial';
      ctx.fillStyle = '#f60';
      ctx.fillRect(125, 1, 62, 20);
      ctx.fillStyle = '#069';
      ctx.fillText('TubeBoost Security 🔒', 2, 15);
      ctx.fillStyle = 'rgba(102, 204, 0, 0.7)';
      ctx.fillText('Device Fingerprint', 4, 45);

      // Add some geometric shapes
      ctx.globalCompositeOperation = 'multiply';
      ctx.fillStyle = 'rgb(255,0,255)';
      ctx.beginPath();
      ctx.arc(50, 50, 50, 0, Math.PI * 2, true);
      ctx.closePath();
      ctx.fill();

      return canvas.toDataURL();
    } catch (e) {
      return '';
    }
  }

  private async getAudioFingerprint(): Promise<string | undefined> {
    try {
      const audioContext = new (window.AudioContext || (window as any).webkitAudioContext)();
      const oscillator = audioContext.createOscillator();
      const analyser = audioContext.createAnalyser();
      const gainNode = audioContext.createGain();
      const scriptProcessor = audioContext.createScriptProcessor(4096, 1, 1);

      oscillator.type = 'triangle';
      oscillator.frequency.setValueAtTime(10000, audioContext.currentTime);

      gainNode.gain.setValueAtTime(0, audioContext.currentTime);
      oscillator.connect(analyser);
      analyser.connect(scriptProcessor);
      scriptProcessor.connect(gainNode);
      gainNode.connect(audioContext.destination);

      oscillator.start(0);

      return new Promise((resolve) => {
        scriptProcessor.onaudioprocess = (event) => {
          const buffer = event.inputBuffer.getChannelData(0);
          let sum = 0;
          for (let i = 0; i < buffer.length; i++) {
            sum += Math.abs(buffer[i]);
          }
          oscillator.stop();
          scriptProcessor.disconnect();
          resolve(sum.toString());
        };
      });
    } catch (e) {
      return undefined;
    }
  }

  private async getMACAddress(): Promise<string | undefined> {
    try {
      // Try to get MAC address via WebRTC
      const rtc = new RTCPeerConnection({ iceServers: [] });
      rtc.createDataChannel('');
      
      return new Promise((resolve) => {
        const timeout = setTimeout(() => resolve(undefined), 1000);
        
        rtc.onicecandidate = (event) => {
          if (event.candidate) {
            const candidate = event.candidate.candidate;
            const macMatch = candidate.match(/([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})/);
            if (macMatch) {
              clearTimeout(timeout);
              resolve(macMatch[0]);
            }
          }
        };

        rtc.createOffer().then(offer => rtc.setLocalDescription(offer));
      });
    } catch (e) {
      return undefined;
    }
  }

  private async getNetworkInterfaces(): Promise<string | undefined> {
    try {
      const connection = (navigator as any).connection || (navigator as any).mozConnection || (navigator as any).webkitConnection;
      if (!connection) return undefined;

      return JSON.stringify({
        effectiveType: connection.effectiveType,
        downlink: connection.downlink,
        rtt: connection.rtt,
        saveData: connection.saveData,
        type: connection.type
      });
    } catch (e) {
      return undefined;
    }
  }

  private getHardwareFingerprint(): string {
    try {
      const hardware = {
        cores: navigator.hardwareConcurrency,
        memory: (navigator as any).deviceMemory,
        maxTouchPoints: navigator.maxTouchPoints,
        cookieEnabled: navigator.cookieEnabled,
        onLine: navigator.onLine,
        doNotTrack: navigator.doNotTrack,
        oscpu: (navigator as any).oscpu,
        buildID: (navigator as any).buildID,
        product: (navigator as any).product,
        productSub: (navigator as any).productSub,
        vendor: navigator.vendor,
        vendorSub: (navigator as any).vendorSub
      };

      return JSON.stringify(hardware);
    } catch (e) {
      return '';
    }
  }

  private getBrowserFingerprint(): string {
    try {
      const browser = {
        plugins: Array.from(navigator.plugins).map(p => ({
          name: p.name,
          description: p.description,
          filename: p.filename
        })),
        mimeTypes: Array.from(navigator.mimeTypes).map(m => ({
          type: m.type,
          description: m.description,
          suffixes: m.suffixes
        })),
        javaEnabled: navigator.javaEnabled(),
        webdriver: (navigator as any).webdriver,
        permissions: (navigator as any).permissions,
        bluetooth: (navigator as any).bluetooth,
        usb: (navigator as any).usb,
        mediaDevices: !!navigator.mediaDevices,
        geolocation: !!navigator.geolocation,
        storage: {
          localStorage: !!window.localStorage,
          sessionStorage: !!window.sessionStorage,
          indexedDB: !!window.indexedDB
        }
      };

      return JSON.stringify(browser);
    } catch (e) {
      return '';
    }
  }

  // Additional security checks
  static async detectVirtualMachine(): Promise<boolean> {
    try {
      // Check for common VM indicators
      const vmIndicators = [
        'VirtualBox',
        'VMware',
        'QEMU',
        'Parallels',
        'VirtualPC',
        'Hyper-V'
      ];

      const userAgent = navigator.userAgent;
      const webglRenderer = await new DeviceFingerprintCollector().getWebGLRenderer();
      
      return vmIndicators.some(indicator => 
        userAgent.includes(indicator) || 
        (webglRenderer && webglRenderer.includes(indicator))
      );
    } catch (e) {
      return false;
    }
  }

  static detectHeadlessBrowser(): boolean {
    try {
      // Check for headless browser indicators
      return (
        navigator.webdriver === true ||
        (window as any).phantom !== undefined ||
        (window as any)._phantom !== undefined ||
        (window as any).__nightmare !== undefined ||
        (window as any).callPhantom !== undefined ||
        (window as any).__fxdriver_evaluate !== undefined ||
        (window as any).__fxdriver_unwrapped !== undefined ||
        (window as any).__webdriver_evaluate !== undefined ||
        (window as any).__selenium_evaluate !== undefined ||
        (window as any).__webdriver_script_function !== undefined ||
        (window as any).__webdriver_script_func !== undefined ||
        (window as any).__webdriver_script_fn !== undefined ||
        navigator.userAgent.includes('HeadlessChrome') ||
        navigator.userAgent.includes('PhantomJS')
      );
    } catch (e) {
      return false;
    }
  }

  static async getIPAddress(): Promise<string | null> {
    try {
      // Use WebRTC to get local IP address
      const rtc = new RTCPeerConnection({ iceServers: [{ urls: 'stun:stun.l.google.com:19302' }] });
      rtc.createDataChannel('');

      return new Promise((resolve) => {
        const timeout = setTimeout(() => resolve(null), 3000);

        rtc.onicecandidate = (event) => {
          if (event.candidate) {
            const candidate = event.candidate.candidate;
            const ipMatch = candidate.match(/([0-9]{1,3}\.){3}[0-9]{1,3}/);
            if (ipMatch && !ipMatch[0].startsWith('127.') && !ipMatch[0].startsWith('169.254.')) {
              clearTimeout(timeout);
              resolve(ipMatch[0]);
            }
          }
        };

        rtc.createOffer().then(offer => rtc.setLocalDescription(offer));
      });
    } catch (e) {
      return null;
    }
  }
}

// Export singleton instance
export const deviceFingerprint = DeviceFingerprintCollector.getInstance();