import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { getQueryFn } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { formatNumber, formatDuration } from "@/lib/utils";
import { TrendingUp, TrendingDown, Eye, Users, Clock, ThumbsUp, MessageSquare, Share2, Calendar, BarChart3, LogOut, User, DollarSign } from "lucide-react";
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, BarChart, Bar, PieChart, Pie, Cell } from "recharts";
import { Link, useLocation } from "wouter";
import { useAuth } from "@/hooks/use-auth";
import { ChannelDropdown } from "@/components/channel-dropdown";
import { DashboardLayout } from "@/components/dashboard-layout";
import { ErrorBoundary } from "@/components/error-boundary";

interface ChannelAnalytics {
  channelId: string;
  channelName: string;
  totalViews: number;
  totalSubscribers: number;
  totalVideos: number;
  averageViewDuration: number;
  subscriberGrowth: {
    daily: number;
    weekly: number;
    monthly: number;
  };
  viewGrowth: {
    daily: number;
    weekly: number;
    monthly: number;
  };
  topVideos: Array<{
    videoId: string;
    title: string;
    views: number;
    likes: number;
    comments: number;
    duration: string;
    publishedAt: string;
    engagement: number;
  }>;
  dailyStats: Array<{
    date: string;
    views: number;
    subscribers: number;
    estimatedWatchTime: number;
    engagement: number;
  }>;
  demographics: {
    topCountries: Array<{ country: string; percentage: number }>;
    ageGroups: Array<{ age: string; percentage: number }>;
    genderSplit: Array<{ gender: string; percentage: number }>;
  };
  revenueEstimate: {
    daily: number;
    monthly: number;
    yearly: number;
    breakdown: {
      baseRevenue: number;
      subscriberBonus: number;
      totalMonthlyViews: number;
      cpmRate: number;
      hasSubscriberBonus: boolean;
    };
  };
}

export default function Analytics() {
  const [timeRange, setTimeRange] = useState<"7d" | "30d" | "90d">("30d");
  const [selectedMetric, setSelectedMetric] = useState<"views" | "subscribers" | "engagement">("views");
  const [selectedChannelId, setSelectedChannelId] = useState<number | null>(null);
  const [location] = useLocation();
  const { user, logout } = useAuth();

  // Fetch analytics data
  const { data: analytics, isLoading } = useQuery<ChannelAnalytics>({
    queryKey: ["/api/analytics", timeRange, selectedChannelId],
    queryFn: () => {
      const params = new URLSearchParams({ timeRange });
      if (selectedChannelId) {
        params.append('channelId', selectedChannelId.toString());
      }
      return fetch(`/api/analytics?${params}`).then(res => res.json());
    },
    retry: false,
    enabled: !!selectedChannelId
  });

  const { data: channel } = useQuery({
    queryKey: ["/api/channel"],
    retry: false
  });

  if (isLoading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="w-8 h-8 border-4 border-google-blue border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
          <p className="text-gray-600">Loading analytics data...</p>
        </div>
      </div>
    );
  }

  if (!analytics && !isLoading) {
    return (
      <DashboardLayout>
        <div className="p-6 bg-gray-50 dark:bg-gray-900">
          <div className="mb-4 sm:mb-6">
            <h2 className="text-xl sm:text-2xl font-semibold text-gray-900 mb-3 sm:mb-4">Channel Analytics</h2>
            
            {/* Channel Selection */}
            <div className="mb-4 sm:mb-6">
              <ChannelDropdown 
                selectedChannelId={selectedChannelId || undefined} 
                onChannelChange={setSelectedChannelId} 
              />
            </div>
          </div>
          
          {!selectedChannelId && (
            <div className="text-center py-20">
              <BarChart3 className="w-16 h-16 text-gray-400 mx-auto mb-4" />
              <h3 className="text-xl font-semibold text-gray-900 mb-2">Select a Channel</h3>
              <p className="text-gray-600">Choose a channel from the dropdown above to view detailed analytics.</p>
            </div>
          )}
        </div>
      </DashboardLayout>
    );
  }

  const chartData = analytics?.dailyStats?.map(stat => ({
    date: new Date(stat.date).toLocaleDateString('en-US', { month: 'short', day: 'numeric' }),
    views: stat.views,
    subscribers: stat.subscribers,
    engagement: stat.engagement,
    watchTime: stat.estimatedWatchTime
  })) || [];

  const getGrowthColor = (growth: number) => {
    if (growth > 0) return "text-green-600";
    if (growth < 0) return "text-red-600";
    return "text-gray-600";
  };

  const getGrowthIcon = (growth: number) => {
    if (growth > 0) return <TrendingUp className="w-4 h-4 text-green-600" />;
    if (growth < 0) return <TrendingDown className="w-4 h-4 text-red-600" />;
    return null;
  };

  return (
    <DashboardLayout>
      <main className="flex-1 p-6 overflow-auto bg-gray-50">
          <div className="mb-6">
            <div className="flex items-center justify-between mb-4">
              <div>
                <h2 className="text-2xl font-semibold text-gray-900">Channel Analytics</h2>
                <p className="text-sm text-gray-600">{analytics?.channelName || "Select a channel"} • Real-time YouTube Data</p>
              </div>
              <div className="flex items-center space-x-4">
                <Select value={timeRange} onValueChange={(value: any) => setTimeRange(value)}>
                  <SelectTrigger className="w-32">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="7d">Last 7 days</SelectItem>
                    <SelectItem value="30d">Last 30 days</SelectItem>
                    <SelectItem value="90d">Last 90 days</SelectItem>
                  </SelectContent>
                </Select>
                <Badge variant="secondary" className="bg-google-blue text-white">
                  <Calendar className="w-3 h-3 mr-1" />
                  Live Data
                </Badge>
              </div>
            </div>
            
            {/* Channel Selection */}
            <div className="mb-6">
              <ChannelDropdown 
                selectedChannelId={selectedChannelId || undefined} 
                onChannelChange={setSelectedChannelId} 
              />
            </div>
          </div>

          <div className="space-y-6">
            {/* Key Metrics */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
              <Card>
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm font-medium text-gray-600">Total Views</p>
                      <p className="text-2xl font-bold text-gray-900">{formatNumber(analytics?.totalViews || 0)}</p>
                      <div className="flex items-center mt-1">
                        {getGrowthIcon(analytics?.viewGrowth?.daily || 0)}
                        <span className={`text-xs ml-1 ${getGrowthColor(analytics?.viewGrowth?.daily || 0)}`}>
                          {(analytics?.viewGrowth?.daily || 0) > 0 ? '+' : ''}{(analytics?.viewGrowth?.daily || 0).toFixed(1)}% today
                        </span>
                      </div>
                    </div>
                    <div className="w-10 h-10 bg-blue-50 rounded-lg flex items-center justify-center">
                      <Eye className="w-5 h-5 text-google-blue" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm font-medium text-gray-600">Subscribers</p>
                      <p className="text-2xl font-bold text-gray-900">{formatNumber(analytics?.totalSubscribers || 0)}</p>
                      <div className="flex items-center mt-1">
                        {getGrowthIcon(analytics?.subscriberGrowth?.daily || 0)}
                        <span className={`text-xs ml-1 ${getGrowthColor(analytics?.subscriberGrowth?.daily || 0)}`}>
                          {(analytics?.subscriberGrowth?.daily || 0) > 0 ? '+' : ''}{(analytics?.subscriberGrowth?.daily || 0).toFixed(1)}% today
                        </span>
                      </div>
                    </div>
                    <div className="w-10 h-10 bg-green-50 rounded-lg flex items-center justify-center">
                      <Users className="w-5 h-5 text-green-600" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm font-medium text-gray-600">Avg. View Duration</p>
                      <p className="text-2xl font-bold text-gray-900">{formatDuration(analytics?.averageViewDuration || 0)}</p>
                      <p className="text-xs text-gray-500 mt-1">Per video view</p>
                    </div>
                    <div className="w-10 h-10 bg-yellow-50 rounded-lg flex items-center justify-center">
                      <Clock className="w-5 h-5 text-yellow-600" />
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm font-medium text-gray-600">Est. Revenue</p>
                      <p className="text-2xl font-bold text-gray-900">${(analytics?.revenueEstimate?.monthly || 0).toFixed(0)}</p>
                      <p className="text-xs text-gray-500 mt-1">Per month</p>
                      {analytics?.revenueEstimate?.breakdown?.hasSubscriberBonus && (
                        <Badge variant="secondary" className="mt-1 text-xs bg-green-100 text-green-800">
                          1M+ Subscriber Bonus
                        </Badge>
                      )}
                    </div>
                    <div className="w-10 h-10 bg-purple-50 rounded-lg flex items-center justify-center">
                      <DollarSign className="w-5 h-5 text-purple-600" />
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Performance Chart */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center justify-between">
                  <span>Performance Trends</span>
                  <Select value={selectedMetric} onValueChange={(value: any) => setSelectedMetric(value)}>
                    <SelectTrigger className="w-32">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="views">Views</SelectItem>
                      <SelectItem value="subscribers">Subscribers</SelectItem>
                      <SelectItem value="engagement">Engagement</SelectItem>
                    </SelectContent>
                  </Select>
                </CardTitle>
              </CardHeader>
              <CardContent>
                <ErrorBoundary fallback={
                  <div className="h-[300px] flex items-center justify-center text-gray-500">
                    <div className="text-center">
                      <BarChart3 className="w-12 h-12 mx-auto mb-2 text-gray-400" />
                      <p>Chart temporarily unavailable</p>
                    </div>
                  </div>
                }>
                  {chartData && chartData.length > 0 ? (
                    <div style={{ width: '100%', height: '300px' }}>
                      <ResponsiveContainer width="100%" height={300}>
                        <LineChart data={chartData}>
                          <CartesianGrid strokeDasharray="3 3" />
                          <XAxis dataKey="date" />
                          <YAxis />
                          <Tooltip />
                          <Line 
                            type="monotone" 
                            dataKey={selectedMetric} 
                            stroke="#4285F4" 
                            strokeWidth={2}
                            dot={{ fill: "#4285F4" }}
                          />
                        </LineChart>
                      </ResponsiveContainer>
                    </div>
                  ) : (
                    <div className="h-[300px] flex items-center justify-center text-gray-500">
                      <div className="text-center">
                        <BarChart3 className="w-12 h-12 mx-auto mb-2 text-gray-400" />
                        <p>No data available for chart</p>
                      </div>
                    </div>
                  )}
                </ErrorBoundary>
              </CardContent>
            </Card>

            {/* Revenue Breakdown */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <DollarSign className="w-5 h-5 text-purple-600" />
                  <span>Revenue Breakdown (Monthly)</span>
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="space-y-4">
                    <h4 className="text-sm font-semibold text-gray-900 mb-3">Calculation Details</h4>
                    <div className="space-y-3">
                      <div className="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                        <span className="text-sm text-gray-600">Monthly Views</span>
                        <span className="text-sm font-semibold text-gray-900">
                          {formatNumber(analytics?.revenueEstimate?.breakdown?.totalMonthlyViews || 0)}
                        </span>
                      </div>
                      <div className="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                        <span className="text-sm text-gray-600">CPM Rate</span>
                        <span className="text-sm font-semibold text-gray-900">
                          ${(analytics?.revenueEstimate?.breakdown?.cpmRate || 0).toFixed(2)}
                        </span>
                      </div>
                      <div className="flex justify-between items-center p-3 bg-blue-50 rounded-lg">
                        <span className="text-sm text-gray-600">Base Ad Revenue</span>
                        <span className="text-sm font-semibold text-blue-600">
                          ${(analytics?.revenueEstimate?.breakdown?.baseRevenue || 0).toFixed(2)}
                        </span>
                      </div>
                      {analytics?.revenueEstimate?.breakdown?.hasSubscriberBonus && (
                        <div className="flex justify-between items-center p-3 bg-green-50 rounded-lg">
                          <span className="text-sm text-gray-600">1M+ Subscriber Bonus</span>
                          <span className="text-sm font-semibold text-green-600">
                            +${(analytics?.revenueEstimate?.breakdown?.subscriberBonus || 0).toFixed(2)}
                          </span>
                        </div>
                      )}
                      <div className="border-t pt-3">
                        <div className="flex justify-between items-center p-3 bg-purple-50 rounded-lg">
                          <span className="text-sm font-semibold text-gray-900">Total Monthly Revenue</span>
                          <span className="text-lg font-bold text-purple-600">
                            ${(analytics?.revenueEstimate?.monthly || 0).toFixed(2)}
                          </span>
                        </div>
                      </div>
                    </div>
                  </div>
                  
                  <div className="space-y-4">
                    <h4 className="text-sm font-semibold text-gray-900 mb-3">Revenue Projections</h4>
                    <div className="space-y-3">
                      <div className="flex justify-between items-center p-3 bg-green-50 rounded-lg">
                        <span className="text-sm text-gray-600">Daily Average</span>
                        <span className="text-lg font-semibold text-green-600">
                          ${(analytics?.revenueEstimate?.daily || 0).toFixed(2)}
                        </span>
                      </div>
                      <div className="flex justify-between items-center p-3 bg-blue-50 rounded-lg">
                        <span className="text-sm text-gray-600">Monthly Total</span>
                        <span className="text-lg font-semibold text-blue-600">
                          ${(analytics?.revenueEstimate?.monthly || 0).toFixed(0)}
                        </span>
                      </div>
                      <div className="flex justify-between items-center p-3 bg-purple-50 rounded-lg">
                        <span className="text-sm text-gray-600">Yearly Projection</span>
                        <span className="text-lg font-semibold text-purple-600">
                          ${(analytics?.revenueEstimate?.yearly || 0).toFixed(0)}
                        </span>
                      </div>
                    </div>
                    
                    <div className="mt-4 p-3 bg-yellow-50 rounded-lg">
                      <p className="text-xs text-yellow-800">
                        <strong>Note:</strong> Revenue estimates are based on industry-standard CPM rates and actual view data. 
                        Channels with 1M+ subscribers receive an additional $5,250 monthly bonus.
                      </p>
                    </div>
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Top Videos and Growth Metrics */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card>
                <CardHeader>
                  <CardTitle>Top Performing Videos</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {(analytics?.topVideos || []).slice(0, 5).map((video, index) => (
                      <div key={video.videoId} className="flex items-center space-x-3 p-3 bg-gray-50 rounded-lg">
                        <div className="w-8 h-8 bg-google-blue rounded-full flex items-center justify-center text-white text-sm font-semibold">
                          {index + 1}
                        </div>
                        <div className="flex-1 min-w-0">
                          <p className="text-sm font-medium text-gray-900 truncate">{video.title}</p>
                          <div className="flex items-center space-x-4 text-xs text-gray-500 mt-1">
                            <span className="flex items-center">
                              <Eye className="w-3 h-3 mr-1" />
                              {formatNumber(video.views)}
                            </span>
                            <span className="flex items-center">
                              <ThumbsUp className="w-3 h-3 mr-1" />
                              {formatNumber(video.likes)}
                            </span>
                            <span className="flex items-center">
                              <MessageSquare className="w-3 h-3 mr-1" />
                              {formatNumber(video.comments)}
                            </span>
                          </div>
                        </div>
                        <Badge variant="secondary" className="text-xs">
                          {video.engagement.toFixed(1)}% eng.
                        </Badge>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle>Growth Metrics</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-3 gap-4">
                    <div className="text-center">
                      <h4 className="text-sm font-medium text-gray-600 mb-2">Daily</h4>
                      <div className="space-y-2">
                        <div className="flex items-center justify-between">
                          <span className="text-sm text-gray-900">Views</span>
                          <span className={`text-sm font-semibold ${getGrowthColor(analytics?.viewGrowth?.daily || 0)}`}>
                            {(analytics?.viewGrowth?.daily || 0) > 0 ? '+' : ''}{(analytics?.viewGrowth?.daily || 0).toFixed(1)}%
                          </span>
                        </div>
                        <div className="flex items-center justify-between">
                          <span className="text-sm text-gray-900">Subs</span>
                          <span className={`text-sm font-semibold ${getGrowthColor(analytics?.subscriberGrowth?.daily || 0)}`}>
                            {(analytics?.subscriberGrowth?.daily || 0) > 0 ? '+' : ''}{(analytics?.subscriberGrowth?.daily || 0).toFixed(1)}%
                          </span>
                        </div>
                      </div>
                    </div>
                    
                    <div className="text-center">
                      <h4 className="text-sm font-medium text-gray-600 mb-2">Weekly</h4>
                      <div className="space-y-2">
                        <div className="flex items-center justify-between">
                          <span className="text-sm text-gray-900">Views</span>
                          <span className={`text-sm font-semibold ${getGrowthColor(analytics?.viewGrowth?.weekly || 0)}`}>
                            {(analytics?.viewGrowth?.weekly || 0) > 0 ? '+' : ''}{(analytics?.viewGrowth?.weekly || 0).toFixed(1)}%
                          </span>
                        </div>
                        <div className="flex items-center justify-between">
                          <span className="text-sm text-gray-900">Subs</span>
                          <span className={`text-sm font-semibold ${getGrowthColor(analytics?.subscriberGrowth?.weekly || 0)}`}>
                            {(analytics?.subscriberGrowth?.weekly || 0) > 0 ? '+' : ''}{(analytics?.subscriberGrowth?.weekly || 0).toFixed(1)}%
                          </span>
                        </div>
                      </div>
                    </div>
                    
                    <div className="text-center">
                      <h4 className="text-sm font-medium text-gray-600 mb-2">Monthly</h4>
                      <div className="space-y-2">
                        <div className="flex items-center justify-between">
                          <span className="text-sm text-gray-900">Views</span>
                          <span className={`text-sm font-semibold ${getGrowthColor(analytics?.viewGrowth?.monthly || 0)}`}>
                            {(analytics?.viewGrowth?.monthly || 0) > 0 ? '+' : ''}{(analytics?.viewGrowth?.monthly || 0).toFixed(1)}%
                          </span>
                        </div>
                        <div className="flex items-center justify-between">
                          <span className="text-sm text-gray-900">Subs</span>
                          <span className={`text-sm font-semibold ${getGrowthColor(analytics?.subscriberGrowth?.monthly || 0)}`}>
                            {(analytics?.subscriberGrowth?.monthly || 0) > 0 ? '+' : ''}{(analytics?.subscriberGrowth?.monthly || 0).toFixed(1)}%
                          </span>
                        </div>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </div>
          </div>
      </main>
    </DashboardLayout>
  );
}