import { useState, useMemo } from "react";
import { useQuery } from "@tanstack/react-query";
import { Link, useLocation } from "wouter";
import { useAuth } from "@/hooks/use-auth";
import { apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { AlertCircle, RefreshCw, DollarSign, Users, Settings, BarChart3, CheckCircle, XCircle, Search, ChevronLeft, ChevronRight, User, LogOut } from "lucide-react";

interface SMMService {
  service: number;
  name: string;
  type: string;
  category: string;
  rate: string;
  min: string;
  max: string;
  refill: boolean;
  cancel: boolean;
}

export default function ApiPage() {
  const { user, logout } = useAuth();
  const [location] = useLocation();
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [searchTerm, setSearchTerm] = useState("");
  const [showRefillOnly, setShowRefillOnly] = useState(false);
  const [currentPage, setCurrentPage] = useState(1);
  const servicesPerPage = 4;

  const { data: services, isLoading, error, refetch } = useQuery({
    queryKey: ["/api/smm-services"],
    queryFn: async () => {
      const response = await apiRequest("/api/smm-services", "GET");
      return response;
    },
  });

  const handleRefresh = async () => {
    setIsRefreshing(true);
    await refetch();
    setIsRefreshing(false);
  };

  const formatPrice = (rate: string) => {
    const price = parseFloat(rate);
    return `$${price.toFixed(2)}`;
  };

  const getCategoryColor = (category: string) => {
    const colors = [
      "bg-blue-100 text-blue-800",
      "bg-green-100 text-green-800", 
      "bg-purple-100 text-purple-800",
      "bg-orange-100 text-orange-800",
      "bg-pink-100 text-pink-800",
      "bg-indigo-100 text-indigo-800",
    ];
    const hash = category.split('').reduce((a, b) => a + b.charCodeAt(0), 0);
    return colors[hash % colors.length];
  };

  // Filter and sort services based on search term and refill filter
  const filteredServices = useMemo(() => {
    if (!services) return [];
    
    let filtered = services;
    
    // Apply search filter if search term exists
    if (searchTerm.trim()) {
      const searchLower = searchTerm.toLowerCase();
      filtered = filtered.filter((service: SMMService) => 
        service.name.toLowerCase().includes(searchLower) ||
        service.category.toLowerCase().includes(searchLower) ||
        service.type.toLowerCase().includes(searchLower) ||
        service.service.toString().includes(searchLower) ||
        service.rate.toLowerCase().includes(searchLower) ||
        service.min.toLowerCase().includes(searchLower) ||
        service.max.toLowerCase().includes(searchLower)
      );
    }

    // Apply refill filter if enabled
    if (showRefillOnly) {
      filtered = filtered.filter((service: SMMService) => service.refill === true);
    }
    
    // Sort by price (cheapest to most expensive)
    return filtered.sort((a: SMMService, b: SMMService) => {
      const priceA = parseFloat(a.rate);
      const priceB = parseFloat(b.rate);
      return priceA - priceB;
    });
  }, [services, searchTerm, showRefillOnly]);

  // Calculate pagination
  const totalPages = Math.ceil(filteredServices.length / servicesPerPage);
  const startIndex = (currentPage - 1) * servicesPerPage;
  const endIndex = startIndex + servicesPerPage;
  const currentServices = filteredServices.slice(startIndex, endIndex);

  // Reset to page 1 when search changes
  const handleSearchChange = (value: string) => {
    setSearchTerm(value);
    setCurrentPage(1);
  };

  const handlePageChange = (page: number) => {
    setCurrentPage(page);
  };

  return (
    <div className="min-h-screen bg-gray-50 flex">
      {/* Sidebar */}
      <div className="w-64 bg-white shadow-sm border-r border-gray-200 flex flex-col">
        {/* Header */}
        <div className="p-4 border-b border-gray-200">
          <div className="flex items-center space-x-3">
            <svg width="24" height="24" viewBox="0 0 24 24" className="text-google-blue">
              <path fill="currentColor" d="M21.58 7.19c-.23-.86-.91-1.54-1.77-1.77C18.25 5 12 5 12 5s-6.25 0-7.81.42c-.86.23-1.54.91-1.77 1.77C2 8.75 2 12 2 12s0 3.25.42 4.81c.23.86.91 1.54 1.77 1.77C5.75 19 12 19 12 19s6.25 0 7.81-.42c.86-.23 1.54-.91 1.77-1.77C22 15.25 22 12 22 12s0-3.25-.42-4.81zM10 15V9l5.2 3L10 15z"/>
            </svg>
            <div>
              <div className="flex items-center space-x-3">
                <h1 className="text-lg font-medium text-gray-900">YouTube Manager</h1>
                <div className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-semibold bg-gradient-to-r from-google-blue to-google-green text-white shadow-sm">
                  v3.2.7
                </div>
              </div>
              <p className="text-xs text-gray-500">Google Internal Tools</p>
            </div>
          </div>
        </div>

        {/* Navigation */}
        <nav className="flex-1 overflow-y-auto">
          <div className="p-4">
            <div className="space-y-1">
              <Link href="/dashboard">
                <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                  location === "/dashboard" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                }`}>
                  <span className="material-icons text-lg">dashboard</span>
                  <span>Dashboard</span>
                </div>
              </Link>
              <Link href="/video-library">
                <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                  location === "/video-library" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                }`}>
                  <span className="material-icons text-lg">video_library</span>
                  <span>Video Library</span>
                </div>
              </Link>
              <Link href="/analytics">
                <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                  location === "/analytics" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                }`}>
                  <span className="material-icons text-lg">analytics</span>
                  <span>Analytics</span>
                </div>
              </Link>
              {user?.role === "admin" && (
                <>
                  <Link href="/api">
                    <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                      location === "/api" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                    }`}>
                      <span className="material-icons text-lg">api</span>
                      <span>API Services</span>
                    </div>
                  </Link>
                  <Link href="/system-users">
                    <div className={`flex items-center space-x-3 px-3 py-2 rounded-md text-sm font-medium cursor-pointer google-transition ${
                      location === "/system-users" ? "bg-google-blue text-white" : "text-gray-600 hover:bg-gray-100 hover:text-gray-900"
                    }`}>
                      <span className="material-icons text-lg">people</span>
                      <span>System Users</span>
                    </div>
                  </Link>
                  <div className="flex items-center space-x-3 px-3 py-2 rounded-md text-gray-600 hover:bg-gray-100 hover:text-gray-900 text-sm cursor-pointer google-transition">
                    <span className="material-icons text-lg">tune</span>
                    <span>Algorithm Tools</span>
                  </div>
                  <div className="flex items-center space-x-3 px-3 py-2 rounded-md text-gray-600 hover:bg-gray-100 hover:text-gray-900 text-sm cursor-pointer google-transition">
                    <span className="material-icons text-lg">settings</span>
                    <span>Settings</span>
                  </div>
                </>
              )}
            </div>
          </div>
        </nav>

        {/* User Profile */}
        <div className="border-t border-gray-200 p-4">
          <div className="flex items-center space-x-3 mb-3">
            <div className="w-8 h-8 bg-gray-100 rounded-full flex items-center justify-center">
              <User className="w-4 h-4 text-gray-600" />
            </div>
            <div className="flex-1">
              <div className="text-sm font-medium text-gray-900">{user?.username}</div>
              <div className="text-xs text-gray-500">{user?.role === "admin" ? "Administrator" : "Client"}</div>
            </div>
          </div>
          <button
            onClick={logout}
            className="w-full flex items-center space-x-2 px-3 py-2 text-sm text-gray-600 hover:bg-gray-100 hover:text-gray-900 rounded-md google-transition"
          >
            <LogOut className="w-4 h-4" />
            <span>Logout</span>
          </button>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 flex flex-col">
        {/* Header */}
        <header className="bg-white border-b border-gray-200 shadow-sm">
          <div className="max-w-full mx-auto px-6 py-3">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-4">
                <div className="flex items-center space-x-3">
                  <Settings className="w-6 h-6 text-google-blue" />
                  <div>
                    <h1 className="text-xl font-medium text-gray-900">API Services</h1>
                    <p className="text-sm text-gray-500">TubeBoost Integration • Social Media Marketing</p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </header>

        {/* Main Content */}
        <main className="flex-1 p-6 overflow-auto">
          <div className="max-w-7xl mx-auto">
            {/* Page Header */}
            <div className="mb-6">
              <div className="flex items-center justify-between">
                <div>
                  <h2 className="text-2xl font-bold text-gray-900">API Services</h2>
                  <p className="text-gray-600 mt-1">TubeBoost API Integration - Social Media Marketing Services</p>
                </div>
                <Button 
                  onClick={handleRefresh}
                  disabled={isRefreshing}
                  className="flex items-center space-x-2"
                >
                  <RefreshCw className={`w-4 h-4 ${isRefreshing ? 'animate-spin' : ''}`} />
                  <span>Refresh Services</span>
                </Button>
              </div>
            </div>

            {/* Search and Filter Controls */}
            <div className="bg-white rounded-lg border border-gray-200 shadow-sm mb-6">
              <div className="p-6">
                <div className="flex flex-col sm:flex-row gap-4">
                  {/* Search Input */}
                  <div className="flex-1">
                    <div className="relative">
                      <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                      <Input
                        type="text"
                        placeholder="Search services by name, category, type, or service ID..."
                        value={searchTerm}
                        onChange={(e) => {
                          setSearchTerm(e.target.value);
                          setCurrentPage(1); // Reset to first page when searching
                        }}
                        className="pl-10 pr-4 py-2 w-full"
                      />
                    </div>
                  </div>
                  
                  {/* Refill Filter Toggle */}
                  <div className="flex items-center space-x-3">
                    <label className="flex items-center space-x-2 cursor-pointer">
                      <input
                        type="checkbox"
                        checked={showRefillOnly}
                        onChange={(e) => {
                          setShowRefillOnly(e.target.checked);
                          setCurrentPage(1); // Reset to first page when filtering
                        }}
                        className="w-4 h-4 text-google-blue bg-gray-100 border-gray-300 rounded focus:ring-google-blue focus:ring-2"
                      />
                      <span className="text-sm font-medium text-gray-700">Refill Only</span>
                    </label>
                    <div className="text-xs text-gray-500 bg-gray-50 px-2 py-1 rounded">
                      {filteredServices.length} services
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* API Status */}
            <div className="bg-white rounded-lg border border-gray-200 shadow-sm mb-6">
              <div className="p-6 border-b border-gray-200">
                <h3 className="text-lg font-semibold flex items-center space-x-2">
                  <Settings className="w-5 h-5" />
                  <span>API Configuration</span>
                </h3>
              </div>
              <div className="p-6">
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div className="flex items-center space-x-3">
                    <div className="w-2 h-2 bg-green-500 rounded-full"></div>
                    <div>
                      <p className="text-sm font-medium">API Status</p>
                      <p className="text-xs text-gray-600">Connected</p>
                    </div>
                  </div>
                  <div className="flex items-center space-x-3">
                    <DollarSign className="w-4 h-4 text-green-600" />
                    <div>
                      <p className="text-sm font-medium">Endpoint</p>
                      <p className="text-xs text-gray-600">smmcost.com/api/v2</p>
                    </div>
                  </div>
                  <div className="flex items-center space-x-3">
                    <BarChart3 className="w-4 h-4 text-blue-600" />
                    <div>
                      <p className="text-sm font-medium">Total Services</p>
                      <p className="text-xs text-gray-600">{services?.length || 0} Available</p>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Search and Pagination Controls */}
            <div className="bg-white rounded-lg border border-gray-200 shadow-sm mb-6">
              <div className="p-6">
                <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-4 sm:space-y-0">
                  <div className="flex flex-col space-y-2">
                    <div className="relative flex-1 max-w-md">
                      <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                      <Input
                        type="text"
                        placeholder="Search services by name, category, type, rate..."
                        value={searchTerm}
                        onChange={(e) => handleSearchChange(e.target.value)}
                        className="pl-10"
                      />
                    </div>
                    <div className="flex items-center space-x-2 text-xs text-gray-500">
                      <DollarSign className="w-3 h-3" />
                      <span>Sorted by price (lowest to highest)</span>
                    </div>
                  </div>
                  <div className="flex items-center space-x-4 text-sm text-gray-600">
                    <span>
                      Showing {startIndex + 1}-{Math.min(endIndex, filteredServices.length)} of {filteredServices.length} services
                    </span>
                    {totalPages > 1 && (
                      <div className="flex items-center space-x-2">
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => handlePageChange(currentPage - 1)}
                          disabled={currentPage === 1}
                        >
                          <ChevronLeft className="w-4 h-4" />
                        </Button>
                        <span className="px-3 py-1 bg-gray-100 rounded text-sm">
                          {currentPage} of {totalPages}
                        </span>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => handlePageChange(currentPage + 1)}
                          disabled={currentPage === totalPages}
                        >
                          <ChevronRight className="w-4 h-4" />
                        </Button>
                      </div>
                    )}
                  </div>
                </div>
              </div>
            </div>

            {/* Error State */}
            {error && (
              <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                <div className="flex items-center space-x-2">
                  <AlertCircle className="h-5 w-5 text-red-500" />
                  <p className="text-red-700">Failed to load services from TubeBoost API. Please check your connection and try again.</p>
                </div>
              </div>
            )}

            {/* Loading State */}
            {isLoading && (
              <div className="flex items-center justify-center py-12">
                <div className="text-center">
                  <div className="w-8 h-8 border-4 border-google-blue border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
                  <p className="text-gray-600">Loading services from TubeBoost API...</p>
                </div>
              </div>
            )}

            {/* Services List */}
            {currentServices.length > 0 && (
              <div className="bg-white rounded-lg border border-gray-200 shadow-sm">
                <div className="p-6 border-b border-gray-200">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-3">
                      <Users className="w-5 h-5" />
                      <span className="text-lg font-semibold">Services</span>
                      {searchTerm && (
                        <span className="bg-blue-100 text-blue-800 px-2 py-1 rounded-full text-sm">
                          Filtered Results
                        </span>
                      )}
                    </div>
                  </div>
                </div>
                <div className="p-6">
                  <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                    {currentServices.map((service: SMMService) => (
                      <div
                        key={service.service}
                        className="border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow"
                      >
                        <div className="flex justify-between items-start mb-3">
                          <div>
                            <h4 className="font-medium text-gray-900">{service.name}</h4>
                            <p className="text-sm text-gray-600">Service #{service.service}</p>
                            <p className="text-sm text-gray-500 mt-1">{service.category}</p>
                          </div>
                          <span className={`px-2 py-1 rounded-full text-xs font-medium ${getCategoryColor(service.category)}`}>
                            {service.type}
                          </span>
                        </div>
                        
                        <div className="space-y-2 mb-4">
                          <div className="flex justify-between text-sm">
                            <span className="text-gray-600">Rate:</span>
                            <span className="font-medium text-green-600">{formatPrice(service.rate)}</span>
                          </div>
                          <div className="flex justify-between text-sm">
                            <span className="text-gray-600">Min/Max:</span>
                            <span className="font-medium">{service.min} - {service.max}</span>
                          </div>
                        </div>

                        <div className="border-t border-gray-200 pt-3 mb-3"></div>

                        <div className="flex justify-between items-center">
                          <div className="flex space-x-3">
                            <div className="flex items-center space-x-1">
                              {service.refill ? (
                                <CheckCircle className="w-4 h-4 text-green-500" />
                              ) : (
                                <XCircle className="w-4 h-4 text-gray-400" />
                              )}
                              <span className="text-xs text-gray-600">Refill</span>
                            </div>
                            <div className="flex items-center space-x-1">
                              {service.cancel ? (
                                <CheckCircle className="w-4 h-4 text-green-500" />
                              ) : (
                                <XCircle className="w-4 h-4 text-gray-400" />
                              )}
                              <span className="text-xs text-gray-600">Cancel</span>
                            </div>
                          </div>
                          <Button size="sm" variant="outline">
                            Order
                          </Button>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            )}

            {/* Empty State */}
            {services && filteredServices.length === 0 && (
              <div className="text-center py-12">
                <AlertCircle className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">
                  {searchTerm ? "No Matching Services" : "No Services Available"}
                </h3>
                <p className="text-gray-600">
                  {searchTerm 
                    ? `No services match your search for "${searchTerm}". Try adjusting your search terms.`
                    : "No services were found from the TubeBoost API."
                  }
                </p>
                {searchTerm && (
                  <Button 
                    variant="outline" 
                    onClick={() => handleSearchChange("")}
                    className="mt-4"
                  >
                    Clear Search
                  </Button>
                )}
              </div>
            )}
          </div>
        </main>
      </div>
    </div>
  );
}