import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { AlertCircle, CheckCircle2, Youtube, BarChart3, TrendingUp, Users, Zap, Crown, Star, MessageSquare, MessageCircle, Target, Shield } from "lucide-react";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { useMutation } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { useAuth } from "@/hooks/use-auth";
import { deviceFingerprint } from "@/lib/fingerprint";

const features = [
  {
    icon: <Shield className="h-8 w-8 text-blue-600" />,
    title: "AI-Powered Content Analysis",
    description: "Advanced machine learning algorithms analyze posts, comments, and user behavior to identify potential violations, spam, and harmful content before it impacts your community."
  },
  {
    icon: <MessageSquare className="h-8 w-8 text-orange-600" />,
    title: "Automated Moderation Actions",
    description: "Intelligent automation that can remove, approve, or flag content based on customizable rules and AI recommendations. Reduce manual workload while maintaining high-quality moderation standards."
  },
  {
    icon: <BarChart3 className="h-8 w-8 text-green-500" />,
    title: "Real-Time Moderation Dashboard",
    description: "Comprehensive moderation queue management with real-time notifications, bulk actions, and detailed analytics to help you stay on top of community management tasks."
  },
  {
    icon: <TrendingUp className="h-8 w-8 text-purple-500" />,
    title: "Multi-Provider AI Integration", 
    description: "Support for multiple AI providers including Anthropic Claude, OpenAI GPT, and Azure OpenAI. Choose the best AI model for your specific moderation needs and customize analysis parameters."
  },
  {
    icon: <Users className="h-8 w-8 text-indigo-500" />,
    title: "Subreddit Analytics & Insights",
    description: "Deep analytics on community health, moderation patterns, user engagement metrics, and trend analysis to help you understand and improve your subreddit's ecosystem."
  },
  {
    icon: <Target className="h-8 w-8 text-red-500" />,
    title: "Proactive Content Monitoring",
    description: "Continuous monitoring of new posts and comments with intelligent risk assessment. Get alerts for potentially problematic content before it gains traction in your community."
  }
];

const pricingPlans = [
  {
    name: "Free",
    price: "$0",
    period: "forever",
    points: "200 Points during Beta - per day",
    channels: "1 Channel",
    features: [
      "Basic channel management",
      "Essential analytics",
      "Video performance tracking",
      "Community support"
    ],
    buttonText: "Get Started Free",
    popular: true,
    tier: "free",
    available: true
  },
  {
    name: "Starter",
    price: "$9.95",
    period: "per month",
    points: "1,500 Points",
    channels: "3 Channels",
    features: [
      "Everything in Free",
      "Advanced analytics dashboard",
      "Performance optimization tools",
      "Priority support",
      "Custom reporting"
    ],
    buttonText: "Coming Soon",
    popular: false,
    tier: "starter",
    available: false
  },
  {
    name: "Professional",
    price: "$14.95",
    period: "per month",
    points: "3,500 Points",
    channels: "10 Channels",
    features: [
      "Everything in Starter",
      "Advanced analytics suite",
      "Multi-channel management",
      "Detailed audience insights",
      "API access",
      "White-label reporting"
    ],
    buttonText: "Coming Soon",
    popular: false,
    tier: "professional",
    available: false
  },
  {
    name: "Enterprise",
    price: "$24.95",
    period: "per month",
    points: "10,000 Points",
    channels: "Unlimited Channels",
    features: [
      "Everything in Professional",
      "AI-powered video ranking suggestions",
      "Advanced automation tools",
      "Dedicated account manager",
      "Custom integrations",
      "Enterprise-grade security"
    ],
    buttonText: "Coming Soon",
    popular: false,
    tier: "enterprise",
    available: false
  }
];

const countries = [
  "Afghanistan", "Albania", "Algeria", "Andorra", "Angola", "Antigua and Barbuda", "Argentina", "Armenia", "Australia", "Austria",
  "Azerbaijan", "Bahamas", "Bahrain", "Bangladesh", "Barbados", "Belarus", "Belgium", "Belize", "Benin", "Bhutan",
  "Bolivia", "Bosnia and Herzegovina", "Botswana", "Brazil", "Brunei", "Bulgaria", "Burkina Faso", "Burundi", "Cabo Verde", "Cambodia",
  "Cameroon", "Canada", "Central African Republic", "Chad", "Chile", "China", "Colombia", "Comoros", "Congo", "Costa Rica",
  "Croatia", "Cuba", "Cyprus", "Czech Republic", "Democratic Republic of the Congo", "Denmark", "Djibouti", "Dominica", "Dominican Republic", "Ecuador",
  "Egypt", "El Salvador", "Equatorial Guinea", "Eritrea", "Estonia", "Eswatini", "Ethiopia", "Fiji", "Finland", "France",
  "Gabon", "Gambia", "Georgia", "Germany", "Ghana", "Greece", "Grenada", "Guatemala", "Guinea", "Guinea-Bissau",
  "Guyana", "Haiti", "Honduras", "Hungary", "Iceland", "India", "Indonesia", "Iran", "Iraq", "Ireland",
  "Israel", "Italy", "Jamaica", "Japan", "Jordan", "Kazakhstan", "Kenya", "Kiribati", "Kuwait", "Kyrgyzstan",
  "Laos", "Latvia", "Lebanon", "Lesotho", "Liberia", "Libya", "Liechtenstein", "Lithuania", "Luxembourg", "Madagascar",
  "Malawi", "Malaysia", "Maldives", "Mali", "Malta", "Marshall Islands", "Mauritania", "Mauritius", "Mexico", "Micronesia",
  "Moldova", "Monaco", "Mongolia", "Montenegro", "Morocco", "Mozambique", "Myanmar", "Namibia", "Nauru", "Nepal",
  "Netherlands", "New Zealand", "Nicaragua", "Niger", "Nigeria", "North Korea", "North Macedonia", "Norway", "Oman", "Pakistan",
  "Palau", "Palestine", "Panama", "Papua New Guinea", "Paraguay", "Peru", "Philippines", "Poland", "Portugal", "Qatar",
  "Romania", "Russia", "Rwanda", "Saint Kitts and Nevis", "Saint Lucia", "Saint Vincent and the Grenadines", "Samoa", "San Marino", "Sao Tome and Principe", "Saudi Arabia",
  "Senegal", "Serbia", "Seychelles", "Sierra Leone", "Singapore", "Slovakia", "Slovenia", "Solomon Islands", "Somalia", "South Africa",
  "South Korea", "South Sudan", "Spain", "Sri Lanka", "Sudan", "Suriname", "Sweden", "Switzerland", "Syria", "Taiwan",
  "Tajikistan", "Tanzania", "Thailand", "Timor-Leste", "Togo", "Tonga", "Trinidad and Tobago", "Tunisia", "Turkey", "Turkmenistan",
  "Tuvalu", "Uganda", "Ukraine", "United Arab Emirates", "United Kingdom", "United States", "Uruguay", "Uzbekistan", "Vanuatu", "Vatican City",
  "Venezuela", "Vietnam", "Yemen", "Zambia", "Zimbabwe"
];

interface RegistrationData {
  firstName: string;
  lastName: string;
  email: string;
  password: string;
  mobileNumber: string;
  country: string;
  city: string;
  company: string;
  subscriptionTier: string;
  accountType: string;
}

interface AnonymousCredentials {
  username: string;
  password: string;
}

export default function Landing() {
  const [showRegisterDialog, setShowRegisterDialog] = useState(false);
  const [selectedTier, setSelectedTier] = useState("free");
  const [registrationData, setRegistrationData] = useState<RegistrationData>({
    firstName: "",
    lastName: "",
    email: "",
    password: "",
    mobileNumber: "",
    country: "",
    city: "",
    company: "",
    subscriptionTier: "free",
    accountType: "regular"
  });
  const [anonymousCredentials, setAnonymousCredentials] = useState<AnonymousCredentials | null>(null);
  const { toast } = useToast();
  const { login } = useAuth();

  const registerMutation = useMutation({
    mutationFn: async (data: RegistrationData) => {
      // Collect device fingerprint for security
      const fingerprintData = await deviceFingerprint.collectFingerprint();
      
      // Get location data if available
      let locationData = null;
      try {
        const ipResponse = await fetch('https://ipapi.co/json/');
        if (ipResponse.ok) {
          const ipData = await ipResponse.json();
          locationData = {
            country: ipData.country_name,
            region: ipData.region,
            city: ipData.city,
            isp: ipData.org,
            isVpn: ipData.threat?.is_anonymous || false,
            isProxy: ipData.threat?.is_known_attacker || false
          };
        }
      } catch (e) {
        console.error('[REGISTRATION] Failed to get location data:', e);
      }

      const requestData = {
        ...data,
        deviceFingerprint: fingerprintData,
        locationData,
        // Use email as username for backend compatibility
        username: data.email
      };

      console.log('[REGISTRATION] Sending registration request', {
        email: data.email,
        fingerprintCollected: !!fingerprintData,
        locationCollected: !!locationData,
        timestamp: new Date().toISOString()
      });

      const result = await apiRequest("/api/register", "POST", requestData);
      console.log('[REGISTRATION] Registration successful', {
        userId: result.user?.id,
        email: result.user?.email,
        timestamp: new Date().toISOString()
      });
      return result;
    },
    onSuccess: (data) => {
      console.log('[REGISTRATION] Registration mutation successful', {
        userId: data.user?.id,
        email: data.user?.email,
        accountType: data.accountType,
        timestamp: new Date().toISOString()
      });
      
      if (data.accountType === "anonymous") {
        // Store the password temporarily in localStorage for display in dashboard
        localStorage.setItem(`temp_password_${data.username}`, data.tempPassword);
        setAnonymousCredentials({
          username: data.username,
          password: data.tempPassword
        });
      } else {
        toast({
          title: "Registration Successful!",
          description: "Your account has been created. You can now sign in.",
          variant: "default"
        });
        setShowRegisterDialog(false);
        console.log('[REGISTRATION] Redirecting to login page');
        window.location.href = "/login";
      }
    },
    onError: (error: any) => {
      console.error('[REGISTRATION] Registration mutation failed', {
        error: error.message,
        stack: error.stack,
        email: registrationData.email,
        timestamp: new Date().toISOString()
      });
      
      // Handle security-related errors with specific messaging
      if (error.message.includes("Registration not allowed")) {
        console.warn('[REGISTRATION] Registration blocked by security system', {
          email: registrationData.email,
          reason: 'multiple_accounts_detected',
          timestamp: new Date().toISOString()
        });
        toast({
          title: "Registration Blocked",
          description: "Our security system has detected multiple accounts from this location. Please contact support if you believe this is an error.",
          variant: "destructive"
        });
      } else if (error.message.includes("Invalid device fingerprint")) {
        console.warn('[REGISTRATION] Device fingerprint validation failed', {
          email: registrationData.email,
          timestamp: new Date().toISOString()
        });
        toast({
          title: "Security Check Failed",
          description: "Device security validation failed. Please try again or contact support.",
          variant: "destructive"
        });
      } else if (error.message.includes("already registered") || error.message.includes("already exists")) {
        console.warn('[REGISTRATION] Email already in use', {
          email: registrationData.email,
          timestamp: new Date().toISOString()
        });
        toast({
          title: "Email Already Registered",
          description: "An account with this email already exists. Please try logging in instead.",
          variant: "destructive"
        });
      } else if (error.message.includes("required")) {
        console.warn('[REGISTRATION] Missing required fields', {
          email: registrationData.email,
          error: error.message,
          timestamp: new Date().toISOString()
        });
        toast({
          title: "Missing Information",
          description: "Please ensure all required fields are filled out correctly.",
          variant: "destructive"
        });
      } else {
        console.error('[REGISTRATION] Unexpected registration error', {
          email: registrationData.email,
          error: error.message,
          timestamp: new Date().toISOString()
        });
        toast({
          title: "Registration Failed",
          description: error.message || "An error occurred during registration. Please try again.",
          variant: "destructive"
        });
      }
    }
  });

  const handleRegister = (accountType: "regular") => {
    if (!registrationData.firstName || !registrationData.lastName || 
        !registrationData.email || !registrationData.password || !registrationData.mobileNumber || 
        !registrationData.country) {
      toast({
        title: "Missing Information",
        description: "Please fill in all required fields marked with *",
        variant: "destructive"
      });
      return;
    }
    
    // Advanced error logging for registration attempt
    console.log('[REGISTRATION] Starting registration process', {
      email: registrationData.email,
      country: registrationData.country,
      subscriptionTier: selectedTier,
      timestamp: new Date().toISOString()
    });
    
    registerMutation.mutate({
      ...registrationData,
      accountType: "regular",
      subscriptionTier: selectedTier
    });
  };

  const openRegisterDialog = (tier: string) => {
    setSelectedTier(tier);
    setShowRegisterDialog(true);
    setAnonymousCredentials(null);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 dark:from-slate-900 dark:via-slate-800 dark:to-slate-900">
      {/* Header */}
      <header className="container mx-auto px-4 py-6">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2">
            <Youtube className="h-8 w-8 text-red-500" />
            <span className="text-2xl font-bold text-slate-900 dark:text-white">TubeBoost</span>
          </div>
          <div className="flex items-center gap-4">
            <Button variant="ghost" onClick={() => window.location.href = "/about"}>
              About
            </Button>
            <Button variant="outline" onClick={() => window.location.href = "/login"}>
              Sign In
            </Button>
            <Button onClick={() => openRegisterDialog("free")}>
              Get Started
            </Button>
          </div>
        </div>
      </header>

      {/* Hero Section */}
      <section className="container mx-auto px-4 py-20 text-center">
        <div className="max-w-4xl mx-auto">
          <Badge className="mb-6 px-4 py-2 text-sm" variant="secondary">
            <Shield className="h-4 w-4 mr-2" />
            Trusted by 5,000+ Reddit Moderators
          </Badge>
          <h1 className="text-5xl md:text-7xl font-bold text-slate-900 dark:text-white mb-6 leading-tight">
            AI-Powered Reddit Moderation
            <span className="text-transparent bg-clip-text bg-gradient-to-r from-blue-500 via-purple-500 to-orange-600"> Platform</span>
          </h1>
          <p className="text-xl text-slate-600 dark:text-slate-300 mb-8 max-w-2xl mx-auto leading-relaxed">
            Revolutionize your Reddit community management with advanced AI-powered moderation tools. Automatically detect violations, manage content, and maintain healthy discussions with intelligent automation trusted by moderators worldwide.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="px-8 py-4 text-lg" onClick={() => openRegisterDialog("free")}>
              Start Free Today
              <Zap className="ml-2 h-5 w-5" />
            </Button>
          </div>
        </div>
      </section>

      {/* Platform Showcase Section */}
      <section className="container mx-auto px-4 py-20">
        <div className="text-center mb-16">
          <Badge className="mb-4 px-4 py-2 text-sm" variant="secondary">
            <Shield className="h-4 w-4 mr-2" />
            AI-Powered Moderation
          </Badge>
          <h2 className="text-4xl font-bold text-slate-900 dark:text-white mb-4">
            Advanced Reddit Moderation Platform
          </h2>
          <p className="text-xl text-slate-600 dark:text-slate-300 max-w-3xl mx-auto">
            Comprehensive moderation suite with multi-provider AI integration, real-time content analysis, and intelligent decision support. Transform your community management with cutting-edge automation.
          </p>
        </div>

        {/* Reddit Moderation Features Showcase */}
        <div className="mb-16">
          <h3 className="text-3xl font-bold text-slate-900 dark:text-white mb-8 text-center">
            <Shield className="inline h-8 w-8 text-blue-600 mr-3" />
            Intelligent Moderation Tools
          </h3>
          <div className="grid md:grid-cols-3 gap-8">
            <Card className="border-0 shadow-lg hover:shadow-xl transition-shadow duration-300">
              <CardHeader className="text-center">
                <div className="mx-auto mb-4 p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg w-fit">
                  <Shield className="h-8 w-8 text-blue-600" />
                </div>
                <CardTitle className="text-xl">AI Content Analysis</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-slate-600 dark:text-slate-300 mb-4">
                  Advanced machine learning algorithms analyze posts and comments for policy violations, spam, and harmful content in real-time.
                </p>
                <div className="bg-slate-50 dark:bg-slate-800 p-4 rounded-lg">
                  <div className="text-sm text-slate-500 mb-2">AI Capabilities:</div>
                  <ul className="text-sm space-y-1">
                    <li>• Rule violation detection</li>
                    <li>• Spam and bot identification</li>
                    <li>• Sentiment analysis</li>
                    <li>• Context-aware moderation</li>
                  </ul>
                </div>
              </CardContent>
            </Card>

            <Card className="border-0 shadow-lg hover:shadow-xl transition-shadow duration-300">
              <CardHeader className="text-center">
                <div className="mx-auto mb-4 p-4 bg-purple-50 dark:bg-purple-900/20 rounded-lg w-fit">
                  <Target className="h-8 w-8 text-purple-500" />
                </div>
                <CardTitle className="text-xl">Automated Moderation Actions</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-slate-600 dark:text-slate-300 mb-4">
                  Intelligent automation that executes moderation actions based on AI analysis and customizable rules. Reduce manual workload while maintaining quality.
                </p>
                <div className="bg-slate-50 dark:bg-slate-800 p-4 rounded-lg">
                  <div className="text-sm text-slate-500 mb-2">Automation Features:</div>
                  <ul className="text-sm space-y-1">
                    <li>• Auto-remove violations</li>
                    <li>• Smart content approval</li>
                    <li>• Bulk action processing</li>
                    <li>• Custom rule triggers</li>
                  </ul>
                </div>
              </CardContent>
            </Card>

            <Card className="border-0 shadow-lg hover:shadow-xl transition-shadow duration-300">
              <CardHeader className="text-center">
                <div className="mx-auto mb-4 p-4 bg-green-50 dark:bg-green-900/20 rounded-lg w-fit">
                  <BarChart3 className="h-8 w-8 text-green-500" />
                </div>
                <CardTitle className="text-xl">Real-Time Dashboard</CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-slate-600 dark:text-slate-300 mb-4">
                  Comprehensive moderation dashboard with real-time queue management, analytics, and detailed insights into community health.
                </p>
                <div className="bg-slate-50 dark:bg-slate-800 p-4 rounded-lg">
                  <div className="text-sm text-slate-500 mb-2">Dashboard Features:</div>
                  <ul className="text-sm space-y-1">
                    <li>• Live moderation queue</li>
                    <li>• Community analytics</li>
                    <li>• Moderation history</li>
                    <li>• Performance metrics</li>
                  </ul>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </section>

      {/* Platform Statistics */}
      <section className="bg-slate-50 dark:bg-slate-900 py-20">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16">
            <h2 className="text-4xl font-bold text-slate-900 dark:text-white mb-4">
              Trusted by Reddit Moderators Worldwide
            </h2>
            <p className="text-xl text-slate-600 dark:text-slate-300 max-w-2xl mx-auto">
              Join thousands of moderators who have revolutionized their community management with AI-powered moderation
            </p>
          </div>
          
          <div className="grid md:grid-cols-4 gap-8 max-w-6xl mx-auto">
            <div className="text-center p-6 bg-white dark:bg-slate-800 rounded-lg shadow-lg">
              <div className="text-4xl font-bold text-blue-600 mb-2">5,000+</div>
              <div className="text-slate-600 dark:text-slate-300">Subreddits Moderated</div>
            </div>
            <div className="text-center p-6 bg-white dark:bg-slate-800 rounded-lg shadow-lg">
              <div className="text-4xl font-bold text-purple-500 mb-2">2M+</div>
              <div className="text-slate-600 dark:text-slate-300">Posts Analyzed</div>
            </div>
            <div className="text-center p-6 bg-white dark:bg-slate-800 rounded-lg shadow-lg">
              <div className="text-4xl font-bold text-green-500 mb-2">98.5%</div>
              <div className="text-slate-600 dark:text-slate-300">Accuracy Rate</div>
            </div>
            <div className="text-center p-6 bg-white dark:bg-slate-800 rounded-lg shadow-lg">
              <div className="text-4xl font-bold text-orange-500 mb-2">24/7</div>
              <div className="text-slate-600 dark:text-slate-300">AI Monitoring</div>
            </div>
          </div>
        </div>
      </section>

      {/* Features Section */}
      <section className="container mx-auto px-4 py-20">
        <div className="text-center mb-16">
          <h2 className="text-4xl font-bold text-slate-900 dark:text-white mb-4">
            Advanced Features for Content Success
          </h2>
          <p className="text-xl text-slate-600 dark:text-slate-300 max-w-2xl mx-auto">
            Comprehensive toolkit designed to help content creators achieve sustainable growth across YouTube and Reddit platforms.
          </p>
        </div>
        
        <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
          {features.map((feature, index) => (
            <Card key={index} className="border-0 shadow-lg hover:shadow-xl transition-shadow duration-300">
              <CardHeader className="text-center pb-4">
                <div className="mx-auto mb-4 p-3 bg-slate-50 dark:bg-slate-800 rounded-full w-fit">
                  {feature.icon}
                </div>
                <CardTitle className="text-xl">{feature.title}</CardTitle>
              </CardHeader>
              <CardContent>
                <CardDescription className="text-center text-slate-600 dark:text-slate-300">
                  {feature.description}
                </CardDescription>
              </CardContent>
            </Card>
          ))}
        </div>
      </section>

      {/* Pricing Section */}
      <section className="container mx-auto px-4 py-20">
        <div className="text-center mb-16">
          <h2 className="text-4xl font-bold text-slate-900 dark:text-white mb-4">
            Choose Your Growth Plan
          </h2>
          <p className="text-xl text-slate-600 dark:text-slate-300 max-w-2xl mx-auto">
            Start free and scale as you grow. All plans include our core features with increasing limits and advanced capabilities.
          </p>
        </div>

        <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-8 max-w-7xl mx-auto">
          {pricingPlans.map((plan, index) => (
            <Card key={index} className={`relative border-2 transition-all duration-300 ${
              plan.available 
                ? plan.popular 
                  ? 'border-blue-500 scale-105 hover:shadow-2xl' 
                  : 'border-slate-200 dark:border-slate-700 hover:shadow-2xl'
                : 'border-slate-200 dark:border-slate-700 opacity-60 bg-slate-50 dark:bg-slate-800/50'
            }`}>
              {plan.popular && plan.available && (
                <Badge className="absolute -top-3 left-1/2 transform -translate-x-1/2 bg-blue-500 text-white px-3 py-1">
                  Beta Access
                </Badge>
              )}
              {!plan.available && (
                <Badge className="absolute -top-3 left-1/2 transform -translate-x-1/2 bg-slate-500 text-white px-3 py-1">
                  Coming Soon
                </Badge>
              )}
              <CardHeader className={`text-center pb-4 ${!plan.available ? 'opacity-70' : ''}`}>
                <CardTitle className="text-2xl font-bold">{plan.name}</CardTitle>
                <div className="mt-4">
                  <span className={`text-4xl font-bold ${plan.available ? 'text-slate-900 dark:text-white' : 'text-slate-500 dark:text-slate-400'}`}>
                    {plan.price}
                  </span>
                  <span className={`ml-2 ${plan.available ? 'text-slate-600 dark:text-slate-300' : 'text-slate-400 dark:text-slate-500'}`}>
                    /{plan.period}
                  </span>
                </div>
                <div className="mt-4 space-y-2">
                  <Badge variant={plan.available ? "secondary" : "outline"} className="px-3 py-1">
                    {plan.points}
                  </Badge>
                  <Badge variant="outline" className="px-3 py-1 ml-2">
                    {plan.channels}
                  </Badge>
                </div>
              </CardHeader>
              <CardContent className={!plan.available ? 'opacity-70' : ''}>
                <ul className="space-y-3 mb-6">
                  {plan.features.map((feature, featureIndex) => (
                    <li key={featureIndex} className="flex items-center gap-2 text-sm">
                      <CheckCircle2 className={`h-4 w-4 flex-shrink-0 ${plan.available ? 'text-green-500' : 'text-slate-400'}`} />
                      <span className={plan.available ? 'text-slate-700 dark:text-slate-300' : 'text-slate-500 dark:text-slate-400'}>
                        {feature}
                      </span>
                    </li>
                  ))}
                </ul>
                <Button 
                  className="w-full" 
                  variant={plan.available ? (plan.popular ? "default" : "outline") : "secondary"}
                  onClick={() => plan.available ? openRegisterDialog(plan.tier) : null}
                  disabled={!plan.available}
                >
                  {plan.buttonText}
                </Button>
              </CardContent>
            </Card>
          ))}
        </div>
      </section>

      {/* Registration Dialog */}
      <Dialog open={showRegisterDialog} onOpenChange={setShowRegisterDialog}>
        <DialogContent className="max-w-md max-h-[85vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Create Your Account</DialogTitle>
            <DialogDescription>
              Choose how you'd like to register for the {pricingPlans.find(p => p.tier === selectedTier)?.name} plan
            </DialogDescription>
          </DialogHeader>

          {anonymousCredentials ? (
            <div className="space-y-4">
              <Alert className="border-green-200 bg-green-50 dark:bg-green-900/20">
                <CheckCircle2 className="h-4 w-4 text-green-600" />
                <AlertDescription className="text-green-800 dark:text-green-200">
                  <strong>Your anonymous account has been created!</strong>
                </AlertDescription>
              </Alert>
              
              <div className="bg-slate-50 dark:bg-slate-800 p-4 rounded-lg border-2 border-red-200 dark:border-red-800">
                <div className="text-center space-y-2">
                  <AlertCircle className="h-8 w-8 text-red-500 mx-auto" />
                  <p className="font-semibold text-red-700 dark:text-red-300">IMPORTANT - Save These Credentials!</p>
                  <div className="space-y-2 text-sm">
                    <div className="flex justify-between items-center bg-white dark:bg-slate-700 p-2 rounded">
                      <span className="text-slate-600 dark:text-slate-300">Username:</span>
                      <span className="font-mono font-bold">{anonymousCredentials.username}</span>
                    </div>
                    <div className="flex justify-between items-center bg-white dark:bg-slate-700 p-2 rounded">
                      <span className="text-slate-600 dark:text-slate-300">Password:</span>
                      <span className="font-mono font-bold">{anonymousCredentials.password}</span>
                    </div>
                  </div>
                  <p className="text-xs text-red-600 dark:text-red-400 mt-2">
                    No password recovery is available for anonymous accounts. Write these down!
                  </p>
                </div>
              </div>
              
              <Button 
                className="w-full" 
                onClick={async () => {
                  try {
                    await login(anonymousCredentials.username, anonymousCredentials.password);
                    window.location.href = "/dashboard";
                  } catch (error) {
                    toast({
                      title: "Login Failed",
                      description: "Failed to automatically log in. Please try signing in manually.",
                      variant: "destructive"
                    });
                    window.location.href = "/login";
                  }
                }}
              >
                Continue to Dashboard
              </Button>
            </div>
          ) : (
            <div className="space-y-4">
              <div className="space-y-3">
                <div>
                  <Label htmlFor="firstName">First Name *</Label>
                  <Input
                    id="firstName"
                    value={registrationData.firstName}
                    onChange={(e) => setRegistrationData({...registrationData, firstName: e.target.value})}
                    placeholder="Enter your first name"
                  />
                </div>
                <div>
                  <Label htmlFor="lastName">Last Name *</Label>
                  <Input
                    id="lastName"
                    value={registrationData.lastName}
                    onChange={(e) => setRegistrationData({...registrationData, lastName: e.target.value})}
                    placeholder="Enter your last name"
                  />
                </div>

                <div>
                  <Label htmlFor="email">Email Address *</Label>
                  <Input
                    id="email"
                    type="email"
                    value={registrationData.email}
                    onChange={(e) => setRegistrationData({...registrationData, email: e.target.value})}
                    placeholder="your@email.com"
                  />
                </div>
                <div>
                  <Label htmlFor="password">Password *</Label>
                  <Input
                    id="password"
                    type="password"
                    value={registrationData.password}
                    onChange={(e) => setRegistrationData({...registrationData, password: e.target.value})}
                    placeholder="Create a strong password"
                  />
                </div>
                <div>
                  <Label htmlFor="mobile">Mobile Number *</Label>
                  <Input
                    id="mobile"
                    value={registrationData.mobileNumber}
                    onChange={(e) => setRegistrationData({...registrationData, mobileNumber: e.target.value})}
                    placeholder="+1 (555) 123-4567"
                  />
                </div>
                <div>
                  <Label htmlFor="country">Country *</Label>
                  <Select value={registrationData.country} onValueChange={(value) => setRegistrationData({...registrationData, country: value})}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select your country" />
                    </SelectTrigger>
                    <SelectContent className="max-h-60 overflow-y-auto">
                      {countries.map((country) => (
                        <SelectItem key={country} value={country}>{country}</SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label htmlFor="city">City</Label>
                  <Input
                    id="city"
                    value={registrationData.city}
                    onChange={(e) => setRegistrationData({...registrationData, city: e.target.value})}
                    placeholder="Enter your city"
                  />
                </div>
                <div>
                  <Label htmlFor="company">Company/Organization</Label>
                  <Input
                    id="company"
                    value={registrationData.company}
                    onChange={(e) => setRegistrationData({...registrationData, company: e.target.value})}
                    placeholder="Your company or organization"
                  />
                </div>
              </div>
              <Button 
                className="w-full" 
                onClick={() => handleRegister("regular")}
                disabled={registerMutation.isPending}
              >
                {registerMutation.isPending ? "Creating Account..." : "Create Account"}
              </Button>
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Footer */}
      <footer className="bg-slate-900 text-white py-12">
        <div className="container mx-auto px-4">
          <div className="text-center">
            <div className="flex items-center justify-center gap-2 mb-4">
              <Youtube className="h-6 w-6 text-red-500" />
              <span className="text-xl font-bold">YTubeManager</span>
            </div>
            <p className="text-slate-400 mb-4">
              Empowering content creators to achieve their YouTube goals
            </p>
            <div className="flex justify-center gap-8 text-sm text-slate-400">
              <a href="#" className="hover:text-white transition-colors">Privacy Policy</a>
              <a href="#" className="hover:text-white transition-colors">Terms of Service</a>
              <a href="#" className="hover:text-white transition-colors">Support</a>
            </div>
          </div>
        </div>
      </footer>
    </div>
  );
}