import { useState, useEffect } from 'react';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { DashboardLayout } from '@/components/dashboard-layout';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { useToast } from '@/hooks/use-toast';
import { apiRequest } from '@/lib/queryClient';
import { 
  Shield, 
  Bot, 
  User, 
  FileText, 
  MessageSquare, 
  AlertTriangle, 
  CheckCircle, 
  XCircle, 
  RefreshCw,
  Search,
  Calendar,
  Download,
  Mail,
  Copy,
  Clock,
  Target,
  Filter,
  Eye,
  ExternalLink
} from 'lucide-react';

interface ModerationLogEntry {
  id: number;
  actionType: string;
  actionReason: string;
  wasAiSuggested: boolean;
  aiConfidenceScore: number;
  moderatorNote: string;
  actionTimestamp: string;
  humanResponse?: string;
  subredditName: string;
  itemType: string;
  author: string;
  title: string;
  content: string;
  permalink: string;
  redditId: string;
  userReports: any[];
  modReports: any[];
  violationsFound: any[];
  severity: string;
  confidenceScore: number;
  suggestedAction: string;
  reasoning: string;
}

export default function ModerationLog() {
  const [selectedSubreddit, setSelectedSubreddit] = useState<string>('');
  const [searchQuery, setSearchQuery] = useState('');
  const [filterAction, setFilterAction] = useState<string>('all');
  const [filterAiSuggested, setFilterAiSuggested] = useState<string>('all');
  const [selectedEntry, setSelectedEntry] = useState<ModerationLogEntry | null>(null);
  const [humanResponseInput, setHumanResponseInput] = useState('');
  const [isGeneratingResponse, setIsGeneratingResponse] = useState(false);
  const [conversationId, setConversationId] = useState('');
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch moderated subreddits
  const { data: moderatedSubreddits, isLoading: isLoadingSubreddits } = useQuery({
    queryKey: ['/api/subreddits/moderated'],
    enabled: true,
  });

  // Auto-select first moderated subreddit or fallback to mildlyinteresting
  useEffect(() => {
    if (moderatedSubreddits && moderatedSubreddits.length > 0 && !selectedSubreddit) {
      setSelectedSubreddit(moderatedSubreddits[0].name);
    } else if (!selectedSubreddit && !moderatedSubreddits?.length) {
      // Fallback to mildlyinteresting for testing
      setSelectedSubreddit('mildlyinteresting');
    }
  }, [moderatedSubreddits, selectedSubreddit]);

  // Fetch moderation log
  const { data: moderationLog, isLoading: isLoadingLog, refetch: refetchLog } = useQuery({
    queryKey: ['/api/reddit/moderation-log', selectedSubreddit],
    queryFn: async () => {
      if (!selectedSubreddit) return [];
      return await apiRequest(`/api/reddit/moderation-log?subreddit=${selectedSubreddit}`, 'GET');
    },
    enabled: !!selectedSubreddit,
  });

  // Generate human-like response mutation
  const generateResponseMutation = useMutation({
    mutationFn: async (entry: ModerationLogEntry) => {
      return await apiRequest(`/api/reddit/generate-human-response`, 'POST', {
        entry,
        subredditName: selectedSubreddit
      });
    },
    onSuccess: (data, variables) => {
      setHumanResponseInput(data.humanResponse);
      toast({
        title: "Human Response Generated",
        description: "AI generated a human-like response for modmail.",
      });
    },
    onError: (error) => {
      toast({
        title: "Generation Failed",
        description: "Failed to generate human-like response. Please try again.",
        variant: "destructive",
      });
    },
  });

  // Save human response mutation
  const saveResponseMutation = useMutation({
    mutationFn: async ({ entryId, response }: { entryId: number; response: string }) => {
      return await apiRequest(`/api/reddit/save-human-response`, 'POST', {
        entryId,
        humanResponse: response
      });
    },
    onSuccess: () => {
      toast({
        title: "Response Saved",
        description: "Human response saved successfully.",
      });
      refetchLog();
      setSelectedEntry(null);
      setHumanResponseInput('');
    },
  });

  // Enhanced send and archive mutation for Modmail
  const sendAndArchiveMutation = useMutation({
    mutationFn: async ({ conversationId, body, autoArchive }: { conversationId: string; body: string; autoArchive: boolean }) => {
      return await apiRequest(`/api/reddit/modmail/send-and-archive`, 'POST', {
        conversationId,
        body,
        subreddit: selectedSubreddit,
        autoArchive
      });
    },
    onSuccess: (data) => {
      toast({
        title: "Message Sent Successfully",
        description: data.archived ? "Reply sent and conversation archived." : "Reply sent successfully.",
      });
      setSelectedEntry(null);
      setHumanResponseInput('');
    },
    onError: (error: any) => {
      toast({
        title: "Failed to Send Message",
        description: error.message || "An error occurred while sending the message",
        variant: "destructive"
      });
    }
  });

  // Filter and search logic
  const filteredLog = moderationLog?.filter((entry: ModerationLogEntry) => {
    const matchesSearch = searchQuery === '' || 
      entry.title?.toLowerCase().includes(searchQuery.toLowerCase()) ||
      entry.author?.toLowerCase().includes(searchQuery.toLowerCase()) ||
      entry.content?.toLowerCase().includes(searchQuery.toLowerCase()) ||
      entry.actionReason?.toLowerCase().includes(searchQuery.toLowerCase());
    
    const matchesAction = filterAction === 'all' || entry.actionType === filterAction;
    const matchesAiSuggested = filterAiSuggested === 'all' || 
      (filterAiSuggested === 'ai' && entry.wasAiSuggested) ||
      (filterAiSuggested === 'manual' && !entry.wasAiSuggested);
    
    return matchesSearch && matchesAction && matchesAiSuggested;
  }) || [];

  const handleGenerateResponse = async (entry: ModerationLogEntry) => {
    setSelectedEntry(entry);
    setIsGeneratingResponse(true);
    try {
      await generateResponseMutation.mutateAsync(entry);
    } finally {
      setIsGeneratingResponse(false);
    }
  };

  const handleSaveResponse = async () => {
    if (!selectedEntry || !humanResponseInput.trim()) return;
    
    await saveResponseMutation.mutateAsync({
      entryId: selectedEntry.id,
      response: humanResponseInput.trim()
    });
  };

  const handleSendToModmail = (autoArchive: boolean = true) => {
    if (!conversationId.trim() || !humanResponseInput.trim()) {
      toast({
        title: "Missing Information", 
        description: "Please provide both conversation ID and response message.",
        variant: "destructive"
      });
      return;
    }

    sendAndArchiveMutation.mutate({
      conversationId: conversationId.trim(),
      body: humanResponseInput,
      autoArchive
    });
  };

  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text);
    toast({
      title: "Copied",
      description: "Text copied to clipboard.",
    });
  };

  const getActionColor = (action: string) => {
    switch (action) {
      case 'approve': return 'bg-green-100 text-green-700 border-green-200';
      case 'remove': return 'bg-red-100 text-red-700 border-red-200';
      case 'spam': return 'bg-orange-100 text-orange-700 border-orange-200';
      default: return 'bg-gray-100 text-gray-700 border-gray-200';
    }
  };

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case 'high': return 'destructive';
      case 'medium': return 'secondary';
      case 'low': return 'outline';
      default: return 'outline';
    }
  };

  return (
    <DashboardLayout>
      <div className="container mx-auto p-4 space-y-6">
        <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold flex items-center gap-2">
            <Shield className="h-8 w-8 text-blue-600" />
            Moderation Log
          </h1>
          <p className="text-muted-foreground mt-1">
            Complete history of moderation actions and AI suggestions for handling modmail complaints
          </p>
        </div>
        <Button onClick={() => refetchLog()} disabled={isLoadingLog}>
          <RefreshCw className={`h-4 w-4 mr-2 ${isLoadingLog ? 'animate-spin' : ''}`} />
          Refresh
        </Button>
      </div>

      {/* Subreddit Selection */}
      <Card>
        <CardHeader>
          <CardTitle className="text-lg">Select Subreddit</CardTitle>
          <CardDescription>Choose a subreddit to view its complete moderation history</CardDescription>
        </CardHeader>
        <CardContent>
          <Select value={selectedSubreddit} onValueChange={setSelectedSubreddit}>
            <SelectTrigger className="w-full">
              <SelectValue placeholder="Select a subreddit to view logs" />
            </SelectTrigger>
            <SelectContent>
              {moderatedSubreddits && Array.isArray(moderatedSubreddits) ? 
                moderatedSubreddits.map((sub: any) => (
                  <SelectItem key={sub.name} value={sub.name}>
                    r/{sub.name}
                  </SelectItem>
                )) : null
              }
            </SelectContent>
          </Select>
          
          {isLoadingSubreddits && (
            <div className="mt-2 text-sm text-muted-foreground">Loading your moderated subreddits...</div>
          )}
        </CardContent>
      </Card>

      {selectedSubreddit && (
        <>
          {/* Filters and Search */}
          <Card>
            <CardHeader>
              <CardTitle className="text-lg">Filter & Search</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div>
                  <label className="text-sm font-medium mb-2 block">Search</label>
                  <div className="relative">
                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                    <Input
                      placeholder="Search titles, authors, content..."
                      value={searchQuery}
                      onChange={(e) => setSearchQuery(e.target.value)}
                      className="pl-10"
                    />
                  </div>
                </div>
                <div>
                  <label className="text-sm font-medium mb-2 block">Action Type</label>
                  <Select value={filterAction} onValueChange={setFilterAction}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">All Actions</SelectItem>
                      <SelectItem value="approve">Approve</SelectItem>
                      <SelectItem value="remove">Remove</SelectItem>
                      <SelectItem value="spam">Spam</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <label className="text-sm font-medium mb-2 block">Source</label>
                  <Select value={filterAiSuggested} onValueChange={setFilterAiSuggested}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">All Sources</SelectItem>
                      <SelectItem value="ai">AI Suggested</SelectItem>
                      <SelectItem value="manual">Manual</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="flex items-end">
                  <Button variant="outline" onClick={() => {
                    setSearchQuery('');
                    setFilterAction('all');
                    setFilterAiSuggested('all');
                  }}>
                    <Filter className="h-4 w-4 mr-2" />
                    Clear Filters
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>

          {/* Moderation Log Entries */}
          <div className="space-y-4">
            <div className="flex items-center justify-between">
              <h2 className="text-xl font-semibold">
                Moderation History for r/{selectedSubreddit}
              </h2>
              <div className="text-sm text-muted-foreground">
                {filteredLog.length} of {moderationLog?.length || 0} entries
              </div>
            </div>

            {isLoadingLog ? (
              <Card>
                <CardContent className="p-6">
                  <div className="flex items-center justify-center">
                    <RefreshCw className="h-6 w-6 animate-spin mr-2" />
                    Loading moderation history...
                  </div>
                </CardContent>
              </Card>
            ) : filteredLog.length === 0 ? (
              <Card>
                <CardContent className="p-6">
                  <div className="text-center text-muted-foreground">
                    {moderationLog?.length === 0 ? 
                      'No moderation actions found for this subreddit.' :
                      'No entries match your current filters.'
                    }
                  </div>
                </CardContent>
              </Card>
            ) : (
              <div className="grid gap-4">
                {filteredLog.map((entry: ModerationLogEntry) => (
                  <Card key={entry.id} className="border-l-4 border-l-blue-500">
                    <CardContent className="p-6">
                      <div className="space-y-4">
                        {/* Header */}
                        <div className="flex items-start justify-between">
                          <div className="space-y-2">
                            <div className="flex items-center gap-2 flex-wrap">
                              {entry.itemType === 'submission' ? (
                                <FileText className="h-4 w-4 text-blue-600" />
                              ) : (
                                <MessageSquare className="h-4 w-4 text-green-600" />
                              )}
                              <Badge variant="outline">
                                {entry.itemType === 'submission' ? 'Post' : 'Comment'}
                              </Badge>
                              <Badge className={getActionColor(entry.actionType)}>
                                {entry.actionType.toUpperCase()}
                              </Badge>
                              {entry.wasAiSuggested && (
                                <Badge variant="default" className="bg-purple-100 text-purple-700">
                                  <Bot className="h-3 w-3 mr-1" />
                                  AI Suggested
                                </Badge>
                              )}
                              {!entry.wasAiSuggested && (
                                <Badge variant="secondary">
                                  <User className="h-3 w-3 mr-1" />
                                  Manual
                                </Badge>
                              )}
                              {entry.severity && (
                                <Badge variant={getSeverityColor(entry.severity)}>
                                  {entry.severity.toUpperCase()} RISK
                                </Badge>
                              )}
                            </div>
                            <div className="text-sm text-muted-foreground">
                              <span>by u/{entry.author}</span> • 
                              <span className="ml-1">
                                {new Date(entry.actionTimestamp).toLocaleString()}
                              </span>
                            </div>
                            {entry.title && (
                              <h3 className="font-semibold text-lg">{entry.title}</h3>
                            )}
                          </div>
                          <div className="flex gap-2">
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => handleGenerateResponse(entry)}
                              disabled={isGeneratingResponse}
                            >
                              <Mail className="h-3 w-3 mr-1" />
                              Generate Response
                            </Button>
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => window.open(`https://reddit.com${entry.permalink}`, '_blank')}
                            >
                              <ExternalLink className="h-3 w-3 mr-1" />
                              View on Reddit
                            </Button>
                          </div>
                        </div>

                        {/* Content Preview */}
                        {entry.content && (
                          <div className="bg-gray-50 dark:bg-gray-900 p-3 rounded-lg">
                            <p className="text-sm line-clamp-3">{entry.content}</p>
                          </div>
                        )}

                        {/* Action Details */}
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          {entry.actionReason && (
                            <div className="bg-blue-50 dark:bg-blue-950 p-3 rounded-lg">
                              <h4 className="font-medium text-sm text-blue-700 dark:text-blue-300 mb-1">
                                Action Reason
                              </h4>
                              <p className="text-sm text-blue-600 dark:text-blue-400">
                                {entry.actionReason}
                              </p>
                            </div>
                          )}
                          
                          {entry.reasoning && (
                            <div className="bg-purple-50 dark:bg-purple-950 p-3 rounded-lg">
                              <h4 className="font-medium text-sm text-purple-700 dark:text-purple-300 mb-1">
                                AI Reasoning
                              </h4>
                              <p className="text-sm text-purple-600 dark:text-purple-400">
                                {entry.reasoning}
                              </p>
                            </div>
                          )}
                        </div>

                        {/* Reports */}
                        {(entry.userReports?.length > 0 || entry.modReports?.length > 0) && (
                          <div className="border-t pt-3">
                            <h4 className="font-medium text-sm mb-2 flex items-center gap-1">
                              <AlertTriangle className="h-3 w-3" />
                              Reports
                            </h4>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                              {entry.userReports?.length > 0 && (
                                <div className="bg-yellow-50 dark:bg-yellow-950 p-2 rounded text-xs">
                                  <span className="font-medium">User Reports ({entry.userReports.length}):</span>
                                  <div className="mt-1 space-y-1">
                                    {entry.userReports.map((report: any, idx: number) => (
                                      <div key={`user-report-${entry.id}-${idx}-${Array.isArray(report) ? report[0] : report}`}>
                                        {Array.isArray(report) ? `"${report[0]}" (${report[1]} reports)` : report}
                                      </div>
                                    ))}
                                  </div>
                                </div>
                              )}
                              {entry.modReports?.length > 0 && (
                                <div className="bg-red-50 dark:bg-red-950 p-2 rounded text-xs">
                                  <span className="font-medium">Mod Reports ({entry.modReports.length}):</span>
                                  <div className="mt-1 space-y-1">
                                    {entry.modReports.map((report: any, idx: number) => (
                                      <div key={`mod-report-${entry.id}-${idx}-${Array.isArray(report) ? report[0] : report}`}>
                                        {Array.isArray(report) ? report[0] : report}
                                      </div>
                                    ))}
                                  </div>
                                </div>
                              )}
                            </div>
                          </div>
                        )}

                        {/* Human Response */}
                        {entry.humanResponse && (
                          <div className="border-t pt-3">
                            <div className="bg-green-50 dark:bg-green-950 p-3 rounded-lg">
                              <div className="flex items-center justify-between mb-2">
                                <h4 className="font-medium text-sm text-green-700 dark:text-green-300 flex items-center gap-1">
                                  <Mail className="h-3 w-3" />
                                  Modmail Response Template
                                </h4>
                                <div className="flex gap-1">
                                  <Button
                                    size="sm"
                                    variant="ghost"
                                    onClick={() => copyToClipboard(entry.humanResponse!)}
                                  >
                                    <Copy className="h-3 w-3" />
                                  </Button>
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    onClick={() => {
                                      setSelectedEntry(entry);
                                      setHumanResponseInput(entry.humanResponse || '');
                                    }}
                                  >
                                    <Mail className="h-3 w-3 mr-1" />
                                    Send to Modmail
                                  </Button>
                                </div>
                              </div>
                              <p className="text-sm text-green-600 dark:text-green-400 whitespace-pre-wrap">
                                {entry.humanResponse}
                              </p>
                            </div>
                          </div>
                        )}
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            )}
          </div>
        </>
      )}

      {/* Human Response Generation Dialog */}
      <Dialog open={!!selectedEntry} onOpenChange={() => setSelectedEntry(null)}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Generate Human-like Modmail Response</DialogTitle>
            <DialogDescription>
              Create a natural, human-sounding response for handling modmail complaints about this moderation action.
            </DialogDescription>
          </DialogHeader>
          
          {selectedEntry && (
            <div className="space-y-4">
              {/* Entry Summary */}
              <div className="bg-gray-50 dark:bg-gray-900 p-3 rounded-lg">
                <h4 className="font-medium text-sm mb-2">Moderation Action Summary</h4>
                <div className="text-sm space-y-1">
                  <div><strong>Action:</strong> {selectedEntry.actionType}</div>
                  <div><strong>Author:</strong> u/{selectedEntry.author}</div>
                  <div><strong>Reason:</strong> {selectedEntry.actionReason}</div>
                  {selectedEntry.wasAiSuggested && (
                    <div><strong>AI Confidence:</strong> {Math.round(selectedEntry.aiConfidenceScore * 100)}%</div>
                  )}
                </div>
              </div>

              {/* Generated Response */}
              <div>
                <div className="flex items-center justify-between mb-2">
                  <label className="text-sm font-medium">Human-like Response</label>
                  <Button
                    size="sm"
                    onClick={() => handleGenerateResponse(selectedEntry)}
                    disabled={isGeneratingResponse}
                  >
                    {isGeneratingResponse ? (
                      <>
                        <RefreshCw className="h-3 w-3 mr-1 animate-spin" />
                        Generating...
                      </>
                    ) : (
                      <>
                        <Bot className="h-3 w-3 mr-1" />
                        Regenerate
                      </>
                    )}
                  </Button>
                </div>
                <Textarea
                  value={humanResponseInput}
                  onChange={(e) => setHumanResponseInput(e.target.value)}
                  placeholder="AI will generate a natural, human-sounding response here..."
                  rows={8}
                  className="w-full"
                />
              </div>

              {/* Modmail Integration */}
              <div className="border-t pt-4">
                <div className="mb-3">
                  <label className="text-sm font-medium text-blue-700 dark:text-blue-300 mb-2 block">
                    🏃‍♂️ Quick Send to Modmail
                  </label>
                  <p className="text-xs text-muted-foreground mb-2">
                    Enter the Modmail conversation ID to send this response directly and optionally auto-archive.
                  </p>
                  <Input
                    value={conversationId}
                    onChange={(e) => setConversationId(e.target.value)}
                    placeholder="Conversation ID (e.g., 2r9fq)"
                    className="w-full"
                  />
                </div>
                <div className="flex gap-2">
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={() => handleSendToModmail(false)}
                    disabled={!conversationId.trim() || !humanResponseInput.trim() || sendAndArchiveMutation.isPending}
                  >
                    <Mail className="h-3 w-3 mr-1" />
                    Send Only
                  </Button>
                  <Button
                    size="sm"
                    onClick={() => handleSendToModmail(true)}
                    disabled={!conversationId.trim() || !humanResponseInput.trim() || sendAndArchiveMutation.isPending}
                  >
                    {sendAndArchiveMutation.isPending ? (
                      <>
                        <RefreshCw className="h-3 w-3 mr-1 animate-spin" />
                        Sending...
                      </>
                    ) : (
                      <>
                        <Mail className="h-3 w-3 mr-1" />
                        Send & Archive
                      </>
                    )}
                  </Button>
                </div>
              </div>

              {/* Actions */}
              <div className="flex gap-2 justify-end">
                <Button variant="outline" onClick={() => setSelectedEntry(null)}>
                  Cancel
                </Button>
                <Button
                  onClick={() => copyToClipboard(humanResponseInput)}
                  disabled={!humanResponseInput.trim()}
                  variant="outline"
                >
                  <Copy className="h-3 w-3 mr-1" />
                  Copy
                </Button>
                <Button
                  onClick={handleSaveResponse}
                  disabled={!humanResponseInput.trim() || saveResponseMutation.isPending}
                >
                  {saveResponseMutation.isPending ? (
                    <>
                      <RefreshCw className="h-3 w-3 mr-1 animate-spin" />
                      Saving...
                    </>
                  ) : (
                    <>
                      <CheckCircle className="h-3 w-3 mr-1" />
                      Save Response
                    </>
                  )}
                </Button>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
      </div>
    </DashboardLayout>
  );
}