import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { DashboardLayout } from "@/components/dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Separator } from "@/components/ui/separator";
import { MessageSquare, TrendingUp, Users, Clock, Plus, BarChart3, ExternalLink, ArrowUp, RefreshCw, AlertCircle, ThumbsUp, ThumbsDown, Send, Reply, Award, Pin, Lock, Shield, Star, Share2, Bookmark, Flag, Eye, Search, Activity, Download, UserPlus } from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { PDFExport } from "@/components/pdf-export";
// Import removed - using console.error for now

interface RedditPost {
  id: string;
  name: string;
  title: string;
  selftext: string;
  selftext_html: string;
  subreddit: string;
  subreddit_id: string;
  subreddit_name_prefixed: string;
  subreddit_type: string;
  url: string;
  permalink: string;
  score: number;
  ups: number;
  downs: number;
  upvote_ratio: number;
  num_comments: number;
  created_utc: number;
  edited: boolean | number;
  is_self: boolean;
  is_video: boolean;
  is_original_content: boolean;
  thumbnail: string;
  thumbnail_width: number;
  thumbnail_height: number;
  over_18: boolean;
  spoiler: boolean;
  locked: boolean;
  stickied: boolean;
  author: string;
  author_fullname: string;
  domain: string;
  post_hint?: string;
  preview?: any;
  media?: any;
  media_embed?: any;
  secure_media?: any;
  secure_media_embed?: any;
  link_flair_text?: string;
  link_flair_type?: string;
  link_flair_background_color?: string;
  link_flair_text_color?: string;
  author_flair_text?: string;
  author_flair_type?: string;
  author_flair_background_color?: string;
  author_flair_text_color?: string;
  gilded: number;
  all_awardings?: any[];
  total_awards_received: number;
  distinguished?: string;
  archived: boolean;
  removed_by_category?: string;
  banned_by?: string;
  clicked: boolean;
  hidden: boolean;
  saved: boolean;
  can_mod_post: boolean;
  is_crosspostable: boolean;
  pinned: boolean;
  top_awarded_type?: string;
  pwls?: number;
  wls?: number;
  contest_mode: boolean;
  mod_note?: string;
  approved_by?: string;
  removal_reason?: string;
  banned_at_utc?: number;
  view_count?: number;
  visited: boolean;
  subreddit_subscribers?: number;
  num_crossposts: number;
  is_meta: boolean;
  send_replies: boolean;
  treatment_tags?: any[];
  likes?: boolean | null;
  suggested_sort?: string;
  user_reports?: any[];
  mod_reports?: any[];
  num_reports?: number;
  mod_reason_by?: string;
  mod_reason_title?: string;
  crosspost_parent_list?: any[];
  crosspost_parent?: string;
  discussion_type?: string;
  no_follow: boolean;
  is_robot_indexable: boolean;
  category?: string;
  link_flair_css_class?: string;
  author_flair_css_class?: string;
  content_categories?: any[];
  whitelist_status?: string;
  gallery_data?: any;
  media_metadata?: any;
  is_gallery: boolean;
}

interface RedditComment {
  id: string;
  author: string;
  body: string;
  body_html: string;
  score: number;
  ups: number;
  downs: number;
  created_utc: number;
  edited: boolean | number;
  permalink: string;
  parent_id: string;
  link_id: string;
  subreddit: string;
  subreddit_id: string;
  depth: number;
  is_submitter: boolean;
  distinguished?: string;
  stickied: boolean;
  score_hidden: boolean;
  archived: boolean;
  no_follow: boolean;
  replies: any[];
  controversiality: number;
  gilded: number;
  all_awardings?: any[];
  total_awards_received: number;
  author_flair_text?: string;
  author_flair_background_color?: string;
  author_flair_text_color?: string;
  likes?: boolean | null;
  saved: boolean;
  can_mod_post: boolean;
  send_replies: boolean;
  collapsed: boolean;
  collapsed_reason?: string;
  associated_award?: any;
  unrepliable_reason?: string;
}

interface ManagedSubreddit {
  id: number;
  subredditName: string;
  displayName: string;
  title: string;
  description: string;
  subscribers: number;
  activeUsers: number;
  isNsfw: boolean;
  subredditType: string;
  createdAt: string;
  updatedAt: string;
}

interface SubredditAnalysis {
  id: number;
  subredditName: string;
  subscribers: number;
  activeUsers: number;
  growthRate: number;
  engagementRate: number;
  bestPostTimes: string[];
  topKeywords: string[];
  competitorPosts: number;
}

interface RedditUserProfile {
  id: string;
  name: string;
  display_name: string;
  created_utc: number;
  link_karma: number;
  comment_karma: number;
  total_karma: number;
  is_employee: boolean;
  is_mod: boolean;
  is_gold: boolean;
  is_premium: boolean;
  has_verified_email: boolean;
  verified: boolean;
  icon_img: string;
  snoovatar_img: string;
  profile_img: string;
  banner_img: string;
  banner_background_image: string;
  banner_background_color: string;
  public_description: string;
  subreddit: any;
  pref_show_snoovatar: boolean;
  accept_followers: boolean;
  accept_pms: boolean;
  accept_chats: boolean;
  hide_from_robots: boolean;
  awarder_karma: number;
  awardee_karma: number;
  is_friend: boolean;
  is_blocked: boolean;
  coins: number;
  has_subscribed: boolean;
  has_subscribed_to_premium: boolean;
  suspension_expiration_utc: number | null;
  is_suspended: boolean;
}

// Helper function to format scores
const formatScore = (score: number): string => {
  if (score >= 1000000) return `${(score / 1000000).toFixed(1)}M`;
  if (score >= 1000) return `${(score / 1000).toFixed(1)}k`;
  return score.toString();
};

// Helper function to format time
const formatTimeAgo = (timestamp: number): string => {
  const now = Date.now();
  const diff = now - (timestamp * 1000);
  const hours = Math.floor(diff / (1000 * 60 * 60));
  const days = Math.floor(hours / 24);
  
  if (days > 0) return `${days}d ago`;
  if (hours > 0) return `${hours}h ago`;
  return "Just now";
};

// Component for displaying Gold Upvotes orders for a specific post
const PostUpvotesOrders: React.FC<{ postId: string }> = ({ postId }) => {
  const { data: ordersData } = useQuery({
    queryKey: ['/api/redupvotes/orders', postId],
    queryFn: async () => {
      const response = await fetch(`/api/redupvotes/orders/${postId}`);
      if (!response.ok) throw new Error('Failed to fetch orders');
      return response.json();
    },
    enabled: !!postId,
  });

  if (!ordersData?.orders || ordersData.orders.length === 0) {
    return null;
  }

  return (
    <div className="mt-3 pt-3 border-t border-gray-200 dark:border-gray-700">
      <h4 className="text-sm font-medium text-gray-900 dark:text-gray-100 mb-2 flex items-center">
        <TrendingUp className="h-4 w-4 mr-1 text-yellow-600" />
        Gold Upvotes Orders
      </h4>
      <div className="space-y-2">
        {ordersData.orders.map((order: any) => (
          <div key={order.id} className="bg-yellow-50 dark:bg-yellow-900/20 rounded-lg p-3 text-sm">
            <div className="flex justify-between items-start">
              <div>
                <p className="font-medium text-yellow-800 dark:text-yellow-200">
                  Order #{order.redupvotesOrderId}
                </p>
                <p className="text-yellow-700 dark:text-yellow-300">
                  {order.quantity} upvotes • Status: {order.status}
                </p>
                {order.charge && (
                  <p className="text-yellow-600 dark:text-yellow-400">
                    Cost: ${order.charge} {order.currency}
                  </p>
                )}
                {order.remains !== null && (
                  <p className="text-yellow-600 dark:text-yellow-400">
                    Remaining: {order.remains}
                  </p>
                )}
              </div>
              <Badge 
                variant={order.status === 'completed' ? 'default' : 
                        order.status === 'in_progress' ? 'secondary' : 'outline'}
                className="text-xs"
              >
                {order.status}
              </Badge>
            </div>
            <p className="text-xs text-yellow-600 dark:text-yellow-400 mt-1">
              Created: {new Date(order.createdAt).toLocaleDateString()}
            </p>
          </div>
        ))}
      </div>
    </div>
  );
};

export default function RedditManagement() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [showCreatePost, setShowCreatePost] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [selectedPost, setSelectedPost] = useState<RedditPost | null>(null);
  const [showComments, setShowComments] = useState(false);
  const [newCommentText, setNewCommentText] = useState('');
  const [replyingTo, setReplyingTo] = useState<string | null>(null);
  const [showCommentsDialog, setShowCommentsDialog] = useState(false);
  const [selectedUser, setSelectedUser] = useState<string | null>(null);
  const [showUserProfile, setShowUserProfile] = useState(false);
  
  // Gold Upvotes state
  const [showUpvoteDialog, setShowUpvoteDialog] = useState(false);
  const [orderingPost, setOrderingPost] = useState<RedditPost | null>(null);
  const [upvoteQuantity, setUpvoteQuantity] = useState('10');
  
  // Subreddit search and analytics state
  const [searchQuery, setSearchQuery] = useState("");
  const [searchResults, setSearchResults] = useState<any[]>([]);
  const [isSearching, setIsSearching] = useState(false);
  const [selectedSubreddit, setSelectedSubreddit] = useState<string | null>(null);
  const [subredditAnalytics, setSubredditAnalytics] = useState<any>(null);
  const [isLoadingAnalytics, setIsLoadingAnalytics] = useState(false);
  const [showAnalyticsModal, setShowAnalyticsModal] = useState(false);
  const [isAddingToManaged, setIsAddingToManaged] = useState(false);
  const [newPost, setNewPost] = useState({
    title: '',
    content: '',
    subreddit: '',
    scheduledAt: ''
  });
  const [analyzeSubreddit, setAnalyzeSubreddit] = useState('');

  const { data: redditStatus } = useQuery({
    queryKey: ['/api/reddit/status'],
    refetchInterval: 30000, // Refresh every 30 seconds
  });

  const { data: posts = [], refetch: refetchPosts, isLoading: postsLoading, error: postsError } = useQuery({
    queryKey: ['/api/reddit/posts'],
    enabled: (redditStatus as any)?.connected === true,
    refetchInterval: 60000, // Refresh every minute
    retry: 3,
    retryDelay: (attemptIndex) => Math.min(1000 * 2 ** attemptIndex, 30000),
  });

  const { data: analytics = [] } = useQuery({
    queryKey: ['/api/reddit/analytics'],
    enabled: (redditStatus as any)?.connected === true
  });

  const { data: managedSubreddits = [] } = useQuery({
    queryKey: ['/api/subreddits/managed'],
    enabled: (redditStatus as any)?.connected === true
  });

  // Comments query for selected post
  const { data: commentsData, refetch: refetchComments, isLoading: commentsLoading } = useQuery({
    queryKey: ['/api/reddit/comments', selectedPost?.id],
    queryFn: async () => {
      if (!selectedPost?.id) return null;
      const response = await fetch(`/api/reddit/comments/${selectedPost.id}`);
      if (!response.ok) {
        throw new Error(`Failed to fetch comments: ${response.status}`);
      }
      return response.json();
    },
    enabled: !!selectedPost && showComments,
    retry: 2,
  });

  // User profile query for selected user
  const { data: userProfile, isLoading: userProfileLoading } = useQuery({
    queryKey: ['/api/reddit/user', selectedUser],
    queryFn: async () => {
      if (!selectedUser) return null;
      const response = await fetch(`/api/reddit/user/${selectedUser}`);
      if (!response.ok) {
        throw new Error(`Failed to fetch user profile: ${response.status}`);
      }
      return response.json();
    },
    enabled: !!selectedUser && showUserProfile,
    retry: 2,
  });

  // Comment posting mutation
  const commentMutation = useMutation({
    mutationFn: async (commentData: { parentId: string; text: string }) => {
      return await apiRequest('/api/reddit/comment', 'POST', commentData);
    },
    onSuccess: () => {
      setNewCommentText('');
      setReplyingTo(null);
      refetchComments();
      toast({
        title: "Comment posted",
        description: "Your comment has been posted successfully.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Failed to post comment",
        description: error.message || "An error occurred while posting your comment.",
        variant: "destructive",
      });
    }
  });

  // Redupvotes order mutation
  const redupvotesMutation = useMutation({
    mutationFn: async ({ redditPostId, redditPostUrl, quantity }: { 
      redditPostId: string; 
      redditPostUrl: string; 
      quantity: number; 
    }) => {
      return apiRequest('/api/redupvotes/order', 'POST', { 
        redditPostId, 
        redditPostUrl, 
        quantity 
      });
    },
    onSuccess: (data) => {
      setShowUpvoteDialog(false);
      setOrderingPost(null);
      setUpvoteQuantity('10');
      queryClient.invalidateQueries({ queryKey: ['/api/redupvotes/orders'] });
      queryClient.invalidateQueries({ queryKey: ['/api/points'] }); // Refresh points balance
      toast({
        title: "Order placed successfully",
        description: `Order ${data.redupvotesOrderId} placed for ${upvoteQuantity} upvotes. 2 points deducted. Remaining: ${data.remainingPoints} points.`,
      });
    },
    onError: (error: any) => {
      const errorMessage = error.message || "Failed to place upvote order";
      const isInsufficientPoints = errorMessage.includes("Insufficient points") || errorMessage.includes("need 2 points");
      
      toast({
        title: isInsufficientPoints ? "Insufficient Points" : "Order failed",
        description: isInsufficientPoints 
          ? "You need at least 2 points to place a Gold Upvotes order. Please add more points to your account."
          : errorMessage,
        variant: "destructive",
      });
    },
  });

  // Query for fetching existing orders for a specific post
  const usePostOrders = (postId: string) => {
    return useQuery({
      queryKey: ['/api/redupvotes/orders', postId],
      queryFn: async () => {
        const response = await fetch(`/api/redupvotes/orders/${postId}`);
        if (!response.ok) throw new Error('Failed to fetch orders');
        return response.json();
      },
      enabled: !!postId,
    });
  };

  // Auto-refresh on page load
  useEffect(() => {
    if ((redditStatus as any)?.connected === true) {
      queryClient.invalidateQueries({ queryKey: ['/api/reddit/posts'] });
    }
  }, [(redditStatus as any)?.connected]);

  // Helper functions
  const formatDate = (timestamp: number) => {
    const date = new Date(timestamp * 1000);
    const now = new Date();
    const diff = now.getTime() - date.getTime();
    const minutes = Math.floor(diff / 60000);
    const hours = Math.floor(diff / 3600000);
    const days = Math.floor(diff / 86400000);

    if (minutes < 60) return `${minutes}m ago`;
    if (hours < 24) return `${hours}h ago`;
    return `${days}d ago`;
  };

  const formatScore = (score: number) => {
    if (score >= 1000) return `${(score / 1000).toFixed(1)}k`;
    return score.toString();
  };

  // Helper function to render comment content with images
  const renderCommentContent = (content: string) => {
    if (!content) return content;
    
    // First decode HTML entities
    const decodedContent = content
      .replace(/&amp;/g, '&')
      .replace(/&lt;/g, '<')
      .replace(/&gt;/g, '>')
      .replace(/&quot;/g, '"')
      .replace(/&#39;/g, "'");
    
    // Regular expression to match image URLs with more specific patterns
    const imageRegex = /(https?:\/\/[^\s\)]+\.(jpg|jpeg|png|gif|webp)(\?[^\s\)]*)?)/gi;
    const allLinkRegex = /(https?:\/\/[^\s\)]+)/gi;
    
    // Find all image URLs first
    const imageUrls = [...decodedContent.matchAll(imageRegex)].map(match => match[0]);
    
    // Skip processing if no URLs are found
    if (imageUrls.length === 0 && !decodedContent.match(allLinkRegex)) {
      return decodedContent;
    }
    
    // Split content by all URLs to handle text and links properly
    const parts = decodedContent.split(allLinkRegex);
    const urls = [...decodedContent.matchAll(allLinkRegex)].map(match => match[0]);
    
    const result = [];
    let urlIndex = 0;
    
    for (let i = 0; i < parts.length; i++) {
      // Add text part if it's not empty
      if (parts[i]) {
        result.push(
          <span key={`text-${i}`} className="whitespace-pre-wrap">
            {parts[i]}
          </span>
        );
      }
      
      // Add URL part if it exists
      if (urlIndex < urls.length) {
        const url = urls[urlIndex];
        const isImage = imageUrls.includes(url);
        
        if (isImage) {
          result.push(
            <div key={`image-${i}-${urlIndex}`} className="my-2">
              <img 
                src={url} 
                alt="Comment image" 
                className="max-w-full h-auto rounded border cursor-pointer"
                style={{ maxHeight: '300px' }}
                onError={(e) => {
                  console.log('Image failed to load:', url);
                  // Fallback to show as link if image fails to load
                  const target = e.target as HTMLImageElement;
                  const parent = target.parentElement;
                  if (parent) {
                    parent.innerHTML = `<a href="${url}" target="_blank" rel="noopener noreferrer" class="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300 underline">${url}</a>`;
                  }
                }}
                onClick={() => window.open(url, '_blank')}
              />
            </div>
          );
        } else {
          result.push(
            <a 
              key={`link-${i}-${urlIndex}`}
              href={url} 
              target="_blank" 
              rel="noopener noreferrer"
              className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300 underline"
            >
              {url}
            </a>
          );
        }
        urlIndex++;
      }
    }
    
    return result.length > 0 ? result : content;
  };

  const handleShowComments = (post: RedditPost) => {
    setSelectedPost(post);
    setShowComments(true);
    setShowCommentsDialog(true);
  };

  const handlePostComment = () => {
    if (!selectedPost || !newCommentText.trim()) return;
    
    // For Reddit API, parentId should be the fullname (t3_ for posts, t1_ for comments)
    const parentId = replyingTo || selectedPost.name; // selectedPost.name is the Reddit fullname
    
    commentMutation.mutate({
      parentId: parentId,
      text: newCommentText.trim()
    });
  };

  const handleUserClick = (username: string) => {
    setSelectedUser(username);
    setShowUserProfile(true);
  };

  // Subreddit search functionality
  const handleSubredditSearch = async () => {
    if (!searchQuery.trim()) return;
    
    setIsSearching(true);
    try {
      const response = await fetch('/api/reddit/search/subreddits', {
        method: 'POST',
        body: JSON.stringify({ query: searchQuery.trim() }),
        headers: { 'Content-Type': 'application/json' }
      });
      
      if (!response.ok) throw new Error('Search failed');
      const data = await response.json();
      setSearchResults(data);
    } catch (error) {
      toast({
        title: "Search failed",
        description: "Failed to search subreddits. Please try again.",
        variant: "destructive",
      });
      setSearchResults([]);
    }
    setIsSearching(false);
  };

  // Get instant analytics for any subreddit
  const handleGetAnalytics = async (subredditName: string) => {
    setSelectedSubreddit(subredditName);
    setIsLoadingAnalytics(true);
    setShowAnalyticsModal(true);
    
    try {
      const response = await fetch('/api/reddit/subreddit/analytics', {
        method: 'POST',
        body: JSON.stringify({ subredditName }),
        headers: { 'Content-Type': 'application/json' }
      });
      
      if (!response.ok) throw new Error('Analytics failed');
      const data = await response.json();
      setSubredditAnalytics(data);
    } catch (error) {
      toast({
        title: "Analytics failed",
        description: `Failed to get analytics for r/${subredditName}. The subreddit may be private or not exist.`,
        variant: "destructive",
      });
      setSubredditAnalytics(null);
    }
    setIsLoadingAnalytics(false);
  };

  // Add subreddit to managed list
  const handleAddToManaged = async () => {
    if (!selectedSubreddit || !subredditAnalytics) return;
    
    setIsAddingToManaged(true);
    try {
      const response = await fetch('/api/reddit/subreddit/add-to-managed', {
        method: 'POST',
        body: JSON.stringify({ 
          subredditName: selectedSubreddit,
          analytics: subredditAnalytics 
        }),
        headers: { 'Content-Type': 'application/json' }
      });
      
      const result = await response.json();
      
      if (response.status === 409) {
        toast({
          title: "Already managed",
          description: "This subreddit is already in your managed list.",
          variant: "destructive",
        });
      } else if (!response.ok) {
        throw new Error(result.message || 'Failed to add subreddit');
      } else {
        toast({
          title: "Added to managed",
          description: `r/${selectedSubreddit} has been added to your managed subreddits.`,
        });
        // Invalidate managed subreddits cache to refresh the list
        queryClient.invalidateQueries({ queryKey: ['/api/subreddits/managed'] });
      }
    } catch (error) {
      toast({
        title: "Failed to add",
        description: `Failed to add r/${selectedSubreddit} to managed list. Please try again.`,
        variant: "destructive",
      });
    } finally {
      setIsAddingToManaged(false);
    }
  };

  // Direct search by entering subreddit name
  const handleDirectAnalytics = () => {
    const cleanName = searchQuery.replace(/^r\//, '').trim();
    if (cleanName) {
      handleGetAnalytics(cleanName);
    }
  };

  const renderMediaPreview = (post: RedditPost) => {
    if (post.is_gallery && post.gallery_data) {
      return (
        <div className="mb-4 p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
          <p className="text-sm text-gray-600 dark:text-gray-400 flex items-center gap-2">
            <Eye className="w-4 h-4" />
            Gallery ({Object.keys(post.gallery_data).length} images)
          </p>
        </div>
      );
    }

    if (post.is_video && post.media) {
      return (
        <div className="mb-4 p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
          <p className="text-sm text-gray-600 dark:text-gray-400 flex items-center gap-2">
            <Eye className="w-4 h-4" />
            Video Content
          </p>
        </div>
      );
    }

    if (!post.is_self && post.url && post.thumbnail && post.thumbnail !== 'self') {
      return (
        <div className="mb-4">
          <img 
            src={post.thumbnail} 
            alt="Post thumbnail"
            className="w-20 h-20 object-cover rounded border"
            onError={(e) => { e.currentTarget.style.display = 'none'; }}
          />
        </div>
      );
    }

    return null;
  };

  // Sync button functionality
  const handleSync = async () => {
    setIsRefreshing(true);
    try {
      await Promise.all([
        queryClient.invalidateQueries({ queryKey: ['/api/reddit/posts'] }),
        queryClient.invalidateQueries({ queryKey: ['/api/reddit/analytics'] }),
        queryClient.invalidateQueries({ queryKey: ['/api/subreddits/managed'] }),
        queryClient.invalidateQueries({ queryKey: ['/api/reddit/status'] })
      ]);
      
      toast({
        title: "Sync completed",
        description: "Reddit data has been refreshed successfully.",
      });
    } catch (error) {
      toast({
        title: "Sync failed",
        description: "Failed to refresh Reddit data. Please try again.",
        variant: "destructive",
      });
      
      console.error("Reddit sync error:", error);
    } finally {
      setIsRefreshing(false);
    }
  };

  // Log posts API errors
  useEffect(() => {
    if (postsError) {
      console.error("Reddit posts API fetch failed:", postsError);
    }
  }, [postsError]);

  const createPostMutation = useMutation({
    mutationFn: async (postData: typeof newPost) => {
      return await apiRequest('/api/reddit/posts', 'POST', postData);
    },
    onSuccess: () => {
      refetchPosts();
      setShowCreatePost(false);
      setNewPost({ title: '', content: '', subreddit: '', scheduledAt: '' });
      toast({
        title: "Success",
        description: "Reddit post created successfully"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to create post",
        variant: "destructive"
      });
    }
  });

  const analyzeSubredditMutation = useMutation({
    mutationFn: async (subreddit: string) => {
      return await apiRequest('/api/reddit/analyze', 'POST', { subreddit });
    },
    onSuccess: () => {
      setAnalyzeSubreddit('');
      toast({
        title: "Success",
        description: "Subreddit analyzed successfully"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to analyze subreddit",
        variant: "destructive"
      });
    }
  });

  // Check if user is connected to Reddit
  if (!redditStatus?.connected) {
    return (
      <DashboardLayout>
        <div className="p-6">
          <div className="max-w-4xl mx-auto">
            <h1 className="text-3xl font-bold mb-6">Reddit Management</h1>
            <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-6 mb-6">
              <h2 className="text-xl font-semibold mb-4">Connect Your Reddit Account</h2>
              <p className="text-gray-600 mb-4">
                To manage your Reddit presence, you need to connect your Reddit account first.
              </p>
              <a
                href="/api/reddit/auth"
                target="_blank"
                rel="noopener noreferrer"
                className="bg-orange-500 hover:bg-orange-600 text-white px-6 py-2 rounded-lg inline-flex items-center gap-2"
              >
                <MessageSquare className="w-5 h-5" />
                Connect to Reddit
              </a>
            </div>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  return (
    <DashboardLayout>
      <div className="p-6">
        <div className="max-w-6xl mx-auto">
          <div className="flex items-center justify-between mb-6">
            <h1 className="text-3xl font-bold">Reddit Management</h1>
            <div className="text-sm text-gray-600">
              Connected as: {redditStatus.redditUser?.username}
            </div>
          </div>

          <Tabs defaultValue="posts" className="space-y-6">
            <TabsList>
              <TabsTrigger value="posts">Posts</TabsTrigger>
              <TabsTrigger value="analytics">Analytics</TabsTrigger>
            </TabsList>

            <TabsContent value="posts" className="space-y-6">
              <div className="flex items-center justify-between">
                <h2 className="text-xl font-semibold">Your Reddit Posts</h2>
                <div className="flex gap-2">
                  <Button 
                    variant="outline" 
                    onClick={handleSync}
                    disabled={isRefreshing}
                  >
                    <RefreshCw className={`w-4 h-4 mr-2 ${isRefreshing ? 'animate-spin' : ''}`} />
                    {isRefreshing ? 'Syncing...' : 'Sync'}
                  </Button>
                  <Button onClick={() => setShowCreatePost(true)}>
                    <Plus className="w-4 h-4 mr-2" />
                    Create Post
                  </Button>
                </div>
              </div>

              {showCreatePost && (
                <Card>
                  <CardHeader>
                    <CardTitle>Create New Reddit Post</CardTitle>
                    <CardDescription>Schedule and manage your Reddit content</CardDescription>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <div>
                      <Label htmlFor="title">Title</Label>
                      <Input
                        id="title"
                        value={newPost.title}
                        onChange={(e) => setNewPost(prev => ({ ...prev, title: e.target.value }))}
                        placeholder="Enter post title"
                      />
                    </div>
                    <div>
                      <Label htmlFor="subreddit">Subreddit</Label>
                      <Select 
                        value={newPost.subreddit} 
                        onValueChange={(value) => setNewPost(prev => ({ ...prev, subreddit: value }))}
                      >
                        <SelectTrigger>
                          <SelectValue placeholder="Select a managed subreddit" />
                        </SelectTrigger>
                        <SelectContent>
                          {(managedSubreddits as ManagedSubreddit[]).map((subreddit) => (
                            <SelectItem key={subreddit.id} value={subreddit.subredditName}>
                              {subreddit.displayName}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label htmlFor="content">Content</Label>
                      <Textarea
                        id="content"
                        value={newPost.content}
                        onChange={(e) => setNewPost(prev => ({ ...prev, content: e.target.value }))}
                        placeholder="Write your post content..."
                        rows={6}
                      />
                    </div>
                    <div>
                      <Label htmlFor="scheduledAt">Schedule For (Optional)</Label>
                      <Input
                        id="scheduledAt"
                        type="datetime-local"
                        value={newPost.scheduledAt}
                        onChange={(e) => setNewPost(prev => ({ ...prev, scheduledAt: e.target.value }))}
                      />
                    </div>
                    <div className="flex gap-2">
                      <Button 
                        onClick={() => createPostMutation.mutate(newPost)}
                        disabled={createPostMutation.isPending || !newPost.title || !newPost.content || !newPost.subreddit}
                      >
                        {createPostMutation.isPending ? 'Creating...' : 'Create Post'}
                      </Button>
                      <Button variant="outline" onClick={() => setShowCreatePost(false)}>
                        Cancel
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              )}

              <div className="grid gap-4">
                {postsLoading ? (
                  <Card>
                    <CardContent className="py-8 text-center">
                      <RefreshCw className="w-12 h-12 mx-auto text-gray-400 mb-4 animate-spin" />
                      <p className="text-gray-600">Loading your Reddit posts...</p>
                    </CardContent>
                  </Card>
                ) : postsError ? (
                  <Card>
                    <CardContent className="py-8 text-center">
                      <AlertCircle className="w-12 h-12 mx-auto text-red-400 mb-4" />
                      <p className="text-gray-600 mb-4">Failed to load Reddit posts</p>
                      <p className="text-sm text-gray-500 mb-4">
                        {(postsError as any)?.message || 'Unknown error occurred'}
                      </p>
                      <Button 
                        variant="outline" 
                        onClick={handleSync}
                        disabled={isRefreshing}
                      >
                        <RefreshCw className={`w-4 h-4 mr-2 ${isRefreshing ? 'animate-spin' : ''}`} />
                        Try Again
                      </Button>
                    </CardContent>
                  </Card>
                ) : (posts as RedditPost[]).length === 0 ? (
                  <Card>
                    <CardContent className="py-8 text-center">
                      <MessageSquare className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                      <p className="text-gray-600">No Reddit posts found. Create your first post!</p>
                    </CardContent>
                  </Card>
                ) : (
                  posts.map((post: RedditPost) => (
                    <Card key={post.id} className="hover:shadow-md transition-shadow">
                      <CardHeader className="pb-3">
                        <div className="flex items-start justify-between">
                          <div className="flex-1">
                            <div className="flex items-center gap-2 mb-2">
                              <Avatar className="w-6 h-6">
                                <AvatarFallback className="text-xs">{post.author[0].toUpperCase()}</AvatarFallback>
                              </Avatar>
                              <span className="text-sm text-gray-600 dark:text-gray-400">
                                u/{post.author}
                              </span>
                              {post.author_flair_text && (
                                <Badge variant="outline" className="text-xs" style={{
                                  backgroundColor: post.author_flair_background_color || undefined,
                                  color: post.author_flair_text_color || undefined
                                }}>
                                  {post.author_flair_text}
                                </Badge>
                              )}
                              <span className="text-xs text-gray-500">•</span>
                              <span className="text-xs text-gray-500">{formatDate(post.created_utc)}</span>
                              {post.edited && (
                                <>
                                  <span className="text-xs text-gray-500">•</span>
                                  <span className="text-xs text-gray-500">edited</span>
                                </>
                              )}
                            </div>
                            <CardTitle className="text-lg mb-2 leading-tight hover:text-blue-600 dark:hover:text-blue-400 cursor-pointer" 
                              onClick={() => window.open(`https://reddit.com${post.permalink}`, '_blank')}>
                              {post.title}
                            </CardTitle>
                            <div className="flex items-center gap-2 mb-3 flex-wrap">
                              <Badge variant="outline">{post.subreddit_name_prefixed}</Badge>
                              {post.over_18 && <Badge variant="destructive" className="text-xs">NSFW</Badge>}
                              {post.spoiler && <Badge variant="secondary" className="text-xs">SPOILER</Badge>}
                              {post.stickied && <Badge className="text-xs bg-green-100 text-green-800"><Pin className="w-3 h-3 mr-1" />Pinned</Badge>}
                              {post.locked && <Badge className="text-xs bg-yellow-100 text-yellow-800"><Lock className="w-3 h-3 mr-1" />Locked</Badge>}
                              {post.distinguished && <Badge className="text-xs bg-purple-100 text-purple-800"><Shield className="w-3 h-3 mr-1" />Mod</Badge>}
                              {post.is_original_content && <Badge className="text-xs bg-blue-100 text-blue-800">OC</Badge>}
                              {post.is_video && <Badge variant="secondary" className="text-xs">Video</Badge>}
                              {post.is_gallery && <Badge variant="secondary" className="text-xs">Gallery</Badge>}
                              {!post.is_self && <Badge variant="secondary" className="text-xs">Link</Badge>}
                              {post.link_flair_text && (
                                <Badge className="text-xs" style={{
                                  backgroundColor: post.link_flair_background_color || undefined,
                                  color: post.link_flair_text_color || undefined
                                }}>
                                  {post.link_flair_text}
                                </Badge>
                              )}
                            </div>
                          </div>
                          <div className="flex items-center gap-2">
                            {post.total_awards_received > 0 && (
                              <div className="flex items-center gap-1 text-xs text-amber-600">
                                <Award className="w-3 h-3" />
                                <span>{post.total_awards_received}</span>
                              </div>
                            )}
                            <Button
                              variant="ghost"
                              size="sm"
                              onClick={() => window.open(`https://reddit.com${post.permalink}`, '_blank')}
                            >
                              <ExternalLink className="w-4 h-4" />
                            </Button>
                          </div>
                        </div>
                      </CardHeader>
                      <CardContent className="pt-0">
                        {renderMediaPreview(post)}
                        
                        {post.selftext && (
                          <div className="mb-4">
                            <div className="text-gray-700 dark:text-gray-300 whitespace-pre-wrap">
                              {post.selftext.length > 400 ? (
                                <>
                                  {post.selftext.substring(0, 400)}...
                                  <Button variant="link" className="p-0 h-auto text-blue-600" 
                                    onClick={() => window.open(`https://reddit.com${post.permalink}`, '_blank')}>
                                    Read more
                                  </Button>
                                </>
                              ) : (
                                post.selftext
                              )}
                            </div>
                          </div>
                        )}

                        {!post.is_self && post.url && (
                          <div className="mb-4 p-3 bg-gray-50 dark:bg-gray-800 rounded-lg border">
                            <div className="flex items-center gap-2 mb-1">
                              <ExternalLink className="w-4 h-4 text-gray-500" />
                              <span className="text-sm text-gray-600 dark:text-gray-400">{post.domain}</span>
                            </div>
                            <a href={post.url} target="_blank" rel="noopener noreferrer" 
                               className="text-sm text-blue-600 dark:text-blue-400 hover:underline truncate block">
                              {post.url}
                            </a>
                          </div>
                        )}

                        <div className="flex items-center justify-between text-sm text-gray-600 dark:text-gray-400 mb-4">
                          <div className="flex items-center gap-4">
                            <div className="flex items-center gap-1">
                              <div className="flex items-center">
                                <ThumbsUp className="w-4 h-4 text-orange-500" />
                                <span className="ml-1 font-medium">{formatScore(post.ups)}</span>
                              </div>
                              <span className="mx-1">•</span>
                              <div className="flex items-center">
                                <ThumbsDown className="w-4 h-4 text-blue-500" />
                                <span className="ml-1">{formatScore(post.downs || 0)}</span>
                              </div>
                              <span className="mx-1">•</span>
                              <span className="text-xs">
                                {Math.round((post.upvote_ratio || 0) * 100)}% upvoted
                              </span>
                            </div>
                            <Button 
                              variant="ghost" 
                              size="sm" 
                              className="flex items-center gap-1 hover:bg-gray-100 dark:hover:bg-gray-800"
                              onClick={() => handleShowComments(post)}
                            >
                              <MessageSquare className="w-4 h-4" />
                              <span>{post.num_comments} comments</span>
                            </Button>
                            
                            <Dialog open={showCommentsDialog && selectedPost?.id === post.id} onOpenChange={(open) => {
                              if (!open) {
                                setShowCommentsDialog(false);
                                setSelectedPost(null);
                                setShowComments(false);
                                setNewCommentText('');
                                setReplyingTo(null);
                              }
                            }}>
                              <DialogContent className="w-[95vw] max-w-4xl h-[90vh] max-h-[90vh] overflow-hidden flex flex-col p-4 sm:p-6" aria-describedby="comments-dialog-description">
                                <DialogHeader>
                                  <DialogTitle className="text-left">{post.title}</DialogTitle>
                                  <div id="comments-dialog-description" className="sr-only">
                                    View and interact with comments on this Reddit post
                                  </div>
                                </DialogHeader>
                                <ScrollArea className="flex-1 w-full min-h-0">
                                  {commentsLoading ? (
                                    <div className="flex items-center justify-center py-8">
                                      <RefreshCw className="w-6 h-6 animate-spin mr-2" />
                                      <span>Loading comments...</span>
                                    </div>
                                  ) : (commentsData as any)?.comments?.length > 0 ? (
                                    <div className="space-y-4">
                                      {(commentsData as any).comments.map((comment: RedditComment) => (
                                        <div key={comment.id} className="border-l-2 border-gray-200 pl-4">
                                          <div className="flex items-center gap-2 mb-2">
                                            <Avatar className="w-5 h-5">
                                              <AvatarFallback className="text-xs">{comment.author[0].toUpperCase()}</AvatarFallback>
                                            </Avatar>
                                            <button 
                                              className="text-sm font-medium text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300 hover:underline"
                                              onClick={() => handleUserClick(comment.author)}
                                            >
                                              u/{comment.author}
                                            </button>
                                            {comment.is_submitter && <Badge className="text-xs">OP</Badge>}
                                            {comment.distinguished && <Badge className="text-xs bg-green-100 text-green-800">MOD</Badge>}
                                            <span className="text-xs text-gray-500">{formatDate(comment.created_utc)}</span>
                                          </div>
                                          <div className="text-sm text-gray-700 dark:text-gray-300 mb-2">
                                            {renderCommentContent(comment.body)}
                                          </div>
                                          <div className="flex items-center gap-4 text-xs text-gray-500">
                                            <div className="flex items-center gap-1">
                                              <ThumbsUp className="w-3 h-3" />
                                              <span>{comment.score}</span>
                                            </div>
                                            <Button 
                                              variant="ghost" 
                                              size="sm" 
                                              className="h-6 px-2 text-xs"
                                              onClick={() => setReplyingTo(comment.id)}
                                            >
                                              <Reply className="w-3 h-3 mr-1" />
                                              Reply
                                            </Button>
                                          </div>
                                        </div>
                                      ))}
                                    </div>
                                  ) : (
                                    <div className="text-center py-8 text-gray-500">
                                      No comments yet. Be the first to comment!
                                    </div>
                                  )}
                                </ScrollArea>
                                <div className="border-t pt-4 flex-shrink-0">
                                  <div className="space-y-3">
                                    {replyingTo && (
                                      <div className="text-sm text-gray-600 bg-gray-50 dark:bg-gray-800 p-2 rounded flex items-center justify-between">
                                        <span>Replying to comment...</span>
                                        <Button variant="ghost" size="sm" onClick={() => setReplyingTo(null)}>Cancel</Button>
                                      </div>
                                    )}
                                    <Textarea
                                      placeholder={replyingTo ? "Write your reply..." : "Write a comment..."}
                                      value={newCommentText}
                                      onChange={(e) => setNewCommentText(e.target.value)}
                                      rows={3}
                                      className="min-h-[80px] resize-none"
                                    />
                                    <div className="flex flex-col sm:flex-row justify-end gap-2">
                                      <Button 
                                        variant="outline" 
                                        className="order-2 sm:order-1"
                                        onClick={() => {
                                          setNewCommentText('');
                                          setReplyingTo(null);
                                          setShowCommentsDialog(false);
                                          setSelectedPost(null);
                                          setShowComments(false);
                                        }}
                                      >
                                        Cancel
                                      </Button>
                                      <Button 
                                        className="order-1 sm:order-2"
                                        onClick={handlePostComment}
                                        disabled={!newCommentText.trim() || commentMutation.isPending}
                                      >
                                        {commentMutation.isPending ? (
                                          <RefreshCw className="w-4 h-4 animate-spin mr-2" />
                                        ) : (
                                          <Send className="w-4 h-4 mr-2" />
                                        )}
                                        {replyingTo ? 'Reply' : 'Comment'}
                                      </Button>
                                    </div>
                                  </div>
                                </div>
                              </DialogContent>
                            </Dialog>
                            {post.view_count && (
                              <div className="flex items-center gap-1">
                                <Eye className="w-4 h-4" />
                                <span>{formatScore(post.view_count)} views</span>
                              </div>
                            )}
                          </div>
                          <div className="flex items-center gap-2">
                            <Button variant="ghost" size="sm" className="text-xs">
                              <Share2 className="w-3 h-3 mr-1" />
                              Share
                            </Button>
                            <Button variant="ghost" size="sm" className="text-xs">
                              <Bookmark className="w-3 h-3 mr-1" />
                              Save
                            </Button>
                            <Button 
                              variant="ghost" 
                              size="sm" 
                              className="text-yellow-600 hover:text-yellow-700 font-medium text-xs"
                              onClick={() => {
                                setOrderingPost(post);
                                setShowUpvoteDialog(true);
                              }}
                            >
                              <TrendingUp className="w-3 h-3 mr-1" />
                              Gold Upvotes
                            </Button>
                          </div>
                        </div>

                        {(post.removed_by_category || post.banned_by) && (
                          <div className="mt-3 p-2 bg-red-50 dark:bg-red-900/20 border border-red-200 rounded text-sm">
                            <div className="flex items-center gap-2 text-red-600 dark:text-red-400">
                              <Flag className="w-4 h-4" />
                              <span>
                                {post.removed_by_category && `Removed: ${post.removed_by_category}`}
                                {post.banned_by && `Banned by: ${post.banned_by}`}
                              </span>
                            </div>
                            {post.removal_reason && (
                              <p className="mt-1 text-red-600 dark:text-red-400">{post.removal_reason}</p>
                            )}
                          </div>
                        )}
                      </CardContent>
                      
                      {/* Gold Upvotes Orders Display */}
                      <PostUpvotesOrders postId={post.id} />
                    </Card>
                  ))
                )}
              </div>
            </TabsContent>

            <TabsContent value="analytics" className="space-y-6">
              {/* Comprehensive Subreddit Search and Analytics Interface */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Search className="w-5 h-5" />
                    Subreddit Discovery & Analytics
                  </CardTitle>
                  <p className="text-sm text-gray-600">
                    Search for any subreddit and get comprehensive analytics instantly - no need to add to your profile.
                  </p>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="flex gap-2">
                    <div className="flex-1">
                      <Input
                        placeholder="Search subreddits or enter r/subredditname..."
                        value={searchQuery}
                        onChange={(e) => setSearchQuery(e.target.value)}
                        onKeyDown={(e) => e.key === 'Enter' && handleSubredditSearch()}
                        className="w-full"
                      />
                    </div>
                    <Button 
                      onClick={handleSubredditSearch}
                      disabled={!searchQuery.trim() || isSearching}
                      className="flex items-center gap-2"
                    >
                      {isSearching ? (
                        <RefreshCw className="w-4 h-4 animate-spin" />
                      ) : (
                        <Search className="w-4 h-4" />
                      )}
                      Search
                    </Button>
                    <Button 
                      onClick={handleDirectAnalytics}
                      disabled={!searchQuery.trim()}
                      variant="outline"
                      className="flex items-center gap-2"
                    >
                      <BarChart3 className="w-4 h-4" />
                      Get Analytics
                    </Button>
                  </div>

                  {/* Search Results */}
                  {searchResults.length > 0 && (
                    <div className="space-y-3">
                      <h4 className="font-semibold">Search Results</h4>
                      <div className="grid gap-3 max-h-96 overflow-y-auto">
                        {searchResults.map((subreddit) => (
                          <div key={subreddit.id} className="p-4 border rounded-lg hover:bg-gray-50 dark:hover:bg-gray-800">
                            <div className="flex items-center justify-between">
                              <div className="flex-1">
                                <div className="flex items-center gap-2 mb-1">
                                  <h5 className="font-medium">r/{subreddit.name}</h5>
                                  {subreddit.isNsfw && <Badge variant="destructive" className="text-xs">NSFW</Badge>}
                                  <Badge variant="outline" className="text-xs">{subreddit.subredditType}</Badge>
                                </div>
                                <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">{subreddit.title}</p>
                                {subreddit.publicDescription && (
                                  <p className="text-xs text-gray-500 mb-2 line-clamp-2">{subreddit.publicDescription}</p>
                                )}
                                <div className="flex items-center gap-4 text-xs text-gray-500">
                                  <span className="flex items-center gap-1">
                                    <Users className="w-3 h-3" />
                                    {subreddit.subscribers?.toLocaleString() || 'N/A'} subscribers
                                  </span>
                                  <span className="flex items-center gap-1">
                                    <Activity className="w-3 h-3" />
                                    {subreddit.activeUsers?.toLocaleString() || 'N/A'} active
                                  </span>
                                </div>
                              </div>
                              <Button 
                                onClick={() => handleGetAnalytics(subreddit.name)}
                                size="sm"
                                className="ml-4"
                              >
                                <BarChart3 className="w-4 h-4 mr-1" />
                                Analytics
                              </Button>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  )}

                  {/* No results message */}
                  {searchResults.length === 0 && !isSearching && (
                    <div className="text-center py-8">
                      <Search className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                      <p className="text-gray-600">
                        Search for subreddits or enter a specific subreddit name to get instant analytics
                      </p>
                      <p className="text-sm text-gray-500 mt-2">
                        Examples: "gaming", "technology", or "r/MachineLearning"
                      </p>
                    </div>
                  )}
                </CardContent>
              </Card>

              {/* Existing Profile Analytics if available */}
              <div className="grid gap-4">
                {analytics.length === 0 ? (
                  <Card>
                    <CardContent className="py-8 text-center">
                      <BarChart3 className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                      <p className="text-gray-600">No saved analytics yet. Use the search above to analyze any subreddit!</p>
                    </CardContent>
                  </Card>
                ) : (
                  analytics.map((data: SubredditAnalysis) => (
                    <Card key={data.id}>
                      <CardHeader>
                        <CardTitle>r/{data.subredditName}</CardTitle>
                        <CardDescription>Saved Subreddit Analytics</CardDescription>
                      </CardHeader>
                      <CardContent>
                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                          <div className="text-center">
                            <div className="text-2xl font-bold text-blue-600">
                              {data.subscribers.toLocaleString()}
                            </div>
                            <div className="text-sm text-gray-600">Subscribers</div>
                          </div>
                          <div className="text-center">
                            <div className="text-2xl font-bold text-green-600">
                              {data.activeUsers.toLocaleString()}
                            </div>
                            <div className="text-sm text-gray-600">Active Users</div>
                          </div>
                          <div className="text-center">
                            <div className="text-2xl font-bold text-purple-600">
                              {data.growthRate.toFixed(1)}%
                            </div>
                            <div className="text-sm text-gray-600">Growth Rate</div>
                          </div>
                          <div className="text-center">
                            <div className="text-2xl font-bold text-orange-600">
                              {data.engagementRate.toFixed(1)}%
                            </div>
                            <div className="text-sm text-gray-600">Engagement</div>
                          </div>
                        </div>
                        
                        <div className="mt-6 grid md:grid-cols-2 gap-4">
                          <div>
                            <h4 className="font-semibold mb-2">Best Post Times</h4>
                            <div className="flex flex-wrap gap-2">
                              {data.bestPostTimes.map((time, index) => (
                                <Badge key={index} variant="secondary">{time}</Badge>
                              ))}
                            </div>
                          </div>
                          <div>
                            <h4 className="font-semibold mb-2">Top Keywords</h4>
                            <div className="flex flex-wrap gap-2">
                              {data.topKeywords.map((keyword, index) => (
                                <Badge key={index} variant="outline">{keyword}</Badge>
                              ))}
                            </div>
                          </div>
                        </div>
                      </CardContent>
                    </Card>
                  ))
                )}
              </div>
            </TabsContent>
          </Tabs>
        </div>
      </div>

      {/* Gold Upvotes Dialog */}
      <Dialog open={showUpvoteDialog} onOpenChange={setShowUpvoteDialog}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center">
              <TrendingUp className="h-5 w-5 mr-2 text-yellow-600" />
              Purchase Reddit Upvotes
            </DialogTitle>
            <div className="text-sm text-gray-600 dark:text-gray-400">
              Boost your post's visibility with high-quality Reddit upvotes
            </div>
          </DialogHeader>
          
          {orderingPost && (
            <div className="space-y-4">
              <div className="p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
                <p className="font-medium text-sm">{orderingPost.title}</p>
                <p className="text-xs text-gray-500 mt-1">
                  r/{orderingPost.subreddit} • {orderingPost.score} points
                </p>
              </div>
              
              <div className="space-y-2">
                <Label htmlFor="quantity">Quantity (minimum 10)</Label>
                <Select value={upvoteQuantity} onValueChange={setUpvoteQuantity}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="10">10 upvotes</SelectItem>
                    <SelectItem value="25">25 upvotes</SelectItem>
                    <SelectItem value="50">50 upvotes</SelectItem>
                    <SelectItem value="100">100 upvotes</SelectItem>
                    <SelectItem value="250">250 upvotes</SelectItem>
                    <SelectItem value="500">500 upvotes</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="bg-yellow-50 dark:bg-yellow-900/20 p-3 rounded-lg">
                <div className="flex items-center justify-between text-sm">
                  <span>Cost:</span>
                  <span className="font-medium">2 Points</span>
                </div>
                <p className="text-xs text-yellow-700 dark:text-yellow-300 mt-1">
                  Fixed cost of 2 points per order, regardless of quantity
                </p>
              </div>

              <div className="flex gap-2">
                <Button
                  variant="outline"
                  onClick={() => {
                    setShowUpvoteDialog(false);
                    setOrderingPost(null);
                    setUpvoteQuantity('10');
                  }}
                  className="flex-1"
                >
                  Cancel
                </Button>
                <Button
                  onClick={() => {
                    if (orderingPost) {
                      redupvotesMutation.mutate({
                        redditPostId: orderingPost.id,
                        redditPostUrl: `https://reddit.com${orderingPost.permalink}`,
                        quantity: parseInt(upvoteQuantity)
                      });
                    }
                  }}
                  disabled={redupvotesMutation.isPending}
                  className="flex-1 bg-yellow-600 hover:bg-yellow-700"
                >
                  {redupvotesMutation.isPending ? (
                    <>
                      <RefreshCw className="h-4 w-4 mr-2 animate-spin" />
                      Placing Order...
                    </>
                  ) : (
                    <>
                      <TrendingUp className="h-4 w-4 mr-2" />
                      Place Order
                    </>
                  )}
                </Button>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* User Profile Modal */}
      <Dialog open={showUserProfile} onOpenChange={setShowUserProfile}>
        <DialogContent className="w-[95vw] max-w-4xl h-[90vh] max-h-[90vh] overflow-y-auto p-4 sm:p-6">
          <DialogHeader>
            <DialogTitle className="text-left flex items-center gap-2">
              <Avatar className="w-8 h-8">
                {userProfile?.icon_img ? (
                  <AvatarImage src={userProfile.icon_img} alt={userProfile.name} />
                ) : (
                  <AvatarFallback>{selectedUser?.[0]?.toUpperCase()}</AvatarFallback>
                )}
              </Avatar>
              Reddit User Profile: u/{selectedUser}
            </DialogTitle>
          </DialogHeader>
          
          {userProfileLoading ? (
            <div className="flex items-center justify-center py-8">
              <RefreshCw className="w-6 h-6 animate-spin mr-2" />
              <span>Loading user profile...</span>
            </div>
          ) : userProfile ? (
            <div className="space-y-6">
              {/* Profile Header */}
              <div className="flex items-start gap-4 p-4 bg-gradient-to-r from-blue-50 to-purple-50 dark:from-blue-900/20 dark:to-purple-900/20 rounded-lg">
                <Avatar className="w-16 h-16">
                  {userProfile.icon_img ? (
                    <AvatarImage src={userProfile.icon_img} alt={userProfile.name} />
                  ) : (
                    <AvatarFallback className="text-xl">{userProfile.name[0].toUpperCase()}</AvatarFallback>
                  )}
                </Avatar>
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-2">
                    <h3 className="text-2xl font-bold">u/{userProfile.name}</h3>
                    {userProfile.is_employee && <Badge className="bg-red-100 text-red-800">Reddit Admin</Badge>}
                    {userProfile.is_mod && <Badge className="bg-green-100 text-green-800">Moderator</Badge>}
                    {userProfile.is_premium && <Badge className="bg-yellow-100 text-yellow-800">Premium</Badge>}
                    {userProfile.is_gold && <Badge className="bg-amber-100 text-amber-800">Gold</Badge>}
                    {userProfile.verified && <Badge className="bg-blue-100 text-blue-800">Verified</Badge>}
                  </div>
                  {userProfile.public_description && (
                    <p className="text-gray-600 dark:text-gray-300 mb-2">{userProfile.public_description}</p>
                  )}
                  <div className="text-sm text-gray-500">
                    Redditor since {new Date(userProfile.created_utc * 1000).toLocaleDateString()}
                  </div>
                </div>
              </div>

              {/* Karma Stats */}
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <div className="text-center p-4 bg-orange-50 dark:bg-orange-900/20 rounded-lg">
                  <div className="text-2xl font-bold text-orange-600">{userProfile.link_karma.toLocaleString()}</div>
                  <div className="text-sm text-gray-600 dark:text-gray-400">Post Karma</div>
                </div>
                <div className="text-center p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                  <div className="text-2xl font-bold text-blue-600">{userProfile.comment_karma.toLocaleString()}</div>
                  <div className="text-sm text-gray-600 dark:text-gray-400">Comment Karma</div>
                </div>
                <div className="text-center p-4 bg-purple-50 dark:bg-purple-900/20 rounded-lg">
                  <div className="text-2xl font-bold text-purple-600">{userProfile.awarder_karma.toLocaleString()}</div>
                  <div className="text-sm text-gray-600 dark:text-gray-400">Awarder Karma</div>
                </div>
                <div className="text-center p-4 bg-green-50 dark:bg-green-900/20 rounded-lg">
                  <div className="text-2xl font-bold text-green-600">{userProfile.awardee_karma.toLocaleString()}</div>
                  <div className="text-sm text-gray-600 dark:text-gray-400">Awardee Karma</div>
                </div>
              </div>

              {/* Account Status */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Shield className="w-5 h-5" />
                    Account Status
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid md:grid-cols-2 gap-4">
                    <div className="space-y-3">
                      <div className="flex justify-between">
                        <span>Email Verified:</span>
                        <Badge variant={userProfile.has_verified_email ? "default" : "secondary"}>
                          {userProfile.has_verified_email ? "Yes" : "No"}
                        </Badge>
                      </div>
                      <div className="flex justify-between">
                        <span>Account Verified:</span>
                        <Badge variant={userProfile.verified ? "default" : "secondary"}>
                          {userProfile.verified ? "Yes" : "No"}
                        </Badge>
                      </div>
                      <div className="flex justify-between">
                        <span>Accepts Followers:</span>
                        <Badge variant={userProfile.accept_followers ? "default" : "secondary"}>
                          {userProfile.accept_followers ? "Yes" : "No"}
                        </Badge>
                      </div>
                      <div className="flex justify-between">
                        <span>Accepts PMs:</span>
                        <Badge variant={userProfile.accept_pms ? "default" : "secondary"}>
                          {userProfile.accept_pms ? "Yes" : "No"}
                        </Badge>
                      </div>
                    </div>
                    <div className="space-y-3">
                      <div className="flex justify-between">
                        <span>Accepts Chats:</span>
                        <Badge variant={userProfile.accept_chats ? "default" : "secondary"}>
                          {userProfile.accept_chats ? "Yes" : "No"}
                        </Badge>
                      </div>
                      <div className="flex justify-between">
                        <span>Hidden from Robots:</span>
                        <Badge variant={userProfile.hide_from_robots ? "destructive" : "default"}>
                          {userProfile.hide_from_robots ? "Yes" : "No"}
                        </Badge>
                      </div>
                      {userProfile.is_suspended && (
                        <div className="flex justify-between">
                          <span>Account Status:</span>
                          <Badge variant="destructive">Suspended</Badge>
                        </div>
                      )}
                      {userProfile.coins > 0 && (
                        <div className="flex justify-between">
                          <span>Reddit Coins:</span>
                          <Badge className="bg-yellow-100 text-yellow-800">{userProfile.coins}</Badge>
                        </div>
                      )}
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Subscriptions */}
              {(userProfile.has_subscribed || userProfile.has_subscribed_to_premium) && (
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <Star className="w-5 h-5" />
                      Subscriptions
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-2">
                      {userProfile.has_subscribed_to_premium && (
                        <Badge className="bg-purple-100 text-purple-800">Reddit Premium Subscriber</Badge>
                      )}
                      {userProfile.has_subscribed && (
                        <Badge className="bg-blue-100 text-blue-800">Has Active Subscriptions</Badge>
                      )}
                    </div>
                  </CardContent>
                </Card>
              )}

              {/* Profile Images */}
              {(userProfile.profile_img || userProfile.banner_img || userProfile.snoovatar_img) && (
                <Card>
                  <CardHeader>
                    <CardTitle>Profile Media</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="grid md:grid-cols-3 gap-4">
                      {userProfile.profile_img && (
                        <div>
                          <h4 className="font-medium mb-2">Profile Image</h4>
                          <img src={userProfile.profile_img} alt="Profile" className="w-full h-32 object-cover rounded" />
                        </div>
                      )}
                      {userProfile.snoovatar_img && (
                        <div>
                          <h4 className="font-medium mb-2">Snoovatar</h4>
                          <img src={userProfile.snoovatar_img} alt="Snoovatar" className="w-full h-32 object-cover rounded" />
                        </div>
                      )}
                      {userProfile.banner_img && (
                        <div>
                          <h4 className="font-medium mb-2">Banner</h4>
                          <img src={userProfile.banner_img} alt="Banner" className="w-full h-32 object-cover rounded" />
                        </div>
                      )}
                    </div>
                  </CardContent>
                </Card>
              )}

              {/* Actions */}
              <div className="flex gap-2 pt-4">
                <Button
                  variant="outline"
                  onClick={() => window.open(`https://reddit.com/u/${userProfile.name}`, '_blank')}
                >
                  <ExternalLink className="w-4 h-4 mr-2" />
                  View on Reddit
                </Button>
                <Button
                  variant="outline"
                  onClick={() => setShowUserProfile(false)}
                >
                  Close
                </Button>
              </div>
            </div>
          ) : (
            <div className="text-center py-8 text-gray-500">
              Failed to load user profile. User may not exist or be private.
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Comprehensive Subreddit Analytics Modal */}
      <Dialog open={showAnalyticsModal} onOpenChange={setShowAnalyticsModal}>
        <DialogContent className="w-[95vw] max-w-6xl h-[90vh] max-h-[90vh] overflow-y-auto p-4 sm:p-6">
          <DialogHeader>
            <DialogTitle className="text-left flex items-center gap-2">
              <BarChart3 className="w-6 h-6" />
              {selectedSubreddit ? `r/${selectedSubreddit} Analytics` : 'Subreddit Analytics'}
            </DialogTitle>
          </DialogHeader>
          
          {isLoadingAnalytics ? (
            <div className="flex items-center justify-center py-12">
              <RefreshCw className="w-8 h-8 animate-spin mr-3" />
              <span className="text-lg">Analyzing subreddit data...</span>
            </div>
          ) : subredditAnalytics ? (
            <div className="space-y-6">
              {/* Basic Info Header */}
              <div className="p-6 bg-gradient-to-r from-blue-50 to-purple-50 dark:from-blue-900/20 dark:to-purple-900/20 rounded-lg">
                <div className="flex items-start justify-between">
                  <div>
                    <h2 className="text-2xl font-bold mb-2">{subredditAnalytics.displayName}</h2>
                    <p className="text-lg text-gray-600 dark:text-gray-300 mb-1">{subredditAnalytics.title}</p>
                    {subredditAnalytics.publicDescription && (
                      <p className="text-sm text-gray-500 max-w-2xl">{subredditAnalytics.publicDescription}</p>
                    )}
                  </div>
                  <div className="flex gap-2">
                    {subredditAnalytics.isNsfw && <Badge variant="destructive">NSFW</Badge>}
                    <Badge variant="outline">{subredditAnalytics.subredditType}</Badge>
                    <Badge variant="secondary">{subredditAnalytics.language?.toUpperCase()}</Badge>
                  </div>
                </div>
              </div>

              {/* Key Metrics */}
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <Card>
                  <CardContent className="p-4 text-center">
                    <div className="text-3xl font-bold text-blue-600 mb-1">
                      {subredditAnalytics.subscribers.toLocaleString()}
                    </div>
                    <div className="text-sm text-gray-600">Subscribers</div>
                  </CardContent>
                </Card>
                <Card>
                  <CardContent className="p-4 text-center">
                    <div className="text-3xl font-bold text-green-600 mb-1">
                      {subredditAnalytics.activeUsers.toLocaleString()}
                    </div>
                    <div className="text-sm text-gray-600">Active Users</div>
                  </CardContent>
                </Card>
                <Card>
                  <CardContent className="p-4 text-center">
                    <div className="text-3xl font-bold text-orange-600 mb-1">
                      {subredditAnalytics.averageUpvotes.toLocaleString()}
                    </div>
                    <div className="text-sm text-gray-600">Avg Upvotes</div>
                  </CardContent>
                </Card>
                <Card>
                  <CardContent className="p-4 text-center">
                    <div className="text-3xl font-bold text-purple-600 mb-1">
                      {subredditAnalytics.engagementRate}%
                    </div>
                    <div className="text-sm text-gray-600">Engagement Rate</div>
                  </CardContent>
                </Card>
              </div>

              {/* Activity & Growth */}
              <div className="grid md:grid-cols-2 gap-6">
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <TrendingUp className="w-5 h-5" />
                      Activity Metrics
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4">
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <div className="text-2xl font-bold text-red-600">
                          {subredditAnalytics.averageComments}
                        </div>
                        <div className="text-sm text-gray-600">Avg Comments</div>
                      </div>
                      <div>
                        <div className="text-2xl font-bold text-indigo-600">
                          {subredditAnalytics.dailyPostingRate}
                        </div>
                        <div className="text-sm text-gray-600">Daily Posts</div>
                      </div>
                    </div>
                    <Separator />
                    <div>
                      <div className="text-lg font-semibold text-green-600">
                        {subredditAnalytics.estimatedGrowthRate}%
                      </div>
                      <div className="text-sm text-gray-600">Estimated Growth Rate</div>
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <Clock className="w-5 h-5" />
                      Best Posting Times
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="flex flex-wrap gap-2">
                      {subredditAnalytics.bestPostingHours.map((time: string, index: number) => (
                        <Badge key={index} variant="secondary" className="text-sm">
                          {time}
                        </Badge>
                      ))}
                    </div>
                  </CardContent>
                </Card>
              </div>

              {/* Content Analysis */}
              <div className="grid md:grid-cols-2 gap-6">
                <Card>
                  <CardHeader>
                    <CardTitle>Content Types</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-3">
                      {Object.entries(subredditAnalytics.postTypes).map(([type, count]) => (
                        <div key={type} className="flex justify-between items-center">
                          <span className="capitalize font-medium">{type}</span>
                          <Badge variant="outline">{count as number}</Badge>
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardHeader>
                    <CardTitle>Popular Keywords</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="flex flex-wrap gap-2">
                      {subredditAnalytics.topKeywords.slice(0, 15).map((keyword: string, index: number) => (
                        <Badge key={index} variant="outline" className="text-xs">
                          {keyword}
                        </Badge>
                      ))}
                    </div>
                  </CardContent>
                </Card>
              </div>

              {/* Subreddit Features */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Shield className="w-5 h-5" />
                    Subreddit Features & Rules
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid md:grid-cols-2 gap-6">
                    <div className="space-y-3">
                      <h4 className="font-semibold">Content Permissions</h4>
                      <div className="space-y-2">
                        <div className="flex justify-between">
                          <span>Images Allowed:</span>
                          <Badge variant={subredditAnalytics.allowImages ? "default" : "secondary"}>
                            {subredditAnalytics.allowImages ? "Yes" : "No"}
                          </Badge>
                        </div>
                        <div className="flex justify-between">
                          <span>Videos Allowed:</span>
                          <Badge variant={subredditAnalytics.allowVideos ? "default" : "secondary"}>
                            {subredditAnalytics.allowVideos ? "Yes" : "No"}
                          </Badge>
                        </div>
                        <div className="flex justify-between">
                          <span>Polls Allowed:</span>
                          <Badge variant={subredditAnalytics.allowPolls ? "default" : "secondary"}>
                            {subredditAnalytics.allowPolls ? "Yes" : "No"}
                          </Badge>
                        </div>
                        <div className="flex justify-between">
                          <span>Galleries Allowed:</span>
                          <Badge variant={subredditAnalytics.allowGalleries ? "default" : "secondary"}>
                            {subredditAnalytics.allowGalleries ? "Yes" : "No"}
                          </Badge>
                        </div>
                      </div>
                    </div>
                    <div className="space-y-3">
                      <h4 className="font-semibold">Moderation</h4>
                      <div className="space-y-2">
                        <div className="flex justify-between">
                          <span>Rules Count:</span>
                          <Badge variant="outline">{subredditAnalytics.rulesCount}</Badge>
                        </div>
                        <div className="flex justify-between">
                          <span>Created:</span>
                          <span className="text-sm text-gray-600">
                            {new Date(subredditAnalytics.createdUtc * 1000).toLocaleDateString()}
                          </span>
                        </div>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Top Performing Posts */}
              {subredditAnalytics.topPosts && subredditAnalytics.topPosts.length > 0 && (
                <Card>
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <Star className="w-5 h-5" />
                      Top Performing Posts
                    </CardTitle>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-4">
                      {subredditAnalytics.topPosts.slice(0, 5).map((post: any, index: number) => (
                        <div key={index} className="p-4 border rounded-lg">
                          <div className="flex items-start justify-between mb-2">
                            <h4 className="font-medium text-sm line-clamp-2 flex-1">{post.title}</h4>
                            <Badge variant="outline" className="ml-2 text-xs">
                              {post.is_self ? 'Text' : 'Link'}
                            </Badge>
                          </div>
                          <div className="flex items-center gap-4 text-xs text-gray-600">
                            <span>u/{post.author}</span>
                            <span className="flex items-center gap-1">
                              <ArrowUp className="w-3 h-3" />
                              {post.upvotes}
                            </span>
                            <span className="flex items-center gap-1">
                              <MessageSquare className="w-3 h-3" />
                              {post.comments}
                            </span>
                            <span>{new Date(post.created_utc * 1000).toLocaleDateString()}</span>
                          </div>
                        </div>
                      ))}
                    </div>
                  </CardContent>
                </Card>
              )}

              {/* PDF Export Section */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Download className="w-5 h-5" />
                    Export Analytics
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <PDFExport 
                    analytics={subredditAnalytics} 
                    subredditName={selectedSubreddit || ''} 
                    onExport={() => {
                      toast({
                        title: "PDF exported",
                        description: "Analytics report has been downloaded successfully.",
                      });
                    }}
                  />
                </CardContent>
              </Card>

              {/* Analysis Footer */}
              <div className="text-center py-4 border-t">
                <p className="text-sm text-gray-500">
                  Analysis based on {subredditAnalytics.totalPostsAnalyzed} recent posts • 
                  Generated in {subredditAnalytics.processingTime}ms • 
                  {new Date(subredditAnalytics.analysisTimestamp).toLocaleString()}
                </p>
                <div className="flex gap-2 justify-center mt-4">
                  <Button
                    onClick={handleAddToManaged}
                    disabled={isAddingToManaged}
                    className="bg-blue-600 hover:bg-blue-700"
                  >
                    <UserPlus className="w-4 h-4 mr-2" />
                    {isAddingToManaged ? 'Adding...' : 'Add to Managed'}
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => window.open(`https://reddit.com/r/${selectedSubreddit}`, '_blank')}
                  >
                    <ExternalLink className="w-4 h-4 mr-2" />
                    View on Reddit
                  </Button>
                  <Button
                    variant="outline"
                    onClick={() => setShowAnalyticsModal(false)}
                  >
                    Close
                  </Button>
                </div>
              </div>
            </div>
          ) : (
            <div className="text-center py-12">
              <BarChart3 className="w-16 h-16 mx-auto text-gray-400 mb-4" />
              <p className="text-lg text-gray-600">Failed to load analytics</p>
              <p className="text-sm text-gray-500 mt-2">
                The subreddit may be private, banned, or doesn't exist.
              </p>
              <Button
                variant="outline"
                onClick={() => setShowAnalyticsModal(false)}
                className="mt-4"
              >
                Close
              </Button>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </DashboardLayout>
  );
}