import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { AlertTriangle, Shield, Users, Globe, Clock, Ban } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { Alert, AlertDescription } from "@/components/ui/alert";

interface SecurityStats {
  totalDeviceFingerprints: number;
  totalIpAddresses: number;
  totalViolations: number;
  blockedDevices: number;
  registrationsToday: number;
  violationsToday: number;
  suspiciousActivities: number;
  averageDevicesPerHousehold: number;
}

interface SecurityViolation {
  id: number;
  violationType: string;
  description: string;
  severity: string;
  action: string;
  userId?: number;
  deviceFingerprint?: string;
  ipAddress?: string;
  userAgent?: string;
  metadata?: any;
  createdAt: string;
}

export default function SecurityDashboard() {
  const [showBlockDialog, setShowBlockDialog] = useState(false);
  const [blockData, setBlockData] = useState({ fingerprint: "", reason: "" });
  const { toast } = useToast();

  const { data: stats, isLoading: statsLoading } = useQuery<SecurityStats>({
    queryKey: ["/api/admin/security/stats"],
    refetchInterval: 30000 // Refresh every 30 seconds
  });

  const { data: violations, isLoading: violationsLoading } = useQuery<SecurityViolation[]>({
    queryKey: ["/api/admin/security/violations"],
    refetchInterval: 10000 // Refresh every 10 seconds
  });

  const blockDeviceMutation = useMutation({
    mutationFn: async (data: { fingerprint: string; reason: string }) => {
      const response = await apiRequest("/api/admin/security/block-device", "POST", data);
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to block device");
      }
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Device Blocked",
        description: "The device fingerprint has been successfully blocked.",
        variant: "default"
      });
      setShowBlockDialog(false);
      setBlockData({ fingerprint: "", reason: "" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/security/stats"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/security/violations"] });
    },
    onError: (error: any) => {
      toast({
        title: "Block Failed",
        description: error.message,
        variant: "destructive"
      });
    }
  });

  const handleBlockDevice = (fingerprint: string) => {
    setBlockData({ fingerprint, reason: "" });
    setShowBlockDialog(true);
  };

  const getSeverityColor = (severity: string) => {
    switch (severity) {
      case "high": return "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200";
      case "medium": return "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200";
      case "low": return "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200";
      default: return "bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200";
    }
  };

  const getViolationIcon = (type: string) => {
    switch (type) {
      case "duplicate_household": return <Users className="h-4 w-4" />;
      case "blocked_device": return <Ban className="h-4 w-4" />;
      case "suspicious_location": return <Globe className="h-4 w-4" />;
      default: return <AlertTriangle className="h-4 w-4" />;
    }
  };

  return (
    <div className="container mx-auto px-4 py-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-slate-900 dark:text-white">Security Dashboard</h1>
          <p className="text-slate-600 dark:text-slate-400">Monitor and manage fraud detection system</p>
        </div>
        <Button 
          onClick={() => setShowBlockDialog(true)}
          variant="destructive"
          className="flex items-center gap-2"
        >
          <Ban className="h-4 w-4" />
          Block Device
        </Button>
      </div>

      <Tabs defaultValue="overview" className="space-y-6">
        <TabsList className="grid w-full grid-cols-3">
          <TabsTrigger value="overview">Overview</TabsTrigger>
          <TabsTrigger value="violations">Violations</TabsTrigger>
          <TabsTrigger value="management">Device Management</TabsTrigger>
        </TabsList>

        <TabsContent value="overview" className="space-y-6">
          {statsLoading ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              {[...Array(8)].map((_, i) => (
                <Card key={i} className="animate-pulse">
                  <CardHeader className="pb-2">
                    <div className="h-4 bg-slate-200 dark:bg-slate-700 rounded w-3/4"></div>
                  </CardHeader>
                  <CardContent>
                    <div className="h-8 bg-slate-200 dark:bg-slate-700 rounded w-1/2"></div>
                  </CardContent>
                </Card>
              ))}
            </div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Device Fingerprints</CardTitle>
                  <Shield className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats ? stats.totalDeviceFingerprints : 0}</div>
                  <p className="text-xs text-muted-foreground">Total registered devices</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">IP Addresses</CardTitle>
                  <Globe className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats ? stats.totalIpAddresses : 0}</div>
                  <p className="text-xs text-muted-foreground">Unique IP addresses tracked</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Security Violations</CardTitle>
                  <AlertTriangle className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats ? stats.totalViolations : 0}</div>
                  <p className="text-xs text-muted-foreground">Total violations detected</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Blocked Devices</CardTitle>
                  <Ban className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats ? stats.blockedDevices : 0}</div>
                  <p className="text-xs text-muted-foreground">Devices currently blocked</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Today's Registrations</CardTitle>
                  <Users className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats ? stats.registrationsToday : 0}</div>
                  <p className="text-xs text-muted-foreground">New accounts created today</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Today's Violations</CardTitle>
                  <AlertTriangle className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats ? stats.violationsToday : 0}</div>
                  <p className="text-xs text-muted-foreground">Violations detected today</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Suspicious Activities</CardTitle>
                  <Clock className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats ? stats.suspiciousActivities : 0}</div>
                  <p className="text-xs text-muted-foreground">Pending review</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                  <CardTitle className="text-sm font-medium">Avg Devices/Household</CardTitle>
                  <Shield className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold">{stats ? stats.averageDevicesPerHousehold.toFixed(1) : "0.0"}</div>
                  <p className="text-xs text-muted-foreground">Average per location</p>
                </CardContent>
              </Card>
            </div>
          )}
        </TabsContent>

        <TabsContent value="violations" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Recent Security Violations</CardTitle>
              <CardDescription>
                Latest security incidents and fraud attempts detected by the system
              </CardDescription>
            </CardHeader>
            <CardContent>
              {violationsLoading ? (
                <div className="space-y-4">
                  {[...Array(5)].map((_, i) => (
                    <div key={i} className="flex items-center space-x-4 p-4 border rounded-lg animate-pulse">
                      <div className="h-10 w-10 bg-slate-200 dark:bg-slate-700 rounded-full"></div>
                      <div className="flex-1 space-y-2">
                        <div className="h-4 bg-slate-200 dark:bg-slate-700 rounded w-3/4"></div>
                        <div className="h-3 bg-slate-200 dark:bg-slate-700 rounded w-1/2"></div>
                      </div>
                    </div>
                  ))}
                </div>
              ) : violations && violations.length > 0 ? (
                <div className="space-y-4">
                  {violations.map((violation: SecurityViolation) => (
                    <div key={violation.id} className="flex items-start space-x-4 p-4 border rounded-lg hover:bg-slate-50 dark:hover:bg-slate-800">
                      <div className="flex-shrink-0 mt-1">
                        {getViolationIcon(violation.violationType)}
                      </div>
                      <div className="flex-1 min-w-0">
                        <div className="flex items-center justify-between">
                          <p className="text-sm font-medium text-slate-900 dark:text-white">
                            {violation.violationType.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase())}
                          </p>
                          <div className="flex items-center space-x-2">
                            <Badge className={getSeverityColor(violation.severity)}>
                              {violation.severity}
                            </Badge>
                            <span className="text-xs text-slate-500">
                              {new Date(violation.createdAt).toLocaleString()}
                            </span>
                          </div>
                        </div>
                        <p className="text-sm text-slate-600 dark:text-slate-400 mt-1">
                          {violation.description}
                        </p>
                        <div className="flex items-center space-x-4 mt-2 text-xs text-slate-500">
                          {violation.ipAddress && (
                            <span>IP: {violation.ipAddress}</span>
                          )}
                          {violation.deviceFingerprint && (
                            <span>Device: {violation.deviceFingerprint.substring(0, 16)}...</span>
                          )}
                          {violation.action && (
                            <span className="bg-slate-100 dark:bg-slate-700 px-2 py-1 rounded">
                              Action: {violation.action}
                            </span>
                          )}
                        </div>
                        {violation.deviceFingerprint && (
                          <div className="mt-2">
                            <Button
                              size="sm"
                              variant="destructive"
                              onClick={() => handleBlockDevice(violation.deviceFingerprint!)}
                            >
                              Block Device
                            </Button>
                          </div>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <Alert>
                  <Shield className="h-4 w-4" />
                  <AlertDescription>
                    No security violations detected. The system is operating normally.
                  </AlertDescription>
                </Alert>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="management" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Device Management</CardTitle>
              <CardDescription>
                Manage blocked devices and monitor suspicious activities
              </CardDescription>
            </CardHeader>
            <CardContent>
              <Alert>
                <AlertTriangle className="h-4 w-4" />
                <AlertDescription>
                  Device blocking is permanent and cannot be undone. Use with caution.
                </AlertDescription>
              </Alert>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      <Dialog open={showBlockDialog} onOpenChange={setShowBlockDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Block Device Fingerprint</DialogTitle>
            <DialogDescription>
              This will permanently block the device from creating new accounts.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="fingerprint">Device Fingerprint</Label>
              <Input
                id="fingerprint"
                value={blockData.fingerprint}
                onChange={(e) => setBlockData(prev => ({ ...prev, fingerprint: e.target.value }))}
                placeholder="Enter device fingerprint to block"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="reason">Reason for Blocking</Label>
              <Input
                id="reason"
                value={blockData.reason}
                onChange={(e) => setBlockData(prev => ({ ...prev, reason: e.target.value }))}
                placeholder="Enter reason for blocking this device"
              />
            </div>
            <div className="flex justify-end space-x-2">
              <Button variant="outline" onClick={() => setShowBlockDialog(false)}>
                Cancel
              </Button>
              <Button
                variant="destructive"
                onClick={() => blockDeviceMutation.mutate(blockData)}
                disabled={!blockData.fingerprint || !blockData.reason || blockDeviceMutation.isPending}
              >
                {blockDeviceMutation.isPending ? "Blocking..." : "Block Device"}
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}