import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { DashboardLayout } from "@/components/dashboard-layout";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";
import { Search, Plus, TrendingUp, Users, Eye, EyeOff, MessageSquare, Star, Shield, Trash2, ExternalLink, RefreshCw, Download, Check, Zap, ChevronDown, ChevronRight, Send, Reply, MoreHorizontal, ArrowUp, ArrowDown, Edit, Flag, Pin, Lock, Archive, AlertTriangle } from "lucide-react";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

interface Subreddit {
  id: string;
  name: string;
  displayName: string;
  title: string;
  description: string;
  subscribers: number;
  activeUsers: number;
  created: number;
  isNsfw: boolean;
  publicDescription: string;
  subredditType: string;
  url: string;
  // Additional Reddit API fields
  allowVideos?: boolean;
  allowImages?: boolean;
  communityIcon?: string;
  bannerImg?: string;
  primaryColor?: string;
  lang?: string;
  whitelist_status?: string;
  wls?: number;
  hide_ads?: boolean;
  prediction_leaderboard_entry_type?: string;
  emojis_enabled?: boolean;
  advertiser_category?: string;
  public_traffic?: boolean;
  all_original_content?: boolean;
  link_flair_enabled?: boolean;
  disable_contributor_requests?: boolean;
  subreddit_type?: string;
  user_is_subscriber?: boolean;
  user_is_moderator?: boolean;
  accounts_active?: number;
  comment_score_hide_mins?: number;
  allow_polls?: boolean;
}

interface SubredditAnalytics {
  id: number;
  subredditName: string;
  subscribers: number;
  activeUsers: number;
  growthRate: number;
  engagementRate: number;
  avgUpvotes: number;
  avgComments: number;
  topPostTime: string;
  contentCategories: string[];
  moderatorCount: number;
  weeklyPosts: number;
  lastUpdated: string;
}

interface ManagedSubreddit {
  id: number;
  subredditName: string;
  displayName: string;
  title: string;
  description: string;
  subscribers: number;
  activeUsers: number;
  isNsfw: boolean;
  subredditType: string;
  isModerator: boolean;
  createdAt: string;
  updatedAt: string;
  analytics?: SubredditAnalytics;
}

interface ModeratedSubreddit {
  id: string;
  name: string;
  displayName: string;
  subscribers: number;
  permissions: string[];
  unmoderatedCount: number;
  reportCount: number;
  modmailCount: number;
}

interface RecentPost {
  id: string;
  fullname: string;
  title: string;
  content: string;
  author: string;
  url?: string;
  permalink: string;
  score: number;
  numComments: number;
  created: string;
  createdUtc: number;
  subreddit: string;
  postHint?: string;
  thumbnail?: string;
  isNsfw: boolean;
  stickied: boolean;
  locked: boolean;
  archived: boolean;
}

interface PostAnalysisResult {
  id: number;
  suggestedAction: 'approve' | 'remove';
  reasoning: string;
  confidenceScore: number;
  severity: 'low' | 'medium' | 'high';
  violationsFound: Array<{
    ruleIndex: number;
    ruleName: string;
    violation: string;
    severity: 'low' | 'medium' | 'high';
  }>;
}

interface SubredditPost {
  id: string;
  fullname: string;
  title: string;
  content: string;
  author: string;
  url?: string;
  permalink: string;
  score: number;
  numComments: number;
  created: string;
  createdUtc: number;
  subreddit: string;
  postHint?: string;
  thumbnail?: string;
  isNsfw: boolean;
  stickied: boolean;
  locked: boolean;
  archived: boolean;
  upvoteRatio: number;
  gilded: number;
  flair?: string;
  linkFlairText?: string;
  domain?: string;
  isVideo: boolean;
  isSelf: boolean;
  distinguished?: string;
  spoiler: boolean;
  crosspostParent?: string;
}

interface Comment {
  id: string;
  fullname: string;
  author: string;
  body: string;
  bodyHtml: string;
  score: number;
  created: string;
  createdUtc: number;
  permalink: string;
  linkId: string;
  parentId: string;
  subreddit: string;
  depth: number;
  replies: Comment[];
  collapsed: boolean;
  distinguished?: string;
  stickied: boolean;
  gilded: number;
  controversiality: number;
  edited: boolean | number;
  archived: boolean;
  scoreHidden: boolean;
  canGild: boolean;
  canMod: boolean;
  isSubmitter: boolean;
}

interface NewPostData {
  title: string;
  content: string;
  type: 'text' | 'link';
  url?: string;
  flair?: string;
  spoiler: boolean;
  nsfw: boolean;
  sendReplies: boolean;
}

interface NewCommentData {
  content: string;
  postId?: string;
  parentId?: string;
}

export default function SubredditManager() {
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [searchQuery, setSearchQuery] = useState('');
  const [searchResults, setSearchResults] = useState<Subreddit[]>([]);
  const [isSearching, setIsSearching] = useState(false);
  
  // Proactive Moderation State
  const [selectedProactiveSubreddit, setSelectedProactiveSubreddit] = useState('');
  const [recentPosts, setRecentPosts] = useState<RecentPost[]>([]);
  const [postAnalysisResults, setPostAnalysisResults] = useState<{[postId: string]: PostAnalysisResult}>({});
  const [isAnalyzingPost, setIsAnalyzingPost] = useState(false);
  const [isBulkAnalyzing, setIsBulkAnalyzing] = useState(false);
  
  // Pagination state with ignored posts filtering
  const [currentPage, setCurrentPage] = useState(1);
  const [ignoredPosts, setIgnoredPosts] = useState<Set<string>>(new Set());
  const postsPerPage = 30;
  const filteredPosts = recentPosts.filter(post => !ignoredPosts.has(post.id));
  const totalPages = Math.ceil(filteredPosts.length / postsPerPage);
  const startIndex = (currentPage - 1) * postsPerPage;
  const endIndex = startIndex + postsPerPage;
  const currentPagePosts = filteredPosts.slice(startIndex, endIndex);
  
  // Comprehensive Interaction State
  const [selectedInteractSubreddit, setSelectedInteractSubreddit] = useState('');
  const [subredditPosts, setSubredditPosts] = useState<SubredditPost[]>([]);
  const [postComments, setPostComments] = useState<{[postId: string]: Comment[]}>({});
  const [expandedPosts, setExpandedPosts] = useState<Set<string>>(new Set());
  const [expandedComments, setExpandedComments] = useState<Set<string>>(new Set());
  const [replyingTo, setReplyingTo] = useState<{[id: string]: boolean}>({});
  const [editingComment, setEditingComment] = useState<{[id: string]: boolean}>({});
  const [newPostData, setNewPostData] = useState<NewPostData>({
    title: '',
    content: '',
    type: 'text',
    url: '',
    flair: '',
    spoiler: false,
    nsfw: false,
    sendReplies: true
  });
  const [newCommentData, setNewCommentData] = useState<{[id: string]: NewCommentData}>({});
  const [postSortType, setPostSortType] = useState<'hot' | 'new' | 'top' | 'rising'>('hot');
  const [loadingStates, setLoadingStates] = useState<{[key: string]: boolean}>({});
  const [errorLogs, setErrorLogs] = useState<Array<{timestamp: Date, error: any, context: string, severity: 'low' | 'medium' | 'high'}>>([]);

  // Helper function to check if error indicates need for re-authentication
  const needsReauth = (error: any): boolean => {
    return error?.message?.includes('403') || 
           error?.message?.includes('needsReauth') ||
           error?.message?.includes('insufficient permissions') ||
           error?.message?.includes('reconnect');
  };

  const { data: redditStatus, refetch: refetchRedditStatus } = useQuery({
    queryKey: ['/api/reddit/status'],
    refetchInterval: 5000, // Check connection status every 5 seconds
    refetchOnWindowFocus: true, // Refresh when user returns to window (after OAuth)
  });

  // Ensure we have proper type checking for redditStatus
  const isRedditConnected = redditStatus && typeof redditStatus === 'object' && 'connected' in redditStatus && (redditStatus as any).connected === true;

  // Check for Reddit connection success in URL parameters
  useEffect(() => {
    const urlParams = new URLSearchParams(window.location.search);
    const redditConnected = urlParams.get('reddit_connected');
    const redditError = urlParams.get('reddit_error');
    
    if (redditConnected === 'true') {
      toast({
        title: "Reddit Connected Successfully!",
        description: "Your Reddit account has been connected. You can now manage subreddits.",
      });
      // Refresh Reddit status immediately
      refetchRedditStatus();
      // Clean up URL
      window.history.replaceState({}, '', window.location.pathname);
    } else if (redditError) {
      toast({
        title: "Reddit Connection Failed",
        description: `Error: ${decodeURIComponent(redditError)}`,
        variant: "destructive"
      });
      // Clean up URL
      window.history.replaceState({}, '', window.location.pathname);
    }
  }, [toast, refetchRedditStatus]);

  const { 
    data: managedSubreddits = [], 
    refetch: refetchManaged,
    isLoading: managedLoading 
  } = useQuery({
    queryKey: ['/api/subreddits/managed'],
    enabled: isRedditConnected
  });

  const { 
    data: moderatedSubreddits = [], 
    refetch: refetchModerated,
    error: moderatedError,
    isError: moderatedIsError,
    isLoading: moderatedLoading
  } = useQuery({
    queryKey: ['/api/subreddits/moderated'],
    enabled: isRedditConnected,
    retry: false
  });

  // Automatic import of moderated subreddits when Reddit is connected and no managed subreddits exist
  useEffect(() => {
    if (isRedditConnected && 
        Array.isArray(moderatedSubreddits) && moderatedSubreddits.length > 0 && 
        Array.isArray(managedSubreddits) && managedSubreddits.length === 0 && 
        !managedLoading && !moderatedLoading) {
      
      console.log('🔄 [AUTO IMPORT] Automatically importing moderated subreddits');
      toast({
        title: "Auto-importing Subreddits",
        description: "Automatically adding your moderated subreddits to management dashboard",
      });
      handleImportModerated();
    }
  }, [isRedditConnected, moderatedSubreddits, managedSubreddits, managedLoading, moderatedLoading]);

  const searchSubredditsMutation = useMutation({
    mutationFn: async (query: string) => {
      const response = await apiRequest('/api/reddit/search/subreddits', 'POST', { query });
      return response as Subreddit[];
    },
    onSuccess: (results: Subreddit[]) => {
      setSearchResults(results);
      setIsSearching(false);
      toast({
        title: "Search Complete",
        description: `Found ${results.length} subreddits`,
      });
    },
    onError: (error: any) => {
      setIsSearching(false);
      setSearchResults([]);
      toast({
        title: "Search Error",
        description: error.message || "Failed to search subreddits",
        variant: "destructive"
      });
    }
  });

  const addSubredditMutation = useMutation({
    mutationFn: async (subredditName: string) => {
      console.log('🔄 [ADD SUBREDDIT] Frontend - Starting mutation for:', subredditName);
      return await apiRequest('/api/subreddits/manage', 'POST', { subredditName });
    },
    onSuccess: (data) => {
      console.log('🔄 [ADD SUBREDDIT] Frontend - Mutation success, refetching managed list');
      refetchManaged();
      setSearchResults([]);
      setSearchQuery('');
      toast({
        title: "Success",
        description: `Added ${data.displayName} to your management dashboard`
      });
    },
    onError: (error: any) => {
      console.error('🔄 [ADD SUBREDDIT] Frontend - Mutation error:', error);
      
      // Handle duplicate subreddit gracefully
      if (error.message && error.message.includes('already in your managed list')) {
        toast({
          title: "Already Added",
          description: "This subreddit is already in your managed list",
          variant: "default"
        });
      } else {
        toast({
          title: "Error",
          description: error.message || "Failed to add subreddit",
          variant: "destructive"
        });
      }
    }
  });

  const removeSubredditMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest(`/api/subreddits/manage/${id}`, 'DELETE');
    },
    onSuccess: () => {
      refetchManaged();
      toast({
        title: "Success",
        description: "Subreddit removed from management"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to remove subreddit",
        variant: "destructive"
      });
    }
  });

  const refreshAnalyticsMutation = useMutation({
    mutationFn: async (subredditName: string) => {
      return await apiRequest('/api/subreddits/analytics/refresh', 'POST', { subredditName });
    },
    onSuccess: () => {
      refetchManaged();
      toast({
        title: "Success",
        description: "Analytics refreshed successfully"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Error",
        description: error.message || "Failed to refresh analytics",
        variant: "destructive"
      });
    }
  });

  const importMutation = useMutation({
    mutationFn: async () => {
      return await apiRequest('/api/subreddits/import-moderated', 'POST');
    },
    onSuccess: (data: any) => {
      refetchManaged();
      refetchModerated();
      toast({
        title: "Import Complete",
        description: `Successfully imported ${data.imported} moderated subreddits and updated ${data.updated} existing ones`
      });
    },
    onError: (error: any) => {
      if (error.message === 'REAUTH_REQUIRED') {
        toast({
          title: "Additional Permissions Required",
          description: "Please reconnect your Reddit account to access moderated subreddits.",
          variant: "destructive",
          action: (
            <Button 
              size="sm" 
              onClick={() => window.open('/api/reddit/auth', '_blank')}
            >
              Reconnect
            </Button>
          ),
        });
      } else {
        toast({
          title: "Import Failed",
          description: error.message || "Failed to import moderated subreddits",
          variant: "destructive"
        });
      }
    }
  });

  const handleImportModerated = () => {
    importMutation.mutate();
  };

  // Proactive Moderation Mutations
  const fetchRecentPostsMutation = useMutation({
    mutationFn: async (subredditName: string) => {
      return await apiRequest(`/api/reddit/recent-posts/${subredditName}?limit=150`, 'GET');
    },
    onSuccess: (data) => {
      setRecentPosts(data || []);
      setIgnoredPosts(new Set()); // Reset ignored posts when loading new posts
      setCurrentPage(1); // Reset to first page when new posts loaded
      toast({
        title: "Posts Loaded",
        description: `Loaded ${data?.length || 0} recent posts for proactive analysis (max 150)`
      });
    },
    onError: (error: any) => {
      console.error('Failed to load recent posts:', error);
      logError(error, 'Failed to load recent posts', 'medium');
      toast({
        title: "Error",
        description: error.message || "Failed to load recent posts",
        variant: "destructive"
      });
    }
  });

  const analyzePostMutation = useMutation({
    mutationFn: async ({ subredditName, postId, postData }: { subredditName: string; postId: string; postData: RecentPost }) => {
      return await apiRequest('/api/reddit/analyze-post', 'POST', {
        subreddit: subredditName,
        postId,
        postData
      });
    },
    onSuccess: (data, variables) => {
      setPostAnalysisResults(prev => ({
        ...prev,
        [variables.postId]: data
      }));
      toast({
        title: "Analysis Complete",
        description: "AI analysis completed for the post"
      });
    },
    onError: (error: any) => {
      toast({
        title: "Analysis Failed",
        description: error.message || "Failed to analyze post",
        variant: "destructive"
      });
    }
  });

  const bulkAnalyzePostsMutation = useMutation({
    mutationFn: async (subredditName: string) => {
      setIsBulkAnalyzing(true);
      const results: {[postId: string]: PostAnalysisResult} = {};
      
      // Only analyze posts on current page (30 posts max)
      for (const post of currentPagePosts) {
        try {
          const analysis = await apiRequest('/api/reddit/analyze-post', 'POST', {
            subreddit: subredditName,
            postId: post.id,
            postData: post
          });
          results[post.id] = analysis;
          
          // Small delay between requests to avoid rate limiting
          await new Promise(resolve => setTimeout(resolve, 800));
        } catch (error) {
          console.error(`Failed to analyze post ${post.id}:`, error);
          logError(error, `AI analysis failed for post ${post.id}`, 'medium');
          // Continue with next post instead of breaking
        }
      }
      
      return results;
    },
    onSuccess: (results) => {
      setPostAnalysisResults(prev => ({ ...prev, ...results }));
      setIsBulkAnalyzing(false);
      toast({
        title: "Bulk Analysis Complete",
        description: `Analyzed ${Object.keys(results).length} posts`
      });
    },
    onError: (error: any) => {
      setIsBulkAnalyzing(false);
      toast({
        title: "Bulk Analysis Failed",
        description: error.message || "Some posts could not be analyzed",
        variant: "destructive"
      });
    }
  });

  // Advanced Error Logging Function
  const logError = (error: any, context: string, severity: 'low' | 'medium' | 'high' = 'medium') => {
    const errorLog = {
      timestamp: new Date(),
      error: {
        message: error.message || error,
        stack: error.stack,
        name: error.name,
        code: error.code,
        status: error.status
      },
      context,
      severity,
      userAgent: navigator.userAgent,
      url: window.location.href,
      subreddit: selectedInteractSubreddit || selectedProactiveSubreddit,
      userId: 'current_user' // This would come from auth context
    };
    
    setErrorLogs(prev => [...prev, errorLog]);
    
    // Send to backend for centralized logging with proper error handling
    apiRequest('/api/error-logs', 'POST', errorLog)
      .catch(logErr => console.error('Failed to send error log:', logErr));
    
    console.error(`🔥 [INTERACTION ERROR - ${severity.toUpperCase()}] ${context}:`, error);
  };

  // Set loading state with error protection
  const setLoadingState = (key: string, loading: boolean) => {
    try {
      setLoadingStates(prev => ({ ...prev, [key]: loading }));
    } catch (error) {
      logError(error, `Setting loading state for ${key}`, 'low');
    }
  };

  // Comprehensive Interaction Mutations
  const fetchSubredditPostsMutation = useMutation({
    mutationFn: async ({ subredditName, sortType, limit = 25 }: { subredditName: string, sortType: string, limit?: number }) => {
      setLoadingState('fetchPosts', true);
      try {
        const response = await apiRequest(`/api/reddit/subreddit/${subredditName}/posts?sort=${sortType}&limit=${limit}`, 'GET');
        if (!response || !Array.isArray(response)) {
          throw new Error('Invalid response format from posts API');
        }
        return response;
      } catch (error) {
        logError(error, `Fetching posts from r/${subredditName} with sort=${sortType}`, 'high');
        throw error;
      } finally {
        setLoadingState('fetchPosts', false);
      }
    },
    onSuccess: (data, variables) => {
      try {
        setSubredditPosts(data || []);
        toast({
          title: "Posts Loaded",
          description: `Loaded ${data?.length || 0} posts from r/${variables.subredditName} (sorted by ${variables.sortType})`
        });
      } catch (error) {
        logError(error, 'Processing fetched posts success', 'medium');
      }
    },
    onError: (error: any, variables) => {
      logError(error, `Failed to fetch posts from r/${variables.subredditName}`, 'high');
      toast({
        title: "Error Loading Posts",
        description: error.message || "Failed to load subreddit posts",
        variant: "destructive"
      });
    }
  });

  const fetchPostCommentsMutation = useMutation({
    mutationFn: async ({ postId, subredditName }: { postId: string, subredditName: string }) => {
      setLoadingState(`comments_${postId}`, true);
      try {
        const response = await apiRequest(`/api/reddit/posts/${postId}/comments?subreddit=${subredditName}`, 'GET');
        if (!response || !Array.isArray(response)) {
          throw new Error('Invalid response format from comments API');
        }
        return { postId, comments: response };
      } catch (error) {
        logError(error, `Fetching comments for post ${postId} in r/${subredditName}`, 'high');
        throw error;
      } finally {
        setLoadingState(`comments_${postId}`, false);
      }
    },
    onSuccess: (data) => {
      try {
        setPostComments(prev => ({
          ...prev,
          [data.postId]: data.comments
        }));
        setExpandedPosts(prev => new Set([...prev, data.postId]));
      } catch (error) {
        logError(error, 'Processing fetched comments success', 'medium');
      }
    },
    onError: (error: any) => {
      logError(error, 'Failed to fetch post comments', 'high');
      toast({
        title: "Error Loading Comments",
        description: error.message || "Failed to load post comments",
        variant: "destructive"
      });
    }
  });

  const createPostMutation = useMutation({
    mutationFn: async ({ subredditName, postData }: { subredditName: string, postData: NewPostData }) => {
      setLoadingState('createPost', true);
      try {
        if (!postData.title.trim()) {
          throw new Error('Post title is required');
        }
        if (postData.type === 'text' && !postData.content.trim()) {
          throw new Error('Post content is required for text posts');
        }
        if (postData.type === 'link' && !postData.url?.trim()) {
          throw new Error('URL is required for link posts');
        }
        
        const response = await apiRequest('/api/reddit/posts/create', 'POST', {
          subreddit: subredditName,
          ...postData
        });
        return response;
      } catch (error) {
        logError(error, `Creating post in r/${subredditName}`, 'high');
        throw error;
      } finally {
        setLoadingState('createPost', false);
      }
    },
    onSuccess: (data, variables) => {
      try {
        // Reset form
        setNewPostData({
          title: '',
          content: '',
          type: 'text',
          url: '',
          flair: '',
          spoiler: false,
          nsfw: false,
          sendReplies: true
        });
        
        // Refresh posts
        fetchSubredditPostsMutation.mutate({
          subredditName: variables.subredditName,
          sortType: postSortType
        });
        
        toast({
          title: "Post Created",
          description: "Your post has been successfully created"
        });
      } catch (error) {
        logError(error, 'Processing create post success', 'medium');
      }
    },
    onError: (error: any) => {
      logError(error, 'Failed to create post', 'high');
      toast({
        title: "Error Creating Post",
        description: error.message || "Failed to create post",
        variant: "destructive"
      });
    }
  });

  const createCommentMutation = useMutation({
    mutationFn: async ({ subredditName, postId, parentId, content }: { subredditName: string, postId?: string, parentId?: string, content: string }) => {
      const commentKey = parentId || postId || 'unknown';
      setLoadingState(`comment_${commentKey}`, true);
      try {
        if (!content.trim()) {
          throw new Error('Comment content is required');
        }
        
        const response = await apiRequest('/api/reddit/comments/create', 'POST', {
          subreddit: subredditName,
          postId,
          parentId,
          content
        });
        return { response, commentKey };
      } catch (error) {
        logError(error, `Creating comment in r/${subredditName}`, 'high');
        throw error;
      } finally {
        setLoadingState(`comment_${commentKey}`, false);
      }
    },
    onSuccess: (data, variables) => {
      try {
        // Clear the reply form
        const commentKey = variables.parentId || variables.postId || 'unknown';
        setNewCommentData(prev => ({
          ...prev,
          [commentKey]: { content: '' }
        }));
        setReplyingTo(prev => ({
          ...prev,
          [commentKey]: false
        }));
        
        // Refresh comments for the post
        if (variables.postId) {
          fetchPostCommentsMutation.mutate({
            postId: variables.postId,
            subredditName: variables.subredditName
          });
        }
        
        toast({
          title: "Comment Posted",
          description: "Your comment has been successfully posted"
        });
      } catch (error) {
        logError(error, 'Processing create comment success', 'medium');
      }
    },
    onError: (error: any) => {
      logError(error, 'Failed to create comment', 'high');
      toast({
        title: "Error Posting Comment",
        description: error.message || "Failed to post comment",
        variant: "destructive"
      });
    }
  });

  const voteOnContentMutation = useMutation({
    mutationFn: async ({ fullname, direction }: { fullname: string, direction: 1 | -1 | 0 }) => {
      setLoadingState(`vote_${fullname}`, true);
      try {
        const response = await apiRequest('/api/reddit/vote', 'POST', {
          fullname,
          direction
        });
        return response;
      } catch (error) {
        logError(error, `Voting on ${fullname} with direction ${direction}`, 'medium');
        throw error;
      } finally {
        setLoadingState(`vote_${fullname}`, false);
      }
    },
    onSuccess: () => {
      toast({
        title: "Vote Recorded",
        description: "Your vote has been recorded"
      });
    },
    onError: (error: any) => {
      logError(error, 'Failed to vote on content', 'medium');
      toast({
        title: "Voting Failed",
        description: error.message || "Failed to record your vote",
        variant: "destructive"
      });
    }
  });

  // Moderation Action Mutation for Proactive Moderation
  const executeModerationActionMutation = useMutation({
    mutationFn: async ({ action, postId, postData, aiAnalysis }: { 
      action: 'approve' | 'remove', 
      postId: string, 
      postData: RecentPost,
      aiAnalysis?: PostAnalysisResult
    }) => {
      // Convert to Reddit fullname format if not already
      const itemId = postId.startsWith('t3_') ? postId : `t3_${postId}`;
      
      const response = await apiRequest('/api/reddit/execute-moderation-action', 'POST', {
        subreddit: selectedProactiveSubreddit,
        itemId: itemId,
        action,
        reason: aiAnalysis?.violationsFound?.map(v => v.violation).join(', ') || `Proactive ${action} via AI analysis`,
        // For proactive moderation, pass AI analysis data directly
        isProactiveModeration: true,
        aiAnalysisData: aiAnalysis ? {
          confidenceScore: aiAnalysis.confidenceScore,
          suggestedAction: aiAnalysis.suggestedAction,
          reasoning: aiAnalysis.reasoning,
          severity: aiAnalysis.severity,
          violationsFound: aiAnalysis.violationsFound
        } : null
      });
      return response;
    },
    onSuccess: (data, variables) => {
      toast({
        title: "Action Executed",
        description: `Post ${variables.action}d successfully. ${data.humanResponse ? 'Human response generated for modmail.' : ''}`,
      });
      
      // Remove the post from recent posts view
      setRecentPosts(prev => prev.filter(post => post.id !== variables.postId));
      
      // Clear analysis result
      setPostAnalysisResults(prev => {
        const newResults = { ...prev };
        delete newResults[variables.postId];
        return newResults;
      });
    },
    onError: (error: any) => {
      toast({
        title: "Action Failed",
        description: error.message || "Failed to execute moderation action",
        variant: "destructive"
      });
    }
  });

  const handleSearch = () => {
    if (searchQuery.trim()) {
      setIsSearching(true);
      searchSubredditsMutation.mutate(searchQuery.trim());
    }
  };

  const formatNumber = (num: number) => {
    if (num >= 1000000) return `${(num / 1000000).toFixed(1)}M`;
    if (num >= 1000) return `${(num / 1000).toFixed(1)}K`;
    return num.toString();
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString();
  };

  // Helper function to check if a subreddit is already managed
  const isSubredditManaged = (subredditName: string) => {
    const cleanName = subredditName.replace(/^r\//, '');
    return Array.isArray(managedSubreddits) && (managedSubreddits as any[]).some((managed: any) => 
      managed?.subredditName === cleanName || managed?.subredditName === `r/${cleanName}`
    );
  };

  // Check if user is connected to Reddit
  if (!isRedditConnected) {
    return (
      <DashboardLayout>
        <div className="p-6">
          <div className="max-w-4xl mx-auto">
            <h1 className="text-3xl font-bold mb-6">Subreddit Manager</h1>
            <div className="bg-gradient-to-r from-orange-50 to-red-50 border border-orange-200 rounded-lg p-6 mb-6">
              <div className="flex items-start gap-4">
                <Shield className="w-8 h-8 text-orange-500 mt-1 flex-shrink-0" />
                <div className="flex-1">
                  <h2 className="text-xl font-semibold mb-3 text-gray-900">Connect Your Reddit Account</h2>
                  <p className="text-gray-700 mb-4">
                    You're logged into TubeBoost, but need to connect your Reddit account to access moderation features. This is a secure OAuth connection.
                  </p>
                  <div className="bg-blue-50 border-l-4 border-blue-400 p-4 mb-4">
                    <div className="flex">
                      <div className="ml-3">
                        <p className="text-sm text-blue-700">
                          <strong>Status:</strong> TubeBoost is ready to connect to Reddit. The 401 errors you see are normal - they'll disappear once you complete the connection below.
                        </p>
                      </div>
                    </div>
                  </div>
                  <div className="bg-white/70 rounded-lg p-4 mb-4">
                    <h3 className="font-medium text-gray-900 mb-2">What you'll get:</h3>
                    <ul className="text-sm text-gray-700 space-y-1">
                      <li>• AI-powered moderation assistance</li>
                      <li>• Subreddit analytics and insights</li>
                      <li>• Automated content management</li>
                      <li>• Comprehensive moderation tools</li>
                    </ul>
                  </div>
                  <div className="space-y-3">
                    <Button
                      onClick={() => {
                        toast({
                          title: "Redirecting to Reddit",
                          description: "You'll be taken to Reddit to authorize TubeBoost, then redirected back here.",
                        });
                        // Open in same window to avoid popup blockers
                        setTimeout(() => {
                          window.location.href = '/api/reddit/auth';
                        }, 1500);
                      }}
                      className="bg-orange-500 hover:bg-orange-600 text-white px-8 py-4 text-lg inline-flex items-center gap-3 w-full sm:w-auto justify-center"
                    >
                      <Shield className="w-6 h-6" />
                      Connect to Reddit Now
                    </Button>
                    <div className="text-sm text-gray-600">
                      <p><strong>What happens next:</strong></p>
                      <ol className="list-decimal list-inside space-y-1 mt-2">
                        <li>You'll be redirected to Reddit's secure OAuth page</li>
                        <li>Reddit will ask you to authorize TubeBoost</li>
                        <li>You'll be redirected back here with full access</li>
                        <li>The 401 errors will disappear and all features will work</li>
                      </ol>
                    </div>
                  </div>
                  {redditStatus && typeof redditStatus === 'object' && 'debugInfo' in redditStatus && (redditStatus as any).debugInfo && (
                    <div className="mt-4 p-3 bg-gray-50 rounded text-xs text-gray-600">
                      <p><strong>Debug Info:</strong> Reddit OAuth is configured and ready</p>
                      <p><strong>Auth Endpoint:</strong> {(redditStatus as any).debugInfo.authEndpoint}</p>
                      <p><strong>Client Configured:</strong> {(redditStatus as any).debugInfo.clientConfigured ? 'Yes' : 'No'}</p>
                    </div>
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  return (
    <DashboardLayout>
      <div className="p-3 sm:p-4 lg:p-6">
        <div className="max-w-7xl mx-auto">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mb-4 sm:mb-6 gap-4">
            <h1 className="text-2xl sm:text-3xl font-bold">Subreddit Manager</h1>
            <div className="text-xs sm:text-sm text-gray-600">
              Connected as: {redditStatus && typeof redditStatus === 'object' && 'redditUser' in redditStatus ? (redditStatus as any).redditUser?.username : 'Unknown'}
            </div>
          </div>

          <Tabs defaultValue="search" className="space-y-4 sm:space-y-6">
            <TabsList className="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-5 w-full h-auto p-1">
              <TabsTrigger value="search" className="text-xs sm:text-sm">Search & Add</TabsTrigger>
              <TabsTrigger value="managed" className="text-xs sm:text-sm">Managed ({Array.isArray(managedSubreddits) ? managedSubreddits.length : 0})</TabsTrigger>
              <TabsTrigger value="interact" className="flex items-center gap-1 sm:gap-2 text-xs sm:text-sm">
                <MessageSquare className="w-3 h-3 sm:w-4 sm:h-4" />
                <span className="hidden sm:inline">Interact & Manage</span>
                <span className="sm:hidden">Interact</span>
              </TabsTrigger>
              <TabsTrigger value="moderated" className="text-xs sm:text-sm">Moderated ({Array.isArray(moderatedSubreddits) ? moderatedSubreddits.length : 0})</TabsTrigger>
              <TabsTrigger value="proactive" className="flex items-center gap-1 sm:gap-2 text-xs sm:text-sm">
                <Shield className="w-3 h-3 sm:w-4 sm:h-4" />
                <span className="hidden sm:inline">Proactive Moderation</span>
                <span className="sm:hidden">Proactive</span>
              </TabsTrigger>
            </TabsList>

            <TabsContent value="search" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Search Subreddits</CardTitle>
                  <CardDescription>Find and add subreddits to your management dashboard</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="flex gap-4 mb-6">
                    <Input
                      placeholder="Search for subreddits (e.g., technology, gaming)"
                      value={searchQuery}
                      onChange={(e) => setSearchQuery(e.target.value)}
                      onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                      className="flex-1"
                    />
                    <Button 
                      onClick={handleSearch}
                      disabled={isSearching || !searchQuery.trim()}
                    >
                      <Search className="w-4 h-4 mr-2" />
                      {isSearching ? 'Searching...' : 'Search'}
                    </Button>
                  </div>

                  {searchResults.length > 0 && (
                    <div className="space-y-4">
                      <h3 className="text-lg font-semibold">Search Results</h3>
                      <div className="grid gap-4">
                        {(searchResults || []).map((subreddit) => (
                          <Card key={subreddit.id} className="p-6">
                            <div className="flex items-start justify-between mb-4">
                              <div className="flex-1">
                                <div className="flex items-center gap-2 mb-3">
                                  <h4 className="text-lg font-semibold">{subreddit.displayName}</h4>
                                  {subreddit.isNsfw && (
                                    <Badge variant="destructive" className="text-xs">NSFW</Badge>
                                  )}
                                  <Badge variant="outline" className="text-xs capitalize">{subreddit.subredditType}</Badge>
                                  {subreddit.user_is_moderator && (
                                    <Badge variant="default" className="text-xs bg-blue-600">
                                      <Shield className="w-3 h-3 mr-1" />
                                      Moderator
                                    </Badge>
                                  )}
                                  {subreddit.user_is_subscriber && (
                                    <Badge variant="outline" className="text-xs bg-green-50 text-green-700">
                                      Subscribed
                                    </Badge>
                                  )}
                                </div>
                                
                                <p className="text-base text-gray-700 dark:text-gray-300 mb-3 font-medium">
                                  {subreddit.title}
                                </p>
                                
                                <p className="text-sm text-gray-600 dark:text-gray-400 mb-4 line-clamp-3">
                                  {subreddit.publicDescription || subreddit.description || 'No description available'}
                                </p>

                                {/* Enhanced Statistics Grid */}
                                <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
                                  <div className="text-center p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                                    <div className="text-lg font-bold text-blue-600 dark:text-blue-400">
                                      {formatNumber(subreddit.subscribers)}
                                    </div>
                                    <div className="text-xs text-gray-600 dark:text-gray-400">Members</div>
                                  </div>
                                  <div className="text-center p-3 bg-green-50 dark:bg-green-900/20 rounded-lg">
                                    <div className="text-lg font-bold text-green-600 dark:text-green-400">
                                      {formatNumber(subreddit.activeUsers || subreddit.accounts_active || 0)}
                                    </div>
                                    <div className="text-xs text-gray-600 dark:text-gray-400">Active</div>
                                  </div>
                                  <div className="text-center p-3 bg-purple-50 dark:bg-purple-900/20 rounded-lg">
                                    <div className="text-lg font-bold text-purple-600 dark:text-purple-400">
                                      {new Date(subreddit.created * 1000).getFullYear()}
                                    </div>
                                    <div className="text-xs text-gray-600 dark:text-gray-400">Created</div>
                                  </div>
                                  <div className="text-center p-3 bg-orange-50 dark:bg-orange-900/20 rounded-lg">
                                    <div className="text-lg font-bold text-orange-600 dark:text-orange-400">
                                      {subreddit.lang?.toUpperCase() || 'EN'}
                                    </div>
                                    <div className="text-xs text-gray-600 dark:text-gray-400">Language</div>
                                  </div>
                                </div>

                                {/* Additional Features */}
                                <div className="space-y-3">
                                  <div>
                                    <h5 className="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">Content Features</h5>
                                    <div className="flex flex-wrap gap-2">
                                      {subreddit.allowImages && (
                                        <Badge variant="outline" className="text-xs">📷 Images</Badge>
                                      )}
                                      {subreddit.allowVideos && (
                                        <Badge variant="outline" className="text-xs">🎥 Videos</Badge>
                                      )}
                                      {subreddit.allow_polls && (
                                        <Badge variant="outline" className="text-xs">📊 Polls</Badge>
                                      )}
                                      {subreddit.link_flair_enabled && (
                                        <Badge variant="outline" className="text-xs">🏷️ Flairs</Badge>
                                      )}
                                      {subreddit.emojis_enabled && (
                                        <Badge variant="outline" className="text-xs">😊 Emojis</Badge>
                                      )}
                                      {subreddit.all_original_content && (
                                        <Badge variant="outline" className="text-xs">✨ OC Only</Badge>
                                      )}
                                    </div>
                                  </div>

                                  {subreddit.whitelist_status && (
                                    <div>
                                      <h5 className="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">Community Status</h5>
                                      <div className="flex gap-2">
                                        <Badge variant="outline" className="text-xs capitalize">
                                          Status: {subreddit.whitelist_status}
                                        </Badge>
                                        {subreddit.public_traffic && (
                                          <Badge variant="outline" className="text-xs">📈 Public Traffic</Badge>
                                        )}
                                        {subreddit.hide_ads && (
                                          <Badge variant="outline" className="text-xs">🚫 No Ads</Badge>
                                        )}
                                      </div>
                                    </div>
                                  )}

                                  {subreddit.advertiser_category && (
                                    <div>
                                      <h5 className="text-sm font-semibold text-gray-700 dark:text-gray-300 mb-2">Category</h5>
                                      <Badge variant="outline" className="text-xs capitalize">
                                        {subreddit.advertiser_category}
                                      </Badge>
                                    </div>
                                  )}
                                </div>
                              </div>
                              
                              <div className="flex gap-2 ml-4">
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => window.open(`https://reddit.com${subreddit.url}`, '_blank')}
                                >
                                  <ExternalLink className="w-4 h-4" />
                                </Button>
                                {isSubredditManaged(subreddit.name) ? (
                                  <Button
                                    size="sm"
                                    variant="outline"
                                    disabled
                                    className="opacity-50"
                                  >
                                    <Check className="w-4 h-4 mr-1" />
                                    Added
                                  </Button>
                                ) : (
                                  <Button
                                    size="sm"
                                    onClick={() => addSubredditMutation.mutate(subreddit.name)}
                                    disabled={addSubredditMutation.isPending}
                                  >
                                    <Plus className="w-4 h-4 mr-1" />
                                    Add
                                  </Button>
                                )}
                              </div>
                            </div>
                          </Card>
                        ))}
                      </div>
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="managed" className="space-y-6">
              <Card>
                <CardHeader>
                  <CardTitle>Managed Subreddits</CardTitle>
                  <CardDescription>Subreddits you're tracking with live analytics</CardDescription>
                </CardHeader>
                <CardContent>
                  {!Array.isArray(managedSubreddits) || managedSubreddits.length === 0 ? (
                    <div className="text-center py-8">
                      <Search className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                      <p className="text-gray-600">No subreddits being managed yet.</p>
                      <p className="text-sm text-gray-500">Use the Search & Add tab to find subreddits to track.</p>
                    </div>
                  ) : (
                    <div className="space-y-6">
                      {Array.isArray(managedSubreddits) && managedSubreddits.map((subreddit: ManagedSubreddit) => (
                        <Card key={subreddit.id} className="p-6">
                          <div className="flex items-start justify-between mb-4">
                            <div>
                              <div className="flex items-center gap-2 mb-2">
                                <h3 className="text-xl font-semibold">{subreddit.displayName}</h3>
                                {subreddit.isModerator && (
                                  <Badge variant="default" className="text-xs bg-blue-600 hover:bg-blue-700">
                                    <Shield className="w-3 h-3 mr-1" />
                                    Moderator
                                  </Badge>
                                )}
                                {subreddit.isNsfw && <Badge variant="destructive" className="text-xs">NSFW</Badge>}
                                <Badge variant="outline" className="text-xs capitalize">{subreddit.subredditType}</Badge>
                              </div>
                              <p className="text-sm text-gray-500">
                                {subreddit.title && <span className="block text-gray-700 dark:text-gray-300 mb-1">{subreddit.title}</span>}
                                Added on {formatDate(subreddit.createdAt)}
                              </p>
                            </div>
                            <div className="flex gap-2">
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => refreshAnalyticsMutation.mutate(subreddit.subredditName)}
                                disabled={refreshAnalyticsMutation.isPending}
                              >
                                <RefreshCw className="w-4 h-4" />
                              </Button>
                              <Button
                                variant="outline"
                                size="sm"
                                onClick={() => window.open(`https://reddit.com/r/${subreddit.subredditName}`, '_blank')}
                              >
                                <ExternalLink className="w-4 h-4" />
                              </Button>
                              <Button
                                variant="destructive"
                                size="sm"
                                onClick={() => removeSubredditMutation.mutate(subreddit.id)}
                                disabled={removeSubredditMutation.isPending}
                              >
                                <Trash2 className="w-4 h-4" />
                              </Button>
                            </div>
                          </div>

                          {subreddit.analytics && (
                            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                              <div className="text-center p-3 bg-blue-50 rounded-lg">
                                <div className="text-2xl font-bold text-blue-600">
                                  {formatNumber(subreddit.analytics.subscribers)}
                                </div>
                                <div className="text-sm text-gray-600">Subscribers</div>
                              </div>
                              <div className="text-center p-3 bg-green-50 rounded-lg">
                                <div className="text-2xl font-bold text-green-600">
                                  {formatNumber(subreddit.analytics.activeUsers)}
                                </div>
                                <div className="text-sm text-gray-600">Active Users</div>
                              </div>
                              <div className="text-center p-3 bg-purple-50 rounded-lg">
                                <div className="text-2xl font-bold text-purple-600">
                                  {subreddit.analytics.growthRate.toFixed(1)}%
                                </div>
                                <div className="text-sm text-gray-600">Growth Rate</div>
                              </div>
                              <div className="text-center p-3 bg-orange-50 rounded-lg">
                                <div className="text-2xl font-bold text-orange-600">
                                  {subreddit.analytics.engagementRate.toFixed(1)}%
                                </div>
                                <div className="text-sm text-gray-600">Engagement</div>
                              </div>
                            </div>
                          )}

                          {subreddit.analytics && (
                            <div className="mt-4 grid md:grid-cols-2 gap-4">
                              <div>
                                <h4 className="font-semibold mb-2">Performance Metrics</h4>
                                <div className="space-y-2 text-sm">
                                  <div className="flex justify-between">
                                    <span>Avg. Upvotes:</span>
                                    <span className="font-medium">{subreddit.analytics.avgUpvotes}</span>
                                  </div>
                                  <div className="flex justify-between">
                                    <span>Avg. Comments:</span>
                                    <span className="font-medium">{subreddit.analytics.avgComments}</span>
                                  </div>
                                  <div className="flex justify-between">
                                    <span>Weekly Posts:</span>
                                    <span className="font-medium">{subreddit.analytics.weeklyPosts}</span>
                                  </div>
                                  <div className="flex justify-between">
                                    <span>Best Post Time:</span>
                                    <span className="font-medium">{subreddit.analytics.topPostTime}</span>
                                  </div>
                                </div>
                              </div>
                              <div>
                                <h4 className="font-semibold mb-2">Content Categories</h4>
                                <div className="flex flex-wrap gap-2">
                                  {subreddit.analytics.contentCategories.map((category, index) => (
                                    <Badge key={index} variant="outline" className="text-xs">
                                      {category}
                                    </Badge>
                                  ))}
                                </div>
                                <p className="text-xs text-gray-500 mt-2">
                                  Last updated: {formatDate(subreddit.analytics.lastUpdated)}
                                </p>
                              </div>
                            </div>
                          )}
                        </Card>
                      ))}
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="moderated" className="space-y-6">
              <Card>
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <div>
                      <CardTitle className="flex items-center gap-2">
                        <Shield className="w-5 h-5 text-blue-600" />
                        My Moderated Subreddits
                      </CardTitle>
                      <CardDescription>Subreddits where you have moderation privileges</CardDescription>
                    </div>
                    <div className="flex gap-2">
                      <Button 
                        onClick={async () => {
                          try {
                            await refetchModerated();
                            toast({
                              title: "Sync Complete",
                              description: "Updated your moderated subreddits from Reddit API",
                            });
                          } catch (error) {
                            toast({
                              title: "Sync Failed",
                              description: "Unable to sync with Reddit API. Please try again.",
                              variant: "destructive",
                            });
                          }
                        }}
                        disabled={moderatedLoading}
                        variant="outline"
                      >
                        {moderatedLoading ? (
                          <>
                            <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                            Syncing...
                          </>
                        ) : (
                          <>
                            <RefreshCw className="w-4 h-4 mr-2" />
                            Sync to Reddit API
                          </>
                        )}
                      </Button>
                      <Button 
                        onClick={handleImportModerated}
                        disabled={importMutation.isPending}
                        variant="outline"
                      >
                        {importMutation.isPending ? (
                          <>
                            <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                            Importing...
                          </>
                        ) : (
                          <>
                            <Download className="w-4 h-4 mr-2" />
                            Import to Managed
                          </>
                        )}
                      </Button>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  {moderatedIsError && needsReauth(moderatedError) ? (
                    <div className="text-center py-8">
                      <Shield className="w-12 h-12 mx-auto text-blue-400 mb-4" />
                      <h3 className="text-lg font-semibold mb-2">Additional Permissions Required</h3>
                      <p className="text-gray-600 mb-4">
                        To view your moderated subreddits, we need additional permissions to access your moderator status.
                      </p>
                      <Button 
                        onClick={() => window.open('/api/reddit/auth', '_blank')}
                        className="bg-blue-600 hover:bg-blue-700"
                      >
                        <Shield className="w-4 h-4 mr-2" />
                        Reconnect with Moderator Permissions
                      </Button>
                      <p className="text-xs text-gray-500 mt-2">
                        This will open a new tab to reconnect your Reddit account with moderator permissions.
                      </p>
                    </div>
                  ) : !Array.isArray(moderatedSubreddits) || moderatedSubreddits.length === 0 ? (
                    <div className="text-center py-8">
                      <Shield className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                      <p className="text-gray-600">No moderated subreddits found.</p>
                      <p className="text-sm text-gray-500">This will show subreddits where you have moderation permissions.</p>
                    </div>
                  ) : (
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>Subreddit</TableHead>
                          <TableHead>Subscribers</TableHead>
                          <TableHead>Permissions</TableHead>
                          <TableHead>Queue</TableHead>
                          <TableHead>Actions</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {Array.isArray(moderatedSubreddits) && moderatedSubreddits.map((subreddit: ModeratedSubreddit, index: number) => (
                          <TableRow key={`${subreddit.id || subreddit.name || index}`}>
                            <TableCell>
                              <div className="font-medium">{subreddit.displayName}</div>
                            </TableCell>
                            <TableCell>{formatNumber(subreddit.subscribers)}</TableCell>
                            <TableCell>
                              <div className="flex flex-wrap gap-1">
                                {(subreddit.permissions || []).map((perm, index) => (
                                  <Badge key={index} variant="outline" className="text-xs">
                                    {perm}
                                  </Badge>
                                ))}
                              </div>
                            </TableCell>
                            <TableCell>
                              <div className="space-y-1 text-sm">
                                <div className="flex items-center gap-1">
                                  <MessageSquare className="w-3 h-3" />
                                  {subreddit.unmoderatedCount} unmoderated
                                </div>
                                <div className="flex items-center gap-1">
                                  <Star className="w-3 h-3" />
                                  {subreddit.reportCount} reports
                                </div>
                              </div>
                            </TableCell>
                            <TableCell>
                              <div className="flex gap-2">
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => window.open(`https://reddit.com/r/${subreddit.name}/about/modqueue`, '_blank')}
                                >
                                  Mod Queue
                                </Button>
                                {isSubredditManaged(subreddit.name) ? (
                                  <Button
                                    variant="outline"
                                    size="sm"
                                    disabled
                                    className="opacity-50"
                                  >
                                    <Check className="w-4 h-4" />
                                  </Button>
                                ) : (
                                  <Button
                                    variant="outline"
                                    size="sm"
                                    onClick={() => addSubredditMutation.mutate(subreddit.name)}
                                    disabled={addSubredditMutation.isPending}
                                  >
                                    <Plus className="w-4 h-4" />
                                  </Button>
                                )}
                              </div>
                            </TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="interact" className="space-y-6">
              <Card>
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <div>
                      <CardTitle className="flex items-center gap-2">
                        <MessageSquare className="w-5 h-5 text-green-600" />
                        Subreddit Interaction Hub
                      </CardTitle>
                      <CardDescription>
                        Complete subreddit management - view posts, comments, create content, and interact with your community
                      </CardDescription>
                    </div>
                    {errorLogs.length > 0 && (
                      <Badge variant="destructive" className="cursor-pointer" 
                        onClick={() => setErrorLogs([])}
                        title="Click to clear error logs"
                      >
                        {errorLogs.length} Errors
                      </Badge>
                    )}
                  </div>
                </CardHeader>
                <CardContent>
                  {/* Subreddit Selection */}
                  <div className="mb-6">
                    <label className="block text-sm font-medium mb-2">Select Subreddit to Manage</label>
                    <div className="grid grid-cols-1 md:grid-cols-4 gap-2">
                      {Array.isArray(managedSubreddits) && managedSubreddits.map((subreddit: ManagedSubreddit) => (
                        <Button
                          key={subreddit.id}
                          variant={selectedInteractSubreddit === subreddit.subredditName ? "default" : "outline"}
                          onClick={() => {
                            setSelectedInteractSubreddit(subreddit.subredditName);
                            setSubredditPosts([]);
                            setPostComments({});
                            setExpandedPosts(new Set());
                            setExpandedComments(new Set());
                          }}
                          className="text-left justify-start"
                        >
                          <MessageSquare className="w-4 h-4 mr-2" />
                          {subreddit.displayName}
                        </Button>
                      ))}
                    </div>
                    {(!Array.isArray(managedSubreddits) || managedSubreddits.length === 0) && (
                      <p className="text-gray-500 text-sm">No managed subreddits available. Add some from the other tabs first.</p>
                    )}
                  </div>

                  {selectedInteractSubreddit && (
                    <div className="space-y-6">
                      {/* Controls */}
                      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between p-3 sm:p-4 bg-green-50 dark:bg-green-950 rounded-lg border gap-3 sm:gap-0">
                        <div>
                          <h3 className="font-semibold text-sm sm:text-base">Managing: r/{selectedInteractSubreddit}</h3>
                          <p className="text-xs sm:text-sm text-gray-600 dark:text-gray-400">
                            {subredditPosts.length} posts loaded
                          </p>
                        </div>
                        <div className="flex flex-col sm:flex-row gap-2">
                          <Select value={postSortType} onValueChange={(value: 'hot' | 'new' | 'top' | 'rising') => setPostSortType(value)}>
                            <SelectTrigger className="w-full sm:w-24">
                              <SelectValue />
                            </SelectTrigger>
                            <SelectContent>
                              <SelectItem value="hot">Hot</SelectItem>
                              <SelectItem value="new">New</SelectItem>
                              <SelectItem value="top">Top</SelectItem>
                              <SelectItem value="rising">Rising</SelectItem>
                            </SelectContent>
                          </Select>
                          <Button
                            onClick={() => fetchSubredditPostsMutation.mutate({
                              subredditName: selectedInteractSubreddit,
                              sortType: postSortType
                            })}
                            disabled={loadingStates.fetchPosts}
                            variant="outline"
                            size="sm"
                            className="whitespace-nowrap"
                          >
                            {loadingStates.fetchPosts ? (
                              <>
                                <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                                <span className="hidden sm:inline">Loading...</span>
                                <span className="sm:hidden">Load</span>
                              </>
                            ) : (
                              <>
                                <RefreshCw className="w-4 h-4 mr-2" />
                                <span className="hidden sm:inline">Load Posts</span>
                                <span className="sm:hidden">Load</span>
                              </>
                            )}
                          </Button>
                        </div>
                      </div>

                      {/* Create New Post */}
                      <Card>
                        <CardHeader>
                          <CardTitle className="text-lg">Create New Post</CardTitle>
                        </CardHeader>
                        <CardContent className="space-y-4">
                          <div className="flex gap-2">
                            <Button
                              variant={newPostData.type === 'text' ? 'default' : 'outline'}
                              size="sm"
                              onClick={() => setNewPostData(prev => ({ ...prev, type: 'text' }))}
                            >
                              Text Post
                            </Button>
                            <Button
                              variant={newPostData.type === 'link' ? 'default' : 'outline'}
                              size="sm"
                              onClick={() => setNewPostData(prev => ({ ...prev, type: 'link' }))}
                            >
                              Link Post
                            </Button>
                          </div>
                          
                          <Input
                            placeholder="Post title"
                            value={newPostData.title}
                            onChange={(e) => setNewPostData(prev => ({ ...prev, title: e.target.value }))}
                            maxLength={300}
                          />
                          
                          {newPostData.type === 'text' ? (
                            <Textarea
                              placeholder="Post content (optional for text posts)"
                              value={newPostData.content}
                              onChange={(e) => setNewPostData(prev => ({ ...prev, content: e.target.value }))}
                              rows={4}
                            />
                          ) : (
                            <Input
                              placeholder="URL"
                              type="url"
                              value={newPostData.url}
                              onChange={(e) => setNewPostData(prev => ({ ...prev, url: e.target.value }))}
                            />
                          )}

                          <div className="flex items-center gap-4">
                            <div className="flex items-center space-x-2">
                              <Checkbox
                                id="spoiler"
                                checked={newPostData.spoiler}
                                onCheckedChange={(checked) => setNewPostData(prev => ({ ...prev, spoiler: !!checked }))}
                              />
                              <label htmlFor="spoiler" className="text-sm">Spoiler</label>
                            </div>
                            <div className="flex items-center space-x-2">
                              <Checkbox
                                id="nsfw"
                                checked={newPostData.nsfw}
                                onCheckedChange={(checked) => setNewPostData(prev => ({ ...prev, nsfw: !!checked }))}
                              />
                              <label htmlFor="nsfw" className="text-sm">NSFW</label>
                            </div>
                          </div>

                          <Button
                            onClick={() => createPostMutation.mutate({
                              subredditName: selectedInteractSubreddit,
                              postData: newPostData
                            })}
                            disabled={loadingStates.createPost || !newPostData.title.trim()}
                            className="w-full"
                          >
                            {loadingStates.createPost ? (
                              <>
                                <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                                Creating Post...
                              </>
                            ) : (
                              <>
                                <Send className="w-4 h-4 mr-2" />
                                Create Post
                              </>
                            )}
                          </Button>
                        </CardContent>
                      </Card>

                      {/* Posts Display */}
                      {subredditPosts.length > 0 ? (
                        <div className="space-y-4">
                          <h3 className="text-lg font-semibold">Posts ({subredditPosts.length})</h3>
                          {subredditPosts.map((post) => (
                            <Card key={post.id} className="p-6">
                              <div className="space-y-4">
                                {/* Post Header */}
                                <div className="flex items-start justify-between">
                                  <div className="flex-1">
                                    <div className="flex items-center gap-2 mb-2">
                                      <h4 className="text-lg font-semibold">{post.title}</h4>
                                      {post.isNsfw && <Badge variant="destructive" className="text-xs">NSFW</Badge>}
                                      {post.stickied && <Badge variant="default" className="text-xs">Pinned</Badge>}
                                      {post.locked && <Badge variant="secondary" className="text-xs">Locked</Badge>}
                                      {post.spoiler && <Badge variant="outline" className="text-xs">Spoiler</Badge>}
                                      {post.linkFlairText && <Badge variant="outline" className="text-xs">{post.linkFlairText}</Badge>}
                                    </div>
                                    <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">
                                      by u/{post.author} • {new Date(post.created).toLocaleString()}
                                      {post.distinguished && ` • ${post.distinguished}`}
                                    </p>
                                    {post.content && (
                                      <p className="text-sm text-gray-700 dark:text-gray-300 mb-3">
                                        {expandedPosts.has(post.id) ? post.content : `${post.content.substring(0, 300)}${post.content.length > 300 ? '...' : ''}`}
                                      </p>
                                    )}
                                    {post.url && post.domain && (
                                      <a href={post.url} target="_blank" rel="noopener noreferrer" 
                                         className="text-blue-600 hover:text-blue-800 text-sm">
                                        {post.domain} ↗
                                      </a>
                                    )}
                                  </div>
                                  <div className="flex gap-2 ml-4">
                                    <Button
                                      variant="outline"
                                      size="sm"
                                      onClick={() => window.open(`https://reddit.com${post.permalink}`, '_blank')}
                                    >
                                      <ExternalLink className="w-4 h-4" />
                                    </Button>
                                  </div>
                                </div>

                                {/* Media Content Display */}
                                {(post.postHint === 'image' || post.isVideo || post.thumbnail) && (
                                  <div className="mt-4">
                                    {post.postHint === 'image' && post.url && !post.url.includes('reddit.com') ? (
                                      <div className="relative group">
                                        <img 
                                          src={post.url}
                                          alt={post.title}
                                          className="max-w-full h-auto max-h-96 rounded-lg shadow-md cursor-pointer hover:shadow-lg transition-shadow"
                                          onClick={() => window.open(post.url, '_blank')}
                                          onError={(e) => {
                                            const target = e.target as HTMLImageElement;
                                            target.style.display = 'none';
                                          }}
                                        />
                                        <div className="absolute top-2 right-2 bg-black bg-opacity-50 text-white px-2 py-1 rounded text-xs opacity-0 group-hover:opacity-100 transition-opacity">
                                          Click to view full size
                                        </div>
                                      </div>
                                    ) : post.isVideo && post.url ? (
                                      <div className="relative bg-gray-100 dark:bg-gray-800 rounded-lg p-4 border-2 border-dashed border-gray-300 dark:border-gray-600">
                                        <div className="flex items-center gap-3">
                                          <div className="w-12 h-12 bg-red-100 dark:bg-red-900 rounded-full flex items-center justify-center">
                                            <span className="text-red-600 text-xl">▶</span>
                                          </div>
                                          <div className="flex-1">
                                            <p className="font-medium text-gray-900 dark:text-gray-100">Video Content</p>
                                            <p className="text-sm text-gray-600 dark:text-gray-400">Click to watch on Reddit</p>
                                          </div>
                                          <Button
                                            variant="outline"
                                            size="sm"
                                            onClick={() => window.open(`https://reddit.com${post.permalink}`, '_blank')}
                                          >
                                            Watch Video
                                          </Button>
                                        </div>
                                      </div>
                                    ) : post.thumbnail && post.thumbnail !== 'self' && post.thumbnail !== 'default' && post.thumbnail !== 'nsfw' && post.thumbnail !== 'spoiler' ? (
                                      <div className="relative">
                                        <img 
                                          src={post.thumbnail}
                                          alt={`Thumbnail for ${post.title}`}
                                          className="max-w-32 h-auto rounded-lg shadow-sm cursor-pointer hover:shadow-md transition-shadow"
                                          onClick={() => post.url ? window.open(post.url, '_blank') : window.open(`https://reddit.com${post.permalink}`, '_blank')}
                                          onError={(e) => {
                                            const target = e.target as HTMLImageElement;
                                            target.style.display = 'none';
                                          }}
                                        />
                                        <div className="absolute bottom-1 right-1 bg-black bg-opacity-60 text-white px-1 py-0.5 rounded text-xs">
                                          Thumbnail
                                        </div>
                                      </div>
                                    ) : null}
                                  </div>
                                )}

                                {/* Reddit Gallery or Multiple Images */}
                                {post.url && post.url.includes('reddit.com/gallery/') && (
                                  <div className="mt-4 p-4 bg-blue-50 dark:bg-blue-950 rounded-lg border">
                                    <div className="flex items-center gap-3">
                                      <div className="w-10 h-10 bg-blue-100 dark:bg-blue-900 rounded-full flex items-center justify-center">
                                        <span className="text-blue-600 text-lg">🖼️</span>
                                      </div>
                                      <div className="flex-1">
                                        <p className="font-medium text-blue-900 dark:text-blue-100">Reddit Gallery</p>
                                        <p className="text-sm text-blue-600 dark:text-blue-400">Multiple images - click to view gallery</p>
                                      </div>
                                      <Button
                                        variant="outline"
                                        size="sm"
                                        onClick={() => window.open(`https://reddit.com${post.permalink}`, '_blank')}
                                      >
                                        View Gallery
                                      </Button>
                                    </div>
                                  </div>
                                )}

                                {/* External Link Preview */}
                                {post.url && !post.isSelf && post.domain && !post.postHint && !post.isVideo && (
                                  <div className="mt-4 p-3 bg-gray-50 dark:bg-gray-900 rounded-lg border">
                                    <div className="flex items-center gap-3">
                                      <div className="w-8 h-8 bg-gray-200 dark:bg-gray-700 rounded flex items-center justify-center">
                                        <ExternalLink className="w-4 h-4 text-gray-600 dark:text-gray-400" />
                                      </div>
                                      <div className="flex-1">
                                        <p className="font-medium text-gray-900 dark:text-gray-100 truncate">{post.domain}</p>
                                        <p className="text-sm text-gray-600 dark:text-gray-400">External link</p>
                                      </div>
                                      <Button
                                        variant="outline"
                                        size="sm"
                                        onClick={() => window.open(post.url, '_blank')}
                                      >
                                        Visit Link
                                      </Button>
                                    </div>
                                  </div>
                                )}

                                {/* Post Stats and Actions */}
                                <div className="flex items-center justify-between">
                                  <div className="flex items-center gap-4 text-sm">
                                    <div className="flex items-center gap-1">
                                      <Button
                                        variant="ghost"
                                        size="sm"
                                        onClick={() => voteOnContentMutation.mutate({ fullname: post.fullname, direction: 1 })}
                                        disabled={loadingStates[`vote_${post.fullname}`]}
                                      >
                                        <ArrowUp className="w-4 h-4" />
                                      </Button>
                                      <span className="font-medium">{post.score}</span>
                                      <Button
                                        variant="ghost"
                                        size="sm"
                                        onClick={() => voteOnContentMutation.mutate({ fullname: post.fullname, direction: -1 })}
                                        disabled={loadingStates[`vote_${post.fullname}`]}
                                      >
                                        <ArrowDown className="w-4 h-4" />
                                      </Button>
                                    </div>
                                    <span>💬 {post.numComments} comments</span>
                                    <span>📈 {Math.round(post.upvoteRatio * 100)}% upvoted</span>
                                    {post.gilded > 0 && <span>🏆 {post.gilded} awards</span>}
                                  </div>
                                  <div className="flex gap-2">
                                    <Button
                                      variant="outline"
                                      size="sm"
                                      onClick={() => {
                                        if (expandedPosts.has(post.id)) {
                                          setExpandedPosts(prev => {
                                            const newSet = new Set(prev);
                                            newSet.delete(post.id);
                                            return newSet;
                                          });
                                        } else {
                                          fetchPostCommentsMutation.mutate({
                                            postId: post.id,
                                            subredditName: selectedInteractSubreddit
                                          });
                                        }
                                      }}
                                      disabled={loadingStates[`comments_${post.id}`]}
                                    >
                                      {loadingStates[`comments_${post.id}`] ? (
                                        <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                                      ) : expandedPosts.has(post.id) ? (
                                        <ChevronDown className="w-4 h-4 mr-2" />
                                      ) : (
                                        <ChevronRight className="w-4 h-4 mr-2" />
                                      )}
                                      {expandedPosts.has(post.id) ? 'Hide' : 'Show'} Comments
                                    </Button>
                                    <Button
                                      variant="outline"
                                      size="sm"
                                      onClick={() => setReplyingTo(prev => ({ 
                                        ...prev, 
                                        [post.id]: !prev[post.id] 
                                      }))}
                                    >
                                      <Reply className="w-4 h-4 mr-2" />
                                      Reply
                                    </Button>
                                  </div>
                                </div>

                                {/* Reply Form */}
                                {replyingTo[post.id] && (
                                  <div className="p-4 bg-gray-50 dark:bg-gray-900 rounded-lg">
                                    <div className="space-y-3">
                                      <Textarea
                                        placeholder="Write your comment..."
                                        value={newCommentData[post.id]?.content || ''}
                                        onChange={(e) => setNewCommentData(prev => ({
                                          ...prev,
                                          [post.id]: { content: e.target.value }
                                        }))}
                                        rows={3}
                                      />
                                      <div className="flex gap-2">
                                        <Button
                                          size="sm"
                                          onClick={() => createCommentMutation.mutate({
                                            subredditName: selectedInteractSubreddit,
                                            postId: post.id,
                                            content: newCommentData[post.id]?.content || ''
                                          })}
                                          disabled={loadingStates[`comment_${post.id}`] || !newCommentData[post.id]?.content?.trim()}
                                        >
                                          {loadingStates[`comment_${post.id}`] ? (
                                            <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                                          ) : (
                                            <Send className="w-4 h-4 mr-2" />
                                          )}
                                          Post Comment
                                        </Button>
                                        <Button
                                          variant="outline"
                                          size="sm"
                                          onClick={() => setReplyingTo(prev => ({ ...prev, [post.id]: false }))}
                                        >
                                          Cancel
                                        </Button>
                                      </div>
                                    </div>
                                  </div>
                                )}

                                {/* Comments Display */}
                                {expandedPosts.has(post.id) && postComments[post.id] && (
                                  <div className="mt-6 space-y-4">
                                    <div className="flex items-center gap-2 pb-2 border-b border-gray-200 dark:border-gray-700">
                                      <MessageSquare className="w-4 h-4 text-gray-600 dark:text-gray-400" />
                                      <h5 className="font-semibold text-gray-900 dark:text-gray-100">
                                        Comments ({postComments[post.id].length})
                                      </h5>
                                    </div>
                                    <div className="space-y-3">
                                      {postComments[post.id].map((comment) => (
                                        <div key={comment.id} 
                                             className={`p-4 border rounded-lg bg-white dark:bg-gray-900 ${
                                               comment.depth > 0 ? 'border-l-4 border-blue-300 dark:border-blue-600' : 'border-gray-200 dark:border-gray-700'
                                             }`}
                                             style={{ marginLeft: `${Math.min(comment.depth * 20, 80)}px` }}>
                                          <div className="flex items-start justify-between mb-3">
                                            <div className="flex flex-col gap-1">
                                              <div className="flex items-center gap-2">
                                                <span className="font-medium text-sm text-blue-600 dark:text-blue-400">u/{comment.author}</span>
                                                {comment.distinguished && <Badge variant="outline" className="text-xs">{comment.distinguished}</Badge>}
                                                {comment.stickied && <Badge variant="default" className="text-xs">Stickied</Badge>}
                                              </div>
                                              <span className="text-xs text-gray-500 dark:text-gray-400">
                                                {new Date(comment.created * 1000).toLocaleString()}
                                                {comment.edited && ' (edited)'}
                                              </span>
                                            </div>
                                            <div className="flex items-center gap-1 text-sm">
                                              <Button
                                                variant="ghost"
                                                size="sm"
                                                onClick={() => voteOnContentMutation.mutate({ 
                                                  fullname: comment.fullname, 
                                                  direction: 1 
                                                })}
                                                disabled={loadingStates[`vote_${comment.fullname}`]}
                                              >
                                                <ArrowUp className="w-3 h-3" />
                                              </Button>
                                              <span className={comment.scoreHidden ? 'text-gray-400' : ''}>
                                                {comment.scoreHidden ? '•' : comment.score}
                                              </span>
                                              <Button
                                                variant="ghost"
                                                size="sm"
                                                onClick={() => voteOnContentMutation.mutate({ 
                                                  fullname: comment.fullname, 
                                                  direction: -1 
                                                })}
                                                disabled={loadingStates[`vote_${comment.fullname}`]}
                                              >
                                                <ArrowDown className="w-3 h-3" />
                                              </Button>
                                            </div>
                                          </div>
                                          <div className="text-sm text-gray-700 dark:text-gray-300 mb-2 prose prose-sm max-w-none">
                                            {comment.bodyHtml ? (
                                              <div 
                                                dangerouslySetInnerHTML={{ 
                                                  __html: comment.bodyHtml
                                                    .replace(/<div class="md">/g, '')
                                                    .replace(/<\/div>/g, '')
                                                    .replace(/&lt;/g, '<')
                                                    .replace(/&gt;/g, '>')
                                                    .replace(/&amp;/g, '&')
                                                    .replace(/&quot;/g, '"')
                                                }}
                                              />
                                            ) : (
                                              <p className="whitespace-pre-wrap">{comment.body}</p>
                                            )}
                                          </div>
                                          <div className="flex gap-2">
                                            <Button
                                              variant="ghost"
                                              size="sm"
                                              onClick={() => setReplyingTo(prev => ({ 
                                                ...prev, 
                                                [comment.id]: !prev[comment.id] 
                                              }))}
                                            >
                                              <Reply className="w-3 h-3 mr-1" />
                                              Reply
                                            </Button>
                                            <Button
                                              variant="ghost"
                                              size="sm"
                                              onClick={() => window.open(`https://reddit.com${comment.permalink}`, '_blank')}
                                            >
                                              <ExternalLink className="w-3 h-3" />
                                            </Button>
                                          </div>

                                          {/* Reply to Comment Form */}
                                          {replyingTo[comment.id] && (
                                            <div className="mt-4 p-4 bg-blue-50 dark:bg-blue-950 rounded-lg border border-blue-200 dark:border-blue-800">
                                              <div className="space-y-3">
                                                <Textarea
                                                  placeholder="Write your reply..."
                                                  value={newCommentData[comment.id]?.content || ''}
                                                  onChange={(e) => setNewCommentData(prev => ({
                                                    ...prev,
                                                    [comment.id]: { content: e.target.value }
                                                  }))}
                                                  rows={2}
                                                />
                                                <div className="flex gap-2">
                                                  <Button
                                                    size="sm"
                                                    onClick={() => createCommentMutation.mutate({
                                                      subredditName: selectedInteractSubreddit,
                                                      parentId: comment.fullname,
                                                      content: newCommentData[comment.id]?.content || ''
                                                    })}
                                                    disabled={loadingStates[`comment_${comment.id}`] || !newCommentData[comment.id]?.content?.trim()}
                                                  >
                                                    {loadingStates[`comment_${comment.id}`] ? (
                                                      <RefreshCw className="w-3 h-3 mr-1 animate-spin" />
                                                    ) : (
                                                      <Send className="w-3 h-3 mr-1" />
                                                    )}
                                                    Reply
                                                  </Button>
                                                  <Button
                                                    variant="outline"
                                                    size="sm"
                                                    onClick={() => setReplyingTo(prev => ({ ...prev, [comment.id]: false }))}
                                                  >
                                                    Cancel
                                                  </Button>
                                                </div>
                                              </div>
                                            </div>
                                          )}
                                        </div>
                                      ))}
                                    </div>
                                  </div>
                                )}
                              </div>
                            </Card>
                          ))}
                        </div>
                      ) : (
                        <div className="text-center py-8">
                          <MessageSquare className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                          <p className="text-gray-600">Click "Load Posts" to start interacting with r/{selectedInteractSubreddit}</p>
                          <p className="text-sm text-gray-500">View posts, comments, create content, and manage your subreddit</p>
                        </div>
                      )}

                      {/* Error Logs Display */}
                      {errorLogs.length > 0 && (
                        <Card>
                          <CardHeader>
                            <CardTitle className="flex items-center gap-2 text-red-600">
                              <AlertTriangle className="w-5 h-5" />
                              Error Logs ({errorLogs.length})
                            </CardTitle>
                          </CardHeader>
                          <CardContent>
                            <div className="space-y-2 max-h-40 overflow-y-auto">
                              {errorLogs.slice(-10).map((log, index) => (
                                <div key={index} className={`p-2 rounded text-xs ${
                                  log.severity === 'high' ? 'bg-red-100 text-red-800' :
                                  log.severity === 'medium' ? 'bg-yellow-100 text-yellow-800' :
                                  'bg-gray-100 text-gray-800'
                                }`}>
                                  <div className="font-medium">{log.context}</div>
                                  <div>{log.error.message}</div>
                                  <div className="text-gray-500">{log.timestamp.toLocaleTimeString()}</div>
                                </div>
                              ))}
                            </div>
                          </CardContent>
                        </Card>
                      )}
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="proactive" className="space-y-6">
              <Card>
                <CardHeader>
                  <div className="flex items-center justify-between">
                    <div>
                      <CardTitle className="flex items-center gap-2">
                        <Shield className="w-5 h-5 text-blue-600" />
                        Proactive Moderation
                      </CardTitle>
                      <CardDescription>
                        Analyze recent posts with AI before they get reported - prevent issues proactively
                      </CardDescription>
                    </div>
                    <div className="flex gap-2">
                      <Button
                        onClick={() => selectedProactiveSubreddit && fetchRecentPostsMutation.mutate(selectedProactiveSubreddit)}
                        disabled={fetchRecentPostsMutation.isPending || !selectedProactiveSubreddit}
                        variant="outline"
                      >
                        {fetchRecentPostsMutation.isPending ? (
                          <>
                            <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                            Loading Posts...
                          </>
                        ) : (
                          <>
                            <RefreshCw className="w-4 h-4 mr-2" />
                            Load Recent Posts
                          </>
                        )}
                      </Button>
                      {recentPosts.length > 0 && (
                        <Button
                          onClick={() => selectedProactiveSubreddit && bulkAnalyzePostsMutation.mutate(selectedProactiveSubreddit)}
                          disabled={isBulkAnalyzing || !selectedProactiveSubreddit}
                          className="bg-blue-600 hover:bg-blue-700"
                        >
                          {isBulkAnalyzing ? (
                            <>
                              <div className="animate-spin rounded-full h-3 w-3 border-b-2 border-white mr-2" />
                              Analyzing All...
                            </>
                          ) : (
                            <>
                              <Zap className="w-4 h-4 mr-2" />
                              Analyze Page Posts ({currentPagePosts.length})
                            </>
                          )}
                        </Button>
                      )}
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="mb-6">
                    <label className="block text-sm font-medium mb-2">Select Subreddit to Analyze</label>
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                      {(managedSubreddits || []).map((subreddit: ManagedSubreddit) => (
                        <Button
                          key={subreddit.id}
                          variant={selectedProactiveSubreddit === subreddit.subredditName ? "default" : "outline"}
                          onClick={() => {
                            setSelectedProactiveSubreddit(subreddit.subredditName);
                            setRecentPosts([]);
                            setPostAnalysisResults({});
                            setIgnoredPosts(new Set()); // Reset ignored posts
                            setCurrentPage(1); // Reset pagination when switching subreddits
                          }}
                          className="text-left justify-start"
                        >
                          <Shield className="w-4 h-4 mr-2" />
                          {subreddit.displayName}
                        </Button>
                      ))}
                    </div>
                    {(managedSubreddits || []).length === 0 && (
                      <p className="text-gray-500 text-sm">No managed subreddits available. Add some from the other tabs first.</p>
                    )}
                  </div>

                  {selectedProactiveSubreddit && (
                    <div className="space-y-4">
                      <div className="flex items-center justify-between p-4 bg-blue-50 dark:bg-blue-950 rounded-lg border">
                        <div>
                          <h3 className="font-semibold">Analyzing: r/{selectedProactiveSubreddit}</h3>
                          <p className="text-sm text-gray-600 dark:text-gray-400">
                            {filteredPosts.length} of {recentPosts.length} posts shown • Page {currentPage} of {totalPages} ({postsPerPage} per page)
                          </p>
                        </div>
                        <div className="flex items-center gap-2">
                          <Badge variant="outline">
                            {Object.keys(postAnalysisResults).length} analyzed
                          </Badge>
                          {recentPosts.length === 150 && (
                            <Badge variant="secondary" className="text-xs">
                              Full load (150)
                            </Badge>
                          )}
                        </div>
                      </div>

                      {recentPosts.length > 0 ? (
                        <div className="space-y-4">
                          {/* Pagination Controls */}
                          {totalPages > 1 && (
                            <div className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-900 rounded-lg">
                              <div className="text-sm text-gray-600 dark:text-gray-400">
                                Showing {startIndex + 1}-{Math.min(endIndex, filteredPosts.length)} of {filteredPosts.length} posts
                              </div>
                              <div className="flex gap-2">
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => setCurrentPage(1)}
                                  disabled={currentPage === 1}
                                >
                                  First
                                </Button>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => setCurrentPage(currentPage - 1)}
                                  disabled={currentPage === 1}
                                >
                                  Previous
                                </Button>
                                <span className="flex items-center px-3 py-1 text-sm font-medium">
                                  Page {currentPage} of {totalPages}
                                </span>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => setCurrentPage(currentPage + 1)}
                                  disabled={currentPage === totalPages}
                                >
                                  Next
                                </Button>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => setCurrentPage(totalPages)}
                                  disabled={currentPage === totalPages}
                                >
                                  Last
                                </Button>
                              </div>
                            </div>
                          )}
                          
                          {currentPagePosts.map((post) => (
                            <Card key={post.id} className="p-6">
                              <div className="space-y-4">
                                <div className="flex items-start justify-between">
                                  <div className="flex-1">
                                    <div className="flex items-center gap-2 mb-2">
                                      <h4 className="text-lg font-semibold">{post.title}</h4>
                                      {post.isNsfw && (
                                        <Badge variant="destructive" className="text-xs">NSFW</Badge>
                                      )}
                                      {post.stickied && (
                                        <Badge variant="default" className="text-xs">Stickied</Badge>
                                      )}
                                      {post.locked && (
                                        <Badge variant="secondary" className="text-xs">Locked</Badge>
                                      )}
                                    </div>
                                    <p className="text-sm text-gray-600 dark:text-gray-400 mb-2">
                                      by u/{post.author} • {new Date(post.created).toLocaleString()}
                                    </p>
                                    {post.content && (
                                      <p className="text-sm text-gray-700 dark:text-gray-300 mb-3 line-clamp-3">
                                        {post.content}
                                      </p>
                                    )}

                                    {/* Image Display */}
                                    {post.url && (post.postHint === 'image' || post.url.match(/\.(jpg|jpeg|png|gif|webp)$/i)) && (
                                      <div className="mt-3 mb-3">
                                        <img 
                                          src={post.url} 
                                          alt={post.title}
                                          className="max-w-full h-auto rounded-lg border border-gray-200 dark:border-gray-700"
                                          style={{ maxHeight: '400px' }}
                                          onError={(e) => {
                                            e.currentTarget.style.display = 'none';
                                          }}
                                        />
                                      </div>
                                    )}

                                    {/* Video Display */}
                                    {post.isVideo && post.url && (
                                      <div className="mt-3 mb-3">
                                        <video 
                                          controls 
                                          className="max-w-full h-auto rounded-lg border border-gray-200 dark:border-gray-700"
                                          style={{ maxHeight: '400px' }}
                                        >
                                          <source src={post.url} type="video/mp4" />
                                          Your browser does not support the video tag.
                                        </video>
                                      </div>
                                    )}

                                    {/* Gallery/Multiple Images */}
                                    {post.galleryData && post.galleryData.items && post.galleryData.items.length > 0 && (
                                      <div className="mt-3 mb-3">
                                        <div className="grid grid-cols-2 gap-2">
                                          {post.galleryData.items.slice(0, 4).map((item, index) => (
                                            <div key={index} className="relative">
                                              <img 
                                                src={item.s?.u?.replace(/&amp;/g, '&') || item.s?.gif?.replace(/&amp;/g, '&')} 
                                                alt={`Gallery image ${index + 1}`}
                                                className="w-full h-32 object-cover rounded border border-gray-200 dark:border-gray-700"
                                                onError={(e) => {
                                                  e.currentTarget.style.display = 'none';
                                                }}
                                              />
                                              {post.galleryData.items.length > 4 && index === 3 && (
                                                <div className="absolute inset-0 bg-black bg-opacity-50 flex items-center justify-center rounded">
                                                  <span className="text-white font-semibold">+{post.galleryData.items.length - 4} more</span>
                                                </div>
                                              )}
                                            </div>
                                          ))}
                                        </div>
                                      </div>
                                    )}

                                    {/* Video Thumbnail */}
                                    {post.thumbnail && post.thumbnail !== 'self' && post.thumbnail !== 'default' && !post.url?.match(/\.(jpg|jpeg|png|gif|webp)$/i) && (
                                      <div className="mt-3 mb-3">
                                        <img 
                                          src={post.thumbnail} 
                                          alt="Post thumbnail"
                                          className="w-32 h-auto rounded border border-gray-200 dark:border-gray-700"
                                          onError={(e) => {
                                            e.currentTarget.style.display = 'none';
                                          }}
                                        />
                                      </div>
                                    )}

                                    {/* External Link Preview */}
                                    {post.url && !post.isSelf && post.domain && !post.postHint && !post.isVideo && (
                                      <div className="mt-3 mb-3 p-3 bg-gray-50 dark:bg-gray-900 rounded-lg border">
                                        <div className="flex items-center gap-3">
                                          <div className="w-8 h-8 bg-gray-200 dark:bg-gray-700 rounded flex items-center justify-center">
                                            <ExternalLink className="w-4 h-4 text-gray-600 dark:text-gray-400" />
                                          </div>
                                          <div className="flex-1">
                                            <p className="font-medium text-gray-900 dark:text-gray-100 truncate">{post.domain}</p>
                                            <p className="text-sm text-gray-600 dark:text-gray-400">External link</p>
                                          </div>
                                          <Button
                                            variant="outline"
                                            size="sm"
                                            onClick={() => window.open(post.url, '_blank')}
                                          >
                                            Visit Link
                                          </Button>
                                        </div>
                                      </div>
                                    )}

                                    <div className="flex items-center gap-4 text-sm text-gray-500">
                                      <span>↑ {post.score} points</span>
                                      <span>💬 {post.numComments} comments</span>
                                      {post.postHint && <span>📎 {post.postHint}</span>}
                                    </div>
                                  </div>
                                  <div className="flex gap-2 ml-4">
                                    <Button
                                      variant="outline"
                                      size="sm"
                                      onClick={() => window.open(`https://reddit.com${post.permalink}`, '_blank')}
                                    >
                                      <ExternalLink className="w-4 h-4" />
                                    </Button>
                                    {!postAnalysisResults[post.id] ? (
                                      <Button
                                        size="sm"
                                        onClick={() => analyzePostMutation.mutate({ 
                                          subredditName: selectedProactiveSubreddit, 
                                          postId: post.id, 
                                          postData: post 
                                        })}
                                        disabled={analyzePostMutation.isPending}
                                        className="bg-blue-600 hover:bg-blue-700"
                                      >
                                        {analyzePostMutation.isPending ? (
                                          <>
                                            <div className="animate-spin rounded-full h-3 w-3 border-b-2 border-white mr-2" />
                                            Analyzing...
                                          </>
                                        ) : (
                                          <>
                                            <Zap className="w-4 h-4 mr-2" />
                                            Analyze
                                          </>
                                        )}
                                      </Button>
                                    ) : (
                                      <Badge variant="default" className="px-3 py-1">
                                        ✓ Analyzed
                                      </Badge>
                                    )}
                                  </div>
                                </div>

                                {/* AI Analysis Results Display - Similar to moderation page */}
                                {postAnalysisResults[post.id] && (
                                  <div className="mt-4 p-4 bg-blue-50 dark:bg-blue-950 rounded-lg border">
                                    <div className="flex items-center justify-between mb-3">
                                      <div className="flex items-center gap-2">
                                        <Zap className="h-4 w-4 text-blue-600" />
                                        <span className="font-semibold text-blue-700 dark:text-blue-300">AI Analysis Results</span>
                                        <Badge variant={postAnalysisResults[post.id].severity === 'high' ? 'destructive' : postAnalysisResults[post.id].severity === 'medium' ? 'default' : 'secondary'}>
                                          {postAnalysisResults[post.id].severity.toUpperCase()} RISK
                                        </Badge>
                                        <span className="text-sm font-medium text-blue-600">
                                          {Math.round(postAnalysisResults[post.id].confidenceScore * 100)}% confidence
                                        </span>
                                      </div>
                                      <Button 
                                        size="sm" 
                                        variant="ghost" 
                                        onClick={() => {
                                          setPostAnalysisResults(prev => {
                                            const newResults = { ...prev };
                                            delete newResults[post.id];
                                            return newResults;
                                          });
                                        }}
                                      >
                                        ✕
                                      </Button>
                                    </div>

                                    <div className="space-y-3">
                                      <div className="bg-white dark:bg-gray-800 p-3 rounded border">
                                        <div className="flex items-center gap-2 mb-2">
                                          <span className="font-medium text-gray-700 dark:text-gray-300">Recommended Action:</span>
                                          <Badge variant={postAnalysisResults[post.id].suggestedAction === 'approve' ? 'default' : 'destructive'}>
                                            {postAnalysisResults[post.id].suggestedAction.toUpperCase()}
                                          </Badge>
                                        </div>
                                        <p className="text-sm text-gray-600 dark:text-gray-400">{postAnalysisResults[post.id].reasoning}</p>
                                      </div>

                                      {postAnalysisResults[post.id].violationsFound && postAnalysisResults[post.id].violationsFound.length > 0 && (
                                        <div className="bg-red-50 dark:bg-red-950 p-3 rounded border border-red-200">
                                          <h4 className="font-medium text-red-700 dark:text-red-300 mb-2">Potential Rule Violations:</h4>
                                          <div className="space-y-2">
                                            {postAnalysisResults[post.id].violationsFound.map((violation, idx) => (
                                              <div key={idx} className="text-sm">
                                                <div className="flex items-center gap-2">
                                                  <Badge variant="destructive" className="text-xs">
                                                    Rule {violation.ruleIndex + 1}
                                                  </Badge>
                                                  <span className="font-medium">{violation.ruleName}</span>
                                                  <Badge variant={violation.severity === 'high' ? 'destructive' : violation.severity === 'medium' ? 'default' : 'secondary'} className="text-xs">
                                                    {violation.severity}
                                                  </Badge>
                                                </div>
                                                <p className="text-red-600 dark:text-red-400 mt-1">{violation.violation}</p>
                                              </div>
                                            ))}
                                          </div>
                                        </div>
                                      )}

                                      <div className="flex gap-2 pt-2">
                                        <Button
                                          size="sm"
                                          variant="outline"
                                          onClick={() => window.open(`https://reddit.com${post.permalink}`, '_blank')}
                                        >
                                          <Eye className="h-3 w-3 mr-1" />
                                          Review on Reddit
                                        </Button>
                                        
                                        {/* Accept AI Suggestion Button */}
                                        {postAnalysisResults[post.id] && (
                                          <Button
                                            size="sm"
                                            variant="default"
                                            onClick={() => executeModerationActionMutation.mutate({
                                              action: postAnalysisResults[post.id].suggestedAction as 'approve' | 'remove',
                                              postId: post.id,
                                              postData: post,
                                              aiAnalysis: postAnalysisResults[post.id]
                                            })}
                                            disabled={executeModerationActionMutation.isPending}
                                            className="bg-blue-600 hover:bg-blue-700"
                                          >
                                            <Zap className="h-3 w-3 mr-1" />
                                            Accept AI Suggestion
                                          </Button>
                                        )}
                                        
                                        {/* Manual Moderation Action Buttons */}
                                        <Button
                                          size="sm"
                                          variant="default"
                                          onClick={() => executeModerationActionMutation.mutate({
                                            action: 'approve',
                                            postId: post.id,
                                            postData: post,
                                            aiAnalysis: postAnalysisResults[post.id]
                                          })}
                                          disabled={executeModerationActionMutation.isPending}
                                          className="bg-green-600 hover:bg-green-700"
                                        >
                                          <Check className="h-3 w-3 mr-1" />
                                          Approve
                                        </Button>
                                        
                                        <Button
                                          size="sm"
                                          variant="destructive"
                                          onClick={() => executeModerationActionMutation.mutate({
                                            action: 'remove',
                                            postId: post.id,
                                            postData: post,
                                            aiAnalysis: postAnalysisResults[post.id]
                                          })}
                                          disabled={executeModerationActionMutation.isPending}
                                        >
                                          <Trash2 className="h-3 w-3 mr-1" />
                                          Remove
                                        </Button>
                                        
                                        {/* Ignore Button */}
                                        <Button
                                          size="sm"
                                          variant="outline"
                                          onClick={() => {
                                            setIgnoredPosts(prev => new Set([...prev, post.id]));
                                            toast({
                                              title: "Post Ignored",
                                              description: "Post removed from view. It won't appear again during this session."
                                            });
                                          }}
                                          disabled={executeModerationActionMutation.isPending}
                                        >
                                          <EyeOff className="h-3 w-3 mr-1" />
                                          Ignore
                                        </Button>
                                      </div>
                                    </div>
                                  </div>
                                )}
                              </div>
                            </Card>
                          ))}
                          
                          {/* Bottom Pagination Controls */}
                          {totalPages > 1 && (
                            <div className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-900 rounded-lg mt-4">
                              <div className="text-sm text-gray-600 dark:text-gray-400">
                                Showing {startIndex + 1}-{Math.min(endIndex, filteredPosts.length)} of {filteredPosts.length} posts
                              </div>
                              <div className="flex gap-2">
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => setCurrentPage(1)}
                                  disabled={currentPage === 1}
                                >
                                  First
                                </Button>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => setCurrentPage(currentPage - 1)}
                                  disabled={currentPage === 1}
                                >
                                  Previous
                                </Button>
                                <span className="flex items-center px-3 py-1 text-sm font-medium">
                                  Page {currentPage} of {totalPages}
                                </span>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => setCurrentPage(currentPage + 1)}
                                  disabled={currentPage === totalPages}
                                >
                                  Next
                                </Button>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  onClick={() => setCurrentPage(totalPages)}
                                  disabled={currentPage === totalPages}
                                >
                                  Last
                                </Button>
                              </div>
                            </div>
                          )}
                        </div>
                      ) : (
                        <div className="text-center py-8">
                          <Eye className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                          <p className="text-gray-600">Click "Load Recent Posts" to start proactive moderation</p>
                          <p className="text-sm text-gray-500">This will load up to 150 recent posts for AI analysis</p>
                        </div>
                      )}
                    </div>
                  )}
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </div>
      </div>
    </DashboardLayout>
  );
}