import { useState, useEffect } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { DashboardLayout } from "@/components/dashboard-layout";
import { ChannelDropdown } from "@/components/channel-dropdown";
import { useAuth } from "@/hooks/use-auth";
import { useToast } from "@/hooks/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Search, Filter, SortAsc, Eye, Heart, TrendingUp, Play, MessageSquare, Share, Clock, Calendar, Star, Brain, Loader2, X } from "lucide-react";
import { useYouTubeStats, formatYouTubeNumber } from "@/hooks/use-youtube-stats";

interface Video {
  id: number;
  channelId: number;
  videoId: string;
  title: string;
  description: string;
  thumbnailUrl: string;
  publishedAt: string;
  duration: string;
  views: number;
  engagementRate: number;
  algorithmScore: number;
  youtubeAdvice: string | null;
  adviceGeneratedAt: string | null;
  createdAt: string;
  updatedAt: string;
}

export default function VideoLibrary() {
  const { user } = useAuth();
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [selectedChannelId, setSelectedChannelId] = useState<number | null>(null);
  const [searchTerm, setSearchTerm] = useState("");
  const [filterCategory, setFilterCategory] = useState("all");
  const [sortBy, setSortBy] = useState("recent");
  const [generatingAdvice, setGeneratingAdvice] = useState<number | null>(null);
  const [watchingVideo, setWatchingVideo] = useState<Video | null>(null);
  const [videoOrdersMap, setVideoOrdersMap] = useState<Record<number, any[]>>({});

  // Function to get order status badge color
  const getOrderStatusColor = (status: string) => {
    switch (status) {
      case 'Pending':
        return 'bg-yellow-100 text-yellow-800';
      case 'In progress':
        return 'bg-blue-100 text-blue-800';
      case 'Processing':
        return 'bg-blue-100 text-blue-800';
      case 'Partial':
        return 'bg-orange-100 text-orange-800';
      case 'Completed':
        return 'bg-green-100 text-green-800';
      case 'Canceled':
        return 'bg-red-100 text-red-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  // Function to get service type display name based on service ID
  const getServiceTypeDisplay = (serviceId: string) => {
    switch (serviceId) {
      case '1800': // Likes service
        return 'likes';
      case '940':  // Views service
      case '2369': // Views service
        return 'views';
      case '9':    // Subscribers service
      case '3108': // Subscribers service
        return 'subscribers';
      default:
        return 'items'; // Generic fallback
    }
  };

  // Fetch videos for selected channel
  const { data: videos = [], isLoading } = useQuery({
    queryKey: ["/api/videos", selectedChannelId],
    queryFn: async () => {
      if (!selectedChannelId) return [];
      const response = await apiRequest(`/api/videos?channelId=${selectedChannelId}`, "GET");
      return response;
    },
    enabled: !!selectedChannelId,
  });

  // Fetch YouTube statistics for all videos
  const videoIds = videos.map((video: any) => video.videoId).filter(Boolean);
  const { data: youtubeStats = [], isLoading: youtubeStatsLoading } = useYouTubeStats(videoIds, !!selectedChannelId && videoIds.length > 0);

  // Create a map of video statistics for quick lookup
  const youtubeStatsMap = youtubeStats.reduce((acc: any, stat: any) => {
    acc[stat.videoId] = stat;
    return acc;
  }, {});

  // Fetch video orders for each video when videos are loaded
  useQuery({
    queryKey: ["video-orders", videos?.map((v: any) => v.id).join(",")],
    queryFn: async () => {
      if (!videos || videos.length === 0) return {};
      
      const ordersMap: Record<number, any[]> = {};
      
      // Fetch orders for each video
      await Promise.all(
        videos.map(async (video: any) => {
          try {
            const response = await fetch(`/api/video-orders/${video.id}`, {
              credentials: "include",
            });
            if (response.ok) {
              const orders = await response.json();
              ordersMap[video.id] = Array.isArray(orders) ? orders : [];
            } else {
              ordersMap[video.id] = [];
            }
          } catch (error) {
            console.error(`Error fetching orders for video ${video.id}:`, error);
            ordersMap[video.id] = [];
          }
        })
      );
      
      setVideoOrdersMap(ordersMap);
      return ordersMap;
    },
    enabled: !!videos && videos.length > 0,
    retry: false,
    refetchInterval: 30000, // Refetch every 30 seconds to update order status
  });

  // Filter and sort videos
  const filteredVideos = videos
    .filter((video: Video) => {
      const matchesSearch = video.title.toLowerCase().includes(searchTerm.toLowerCase());
      const matchesCategory = 
        filterCategory === "all" || 
        (filterCategory === "top" && video.algorithmScore >= 8.0) ||
        (filterCategory === "underperforming" && video.algorithmScore < 7.0);
      return matchesSearch && matchesCategory;
    })
    .sort((a: Video, b: Video) => {
      switch (sortBy) {
        case "views": return b.views - a.views;
        case "engagement": return b.engagementRate - a.engagementRate;
        case "algorithm": return b.algorithmScore - a.algorithmScore;
        case "recent": return new Date(b.publishedAt || 0).getTime() - new Date(a.publishedAt || 0).getTime();
        default: return 0;
      }
    });

  const parseAdvice = (adviceJson: string | null) => {
    if (!adviceJson) return null;
    try {
      return JSON.parse(adviceJson);
    } catch {
      return null;
    }
  };

  const generateAdviceMutation = useMutation({
    mutationFn: async (videoId: number) => {
      console.log("🚀 Making API request to:", `/api/videos/${videoId}/generate-advice`);
      const response = await fetch(`/api/videos/${videoId}/generate-advice`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        credentials: 'include', // Include cookies for authentication
      });
      console.log("📡 API response received:", response.status, response.statusText);
      
      if (!response.ok) {
        if (response.headers.get('content-type')?.includes('text/html')) {
          throw new Error('Authentication required or server error');
        }
        const error = await response.json().catch(() => ({ message: 'Network error' }));
        throw new Error(error.message || 'Failed to generate advice');
      }
      
      const data = await response.json();
      return data;
    },
    onSuccess: (data, videoId) => {
      toast({
        title: "AI Advice Generated",
        description: "Claude Sonnet has analyzed your video and provided optimization recommendations.",
      });
      
      queryClient.invalidateQueries({ queryKey: ["/api/videos"] });
      setGeneratingAdvice(null);
    },
    onError: (error: Error) => {
      toast({
        title: "Failed to Generate Advice",
        description: error.message,
        variant: "destructive",
      });
      setGeneratingAdvice(null);
    },
  });

  const handleGenerateAdvice = (videoId: number) => {
    console.log("🎯 Generate Advice button clicked for video ID:", videoId);
    console.log("🔍 Current generating advice state:", generatingAdvice);
    console.log("💫 About to call generateAdviceMutation.mutate with:", videoId);
    setGeneratingAdvice(videoId);
    generateAdviceMutation.mutate(videoId);
  };

  if (isLoading) {
    return (
      <DashboardLayout>
        <div className="min-h-screen bg-gray-50 flex items-center justify-center">
          <div className="text-center">
            <div className="animate-spin rounded-full h-16 w-16 border-b-2 border-google-blue mx-auto mb-4"></div>
            <h2 className="text-xl font-semibold text-gray-900 mb-2">Loading Video Library</h2>
            <p className="text-gray-600">Analyzing video performance data...</p>
          </div>
        </div>
      </DashboardLayout>
    );
  }

  return (
    <DashboardLayout>
      <div className="p-6 bg-gray-50 dark:bg-gray-900">
        <div className="mb-4 sm:mb-6">
          <h2 className="text-xl sm:text-2xl font-semibold text-gray-900 mb-3 sm:mb-4">Video Library</h2>
          
          {/* Channel Selection */}
          <div className="mb-4 sm:mb-6">
            <ChannelDropdown 
              selectedChannelId={selectedChannelId || undefined} 
              onChannelChange={setSelectedChannelId} 
            />
          </div>
        </div>
        
        {!selectedChannelId ? (
          <div className="text-center py-20">
            <div className="w-16 h-16 bg-google-blue rounded-full flex items-center justify-center mx-auto mb-4">
              <span className="material-icons text-white text-2xl">video_library</span>
            </div>
            <h3 className="text-xl font-semibold text-gray-900 mb-2">Select a Channel</h3>
            <p className="text-gray-600">Choose a channel from the dropdown above to view detailed video analytics.</p>
          </div>
        ) : (
          <div className="space-y-6">
            {/* Filters and Search */}
            <div className="bg-white rounded-lg border border-gray-200 p-3 sm:p-4">
              <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-3 sm:space-y-0">
                <div className="flex items-center space-x-3 sm:space-x-4">
                  <div className="relative flex-1 sm:flex-none">
                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                    <Input
                      type="text"
                      placeholder="Search videos..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="pl-10 w-full sm:w-64"
                    />
                  </div>
                  
                  <Select value={filterCategory} onValueChange={setFilterCategory}>
                    <SelectTrigger className="w-40">
                      <Filter className="w-4 h-4 mr-2" />
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">All Videos</SelectItem>
                      <SelectItem value="top">Top Performing</SelectItem>
                      <SelectItem value="underperforming">Needs Attention</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                
                <Select value={sortBy} onValueChange={setSortBy}>
                  <SelectTrigger className="w-40">
                    <SortAsc className="w-4 h-4 mr-2" />
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="recent">Most Recent</SelectItem>
                    <SelectItem value="views">Most Views</SelectItem>
                    <SelectItem value="engagement">Best Engagement</SelectItem>
                    <SelectItem value="algorithm">Algorithm Score</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            {/* Video Grid */}
            {filteredVideos.length === 0 ? (
              <div className="text-center py-12">
                <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                  <Search className="w-8 h-8 text-gray-400" />
                </div>
                <h3 className="text-lg font-semibold text-gray-900 mb-2">No videos found</h3>
                <p className="text-gray-600">Try adjusting your search terms or filters.</p>
              </div>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {filteredVideos.map((video: Video) => {
                  const advice = parseAdvice(video.youtubeAdvice);
                  
                  return (
                    <Card key={video.id} className="overflow-hidden hover:shadow-lg transition-shadow">
                      <div className="relative">
                        <img 
                          src={video.thumbnailUrl} 
                          alt={video.title}
                          className="w-full h-48 object-cover"
                        />
                        <div className="absolute bottom-2 right-2 bg-black bg-opacity-75 text-white text-xs px-2 py-1 rounded">
                          {youtubeStatsMap[video.videoId] ? 
                            youtubeStatsMap[video.videoId].duration : 
                            (youtubeStatsLoading ? 'Loading...' : 
                              (video.duration && typeof video.duration === 'number' ? `${Math.floor(video.duration / 60)}:${(video.duration % 60).toString().padStart(2, '0')}` : '0:00')
                            )
                          }
                        </div>
                        <div className="absolute top-2 left-2">
                          <Badge 
                            variant={video.algorithmScore >= 8 ? "default" : video.algorithmScore >= 7 ? "secondary" : "destructive"}
                            className="text-xs"
                          >
                            Score: {video.algorithmScore?.toFixed(1) || 'N/A'}
                          </Badge>
                        </div>
                      </div>
                      
                      <CardContent className="p-4">
                        <h3 className="font-semibold text-gray-900 mb-2 line-clamp-2">{video.title}</h3>
                        
                        {/* Core YouTube Stats */}
                        <div className="grid grid-cols-2 gap-2 text-sm text-gray-600 mb-3">
                          <div className="flex items-center">
                            <Eye className="w-4 h-4 mr-1" />
                            {youtubeStatsMap[video.videoId] ? 
                              formatYouTubeNumber(youtubeStatsMap[video.videoId].viewCount) : 
                              (youtubeStatsLoading ? 'Loading...' : video.views?.toLocaleString() || 'N/A')
                            }
                          </div>
                          <div className="flex items-center">
                            <Heart className="w-4 h-4 mr-1" />
                            {youtubeStatsMap[video.videoId] ? 
                              formatYouTubeNumber(youtubeStatsMap[video.videoId].likeCount) : 
                              (youtubeStatsLoading ? 'Loading...' : 'N/A')
                            }
                          </div>
                          <div className="flex items-center">
                            <MessageSquare className="w-4 h-4 mr-1" />
                            {youtubeStatsMap[video.videoId] ? 
                              formatYouTubeNumber(youtubeStatsMap[video.videoId].commentCount) : 
                              (youtubeStatsLoading ? 'Loading...' : 'N/A')
                            }
                          </div>
                          <div className="flex items-center">
                            <Star className="w-4 h-4 mr-1" />
                            {youtubeStatsMap[video.videoId] ? 
                              formatYouTubeNumber(youtubeStatsMap[video.videoId].favoriteCount) : 
                              (youtubeStatsLoading ? 'Loading...' : 'N/A')
                            }
                          </div>
                        </div>

                        {/* Comprehensive YouTube Data - Expandable */}
                        {youtubeStatsMap[video.videoId] && (
                          <details className="mb-3">
                            <summary className="text-xs font-medium text-muted-foreground cursor-pointer hover:text-foreground transition-colors">
                              📊 Complete YouTube Data
                            </summary>
                            <div className="mt-2 space-y-3 text-xs">
                              
                              {/* Video Details */}
                              <div className="bg-blue-50 dark:bg-blue-950 rounded-lg p-2">
                                <h5 className="font-medium text-blue-900 dark:text-blue-100 mb-1">Video Details</h5>
                                <div className="grid grid-cols-2 gap-1 text-blue-800 dark:text-blue-200">
                                  <div>Status: <span className="font-medium">{youtubeStatsMap[video.videoId].privacyStatus}</span></div>
                                  <div>Upload: <span className="font-medium">{youtubeStatsMap[video.videoId].uploadStatus}</span></div>
                                  <div>Definition: <span className="font-medium">{youtubeStatsMap[video.videoId].definition?.toUpperCase()}</span></div>
                                  <div>Dimension: <span className="font-medium">{youtubeStatsMap[video.videoId].dimension}</span></div>
                                  <div>Captions: <span className="font-medium">{youtubeStatsMap[video.videoId].caption === 'true' ? 'Yes' : 'No'}</span></div>
                                  <div>Embeddable: <span className="font-medium">{youtubeStatsMap[video.videoId].embeddable ? 'Yes' : 'No'}</span></div>
                                  <div>Made for Kids: <span className="font-medium">{youtubeStatsMap[video.videoId].madeForKids ? 'Yes' : 'No'}</span></div>
                                  <div>Licensed: <span className="font-medium">{youtubeStatsMap[video.videoId].licensedContent ? 'Yes' : 'No'}</span></div>
                                </div>
                                {youtubeStatsMap[video.videoId].publishedAt && (
                                  <div className="mt-1 text-blue-700 dark:text-blue-300">
                                    Published: {new Date(youtubeStatsMap[video.videoId].publishedAt).toLocaleDateString()} at {new Date(youtubeStatsMap[video.videoId].publishedAt).toLocaleTimeString()}
                                  </div>
                                )}
                              </div>

                              {/* Technical Info */}
                              <div className="bg-green-50 dark:bg-green-950 rounded-lg p-2">
                                <h5 className="font-medium text-green-900 dark:text-green-100 mb-1">Technical Info</h5>
                                <div className="grid grid-cols-2 gap-1 text-green-800 dark:text-green-200">
                                  <div>License: <span className="font-medium">{youtubeStatsMap[video.videoId].license}</span></div>
                                  <div>Projection: <span className="font-medium">{youtubeStatsMap[video.videoId].projection}</span></div>
                                  <div>Live Content: <span className="font-medium">{youtubeStatsMap[video.videoId].liveBroadcastContent}</span></div>
                                  <div>Public Stats: <span className="font-medium">{youtubeStatsMap[video.videoId].publicStatsViewable ? 'Yes' : 'No'}</span></div>
                                </div>
                                {youtubeStatsMap[video.videoId].defaultLanguage && (
                                  <div className="mt-1 text-green-700 dark:text-green-300">
                                    Language: {youtubeStatsMap[video.videoId].defaultLanguage}
                                    {youtubeStatsMap[video.videoId].defaultAudioLanguage && youtubeStatsMap[video.videoId].defaultAudioLanguage !== youtubeStatsMap[video.videoId].defaultLanguage && 
                                      ` | Audio: ${youtubeStatsMap[video.videoId].defaultAudioLanguage}`}
                                  </div>
                                )}
                              </div>

                              {/* Channel Info */}
                              <div className="bg-purple-50 dark:bg-purple-950 rounded-lg p-2">
                                <h5 className="font-medium text-purple-900 dark:text-purple-100 mb-1">Channel Info</h5>
                                <div className="text-purple-800 dark:text-purple-200">
                                  <div>Channel: <span className="font-medium">{youtubeStatsMap[video.videoId].channelTitle}</span></div>
                                  <div className="text-xs text-purple-600 dark:text-purple-400 mt-1">ID: {youtubeStatsMap[video.videoId].channelId}</div>
                                </div>
                              </div>

                              {/* Video Tags */}
                              {youtubeStatsMap[video.videoId].tags && youtubeStatsMap[video.videoId].tags.length > 0 && (
                                <div className="bg-orange-50 dark:bg-orange-950 rounded-lg p-2">
                                  <h5 className="font-medium text-orange-900 dark:text-orange-100 mb-1">Tags ({youtubeStatsMap[video.videoId].tags.length})</h5>
                                  <div className="flex flex-wrap gap-1">
                                    {youtubeStatsMap[video.videoId].tags.slice(0, 6).map((tag: string, idx: number) => (
                                      <span key={idx} className="bg-orange-200 dark:bg-orange-800 text-orange-800 dark:text-orange-200 px-1 py-0.5 rounded text-xs">
                                        {tag}
                                      </span>
                                    ))}
                                    {youtubeStatsMap[video.videoId].tags.length > 6 && (
                                      <span className="text-orange-600 dark:text-orange-400 text-xs">
                                        +{youtubeStatsMap[video.videoId].tags.length - 6} more
                                      </span>
                                    )}
                                  </div>
                                </div>
                              )}

                              {/* Topic Categories */}
                              {youtubeStatsMap[video.videoId].topicCategories && youtubeStatsMap[video.videoId].topicCategories.length > 0 && (
                                <div className="bg-indigo-50 dark:bg-indigo-950 rounded-lg p-2">
                                  <h5 className="font-medium text-indigo-900 dark:text-indigo-100 mb-1">Topics</h5>
                                  <div className="text-indigo-800 dark:text-indigo-200">
                                    {youtubeStatsMap[video.videoId].topicCategories.map((topic: string, idx: number) => (
                                      <div key={idx} className="text-xs">
                                        {topic.replace('https://en.wikipedia.org/wiki/', '').replace(/_/g, ' ')}
                                      </div>
                                    ))}
                                  </div>
                                </div>
                              )}

                              {/* Recording Date */}
                              {youtubeStatsMap[video.videoId].recordingDate && (
                                <div className="bg-gray-50 dark:bg-gray-900 rounded-lg p-2">
                                  <h5 className="font-medium text-gray-900 dark:text-gray-100 mb-1">Recording Date</h5>
                                  <div className="text-gray-800 dark:text-gray-200 text-xs">
                                    {new Date(youtubeStatsMap[video.videoId].recordingDate).toLocaleDateString()}
                                  </div>
                                </div>
                              )}

                              {/* Video Description Preview */}
                              {youtubeStatsMap[video.videoId].description && (
                                <div className="bg-slate-50 dark:bg-slate-900 rounded-lg p-2">
                                  <h5 className="font-medium text-slate-900 dark:text-slate-100 mb-1">Description</h5>
                                  <div className="text-slate-800 dark:text-slate-200 text-xs line-clamp-3">
                                    {youtubeStatsMap[video.videoId].description}
                                  </div>
                                </div>
                              )}

                            </div>
                          </details>
                        )}
                        
                        <div className="flex items-center justify-between text-xs text-gray-500 mb-3">
                          <span>{video.publishedAt ? new Date(video.publishedAt).toLocaleDateString() : 'N/A'}</span>
                        </div>
                        
                        <div className="flex space-x-2">
                          <Button 
                            size="sm" 
                            variant="outline"
                            onClick={() => setWatchingVideo(video)}
                            className="flex-1"
                          >
                            <Play className="w-4 h-4 mr-1" />
                            Watch
                          </Button>
                          
                          {parseAdvice(video.youtubeAdvice) && (
                            <Dialog>
                              <DialogTrigger asChild>
                                <Button size="sm" variant="outline">
                                  <Star className="w-4 h-4 mr-1" />
                                  AI Tips
                                </Button>
                              </DialogTrigger>
                              <DialogContent className="max-w-2xl">
                                <DialogHeader>
                                  <DialogTitle className="flex items-center space-x-2">
                                    <Brain className="w-5 h-5 text-purple-600" />
                                    <span>Claude Sonnet 4 - Video Optimization Advice</span>
                                  </DialogTitle>
                                </DialogHeader>
                                    <div className="space-y-4">
                                      {parseAdvice(video.youtubeAdvice)?.keyStrengths && (
                                        <div>
                                          <h4 className="font-semibold mb-2 text-green-700">Key Strengths:</h4>
                                          <p className="text-sm text-gray-600">{parseAdvice(video.youtubeAdvice).keyStrengths}</p>
                                        </div>
                                      )}
                                      
                                      {parseAdvice(video.youtubeAdvice)?.mainIssues && (
                                        <div>
                                          <h4 className="font-semibold mb-2 text-red-700">Main Issues:</h4>
                                          <p className="text-sm text-gray-600">{parseAdvice(video.youtubeAdvice).mainIssues}</p>
                                        </div>
                                      )}
                                      
                                      {parseAdvice(video.youtubeAdvice)?.actionableSteps && (
                                        <div>
                                          <h4 className="font-semibold mb-2 text-blue-700">Actionable Steps:</h4>
                                          <ul className="list-disc pl-5 space-y-1">
                                            {parseAdvice(video.youtubeAdvice).actionableSteps.map((step: string, index: number) => (
                                              <li key={index} className="text-sm text-gray-600">{step}</li>
                                            ))}
                                          </ul>
                                        </div>
                                      )}
                                      
                                      {parseAdvice(video.youtubeAdvice)?.expectedImpact && (
                                        <div>
                                          <h4 className="font-semibold mb-2 text-purple-700">Expected Impact:</h4>
                                          <p className="text-sm text-gray-600">{parseAdvice(video.youtubeAdvice).expectedImpact}</p>
                                        </div>
                                      )}
                                      
                                      {parseAdvice(video.youtubeAdvice)?.timeframe && (
                                        <div>
                                          <h4 className="font-semibold mb-2 text-orange-700">Timeframe:</h4>
                                          <p className="text-sm text-gray-600">{parseAdvice(video.youtubeAdvice).timeframe}</p>
                                        </div>
                                      )}
                                    </div>
                              </DialogContent>
                            </Dialog>
                          )}
                        </div>

                        {/* Video Orders Section */}
                        {videoOrdersMap[video.id] && videoOrdersMap[video.id].length > 0 && (
                          <div className="mt-4 pt-3 border-t border-gray-100">
                            <div className="flex items-center justify-between mb-3">
                              <span className="text-sm font-medium text-gray-900">Orders</span>
                              <span className="text-xs text-gray-500">{videoOrdersMap[video.id].length} orders</span>
                            </div>
                            <div className="space-y-2">
                              {videoOrdersMap[video.id].slice(0, 3).map((order: any) => (
                                <div key={order.id} className="bg-gray-50 rounded-lg p-3 text-xs">
                                  <div className="flex items-center justify-between mb-2">
                                    <div className="flex items-center space-x-2">
                                      <div className={`px-2 py-1 rounded text-xs font-medium ${getOrderStatusColor(order.status)}`}>
                                        {order.status}
                                      </div>
                                      <span className="text-gray-600">{order.quantity} {getServiceTypeDisplay(order.service)}</span>
                                      <span className="text-gray-500">#{order.orderId?.slice(-6)}</span>
                                    </div>
                                    <span className="text-gray-500">
                                      {order.createdAt && new Date(order.createdAt).toLocaleDateString()}
                                    </span>
                                  </div>
                                  
                                  {/* Order Details */}
                                  {(order.charge || order.remains) && (
                                    <div className="grid grid-cols-2 gap-2 text-xs">
                                      {order.charge && (
                                        <div>
                                          <span className="text-gray-500">Charge: </span>
                                          <span className="font-medium">${order.charge} {order.currency || 'USD'}</span>
                                        </div>
                                      )}
                                      {order.remains && (
                                        <div>
                                          <span className="text-gray-500">Remaining: </span>
                                          <span className="font-medium">{order.remains}</span>
                                        </div>
                                      )}
                                    </div>
                                  )}
                                </div>
                              ))}
                              {videoOrdersMap[video.id].length > 3 && (
                                <div className="text-xs text-gray-500 text-center py-2">
                                  +{videoOrdersMap[video.id].length - 3} more orders
                                </div>
                              )}
                            </div>
                          </div>
                        )}
                        
                        {/* Claude Sonnet 4 AI Advice Section */}
                        <div className="mt-4 pt-3 border-t border-gray-100">
                          <div className="flex items-center justify-between mb-3">
                            <div className="flex items-center space-x-2">
                              <Brain className="w-4 h-4 text-purple-600" />
                              <span className="text-sm font-medium text-gray-900">OpenWeb 3.2 AI Advice</span>
                            </div>
                            <Button
                              size="sm"
                              variant="outline"
                              onClick={() => handleGenerateAdvice(video.id)}
                              disabled={generatingAdvice === video.id}
                              className="text-xs"
                            >
                              {generatingAdvice === video.id ? (
                                <>
                                  <Loader2 className="w-3 h-3 mr-1 animate-spin" />
                                  Generating...
                                </>
                              ) : (
                                <>
                                  <Brain className="w-3 h-3 mr-1" />
                                  Generate
                                </>
                              )}
                            </Button>
                          </div>
                          
                          {video.youtubeAdvice ? (
                            <div className="bg-purple-50 rounded-lg p-3">
                              <div className="text-sm text-gray-700 mb-2">
                                <span className="font-medium text-purple-700">AI Analysis Complete</span>
                              </div>
                              <p className="text-xs text-gray-600">
                                Claude Sonnet has analyzed this video and provided optimization recommendations. Click "AI Tips" above to view the full analysis.
                              </p>
                            </div>
                          ) : (
                            <div className="bg-gray-50 rounded-lg p-3 text-center">
                              <p className="text-xs text-gray-500 mb-2">
                                No AI advice generated yet.
                              </p>
                              <p className="text-xs text-gray-400">
                                Click "Generate" to get Claude Sonnet 4's optimization recommendations for this video's metadata, thumbnail, and content strategy.
                              </p>
                            </div>
                          )}
                        </div>
                      </CardContent>
                    </Card>
                  );
                })}
              </div>
            )}
          </div>
        )}
        
        {/* Video Watch Modal */}
        {watchingVideo && (
          <Dialog open={!!watchingVideo} onOpenChange={() => setWatchingVideo(null)}>
            <DialogContent className="max-w-4xl w-full h-[80vh]">
              <DialogHeader>
                <DialogTitle className="flex items-center justify-between">
                  <span className="truncate">{watchingVideo.title}</span>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => setWatchingVideo(null)}
                  >
                    <X className="w-4 h-4" />
                  </Button>
                </DialogTitle>
              </DialogHeader>
              <div className="flex-1 min-h-0">
                <iframe
                  width="100%"
                  height="100%"
                  src={`https://www.youtube.com/embed/${watchingVideo.videoId}?autoplay=1&rel=0`}
                  title={watchingVideo.title}
                  frameBorder="0"
                  allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                  allowFullScreen
                  className="rounded-lg"
                />
              </div>
            </DialogContent>
          </Dialog>
        )}
      </div>
    </DashboardLayout>
  );
}