import bcrypt from "bcryptjs";
import { randomBytes } from "crypto";
import { storage } from "./storage";
import type { User, InsertUser } from "@shared/schema";

export class AuthService {
  private static readonly SESSION_DURATION = 24 * 60 * 60 * 1000; // 24 hours
  private static readonly SALT_ROUNDS = 12;

  static async hashPassword(password: string): Promise<string> {
    return await bcrypt.hash(password, this.SALT_ROUNDS);
  }

  static async verifyPassword(password: string, hash: string): Promise<boolean> {
    return await bcrypt.compare(password, hash);
  }

  static generateSessionId(): string {
    return randomBytes(32).toString("hex");
  }

  static async createSession(userId: number): Promise<string> {
    const sessionId = this.generateSessionId();
    const expiresAt = new Date(Date.now() + this.SESSION_DURATION);
    
    await storage.createSession({
      id: sessionId,
      userId,
      expiresAt
    });

    return sessionId;
  }

  static async validateSession(sessionId: string): Promise<User | null> {
    const session = await storage.getSession(sessionId);
    
    if (!session || session.expiresAt < new Date()) {
      if (session) {
        await storage.deleteSession(sessionId);
      }
      return null;
    }

    const user = await storage.getUser(session.userId);
    return user || null;
  }

  static async login(email: string, password: string): Promise<{ user: User; sessionId: string } | null> {
    const user = await storage.getUserByEmail(email);
    
    if (!user || !user.isActive) {
      return null;
    }

    const isValidPassword = await this.verifyPassword(password, user.password);
    if (!isValidPassword) {
      return null;
    }

    // Update last login
    await storage.updateUser(user.id, { lastLogin: new Date() });

    const sessionId = await this.createSession(user.id);
    
    return { user, sessionId };
  }

  static async logout(sessionId: string): Promise<void> {
    await storage.deleteSession(sessionId);
  }

  static async register(userData: InsertUser): Promise<User> {
    const hashedPassword = await this.hashPassword(userData.password);
    
    return await storage.createUser({
      ...userData,
      password: hashedPassword
    });
  }

  static async createAdminUser(): Promise<User | null> {
    try {
      // Check if admin user already exists
      const existingUser = await storage.getUserByEmail("kwright@staffaccess.space");
      if (existingUser) {
        console.log("Admin user already exists");
        return existingUser;
      }

      const adminData: InsertUser = {
        username: "kwright",
        email: "kwright@staffaccess.space",
        password: "Maniac20!",
        role: "admin",
        isActive: true
      };

      const adminUser = await this.register(adminData);
      console.log("Admin user created successfully");
      return adminUser;
    } catch (error) {
      console.error("Failed to create admin user:", error);
      return null;
    }
  }
}