/**
 * Azure AI Service for Reddit Moderation
 * Uses Azure AI Studio endpoint with custom agent for OpenWeb 3.2 AI
 */

export interface ModerationAnalysis {
  suggestedAction: 'approve' | 'remove' | 'spam' | 'manual_review';
  reasoning: string;
  confidenceScore: number;
  severity: 'low' | 'medium' | 'high';
  violationsFound: Array<{
    ruleIndex: number;
    ruleName: string;
    violation: string;
    severity: 'low' | 'medium' | 'high';
  }>;
  removalReason?: string;
  requiresHumanReview: boolean;
  additionalActions: string[];
}

const AZURE_API_ENDPOINT = "https://rmildlyinteresting-ai-resource.services.ai.azure.com/api/projects/rmildlyinteresting_ai";
const AZURE_API_TOKEN = "3oazOCsRFOgFZzPRdPJwjCPjXyTsEaDufsiFASDz0BRk0hUFUQbPJQQJ99BGACHYHv6XJ3w3AAAAACOGuYMy";
const AZURE_AGENT_ID = "asst_Lr5MKoJCvjM8vutfvtaj3RIx";

export class AzureAIService {
  /**
   * Moderate content using Azure Llama 3.2 90B for Reddit posts
   */
  static async moderateContent(
    title: string,
    content: string,
    userReports: any[],
    modReports: any[],
    subredditRules: any
  ): Promise<ModerationAnalysis> {
    try {
      console.log(`🤖 [MODERATION AI] Starting Azure AI analysis for content: "${title.substring(0, 50)}..."`);

      // Use the new Azure AI endpoint with agent ID
      if (!AZURE_API_TOKEN) {
        throw new Error('Azure AI API token not configured');
      }

      const prompt = `Analyze this Reddit post for potential rule violations and provide moderation recommendations.

POST DETAILS:
Title: "${title}"
Content: "${content || 'No content (link post)'}"

SUBREDDIT RULES:
${JSON.stringify(subredditRules, null, 2)}

USER REPORTS: ${userReports.length > 0 ? JSON.stringify(userReports) : 'None'}
MOD REPORTS: ${modReports.length > 0 ? JSON.stringify(modReports) : 'None'}

Provide your analysis in this exact JSON format (respond with ONLY the JSON, no additional text):
{
  "suggestedAction": "approve|remove|spam|manual_review",
  "reasoning": "Clear explanation of your recommendation",
  "confidenceScore": 0.85,
  "severity": "low|medium|high",
  "violationsFound": [
    {
      "ruleIndex": 0,
      "ruleName": "Rule name",
      "violation": "Description of violation",
      "severity": "low|medium|high"
    }
  ],
  "removalReason": "Brief reason for removal if removing",
  "requiresHumanReview": false,
  "additionalActions": ["List any additional recommended actions"]
}`;

      // Use Azure AI Foundry chat completions endpoint
      const response = await fetch(`${AZURE_API_ENDPOINT}/chat/completions?api-version=2024-05-01-preview`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'api-key': AZURE_API_TOKEN
        },
        body: JSON.stringify({
          messages: [
            {
              role: "system",
              content: "You are an expert Reddit moderator AI assistant. Analyze posts for rule violations and provide detailed moderation recommendations in JSON format."
            },
            {
              role: "user", 
              content: prompt
            }
          ],
          max_tokens: 1000,
          temperature: 0.3,
          top_p: 0.1,
          model: "gpt-4o-mini"
        })
      });

      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`Azure AI request failed: ${response.status} - ${errorText}`);
      }

      const data = await response.json();
      
      if (!data.choices || data.choices.length === 0) {
        throw new Error('No response choices from Azure AI');
      }

      let analysisText = data.choices[0].message?.content || '';
      
      // Clean markdown code blocks if present
      if (analysisText.startsWith('```json')) {
        analysisText = analysisText.replace(/^```json\s*/, '').replace(/\s*```$/, '');
      } else if (analysisText.startsWith('```')) {
        analysisText = analysisText.replace(/^```\s*/, '').replace(/\s*```$/, '');
      }

      console.log(`🤖 [MODERATION AI] Azure AI analysis result: ${analysisText.substring(0, 200)}...`);

      const analysis = JSON.parse(analysisText);
      
      // Validate and ensure all required fields exist
      const validatedAnalysis: ModerationAnalysis = {
        suggestedAction: analysis.suggestedAction || 'manual_review',
        reasoning: analysis.reasoning || 'Analysis could not be completed',
        confidenceScore: Math.max(0, Math.min(1, analysis.confidenceScore || 0.5)),
        severity: analysis.severity || 'medium',
        violationsFound: Array.isArray(analysis.violationsFound) ? analysis.violationsFound : [],
        removalReason: analysis.removalReason || null,
        requiresHumanReview: analysis.requiresHumanReview !== false,
        additionalActions: Array.isArray(analysis.additionalActions) ? analysis.additionalActions : []
      };

      console.log(`🤖 [MODERATION AI] Azure analysis complete: ${validatedAnalysis.suggestedAction} (${Math.round(validatedAnalysis.confidenceScore * 100)}% confidence)`);
      return validatedAnalysis;

    } catch (error) {
      console.error('🤖 [MODERATION AI] Error analyzing content with Azure AI:', error);
      
      // Return safe fallback analysis
      return {
        suggestedAction: 'manual_review',
        reasoning: 'AI analysis failed - requires human review',
        confidenceScore: 0,
        severity: 'medium',
        violationsFound: [],
        removalReason: null,
        requiresHumanReview: true,
        additionalActions: ['Requires manual review due to AI error']
      };
    }
  }

  /**
   * Generate human-like response for modmail complaints using Azure AI
   */
  static async generateHumanResponse(
    postTitle: string,
    removalReason: string,
    username: string,
    subredditName: string
  ): Promise<string> {
    try {
      console.log(`🤖 [MODERATION AI] Generating human response for modmail complaint about "${postTitle}"`);

      if (!AZURE_API_TOKEN) {
        throw new Error('Azure AI API token not configured');
      }

      const prompt = `You are a professional Reddit moderator responding to a user complaint about a removed post. Generate a helpful, empathetic response that addresses their concern while explaining the moderation decision.

POST DETAILS:
- Post Title: "${postTitle}"
- Removal Reason: "${removalReason}"
- User: ${username}
- Subreddit: r/${subredditName}

Write a professional, human-sounding response that:
1. Acknowledges their concern respectfully
2. Explains why the post was removed without being defensive
3. Offers constructive guidance for future posts
4. Maintains a friendly but professional tone
5. Does NOT reveal that AI was involved in the decision

Keep the response concise (2-3 paragraphs) and helpful. Address the user by their username.`;

      // Create a thread for the agent conversation
      const threadResponse = await fetch(`${AZURE_API_ENDPOINT}/threads?api-version=2025-05-01`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${AZURE_API_TOKEN}`
        },
        body: JSON.stringify({})
      });

      if (!threadResponse.ok) {
        const threadError = await threadResponse.text();
        throw new Error(`Thread creation failed: ${threadResponse.status} - ${threadError}`);
      }

      const threadData = await threadResponse.json();
      const threadId = threadData.id;

      // Add message to thread
      const messageResponse = await fetch(`${AZURE_API_ENDPOINT}/threads/${threadId}/messages?api-version=2025-05-01`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${AZURE_API_TOKEN}`
        },
        body: JSON.stringify({
          role: "user",
          content: prompt
        })
      });

      if (!messageResponse.ok) {
        const messageError = await messageResponse.text();
        throw new Error(`Message creation failed: ${messageResponse.status} - ${messageError}`);
      }

      // Create run with the specific agent
      const runResponse = await fetch(`${AZURE_API_ENDPOINT}/threads/${threadId}/runs?api-version=2025-05-01`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${AZURE_API_TOKEN}`
        },
        body: JSON.stringify({
          assistant_id: AZURE_AGENT_ID,
          max_prompt_tokens: 512,
          temperature: 0.7,
          top_p: 0.8
        })
      });

      if (!runResponse.ok) {
        const runError = await runResponse.text();
        throw new Error(`Run creation failed: ${runResponse.status} - ${runError}`);
      }

      const runData = await runResponse.json();
      const runId = runData.id;

      // Poll for completion
      let run = runData;
      while (run.status === 'in_progress' || run.status === 'queued') {
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        const statusResponse = await fetch(`${AZURE_API_ENDPOINT}/threads/${threadId}/runs/${runId}?api-version=2025-05-01`, {
          method: 'GET',
          headers: {
            'Authorization': `Bearer ${AZURE_API_TOKEN}`
          }
        });

        if (!statusResponse.ok) {
          throw new Error(`Status check failed: ${statusResponse.status}`);
        }

        run = await statusResponse.json();
      }

      if (run.status !== 'completed') {
        throw new Error(`Run failed with status: ${run.status}`);
      }

      // Get the response messages
      const messagesResponse = await fetch(`${AZURE_API_ENDPOINT}/threads/${threadId}/messages?api-version=2025-05-01`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${AZURE_API_TOKEN}`
        }
      });

      if (!messagesResponse.ok) {
        const errorText = await messagesResponse.text();
        console.error(`❌ [MODERATION AI] Azure API error for human response: ${messagesResponse.status} - ${errorText}`);
        throw new Error(`Azure API request failed: ${messagesResponse.status}`);
      }

      const messagesData = await messagesResponse.json();
      
      if (!messagesData.data || messagesData.data.length === 0) {
        throw new Error('No messages found in response');
      }

      // Get the assistant's response (latest message)
      const assistantMessage = messagesData.data[0];
      if (!assistantMessage.content || assistantMessage.content.length === 0) {
        throw new Error('No content in assistant message');
      }

      const humanResponse = assistantMessage.content[0].text?.value || '';
      console.log(`🤖 [MODERATION AI] Generated human response: ${humanResponse.substring(0, 100)}...`);
      
      return humanResponse;

    } catch (error) {
      console.error('🤖 [MODERATION AI] Error generating human response with Azure AI:', error);
      
      // Return fallback response
      return `Hi ${username},

Thank you for reaching out about your post "${postTitle}". After reviewing the content, we found that it didn't align with our subreddit guidelines, specifically: ${removalReason}.

We appreciate your participation in r/${subredditName} and encourage you to review our rules before submitting future posts. If you have any questions about our guidelines, please don't hesitate to ask.

Best regards,
The Mod Team`;
    }
  }
}