// Set database configuration before any imports
if (!process.env.DATABASE_URL) {
  process.env.DATABASE_URL = 'postgresql://neondb_owner:Maniac20!@userza1004.openweb.direct/tube';
  process.env.PGDATABASE = 'tube';
  process.env.PGHOST = 'userza1004.openweb.direct';
  process.env.PGPORT = '5432';
  process.env.PGUSER = 'neondb_owner';
  process.env.PGPASSWORD = 'Maniac20!';
}

import express, { type Request, Response, NextFunction } from "express";
import cookieParser from "cookie-parser";
import { registerRoutes } from "./routes";
import { setupVite, serveStatic, log } from "./vite";
import { AuthService } from "./auth";
import { storage } from "./storage";
import { PointScheduler } from "./point-scheduler";

const app = express();
app.use(express.json());
app.use(express.urlencoded({ extended: false }));
app.use(cookieParser());

app.use((req, res, next) => {
  const start = Date.now();
  const path = req.path;
  let capturedJsonResponse: Record<string, any> | undefined = undefined;

  const originalResJson = res.json;
  res.json = function (bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };

  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path.startsWith("/api")) {
      let logLine = `${req.method} ${path} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }

      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "…";
      }

      log(logLine);
    }
  });

  next();
});

(async () => {
  // Create admin user on startup
  await AuthService.createAdminUser();

  // Initialize default system settings for points
  try {
    const defaultPointsSetting = await storage.getSystemSetting('default_daily_points');
    if (!defaultPointsSetting) {
      await storage.setSystemSetting({
        key: 'default_daily_points',
        value: '600',
        description: 'Default daily point allocation for all users'
      });
      console.log('Initialized default daily points setting: 600');
    }
  } catch (error) {
    console.error('Error initializing point system settings:', error);
  }

  // Start the point scheduler for daily resets
  PointScheduler.start();
  
  const server = await registerRoutes(app);

  app.use((err: any, _req: Request, res: Response, _next: NextFunction) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";

    res.status(status).json({ message });
    throw err;
  });

  // importantly only setup vite in development and after
  // setting up all the other routes so the catch-all route
  // doesn't interfere with the other routes
  if (app.get("env") === "development") {
    await setupVite(app, server);
  } else {
    serveStatic(app);
  }

  // ALWAYS serve the app on port 5000
  // this serves both the API and the client.
  // It is the only port that is not firewalled.
  const port = 5000;
  server.listen({
    port,
    host: "0.0.0.0",
    reusePort: true,
  }, () => {
    log(`serving on port ${port}`);
  });
})();
