import { storage } from "./storage";

export class OrderStatusMonitor {
  private static instance: OrderStatusMonitor;
  private monitoringInterval: NodeJS.Timeout | null = null;
  private readonly POLL_INTERVAL = 30000; // 30 seconds
  private readonly API_KEY = process.env.SMMCOST_API_KEY || "91e5a8595f63688b6c3bd59fa80218de";

  private constructor() {
    // Use the existing storage instance
  }

  static getInstance(): OrderStatusMonitor {
    if (!OrderStatusMonitor.instance) {
      OrderStatusMonitor.instance = new OrderStatusMonitor();
    }
    return OrderStatusMonitor.instance;
  }

  async startMonitoring(): Promise<void> {
    console.log("🔄 Starting SMMCost Order Status Monitor with 30-second polling...");
    
    // Initial check
    await this.checkAllActiveOrders();
    
    // Set up recurring monitoring
    this.monitoringInterval = setInterval(async () => {
      try {
        await this.checkAllActiveOrders();
      } catch (error) {
        console.error("❌ Error in order status monitoring:", error);
      }
    }, this.POLL_INTERVAL);
  }

  stopMonitoring(): void {
    if (this.monitoringInterval) {
      clearInterval(this.monitoringInterval);
      this.monitoringInterval = null;
      console.log("⏹️ Stopped SMMCost Order Status Monitor");
    }
  }

  private async checkAllActiveOrders(): Promise<void> {
    try {
      // Get all active video orders
      const activeVideoOrders = await storage.getActiveVideoOrders();
      const activeSubscriberOrders = await storage.getActiveSubscriberOrders();
      
      const allActiveOrders = [...activeVideoOrders, ...activeSubscriberOrders];
      
      if (allActiveOrders.length === 0) {
        return; // No active orders to check
      }

      console.log(`🔍 Checking status of ${allActiveOrders.length} active orders...`);

      // Check status for each active order
      for (const order of allActiveOrders) {
        try {
          await this.updateOrderStatus(order);
          // Small delay between API calls to avoid rate limiting
          await new Promise(resolve => setTimeout(resolve, 100));
        } catch (error) {
          console.error(`❌ Failed to update status for order ${order.orderId}:`, error);
        }
      }
    } catch (error) {
      console.error("❌ Error checking active orders:", error);
    }
  }

  private async updateOrderStatus(order: any): Promise<void> {
    try {
      const response = await fetch("https://smmcost.com/api/v2", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          key: this.API_KEY,
          action: "status",
          order: order.orderId
        })
      });

      if (!response.ok) {
        throw new Error(`SMMCost API error: ${response.status}`);
      }

      const statusResult = await response.json();

      if (statusResult.error) {
        console.error(`SMMCost API error for order ${order.orderId}:`, statusResult.error);
        return;
      }

      // Update order status in database if it changed
      const newStatus = statusResult.status;
      const newRemains = statusResult.remains || order.remains;
      const newStartCount = statusResult.start_count || order.startCount;

      if (newStatus !== order.status || newRemains !== order.remains) {
        console.log(`📊 Updating order ${order.orderId}: ${order.status} → ${newStatus} (${newRemains} remaining)`);
        
        if (order.videoId) {
          // Update video order
          await storage.updateVideoOrder(order.id, {
            status: newStatus,
            remains: newRemains.toString(),
            startCount: newStartCount?.toString() || order.startCount
          });
        } else {
          // Update subscriber order
          await storage.updateSubscriberOrder(order.id, {
            status: newStatus,
            remains: newRemains.toString(),
            startCount: newStartCount?.toString() || order.startCount
          });
        }
      }
    } catch (error) {
      console.error(`Failed to update order ${order.orderId}:`, error);
    }
  }

  async getOrderStatusById(orderId: string): Promise<any> {
    try {
      const response = await fetch("https://smmcost.com/api/v2", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          key: this.API_KEY,
          action: "status",
          order: orderId
        })
      });

      if (!response.ok) {
        throw new Error(`SMMCost API error: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.error) {
        throw new Error(`SMMCost API error: ${result.error}`);
      }

      return result;
    } catch (error) {
      console.error(`Failed to get status for order ${orderId}:`, error);
      throw error;
    }
  }
}

// Initialize and start monitoring when server starts
export function initializeOrderStatusMonitor(): void {
  const monitor = OrderStatusMonitor.getInstance();
  monitor.startMonitoring().catch(error => {
    console.error("Failed to start order status monitor:", error);
  });
}