import { TubeBoostAPI } from "./smmcost";
import { storage } from "./storage";

interface OrderStatusUpdate {
  orderId: string;
  status: string;
  charge?: string;
  startCount?: string;
  remains?: string;
  currency?: string;
}

export class OrderTracker {
  private static updateInterval: NodeJS.Timeout | null = null;
  private static readonly UPDATE_INTERVAL = 30000; // 30 seconds

  static start() {
    if (this.updateInterval) {
      return;
    }

    console.log("Order tracker started - updating all order statuses every 30 seconds");
    
    // Run immediately
    this.updateAllOrderStatuses();
    
    // Then run every 30 seconds
    this.updateInterval = setInterval(() => {
      this.updateAllOrderStatuses();
    }, this.UPDATE_INTERVAL);
  }

  static stop() {
    if (this.updateInterval) {
      clearInterval(this.updateInterval);
      this.updateInterval = null;
      console.log("Order tracker stopped");
    }
  }

  static async updateAllOrderStatuses() {
    try {
      console.log("🔄 Starting order status update cycle...");
      
      // Get all active video orders
      const activeVideoOrders = await this.getActiveVideoOrders();
      console.log(`Found ${activeVideoOrders.length} active video orders`);
      
      // Get all active subscriber orders
      const activeSubscriberOrders = await this.getActiveSubscriberOrders();
      console.log(`Found ${activeSubscriberOrders.length} active subscriber orders`);
      
      let videoUpdated = 0;
      let subscriberUpdated = 0;

      // Update video orders
      for (const order of activeVideoOrders) {
        try {
          const statusResponse = await TubeBoostAPI.getSingleOrderStatus(order.orderId);
          
          if ('error' in statusResponse) {
            console.log(`❌ Error fetching status for video order ${order.orderId}:`, statusResponse.error);
            continue;
          }

          // Update if status changed
          if (this.hasStatusChanged(order, statusResponse)) {
            await storage.updateVideoOrder(order.id, {
              status: statusResponse.status,
              charge: statusResponse.charge,
              startCount: statusResponse.start_count,
              remains: statusResponse.remains,
              currency: statusResponse.currency
            });
            
            console.log(`✅ Updated video order ${order.orderId}: ${statusResponse.status}`);
            videoUpdated++;
          }
        } catch (error) {
          console.error(`Error updating video order ${order.orderId}:`, error);
        }
      }

      // Update subscriber orders
      for (const order of activeSubscriberOrders) {
        try {
          const statusResponse = await TubeBoostAPI.getSingleOrderStatus(order.orderId);
          
          if ('error' in statusResponse) {
            console.log(`❌ Error fetching status for subscriber order ${order.orderId}:`, statusResponse.error);
            continue;
          }

          // Update if status changed
          if (this.hasStatusChanged(order, statusResponse)) {
            await storage.updateSubscriberOrder(order.id, {
              status: statusResponse.status,
              charge: statusResponse.charge,
              startCount: statusResponse.start_count,
              remains: statusResponse.remains,
              currency: statusResponse.currency
            });
            
            console.log(`✅ Updated subscriber order ${order.orderId}: ${statusResponse.status}`);
            subscriberUpdated++;
          }
        } catch (error) {
          console.error(`Error updating subscriber order ${order.orderId}:`, error);
        }
      }

      console.log(`📊 Order update complete: ${videoUpdated} video orders, ${subscriberUpdated} subscriber orders updated`);
      
    } catch (error) {
      console.error("Error in order status update cycle:", error);
    }
  }

  private static async getActiveVideoOrders() {
    try {
      return await storage.getActiveVideoOrders();
    } catch (error) {
      console.error("Error fetching active video orders:", error);
      return [];
    }
  }

  private static async getActiveSubscriberOrders() {
    try {
      return await storage.getActiveSubscriberOrders();
    } catch (error) {
      console.error("Error fetching active subscriber orders:", error);
      return [];
    }
  }

  private static hasStatusChanged(order: any, statusResponse: any): boolean {
    return (
      order.status !== statusResponse.status ||
      order.charge !== statusResponse.charge ||
      order.startCount !== statusResponse.start_count ||
      order.remains !== statusResponse.remains ||
      order.currency !== statusResponse.currency
    );
  }

  // Manual force update for specific order
  static async forceUpdateOrder(orderId: string, type: 'video' | 'subscriber'): Promise<boolean> {
    try {
      const statusResponse = await TubeBoostAPI.getSingleOrderStatus(orderId);
      
      if ('error' in statusResponse) {
        console.log(`❌ Error fetching status for ${type} order ${orderId}:`, statusResponse.error);
        return false;
      }

      if (type === 'video') {
        const order = await storage.getVideoOrderByOrderId(orderId);
        if (order) {
          await storage.updateVideoOrder(order.id, {
            status: statusResponse.status,
            charge: statusResponse.charge,
            startCount: statusResponse.start_count,
            remains: statusResponse.remains,
            currency: statusResponse.currency
          });
          console.log(`✅ Force updated video order ${orderId}: ${statusResponse.status}`);
          return true;
        }
      } else {
        const order = await storage.getSubscriberOrderByOrderId(orderId);
        if (order) {
          await storage.updateSubscriberOrder(order.id, {
            status: statusResponse.status,
            charge: statusResponse.charge,
            startCount: statusResponse.start_count,
            remains: statusResponse.remains,
            currency: statusResponse.currency
          });
          console.log(`✅ Force updated subscriber order ${orderId}: ${statusResponse.status}`);
          return true;
        }
      }
      
      return false;
    } catch (error) {
      console.error(`Error force updating ${type} order ${orderId}:`, error);
      return false;
    }
  }
}