import type { Express, RequestHandler } from "express";
import { storage } from "./storage";
import { AuthService } from "./auth";

export interface RedditTokens {
  access_token: string;
  refresh_token: string;
  expires_in: number;
  scope: string;
  token_type: string;
}

export interface RedditUser {
  id: string;
  name: string;
  icon_img: string;
  created_utc: number;
  link_karma: number;
  comment_karma: number;
  verified: boolean;
  has_verified_email: boolean;
}

export class RedditAuthService {
  private static readonly USER_AGENT = 'TubeBoost:v1.3.1 (by /u/tubeboost)';
  private static readonly SCOPE = 'identity read submit edit modposts mysubreddits modmail';

  private static get CLIENT_ID(): string | undefined {
    return process.env.REDDIT_CLIENT_ID;
  }

  private static get CLIENT_SECRET(): string | undefined {
    return process.env.REDDIT_CLIENT_SECRET;
  }

  static getRedirectUri(hostname: string): string {
    console.log('🔶 [REDDIT AUTH] Determining redirect URI for hostname:', hostname);
    
    // Handle both development and production environments
    if (hostname.includes('localhost') || hostname.includes('127.0.0.1')) {
      const redirectUri = `http://${hostname}/api/reddit/callback`;
      console.log('🔶 [REDDIT AUTH] Using localhost redirect URI:', redirectUri);
      return redirectUri;
    } else if (hostname.includes('replit.app') || hostname.includes('replit.dev') || hostname.includes('pike.replit.dev')) {
      const redirectUri = `https://${hostname}/api/reddit/callback`;
      console.log('🔶 [REDDIT AUTH] Using Replit redirect URI:', redirectUri);
      return redirectUri;
    } else if (hostname === 'tubeboost.cc' || hostname.includes('tubeboost.cc')) {
      const redirectUri = 'https://tubeboost.cc/api/reddit/callback';
      console.log('🔶 [REDDIT AUTH] Using production redirect URI:', redirectUri);
      return redirectUri;
    } else {
      // Fallback for any other domains
      const redirectUri = `https://${hostname}/api/reddit/callback`;
      console.log('🔶 [REDDIT AUTH] Using fallback HTTPS redirect URI:', redirectUri);
      return redirectUri;
    }
  }

  static getAuthUrl(hostname: string, state?: string): string {
    const clientId = this.CLIENT_ID;
    console.log('Reddit CLIENT_ID check:', clientId ? 'Found' : 'Missing', 'Length:', clientId?.length || 0);
    
    if (!clientId) {
      throw new Error('REDDIT_CLIENT_ID not configured');
    }

    const redirectUri = this.getRedirectUri(hostname);
    const stateParam = state || Math.random().toString(36).substring(2, 15);
    
    const params = new URLSearchParams({
      client_id: this.CLIENT_ID,
      response_type: 'code',
      state: stateParam,
      redirect_uri: redirectUri,
      duration: 'permanent',
      scope: this.SCOPE
    });

    return `https://www.reddit.com/api/v1/authorize?${params.toString()}`;
  }

  static async exchangeCodeForTokens(
    code: string,
    hostname: string
  ): Promise<RedditTokens> {
    if (!this.CLIENT_ID || !this.CLIENT_SECRET) {
      throw new Error('Reddit client credentials not configured');
    }

    const redirectUri = this.getRedirectUri(hostname);
    
    const response = await fetch('https://www.reddit.com/api/v1/access_token', {
      method: 'POST',
      headers: {
        'Authorization': `Basic ${Buffer.from(`${this.CLIENT_ID}:${this.CLIENT_SECRET}`).toString('base64')}`,
        'Content-Type': 'application/x-www-form-urlencoded',
        'User-Agent': this.USER_AGENT
      },
      body: new URLSearchParams({
        grant_type: 'authorization_code',
        code,
        redirect_uri: redirectUri
      })
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Reddit token exchange failed: ${response.status} ${errorText}`);
    }

    return await response.json();
  }

  static async refreshTokens(refreshToken: string): Promise<RedditTokens> {
    if (!this.CLIENT_ID || !this.CLIENT_SECRET) {
      throw new Error('Reddit client credentials not configured');
    }

    const response = await fetch('https://www.reddit.com/api/v1/access_token', {
      method: 'POST',
      headers: {
        'Authorization': `Basic ${Buffer.from(`${this.CLIENT_ID}:${this.CLIENT_SECRET}`).toString('base64')}`,
        'Content-Type': 'application/x-www-form-urlencoded',
        'User-Agent': this.USER_AGENT
      },
      body: new URLSearchParams({
        grant_type: 'refresh_token',
        refresh_token: refreshToken
      })
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Reddit token refresh failed: ${response.status} ${errorText}`);
    }

    return await response.json();
  }

  static async getRedditUser(accessToken: string): Promise<RedditUser> {
    const response = await fetch('https://oauth.reddit.com/api/v1/me', {
      headers: {
        'Authorization': `Bearer ${accessToken}`,
        'User-Agent': this.USER_AGENT
      }
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Reddit user fetch failed: ${response.status} ${errorText}`);
    }

    return await response.json();
  }

  static async getClientCredentialsToken(): Promise<string> {
    const clientId = this.CLIENT_ID;
    const clientSecret = this.CLIENT_SECRET;
    
    if (!clientId || !clientSecret) {
      throw new Error('Reddit client credentials not configured');
    }

    const auth = Buffer.from(`${clientId}:${clientSecret}`).toString('base64');
    
    const response = await fetch('https://www.reddit.com/api/v1/access_token', {
      method: 'POST',
      headers: {
        'Authorization': `Basic ${auth}`,
        'Content-Type': 'application/x-www-form-urlencoded',
        'User-Agent': this.USER_AGENT
      },
      body: 'grant_type=client_credentials'
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to get client credentials token: ${response.status} ${errorText}`);
    }
    
    const data = await response.json();
    return data.access_token;
  }

  static async makeRedditApiCall(
    endpoint: string,
    method: 'GET' | 'POST' = 'GET',
    accessToken: string,
    body?: any
  ): Promise<any> {
    const headers: Record<string, string> = {
      'Authorization': `Bearer ${accessToken}`,
      'User-Agent': this.USER_AGENT
    };

    let requestBody: string | FormData | undefined;

    if (method === 'POST' && body) {
      // For Reddit's /api/submit endpoint, use form data
      if (endpoint.includes('/api/submit')) {
        const formData = new URLSearchParams();
        for (const [key, value] of Object.entries(body)) {
          formData.append(key, String(value));
        }
        requestBody = formData.toString();
        headers['Content-Type'] = 'application/x-www-form-urlencoded';
      } else {
        // For other endpoints, use JSON
        headers['Content-Type'] = 'application/json';
        requestBody = JSON.stringify(body);
      }
    }

    // Ensure endpoint starts with / for proper URL construction
    const cleanEndpoint = endpoint.startsWith('/') ? endpoint : `/${endpoint}`;
    const url = `https://oauth.reddit.com${cleanEndpoint}`;

    console.log(`🔗 [REDDIT API] Making ${method} request to: ${url}`);

    const response = await fetch(url, {
      method,
      headers,
      body: requestBody
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error(`🚨 [REDDIT API] Error ${response.status}: ${errorText}`);
      throw new Error(`Reddit API call failed: ${response.status} ${errorText}`);
    }

    return await response.json();
  }

  static async makePublicApiCall(endpoint: string, method: 'GET' | 'POST' = 'GET'): Promise<any> {
    const clientToken = await this.getClientCredentialsToken();
    return this.makeRedditApiCall(endpoint, method, clientToken);
  }
}

export async function setupRedditAuth(app: Express): Promise<void> {
  // Reddit OAuth initiation
  app.get('/api/reddit/auth', (req, res) => {
    try {
      console.log('🔶 [REDDIT AUTH] Initiating OAuth flow');
      console.log('🔶 [REDDIT AUTH] Client ID available:', !!process.env.REDDIT_CLIENT_ID);
      console.log('🔶 [REDDIT AUTH] Hostname:', req.hostname);
      console.log('🔶 [REDDIT AUTH] Full headers:', JSON.stringify(req.headers, null, 2));
      console.log('🔶 [REDDIT AUTH] User agent:', req.get('User-Agent'));
      console.log('🔶 [REDDIT AUTH] Request ID:', req.get('x-request-id') || 'none');

      const authUrl = RedditAuthService.getAuthUrl(req.hostname);
      console.log('🔶 [REDDIT AUTH] Generated auth URL:', authUrl);
      console.log('🔶 [REDDIT AUTH] ⚠️  IMPORTANT: Reddit will redirect to the callback URL above after authorization');
      
      res.redirect(authUrl);
    } catch (error) {
      console.error('🔴 [REDDIT AUTH] Initiation failed:', error);
      console.error('🔴 [REDDIT AUTH] Error stack:', error instanceof Error ? error.stack : 'No stack trace');
      console.error('🔴 [REDDIT AUTH] Environment check:', {
        clientId: !!process.env.REDDIT_CLIENT_ID,
        clientSecret: !!process.env.REDDIT_CLIENT_SECRET,
        nodeEnv: process.env.NODE_ENV
      });
      
      res.status(500).json({ 
        message: 'Failed to initiate Reddit authentication',
        error: error instanceof Error ? error.message : 'Unknown error',
        timestamp: new Date().toISOString()
      });
    }
  });

  // Reddit OAuth callback
  app.get('/api/reddit/callback', async (req, res) => {
    try {
      console.log('🔶 [REDDIT CALLBACK] Processing OAuth callback');
      console.log('🔶 [REDDIT CALLBACK] Query params:', req.query);
      console.log('🔶 [REDDIT CALLBACK] Request timestamp:', new Date().toISOString());
      console.log('🔶 [REDDIT CALLBACK] User agent:', req.get('User-Agent'));
      console.log('🔶 [REDDIT CALLBACK] Request hostname:', req.hostname);
      console.log('🔶 [REDDIT CALLBACK] Request URL:', req.url);
      console.log('🔶 [REDDIT CALLBACK] Full request headers:', JSON.stringify(req.headers, null, 2));
      
      const { code, state, error } = req.query;

      if (error) {
        console.error('🔴 [REDDIT CALLBACK] OAuth error from Reddit:', error);
        console.error('🔴 [REDDIT CALLBACK] Error description:', req.query.error_description);
        return res.redirect(`/?reddit_error=${encodeURIComponent(error as string)}`);
      }

      if (!code) {
        console.error('🔴 [REDDIT CALLBACK] No authorization code received');
        console.error('🔴 [REDDIT CALLBACK] Full query object:', JSON.stringify(req.query, null, 2));
        return res.redirect('/?reddit_error=no_code_received');
      }

      console.log('🔶 [REDDIT CALLBACK] Authorization code received, length:', (code as string).length);

      // Exchange code for tokens
      console.log('🔶 [REDDIT CALLBACK] Exchanging code for tokens...');
      const tokens = await RedditAuthService.exchangeCodeForTokens(
        code as string,
        req.hostname
      );
      console.log('🔶 [REDDIT CALLBACK] Tokens received successfully:', {
        hasAccessToken: !!tokens.access_token,
        hasRefreshToken: !!tokens.refresh_token,
        expiresIn: tokens.expires_in,
        scope: tokens.scope
      });

      // Get Reddit user info
      console.log('🔶 [REDDIT CALLBACK] Fetching Reddit user info...');
      const redditUser = await RedditAuthService.getRedditUser(tokens.access_token);
      console.log('🔶 [REDDIT CALLBACK] Reddit user fetched:', {
        id: redditUser.id,
        name: redditUser.name,
        verified: redditUser.verified,
        linkKarma: redditUser.link_karma,
        commentKarma: redditUser.comment_karma
      });

      // Store Reddit connection in database
      console.log('🔶 [REDDIT CALLBACK] Storing Reddit connection in database');
      
      // Get current user ID from session cookie
      const sessionId = req.cookies?.sessionId;
      console.log('🔶 [REDDIT CALLBACK] Session cookie check:', {
        hasSessionId: !!sessionId,
        sessionIdLength: sessionId ? sessionId.length : 0
      });
      
      if (!sessionId) {
        console.error('🔴 [REDDIT CALLBACK] No session cookie found');
        return res.redirect('/?reddit_error=authentication_required');
      }
      
      console.log('🔶 [REDDIT CALLBACK] Validating session...');
      const user = await AuthService.validateSession(sessionId);
      console.log('🔶 [REDDIT CALLBACK] Session validation result:', {
        hasUser: !!user,
        userId: user?.id,
        username: user?.username
      });
      
      if (!user) {
        console.error('🔴 [REDDIT CALLBACK] Invalid session');
        return res.redirect('/?reddit_error=invalid_session');
      }
      
      const userId = user.id;
      console.log('🔶 [REDDIT CALLBACK] Using userId for connection:', userId);
      
      const connectionData = {
        userId: userId,
        redditUserId: redditUser.id,
        redditUsername: redditUser.name,
        accessToken: tokens.access_token,
        refreshToken: tokens.refresh_token,
        expiresAt: new Date(Date.now() + tokens.expires_in * 1000),
        scope: tokens.scope
      };
      
      try {
        console.log('🔶 [REDDIT CALLBACK] About to save connection data:', connectionData);
        const savedConnection = await storage.createRedditConnection(connectionData);
        console.log('🔶 [REDDIT CALLBACK] Reddit connection saved successfully:', {
          id: savedConnection.id,
          userId: savedConnection.userId,
          redditUsername: savedConnection.redditUsername,
          expiresAt: savedConnection.expiresAt
        });
        
        // Verify the connection was saved by reading back from database
        console.log('🔶 [REDDIT CALLBACK] Verifying connection was saved...');
        const verifyConnection = await storage.getRedditConnection(userId);
        console.log('🔶 [REDDIT CALLBACK] Verification result:', verifyConnection ? {
          id: verifyConnection.id,
          userId: verifyConnection.userId,
          redditUsername: verifyConnection.redditUsername,
          expiresAt: verifyConnection.expiresAt,
          found: true,
          username: verifyConnection.redditUsername,
          expires: verifyConnection.expiresAt
        } : 'NO CONNECTION FOUND IN DATABASE');
      } catch (error) {
        console.error('🔴 [REDDIT CALLBACK] Failed to save Reddit connection:', error);
        console.error('🔴 [REDDIT CALLBACK] Error type:', typeof error);
        console.error('🔴 [REDDIT CALLBACK] Error name:', error instanceof Error ? error.name : 'Unknown');
        console.error('🔴 [REDDIT CALLBACK] Error message:', error instanceof Error ? error.message : String(error));
        console.error('🔴 [REDDIT CALLBACK] Error stack:', error instanceof Error ? error.stack : 'No stack');
        console.error('🔴 [REDDIT CALLBACK] Connection data that failed to save:', JSON.stringify(connectionData, null, 2));
        console.error('🔴 [REDDIT CALLBACK] Database connection status: checking...');
        
        // Try to check database connectivity
        try {
          const testUser = await AuthService.validateSession(req.cookies?.sessionId);
          console.log('🔴 [REDDIT CALLBACK] Database connectivity test - user lookup:', testUser ? 'SUCCESS' : 'FAILED');
        } catch (dbError) {
          console.error('🔴 [REDDIT CALLBACK] Database connectivity test failed:', dbError);
        }
        
        throw error;
      }

      // Redirect to dashboard with success message
      console.log('🔶 [REDDIT CALLBACK] Redirecting to dashboard with success flag');
      res.redirect('/?reddit_connected=true');
    } catch (error) {
      console.error('🔴 [REDDIT CALLBACK] Callback processing failed:', error);
      console.error('🔴 [REDDIT CALLBACK] Error stack:', error instanceof Error ? error.stack : 'No stack trace');
      console.error('🔴 [REDDIT CALLBACK] Request details:', {
        hostname: req.hostname,
        userAgent: req.get('User-Agent'),
        query: req.query,
        timestamp: new Date().toISOString(),
        headers: req.headers
      });
      res.redirect(`/?reddit_error=${encodeURIComponent(error instanceof Error ? error.message : 'Unknown error')}`);
    }
  });

  // Get Reddit connection status
  app.get('/api/reddit/status', async (req, res) => {
    console.log('🔶 [REDDIT STATUS] Checking Reddit connection status');
    console.log('🔶 [REDDIT STATUS] Environment check:', {
      clientId: !!process.env.REDDIT_CLIENT_ID,
      clientSecret: !!process.env.REDDIT_CLIENT_SECRET,
      userAgent: 'TubeBoost:v1.3.1 (by /u/tubeboost)'
    });
    
    // Get current user from session - use session cookie
    const sessionId = req.cookies?.sessionId;
    if (!sessionId) {
      return res.json({
        connected: false,
        message: 'Authentication required',
        redditUser: null,
        debugInfo: {
          clientConfigured: !!process.env.REDDIT_CLIENT_ID,
          authEndpoint: '/api/reddit/auth',
          callbackEndpoint: '/api/reddit/callback',
          hasConnection: false,
          timestamp: new Date().toISOString()
        }
      });
    }
    
    // Validate session and get user
    const user = await AuthService.validateSession(sessionId);
    if (!user) {
      return res.json({
        connected: false,
        message: 'Invalid session',
        redditUser: null,
        debugInfo: {
          clientConfigured: !!process.env.REDDIT_CLIENT_ID,
          authEndpoint: '/api/reddit/auth',
          callbackEndpoint: '/api/reddit/callback',
          hasConnection: false,
          timestamp: new Date().toISOString()
        }
      });
    }
    
    const userId = user.id;
    
    // Check if user has Reddit connection
    console.log('🔶 [REDDIT STATUS] Checking connection for userId:', userId);
    
    let connected = false;
    let message = 'Reddit integration framework ready';
    let redditUser = null;
    
    try {
      const connection = await storage.getRedditConnection(userId);
      console.log('🔶 [REDDIT STATUS] Database lookup result:', {
        found: !!connection,
        userId: connection?.userId,
        redditUsername: connection?.redditUsername,
        expiresAt: connection?.expiresAt,
        isExpired: connection ? connection.expiresAt <= new Date() : false
      });
      
      if (connection) {
        // Check if token is still valid
        const now = new Date();
        if (connection.expiresAt > now) {
          connected = true;
          message = 'Connected to Reddit';
          redditUser = {
            id: connection.redditUserId,
            username: connection.redditUsername
          };
        } else {
          message = 'Reddit token expired, please reconnect';
        }
      }
    } catch (error) {
      console.error('🔴 [REDDIT STATUS] Database lookup failed:', error);
      connected = false;
      message = 'Database error checking Reddit connection';
      redditUser = null;
    }
    
    const statusResponse = { 
      connected,
      message,
      redditUser,
      debugInfo: {
        clientConfigured: !!process.env.REDDIT_CLIENT_ID,
        authEndpoint: '/api/reddit/auth',
        callbackEndpoint: '/api/reddit/callback',
        hasConnection: connected,
        timestamp: new Date().toISOString()
      }
    };
    
    console.log('🔶 [REDDIT STATUS] Status response:', statusResponse);
    res.json(statusResponse);
  });

  // Disconnect Reddit account
  app.post('/api/reddit/disconnect', async (req, res) => {
    try {
      // Get current user from session cookie
      const sessionId = req.cookies?.sessionId;
      if (!sessionId) {
        return res.status(401).json({ message: 'Authentication required' });
      }
      
      const user = await AuthService.validateSession(sessionId);
      if (!user) {
        return res.status(401).json({ message: 'Invalid session' });
      }
      
      const userId = user.id;
      
      console.log('🔶 [REDDIT DISCONNECT] Disconnecting Reddit for userId:', userId);
      
      // Remove Reddit connection from database
      await storage.deleteRedditConnection(userId);
      
      console.log('🔶 [REDDIT DISCONNECT] Reddit connection deleted successfully');
      
      res.json({ 
        success: true,
        message: 'Reddit account disconnected successfully'
      });
    } catch (error) {
      console.error('🔴 [REDDIT DISCONNECT] Error:', error);
      res.status(500).json({ 
        message: 'Failed to disconnect Reddit account',
        error: error instanceof Error ? error.message : 'Unknown error'
      });
    }
  });
}

export const isRedditAuthenticated: RequestHandler = async (req, res, next) => {
  try {
    // Check if user has valid Reddit tokens
    // const connection = await storage.getRedditConnection(userId);
    // if (!connection || isTokenExpired(connection.expiresAt)) {
    //   return res.status(401).json({ message: 'Reddit authentication required' });
    // }
    
    // For now, allow all requests to pass through
    next();
  } catch (error) {
    console.error('Reddit auth check error:', error);
    res.status(500).json({ 
      message: 'Failed to verify Reddit authentication',
      error: error instanceof Error ? error.message : 'Unknown error'
    });
  }
};