import { users, channels, userChannels, videos, agSessions, sessions, videoOrders, channelViewsLimits, refillOrders, subscriberOrders, subscriberRefillOrders, systemSettings, userPointAllocations, pointTransactions, deviceFingerprints, ipAddressLogs, securityViolations, queueItems, redditConnections, redditPosts, subredditAnalytics, managedSubreddits, subredditAnalyticsExtended, type User, type InsertUser, type Channel, type InsertChannel, type UserChannel, type InsertUserChannel, type Video, type InsertVideo, type AgSession, type InsertAgSession, type Session, type InsertSession, type VideoOrder, type InsertVideoOrder, type ChannelViewsLimit, type InsertChannelViewsLimit, type RefillOrder, type InsertRefillOrder, type SubscriberOrder, type InsertSubscriberOrder, type SubscriberRefillOrder, type InsertSubscriberRefillOrder, type SystemSetting, type InsertSystemSetting, type UserPointAllocation, type InsertUserPointAllocation, type PointTransaction, type InsertPointTransaction, type DeviceFingerprint, type InsertDeviceFingerprint, type IpAddressLog, type InsertIpAddressLog, type SecurityViolation, type InsertSecurityViolation, type QueueItem, type InsertQueueItem, type RedditConnection, type InsertRedditConnection, type RedditPost, type InsertRedditPost, type SubredditAnalytics, type InsertSubredditAnalytics, type ManagedSubreddit, type InsertManagedSubreddit, type SubredditAnalyticsExtended, type InsertSubredditAnalyticsExtended } from "@shared/schema";
import { moderationQueueItems, aiAnalysisResults, moderationActions, subredditRulesCache, type ModerationQueueItem, type InsertModerationQueueItem, type AiAnalysisResult, type InsertAiAnalysisResult, type ModerationAction, type InsertModerationAction, type SubredditRulesCache, type InsertSubredditRulesCache } from "@shared/reddit-schema";
import { db } from "./db";
import { eq, desc, and, or, gte, lt, lte } from "drizzle-orm";

export interface IStorage {
  // User methods
  getUser(id: number): Promise<User | undefined>;
  getUserByUsername(username: string): Promise<User | undefined>;
  getUserByEmail(email: string): Promise<User | undefined>;
  getAllUsers(): Promise<User[]>;
  createUser(user: InsertUser): Promise<User>;
  updateUser(id: number, updates: Partial<User>): Promise<User | undefined>;
  deleteUser(id: number): Promise<void>;

  // Session methods
  createSession(session: InsertSession): Promise<Session>;
  getSession(sessionId: string): Promise<Session | undefined>;
  deleteSession(sessionId: string): Promise<void>;

  // Channel methods
  getChannelByUserId(userId: number): Promise<Channel | undefined>;
  getChannelsByUserId(userId: number): Promise<Channel[]>;
  getAllChannels(): Promise<Channel[]>;
  getChannelByChannelId(channelId: string): Promise<Channel | undefined>;
  getChannelById(id: number): Promise<Channel | undefined>;
  getActiveChannel(userId: number): Promise<Channel | undefined>;
  createChannel(channel: InsertChannel): Promise<Channel>;
  updateChannel(id: number, updates: Partial<Channel>): Promise<Channel | undefined>;
  setActiveChannel(userId: number, channelId: number): Promise<Channel | undefined>;
  
  // User Channel methods (for channel sharing)
  addUserToChannel(userId: number, channelId: number): Promise<UserChannel>;
  getUserChannels(userId: number): Promise<Channel[]>;
  hideChannelForUser(userId: number, channelId: number): Promise<void>;
  showChannelForUser(userId: number, channelId: number): Promise<void>;
  removeUserFromChannel(userId: number, channelId: number): Promise<void>;
  removeChannelFromUser(userId: number, channelId: number): Promise<void>;

  // Video methods
  getVideosByChannelId(channelId: number): Promise<Video[]>;
  getVideoById(id: number): Promise<Video | undefined>;
  createVideo(video: InsertVideo): Promise<Video>;
  updateVideo(id: number, updates: Partial<Video>): Promise<Video | undefined>;
  deleteVideosByChannelId(channelId: number): Promise<void>;

  // AG Session methods
  createAgSession(session: InsertAgSession): Promise<AgSession>;
  getAgSessionsByChannelId(channelId: number): Promise<AgSession[]>;
  updateAgSession(id: number, updates: Partial<AgSession>): Promise<AgSession | undefined>;
  getActiveAgSession(channelId: number): Promise<AgSession | undefined>;

  // Video Order methods
  createVideoOrder(order: InsertVideoOrder): Promise<VideoOrder>;
  getVideoOrdersByVideoId(videoId: number): Promise<VideoOrder[]>;
  getVideoOrdersByChannelId(channelId: number): Promise<VideoOrder[]>;
  getVideoOrdersByYouTubeVideoId(youtubeVideoId: string, channelId: number): Promise<VideoOrder[]>;
  updateVideoOrder(id: number, updates: Partial<VideoOrder>): Promise<VideoOrder | undefined>;
  getVideoOrderById(id: number): Promise<VideoOrder | undefined>;
  getVideoOrderByOrderId(orderId: string): Promise<VideoOrder | undefined>;
  getActiveVideoOrders(): Promise<VideoOrder[]>;
  getAllVideoOrders(): Promise<VideoOrder[]>;

  // Channel Views Limit methods
  getChannelViewsLimit(channelId: number): Promise<ChannelViewsLimit | undefined>;
  createChannelViewsLimit(limit: InsertChannelViewsLimit): Promise<ChannelViewsLimit>;
  updateChannelViewsLimit(channelId: number, updates: Partial<ChannelViewsLimit>): Promise<ChannelViewsLimit | undefined>;
  resetChannelViewsLimit(channelId: number): Promise<ChannelViewsLimit | undefined>;

  // Refill Order methods
  createRefillOrder(refill: InsertRefillOrder): Promise<RefillOrder>;
  getRefillOrdersByVideoOrderId(videoOrderId: number): Promise<RefillOrder[]>;
  updateRefillOrder(id: number, updates: Partial<RefillOrder>): Promise<RefillOrder | undefined>;
  getRefillOrderById(id: number): Promise<RefillOrder | undefined>;

  // Subscriber Order methods
  createSubscriberOrder(order: InsertSubscriberOrder): Promise<SubscriberOrder>;
  getSubscriberOrdersByChannelId(channelId: number): Promise<SubscriberOrder[]>;
  updateSubscriberOrder(id: number, updates: Partial<SubscriberOrder>): Promise<SubscriberOrder | undefined>;
  getSubscriberOrderById(id: number): Promise<SubscriberOrder | undefined>;
  getSubscriberOrderByOrderId(orderId: string): Promise<SubscriberOrder | undefined>;
  getSubscriberOrdersCountToday(channelId: number): Promise<number>;
  getActiveSubscriberOrders(): Promise<SubscriberOrder[]>;
  getAllSubscriberOrders(): Promise<SubscriberOrder[]>;

  // Subscriber Refill Order methods
  createSubscriberRefillOrder(refill: InsertSubscriberRefillOrder): Promise<SubscriberRefillOrder>;
  getSubscriberRefillOrdersBySubscriberOrderId(subscriberOrderId: number): Promise<SubscriberRefillOrder[]>;
  updateSubscriberRefillOrder(id: number, updates: Partial<SubscriberRefillOrder>): Promise<SubscriberRefillOrder | undefined>;
  getSubscriberRefillOrderById(id: number): Promise<SubscriberRefillOrder | undefined>;

  // System Settings methods
  getSystemSetting(key: string): Promise<SystemSetting | undefined>;
  setSystemSetting(setting: InsertSystemSetting): Promise<SystemSetting>;
  updateSystemSetting(key: string, value: string): Promise<SystemSetting | undefined>;

  // User Point Allocation methods
  getUserPointAllocation(userId: number, date: string): Promise<UserPointAllocation | undefined>;
  createUserPointAllocation(allocation: InsertUserPointAllocation): Promise<UserPointAllocation>;
  updateUserPointAllocation(id: number, updates: Partial<UserPointAllocation>): Promise<UserPointAllocation | undefined>;
  getUserPointAllocations(userId: number): Promise<UserPointAllocation[]>;
  getAllUserPointAllocationsForDate(date: string): Promise<UserPointAllocation[]>;
  
  // Point Transaction methods
  createPointTransaction(transaction: InsertPointTransaction): Promise<PointTransaction>;
  getPointTransactionsByUserId(userId: number): Promise<PointTransaction[]>;
  getPointTransactionsByAllocationId(allocationId: number): Promise<PointTransaction[]>;
  
  // Point system utilities
  ensureDailyPointAllocation(userId: number, date: string): Promise<UserPointAllocation>;
  deductPointsInternal(userId: number, points: number, type: string, description?: string, relatedId?: number): Promise<boolean>;
  getUserRemainingPoints(userId: number): Promise<number>;
  getUserPoints(userId: number): Promise<{ remainingPoints: number; totalPoints: number; usedPoints: number }>;
  deductPoints(userId: number, amount: number, description: string): Promise<void>;
  addPoints(userId: number, amount: number, description: string, adminId?: number): Promise<void>;
  getPointTransactionHistory(userId: number): Promise<PointTransaction[]>;

  // Security methods
  createDeviceFingerprint(fingerprint: InsertDeviceFingerprint): Promise<DeviceFingerprint>;
  getDeviceFingerprintByHash(fingerprint: string): Promise<DeviceFingerprint | undefined>;
  getDeviceFingerprintsByUserId(userId: string): Promise<DeviceFingerprint[]>;
  updateDeviceFingerprint(id: number, updates: Partial<DeviceFingerprint>): Promise<DeviceFingerprint | undefined>;
  
  createIpAddressLog(log: InsertIpAddressLog): Promise<IpAddressLog>;
  getIpAddressLog(hashedIp: string): Promise<IpAddressLog | undefined>;
  updateIpAddressLog(hashedIp: string, updates: Partial<IpAddressLog>): Promise<IpAddressLog | undefined>;
  
  createSecurityViolation(violation: InsertSecurityViolation): Promise<SecurityViolation>;
  getSecurityViolationsByUserId(userId: number): Promise<SecurityViolation[]>;
  getRecentSecurityViolations(limit?: number): Promise<SecurityViolation[]>;

  // Queue methods
  createQueueItem(queueItem: InsertQueueItem): Promise<QueueItem>;
  getQueueItemsByUserId(userId: number): Promise<QueueItem[]>;
  getQueueItemById(id: number): Promise<QueueItem | undefined>;
  updateQueueItem(id: number, updates: Partial<QueueItem>): Promise<QueueItem | undefined>;
  deleteQueueItem(id: number): Promise<void>;
  getNextQueuedItem(userId: number): Promise<QueueItem | undefined>;
  getRunningQueueItems(userId: number): Promise<QueueItem[]>;
  getStuckRunningItems(timeoutMs: number): Promise<QueueItem[]>;
  getStuckQueuedItems(timeoutMs: number): Promise<QueueItem[]>;
  getStuckItemsByUser(userId: number): Promise<QueueItem[]>;
  getAllStuckItems(): Promise<QueueItem[]>;

  // Reddit connection methods
  createRedditConnection(connection: InsertRedditConnection): Promise<RedditConnection>;
  getRedditConnection(userId: number): Promise<RedditConnection | undefined>;
  updateRedditConnection(userId: number, updates: Partial<RedditConnection>): Promise<RedditConnection | undefined>;
  deleteRedditConnection(userId: number): Promise<void>;

  // Reddit moderation methods
  upsertSubredditRules(rules: InsertSubredditRulesCache): Promise<SubredditRulesCache>;
  getSubredditRules(subredditName: string): Promise<SubredditRulesCache | undefined>;
  createModqueueItem(item: InsertModerationQueueItem): Promise<ModerationQueueItem>;
  getModqueueItem(redditId: string): Promise<ModerationQueueItem | undefined>;
  createAiAnalysisResult(analysis: InsertAiAnalysisResult): Promise<AiAnalysisResult>;
  getAiAnalysisResult(id: number): Promise<AiAnalysisResult | undefined>;
  createModerationAction(action: InsertModerationAction): Promise<ModerationAction>;
  getModerationHistory(subredditName: string, limit: number): Promise<any[]>;
}

export class DatabaseStorage implements IStorage {
  async getUser(id: number): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user || undefined;
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.username, username));
    return user || undefined;
  }

  async getUserByEmail(email: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.email, email));
    return user || undefined;
  }

  async getAllUsers(): Promise<User[]> {
    const allUsers = await db.select().from(users);
    return allUsers;
  }

  async createUser(insertUser: InsertUser): Promise<User> {
    const [user] = await db
      .insert(users)
      .values(insertUser)
      .returning();
    return user;
  }

  async deleteUser(id: number): Promise<void> {
    await db.delete(users).where(eq(users.id, id));
  }

  async updateUser(id: number, updates: Partial<User>): Promise<User | undefined> {
    const [user] = await db
      .update(users)
      .set(updates)
      .where(eq(users.id, id))
      .returning();
    return user || undefined;
  }

  // Session methods
  async createSession(insertSession: InsertSession): Promise<Session> {
    const [session] = await db
      .insert(sessions)
      .values(insertSession)
      .returning();
    return session;
  }

  async getSession(sessionId: string): Promise<Session | undefined> {
    const [session] = await db.select().from(sessions).where(eq(sessions.id, sessionId));
    return session || undefined;
  }

  async deleteSession(sessionId: string): Promise<void> {
    await db.delete(sessions).where(eq(sessions.id, sessionId));
  }

  async getChannelByUserId(userId: number): Promise<Channel | undefined> {
    const [channel] = await db.select().from(channels).where(eq(channels.userId, userId));
    return channel || undefined;
  }

  async getChannelsByUserId(userId: number): Promise<Channel[]> {
    // Get owned channels
    const ownedChannels = await db.select().from(channels).where(eq(channels.userId, userId)).orderBy(desc(channels.createdAt));
    
    // Get shared channels through userChannels junction table
    const sharedChannels = await this.getUserChannels(userId);
    
    // Combine and deduplicate based on channel ID
    const allChannels = [...ownedChannels, ...sharedChannels];
    const uniqueChannels = allChannels.filter((channel, index, self) => 
      index === self.findIndex(c => c.id === channel.id)
    );
    
    return uniqueChannels.sort((a, b) => new Date(b.createdAt!).getTime() - new Date(a.createdAt!).getTime());
  }

  async getAllChannels(): Promise<Channel[]> {
    const allChannels = await db.select().from(channels).orderBy(desc(channels.createdAt));
    return allChannels;
  }

  async getChannelById(id: number): Promise<Channel | undefined> {
    const [channel] = await db.select().from(channels).where(eq(channels.id, id));
    return channel || undefined;
  }

  async getActiveChannel(userId: number): Promise<Channel | undefined> {
    // Get active channel through userChannels table
    const [result] = await db
      .select({
        id: channels.id,
        userId: channels.userId,
        channelId: channels.channelId,
        channelName: channels.channelName,
        channelHandle: channels.channelHandle,
        customUrl: channels.customUrl,
        agScore: channels.agScore,
        lastReplenishTime: channels.lastReplenishTime,
        totalViews: channels.totalViews,
        subscribers: channels.subscribers,
        watchTimeHours: channels.watchTimeHours,
        isActive: channels.isActive,
        createdAt: channels.createdAt,
        updatedAt: channels.updatedAt,
      })
      .from(userChannels)
      .innerJoin(channels, eq(userChannels.channelId, channels.id))
      .where(and(
        eq(userChannels.userId, userId),
        eq(userChannels.isCurrentActive, true),
        eq(userChannels.isVisible, true)
      ));
    
    return result || undefined;
  }

  async setActiveChannel(userId: number, channelId: number): Promise<Channel | undefined> {
    // Get user to check if they're admin
    const user = await this.getUser(userId);
    if (!user) {
      return undefined;
    }

    // Check if channel exists
    const [channel] = await db
      .select()
      .from(channels)
      .where(eq(channels.id, channelId));
    
    if (!channel) {
      return undefined; // Channel doesn't exist
    }

    // For admin users, allow access to any channel
    if (user.role === 'admin') {
      // Check if admin already has access to this channel
      const [existingUserChannel] = await db
        .select()
        .from(userChannels)
        .where(and(
          eq(userChannels.userId, userId),
          eq(userChannels.channelId, channelId)
        ));

      // If admin doesn't have access, add them to the channel
      if (!existingUserChannel) {
        await db
          .insert(userChannels)
          .values({
            userId: userId,
            channelId: channelId,
            isVisible: true,
            isCurrentActive: false
          });
      }
    } else {
      // For non-admin users, check if they have access through userChannels
      const [userChannel] = await db
        .select()
        .from(userChannels)
        .where(and(
          eq(userChannels.userId, userId),
          eq(userChannels.channelId, channelId),
          eq(userChannels.isVisible, true)
        ));
      
      if (!userChannel) {
        return undefined; // User doesn't have access to this channel
      }
    }

    // Deactivate all channels for this user by updating userChannels
    await db
      .update(userChannels)
      .set({ isCurrentActive: false })
      .where(eq(userChannels.userId, userId));
    
    // Activate the selected channel for this user
    await db
      .update(userChannels)
      .set({ isCurrentActive: true })
      .where(and(
        eq(userChannels.userId, userId),
        eq(userChannels.channelId, channelId)
      ));
    
    return channel;
  }

  async getChannelByChannelId(channelId: string): Promise<Channel | undefined> {
    const [channel] = await db.select().from(channels).where(eq(channels.channelId, channelId));
    return channel || undefined;
  }

  async createChannel(insertChannel: InsertChannel): Promise<Channel> {
    const [channel] = await db
      .insert(channels)
      .values(insertChannel)
      .returning();
    
    // Automatically add the creator to the userChannels table
    await this.addUserToChannel(insertChannel.userId, channel.id);
    
    return channel;
  }

  async updateChannel(id: number, updates: Partial<Channel>): Promise<Channel | undefined> {
    const [channel] = await db
      .update(channels)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(channels.id, id))
      .returning();
    return channel || undefined;
  }

  async getVideosByChannelId(channelId: number): Promise<Video[]> {
    return await db.select().from(videos).where(eq(videos.channelId, channelId)).orderBy(desc(videos.publishedAt));
  }

  async getVideoById(id: number): Promise<Video | undefined> {
    const [video] = await db.select().from(videos).where(eq(videos.id, id));
    return video || undefined;
  }

  async createVideo(insertVideo: InsertVideo): Promise<Video> {
    const [video] = await db
      .insert(videos)
      .values(insertVideo)
      .returning();
    return video;
  }

  async updateVideo(id: number, updates: Partial<Video>): Promise<Video | undefined> {
    const [video] = await db
      .update(videos)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(videos.id, id))
      .returning();
    return video || undefined;
  }

  async createAgSession(insertSession: InsertAgSession): Promise<AgSession> {
    const [session] = await db
      .insert(agSessions)
      .values(insertSession)
      .returning();
    return session;
  }

  async getAgSessionsByChannelId(channelId: number): Promise<AgSession[]> {
    return await db.select().from(agSessions).where(eq(agSessions.channelId, channelId)).orderBy(desc(agSessions.startedAt));
  }

  async updateAgSession(id: number, updates: Partial<AgSession>): Promise<AgSession | undefined> {
    const [session] = await db
      .update(agSessions)
      .set(updates)
      .where(eq(agSessions.id, id))
      .returning();
    return session || undefined;
  }

  async getActiveAgSession(channelId: number): Promise<AgSession | undefined> {
    const [session] = await db
      .select()
      .from(agSessions)
      .where(and(eq(agSessions.channelId, channelId), eq(agSessions.status, 'running')));
    return session || undefined;
  }

  async deleteVideosByChannelId(channelId: number): Promise<void> {
    // First get all video IDs for this channel
    const channelVideos = await db
      .select({ id: videos.id })
      .from(videos)
      .where(eq(videos.channelId, channelId));
    
    // Delete all video orders for these videos first
    for (const video of channelVideos) {
      await db.delete(videoOrders).where(eq(videoOrders.videoId, video.id));
    }
    
    // Now we can safely delete the videos
    await db.delete(videos).where(eq(videos.channelId, channelId));
  }

  // Video Order methods
  async createVideoOrder(insertOrder: InsertVideoOrder): Promise<VideoOrder> {
    const [order] = await db
      .insert(videoOrders)
      .values(insertOrder)
      .returning();
    return order;
  }

  async getVideoOrdersByVideoId(videoId: number): Promise<VideoOrder[]> {
    return await db
      .select()
      .from(videoOrders)
      .where(eq(videoOrders.videoId, videoId))
      .orderBy(desc(videoOrders.createdAt));
  }

  async getVideoOrdersByChannelId(channelId: number): Promise<VideoOrder[]> {
    return await db
      .select()
      .from(videoOrders)
      .where(eq(videoOrders.channelId, channelId))
      .orderBy(desc(videoOrders.createdAt));
  }

  async getVideoOrdersByYouTubeVideoId(youtubeVideoId: string, channelId: number): Promise<VideoOrder[]> {
    // Get all videos in the channel with this YouTube video ID (across all sync versions)
    const channelVideos = await db
      .select({ id: videos.id })
      .from(videos)
      .where(and(eq(videos.videoId, youtubeVideoId), eq(videos.channelId, channelId)));
    
    if (channelVideos.length === 0) {
      return [];
    }
    
    // Get orders for all video versions
    const allOrders: VideoOrder[] = [];
    for (const video of channelVideos) {
      const ordersForVideo = await db
        .select()
        .from(videoOrders)
        .where(eq(videoOrders.videoId, video.id))
        .orderBy(desc(videoOrders.createdAt));
      allOrders.push(...ordersForVideo);
    }
    
    // Sort all orders by creation date
    return allOrders.sort((a: VideoOrder, b: VideoOrder) => {
      const dateA = a.createdAt ? new Date(a.createdAt).getTime() : 0;
      const dateB = b.createdAt ? new Date(b.createdAt).getTime() : 0;
      return dateB - dateA;
    });
  }

  async updateVideoOrder(id: number, updates: Partial<VideoOrder>): Promise<VideoOrder | undefined> {
    const [order] = await db
      .update(videoOrders)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(videoOrders.id, id))
      .returning();
    return order || undefined;
  }

  async getVideoOrderById(id: number): Promise<VideoOrder | undefined> {
    const [order] = await db.select().from(videoOrders).where(eq(videoOrders.id, id));
    return order || undefined;
  }

  async getVideoOrderByOrderId(orderId: string): Promise<VideoOrder | undefined> {
    const [order] = await db.select().from(videoOrders).where(eq(videoOrders.orderId, orderId));
    return order || undefined;
  }

  async getActiveVideoOrders(): Promise<VideoOrder[]> {
    return await db
      .select()
      .from(videoOrders)
      .where(and(
        eq(videoOrders.orderId, videoOrders.orderId), // Only orders with valid order IDs
        or(
          eq(videoOrders.status, 'Pending'),
          eq(videoOrders.status, 'Processing'),
          eq(videoOrders.status, 'In progress'),
          eq(videoOrders.status, 'Partial')
        )
      ))
      .orderBy(desc(videoOrders.createdAt));
  }

  async getAllVideoOrders(): Promise<VideoOrder[]> {
    return await db
      .select()
      .from(videoOrders)
      .orderBy(desc(videoOrders.createdAt));
  }

  // Channel Views Limit methods
  async getChannelViewsLimit(channelId: number): Promise<ChannelViewsLimit | undefined> {
    const [limit] = await db
      .select()
      .from(channelViewsLimits)
      .where(eq(channelViewsLimits.channelId, channelId));
    return limit || undefined;
  }

  async createChannelViewsLimit(insertLimit: InsertChannelViewsLimit): Promise<ChannelViewsLimit> {
    const [limit] = await db
      .insert(channelViewsLimits)
      .values(insertLimit)
      .returning();
    return limit;
  }

  async updateChannelViewsLimit(channelId: number, updates: Partial<ChannelViewsLimit>): Promise<ChannelViewsLimit | undefined> {
    const [limit] = await db
      .update(channelViewsLimits)
      .set(updates)
      .where(eq(channelViewsLimits.channelId, channelId))
      .returning();
    return limit || undefined;
  }

  async resetChannelViewsLimit(channelId: number): Promise<ChannelViewsLimit | undefined> {
    const [limit] = await db
      .update(channelViewsLimits)
      .set({ 
        usedCount: 0, 
        lastResetTime: new Date() 
      })
      .where(eq(channelViewsLimits.channelId, channelId))
      .returning();
    return limit || undefined;
  }

  async createRefillOrder(insertRefill: InsertRefillOrder): Promise<RefillOrder> {
    const [refill] = await db
      .insert(refillOrders)
      .values(insertRefill)
      .returning();
    return refill;
  }

  async getRefillOrdersByVideoOrderId(videoOrderId: number): Promise<RefillOrder[]> {
    return await db
      .select()
      .from(refillOrders)
      .where(eq(refillOrders.originalOrderId, videoOrderId))
      .orderBy(desc(refillOrders.createdAt));
  }

  async updateRefillOrder(id: number, updates: Partial<RefillOrder>): Promise<RefillOrder | undefined> {
    const [refill] = await db
      .update(refillOrders)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(refillOrders.id, id))
      .returning();
    return refill || undefined;
  }

  async getRefillOrderById(id: number): Promise<RefillOrder | undefined> {
    const [refill] = await db
      .select()
      .from(refillOrders)
      .where(eq(refillOrders.id, id));
    return refill || undefined;
  }

  // User Channel methods (for channel sharing)
  async addUserToChannel(userId: number, channelId: number): Promise<UserChannel> {
    const [userChannel] = await db
      .insert(userChannels)
      .values({ userId, channelId })
      .onConflictDoUpdate({
        target: [userChannels.userId, userChannels.channelId],
        set: { isVisible: true }
      })
      .returning();
    return userChannel;
  }

  async getUserChannels(userId: number): Promise<Channel[]> {
    const result = await db
      .select({
        id: channels.id,
        userId: channels.userId,
        channelId: channels.channelId,
        channelName: channels.channelName,
        channelHandle: channels.channelHandle,
        customUrl: channels.customUrl,
        agScore: channels.agScore,
        lastReplenishTime: channels.lastReplenishTime,
        totalViews: channels.totalViews,
        subscribers: channels.subscribers,
        watchTimeHours: channels.watchTimeHours,
        isActive: channels.isActive,
        createdAt: channels.createdAt,
        updatedAt: channels.updatedAt,
      })
      .from(userChannels)
      .innerJoin(channels, eq(userChannels.channelId, channels.id))
      .where(and(
        eq(userChannels.userId, userId),
        eq(userChannels.isVisible, true)
      ));
    return result;
  }

  async hideChannelForUser(userId: number, channelId: number): Promise<void> {
    await db
      .update(userChannels)
      .set({ isVisible: false, isCurrentActive: false })
      .where(and(
        eq(userChannels.userId, userId),
        eq(userChannels.channelId, channelId)
      ));
  }

  async showChannelForUser(userId: number, channelId: number): Promise<void> {
    await db
      .update(userChannels)
      .set({ isVisible: true })
      .where(and(
        eq(userChannels.userId, userId),
        eq(userChannels.channelId, channelId)
      ));
  }

  async removeChannelFromUser(userId: number, channelId: number): Promise<void> {
    // Remove the user-channel relationship completely (for client removal)
    await db
      .delete(userChannels)
      .where(and(
        eq(userChannels.userId, userId),
        eq(userChannels.channelId, channelId)
      ));
  }

  async removeUserFromChannel(userId: number, channelId: number): Promise<void> {
    await db
      .delete(userChannels)
      .where(and(
        eq(userChannels.userId, userId),
        eq(userChannels.channelId, channelId)
      ));
  }

  // Subscriber Order methods
  async createSubscriberOrder(insertOrder: InsertSubscriberOrder): Promise<SubscriberOrder> {
    const [order] = await db
      .insert(subscriberOrders)
      .values(insertOrder)
      .returning();
    return order;
  }

  async getSubscriberOrdersByChannelId(channelId: number): Promise<SubscriberOrder[]> {
    return await db
      .select()
      .from(subscriberOrders)
      .where(eq(subscriberOrders.channelId, channelId))
      .orderBy(desc(subscriberOrders.createdAt));
  }

  async updateSubscriberOrder(id: number, updates: Partial<SubscriberOrder>): Promise<SubscriberOrder | undefined> {
    const [order] = await db
      .update(subscriberOrders)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(subscriberOrders.id, id))
      .returning();
    return order;
  }

  async getSubscriberOrderById(id: number): Promise<SubscriberOrder | undefined> {
    const [order] = await db
      .select()
      .from(subscriberOrders)
      .where(eq(subscriberOrders.id, id));
    return order;
  }

  async getSubscriberOrdersCountToday(channelId: number): Promise<number> {
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    
    const tomorrow = new Date(today);
    tomorrow.setDate(tomorrow.getDate() + 1);

    const orders = await db
      .select()
      .from(subscriberOrders)
      .where(and(
        eq(subscriberOrders.channelId, channelId),
        and(
          gte(subscriberOrders.createdAt, today),
          lt(subscriberOrders.createdAt, tomorrow)
        )
      ));
    
    return orders.length;
  }

  async getSubscriberOrderByOrderId(orderId: string): Promise<SubscriberOrder | undefined> {
    const [order] = await db.select().from(subscriberOrders).where(eq(subscriberOrders.orderId, orderId));
    return order || undefined;
  }

  async getActiveSubscriberOrders(): Promise<SubscriberOrder[]> {
    return await db
      .select()
      .from(subscriberOrders)
      .where(and(
        eq(subscriberOrders.orderId, subscriberOrders.orderId), // Only orders with valid order IDs
        or(
          eq(subscriberOrders.status, 'Pending'),
          eq(subscriberOrders.status, 'Processing'),
          eq(subscriberOrders.status, 'In progress'),
          eq(subscriberOrders.status, 'Partial')
        )
      ))
      .orderBy(desc(subscriberOrders.createdAt));
  }

  async getAllSubscriberOrders(): Promise<SubscriberOrder[]> {
    return await db
      .select()
      .from(subscriberOrders)
      .orderBy(desc(subscriberOrders.createdAt));
  }

  // Subscriber Refill Order methods
  async createSubscriberRefillOrder(insertRefill: InsertSubscriberRefillOrder): Promise<SubscriberRefillOrder> {
    const [refill] = await db
      .insert(subscriberRefillOrders)
      .values(insertRefill)
      .returning();
    return refill;
  }

  async getSubscriberRefillOrdersBySubscriberOrderId(subscriberOrderId: number): Promise<SubscriberRefillOrder[]> {
    return await db
      .select()
      .from(subscriberRefillOrders)
      .where(eq(subscriberRefillOrders.originalOrderId, subscriberOrderId))
      .orderBy(desc(subscriberRefillOrders.createdAt));
  }

  async updateSubscriberRefillOrder(id: number, updates: Partial<SubscriberRefillOrder>): Promise<SubscriberRefillOrder | undefined> {
    const [refill] = await db
      .update(subscriberRefillOrders)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(subscriberRefillOrders.id, id))
      .returning();
    return refill || undefined;
  }

  async getSubscriberRefillOrderById(id: number): Promise<SubscriberRefillOrder | undefined> {
    const [refill] = await db
      .select()
      .from(subscriberRefillOrders)
      .where(eq(subscriberRefillOrders.id, id));
    return refill || undefined;
  }

  // System Settings methods
  async getSystemSetting(key: string): Promise<SystemSetting | undefined> {
    const [setting] = await db
      .select()
      .from(systemSettings)
      .where(eq(systemSettings.key, key));
    return setting || undefined;
  }

  async setSystemSetting(insertSetting: InsertSystemSetting): Promise<SystemSetting> {
    const [setting] = await db
      .insert(systemSettings)
      .values(insertSetting)
      .onConflictDoUpdate({
        target: systemSettings.key,
        set: {
          value: insertSetting.value,
          description: insertSetting.description,
          updatedAt: new Date(),
        },
      })
      .returning();
    return setting;
  }

  async updateSystemSetting(key: string, value: string): Promise<SystemSetting | undefined> {
    const [setting] = await db
      .update(systemSettings)
      .set({ value, updatedAt: new Date() })
      .where(eq(systemSettings.key, key))
      .returning();
    return setting || undefined;
  }

  // User Point Allocation methods
  async getUserPointAllocation(userId: number, date: string): Promise<UserPointAllocation | undefined> {
    const [allocation] = await db
      .select()
      .from(userPointAllocations)
      .where(and(eq(userPointAllocations.userId, userId), eq(userPointAllocations.date, date)));
    return allocation || undefined;
  }

  async createUserPointAllocation(insertAllocation: InsertUserPointAllocation): Promise<UserPointAllocation> {
    const [allocation] = await db
      .insert(userPointAllocations)
      .values(insertAllocation)
      .returning();
    return allocation;
  }

  async updateUserPointAllocation(id: number, updates: Partial<UserPointAllocation>): Promise<UserPointAllocation | undefined> {
    const [allocation] = await db
      .update(userPointAllocations)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(userPointAllocations.id, id))
      .returning();
    return allocation || undefined;
  }

  async getUserPointAllocations(userId: number): Promise<UserPointAllocation[]> {
    return await db
      .select()
      .from(userPointAllocations)
      .where(eq(userPointAllocations.userId, userId))
      .orderBy(desc(userPointAllocations.date));
  }

  async getAllUserPointAllocationsForDate(date: string): Promise<UserPointAllocation[]> {
    return await db
      .select()
      .from(userPointAllocations)
      .where(eq(userPointAllocations.date, date));
  }

  // Point Transaction methods
  async createPointTransaction(insertTransaction: InsertPointTransaction): Promise<PointTransaction> {
    const [transaction] = await db
      .insert(pointTransactions)
      .values(insertTransaction)
      .returning();
    return transaction;
  }

  async getPointTransactionsByUserId(userId: number): Promise<PointTransaction[]> {
    return await db
      .select()
      .from(pointTransactions)
      .where(eq(pointTransactions.userId, userId))
      .orderBy(desc(pointTransactions.createdAt));
  }

  async getPointTransactionsByAllocationId(allocationId: number): Promise<PointTransaction[]> {
    return await db
      .select()
      .from(pointTransactions)
      .where(eq(pointTransactions.allocationId, allocationId))
      .orderBy(desc(pointTransactions.createdAt));
  }



  // Point system utilities
  async ensureDailyPointAllocation(userId: number, date: string): Promise<UserPointAllocation> {
    // Check if allocation already exists
    let allocation = await this.getUserPointAllocation(userId, date);
    
    if (!allocation) {
      // Get default points from system settings
      const defaultPointsSetting = await this.getSystemSetting('default_daily_points');
      const defaultPoints = defaultPointsSetting ? parseInt(defaultPointsSetting.value) : 600;
      
      // Create new allocation
      allocation = await this.createUserPointAllocation({
        userId,
        date,
        totalPoints: defaultPoints,
        usedPoints: 0,
        remainingPoints: defaultPoints,
        isCustomAllocation: false,
      });
    }
    
    return allocation;
  }

  async deductPointsInternal(userId: number, points: number, type: string, description?: string, relatedId?: number): Promise<boolean> {
    const today = new Date().toISOString().split('T')[0];
    const allocation = await this.ensureDailyPointAllocation(userId, today);
    
    if (allocation.remainingPoints < points) {
      return false; // Insufficient points
    }
    
    // Update allocation
    const newUsedPoints = allocation.usedPoints + points;
    const newRemainingPoints = allocation.remainingPoints - points;
    
    await this.updateUserPointAllocation(allocation.id, {
      usedPoints: newUsedPoints,
      remainingPoints: newRemainingPoints,
    });
    
    // Create transaction record
    await this.createPointTransaction({
      userId,
      allocationId: allocation.id,
      points: -points, // negative for deduction
      type,
      description,
      relatedId,
    });
    
    return true;
  }

  async getUserRemainingPoints(userId: number): Promise<number> {
    const today = new Date().toISOString().split('T')[0];
    const allocation = await this.ensureDailyPointAllocation(userId, today);
    return allocation.remainingPoints;
  }

  async getUserPoints(userId: number): Promise<{ remainingPoints: number; totalPoints: number; usedPoints: number }> {
    const today = new Date().toISOString().split('T')[0];
    const allocation = await this.ensureDailyPointAllocation(userId, today);
    
    return {
      remainingPoints: allocation.remainingPoints,
      totalPoints: allocation.totalPoints,
      usedPoints: allocation.usedPoints
    };
  }

  async deductPoints(userId: number, amount: number, description: string): Promise<void> {
    const success = await this.deductPointsInternal(userId, amount, 'order', description);
    if (!success) {
      throw new Error('Insufficient points for this operation');
    }
  }

  async addPoints(userId: number, amount: number, description: string, adminId?: number): Promise<void> {
    const today = new Date().toISOString().split('T')[0];
    const allocation = await this.ensureDailyPointAllocation(userId, today);
    
    // Update allocation
    const newTotalPoints = allocation.totalPoints + amount;
    const newRemainingPoints = allocation.remainingPoints + amount;
    
    await this.updateUserPointAllocation(allocation.id, {
      totalPoints: newTotalPoints,
      remainingPoints: newRemainingPoints,
    });
    
    // Create transaction record
    await this.createPointTransaction({
      userId,
      allocationId: allocation.id,
      points: amount, // positive for addition
      type: 'admin_addition',
      description,
      relatedId: adminId,
    });
  }

  async getPointTransactionHistory(userId: number): Promise<PointTransaction[]> {
    return await this.getPointTransactionsByUserId(userId);
  }

  // Security method implementations
  async createDeviceFingerprint(fingerprint: InsertDeviceFingerprint): Promise<DeviceFingerprint> {
    const [result] = await db
      .insert(deviceFingerprints)
      .values(fingerprint)
      .returning();
    return result;
  }

  async getDeviceFingerprintByHash(fingerprint: string): Promise<DeviceFingerprint | undefined> {
    const [result] = await db
      .select()
      .from(deviceFingerprints)
      .where(eq(deviceFingerprints.fingerprintHash, fingerprint))
      .limit(1);
    return result || undefined;
  }

  async getDeviceFingerprintsByUserId(userId: string): Promise<DeviceFingerprint[]> {
    return await db
      .select()
      .from(deviceFingerprints)
      .where(eq(deviceFingerprints.userId, userId))
      .orderBy(desc(deviceFingerprints.createdAt));
  }

  async updateDeviceFingerprint(id: number, updates: Partial<DeviceFingerprint>): Promise<DeviceFingerprint | undefined> {
    const [result] = await db
      .update(deviceFingerprints)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(deviceFingerprints.id, id))
      .returning();
    return result || undefined;
  }

  async createIpAddressLog(log: InsertIpAddressLog): Promise<IpAddressLog> {
    const [result] = await db
      .insert(ipAddressLogs)
      .values(log)
      .returning();
    return result;
  }

  async getIpAddressLog(hashedIp: string): Promise<IpAddressLog | undefined> {
    const [result] = await db
      .select()
      .from(ipAddressLogs)
      .where(eq(ipAddressLogs.ipHash, hashedIp))
      .limit(1);
    return result || undefined;
  }

  async updateIpAddressLog(hashedIp: string, updates: Partial<IpAddressLog>): Promise<IpAddressLog | undefined> {
    const [result] = await db
      .update(ipAddressLogs)
      .set({ ...updates, lastSeen: new Date() })
      .where(eq(ipAddressLogs.ipHash, hashedIp))
      .returning();
    return result || undefined;
  }

  async createSecurityViolation(violation: InsertSecurityViolation): Promise<SecurityViolation> {
    const [result] = await db
      .insert(securityViolations)
      .values(violation)
      .returning();
    return result;
  }

  async getSecurityViolationsByUserId(userId: number): Promise<SecurityViolation[]> {
    return await db
      .select()
      .from(securityViolations)
      .where(eq(securityViolations.userId, userId))
      .orderBy(desc(securityViolations.createdAt));
  }

  async getRecentSecurityViolations(limit = 50): Promise<SecurityViolation[]> {
    return await db
      .select()
      .from(securityViolations)
      .orderBy(desc(securityViolations.createdAt))
      .limit(limit);
  }

  // Queue method implementations
  async createQueueItem(queueItem: InsertQueueItem): Promise<QueueItem> {
    const [result] = await db
      .insert(queueItems)
      .values(queueItem)
      .returning();
    return result;
  }

  async getQueueItemsByUserId(userId: number): Promise<QueueItem[]> {
    return await db
      .select()
      .from(queueItems)
      .where(eq(queueItems.userId, userId))
      .orderBy(desc(queueItems.priority), queueItems.createdAt);
  }

  async getQueueItemById(id: number): Promise<QueueItem | undefined> {
    const [result] = await db
      .select()
      .from(queueItems)
      .where(eq(queueItems.id, id))
      .limit(1);
    return result || undefined;
  }

  async updateQueueItem(id: number, updates: Partial<QueueItem>): Promise<QueueItem | undefined> {
    const [result] = await db
      .update(queueItems)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(queueItems.id, id))
      .returning();
    return result || undefined;
  }

  async deleteQueueItem(id: number): Promise<void> {
    await db.delete(queueItems).where(eq(queueItems.id, id));
  }

  async getNextQueuedItem(userId: number): Promise<QueueItem | undefined> {
    const [result] = await db
      .select()
      .from(queueItems)
      .where(and(eq(queueItems.userId, userId), eq(queueItems.status, "queued")))
      .orderBy(desc(queueItems.priority), queueItems.createdAt)
      .limit(1);
    return result || undefined;
  }

  async getRunningQueueItems(userId: number): Promise<QueueItem[]> {
    return await db
      .select()
      .from(queueItems)
      .where(and(eq(queueItems.userId, userId), eq(queueItems.status, "running")));
  }



  async createModerationAction(action: InsertModerationAction): Promise<ModerationAction> {
    const [result] = await db
      .insert(moderationActions)
      .values(action)
      .returning();
    return result;
  }

  async getModerationHistory(subredditName: string, limit: number = 50): Promise<any[]> {
    return await db
      .select({
        id: moderationActions.id,
        actionType: moderationActions.actionType,
        actionReason: moderationActions.actionReason,
        wasAiSuggested: moderationActions.wasAiSuggested,
        aiConfidenceScore: moderationActions.aiConfidenceScore,
        moderatorNote: moderationActions.moderatorNote,
        actionTimestamp: moderationActions.actionTimestamp,
        itemType: moderationQueueItems.itemType,
        author: moderationQueueItems.author,
        title: moderationQueueItems.title,
        content: moderationQueueItems.content
      })
      .from(moderationActions)
      .leftJoin(moderationQueueItems, eq(moderationActions.modqueueItemId, moderationQueueItems.id))
      .where(eq(moderationQueueItems.subredditName, subredditName))
      .orderBy(desc(moderationActions.actionTimestamp))
      .limit(limit);
  }

  async upsertSubredditRules(rules: InsertSubredditRulesCache): Promise<SubredditRulesCache> {
    const [result] = await db
      .insert(subredditRulesCache)
      .values(rules)
      .onConflictDoUpdate({
        target: subredditRulesCache.subredditName,
        set: {
          rulesJson: rules.rulesJson,
          description: rules.description,
          sidebarHtml: rules.sidebarHtml,
          subredditType: rules.subredditType,
          submissionType: rules.submissionType,
          lastUpdated: new Date()
        }
      })
      .returning();
    return result;
  }

  async getSubredditRules(subredditName: string): Promise<SubredditRulesCache | undefined> {
    const [result] = await db
      .select()
      .from(subredditRulesCache)
      .where(eq(subredditRulesCache.subredditName, subredditName))
      .limit(1);
    return result;
  }

  // Queue management methods
  async getAllStuckItems(): Promise<QueueItem[]> {
    return await db
      .select()
      .from(queueItems)
      .where(or(
        eq(queueItems.status, "queued"),
        eq(queueItems.status, "running")
      ));
  }

  async getAllQueueItems(): Promise<QueueItem[]> {
    return await db.select().from(queueItems);
  }

  async getStuckRunningItems(timeoutMs: number): Promise<QueueItem[]> {
    const cutoffTime = new Date(Date.now() - timeoutMs);
    return await db
      .select()
      .from(queueItems)
      .where(and(
        eq(queueItems.status, "running"),
        lt(queueItems.updatedAt, cutoffTime)
      ));
  }

  // Reddit connection methods
  async createRedditConnection(connection: InsertRedditConnection): Promise<RedditConnection> {
    const [redditConnection] = await db
      .insert(redditConnections)
      .values(connection)
      .onConflictDoUpdate({
        target: redditConnections.userId,
        set: {
          redditUserId: connection.redditUserId,
          redditUsername: connection.redditUsername,
          accessToken: connection.accessToken,
          refreshToken: connection.refreshToken,
          expiresAt: connection.expiresAt,
          scope: connection.scope,
          updatedAt: new Date(),
        },
      })
      .returning();
    return redditConnection;
  }

  async getRedditConnection(userId: number): Promise<RedditConnection | undefined> {
    const [connection] = await db
      .select()
      .from(redditConnections)
      .where(eq(redditConnections.userId, userId));
    return connection || undefined;
  }

  async updateRedditConnection(userId: number, updates: Partial<RedditConnection>): Promise<RedditConnection | undefined> {
    const [updated] = await db
      .update(redditConnections)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(redditConnections.userId, userId))
      .returning();
    return updated || undefined;
  }

  async deleteRedditConnection(userId: number): Promise<void> {
    await db
      .delete(redditConnections)
      .where(eq(redditConnections.userId, userId));
  }

  // Reddit post management methods
  async createRedditPost(post: InsertRedditPost): Promise<RedditPost> {
    const [result] = await db
      .insert(redditPosts)
      .values(post)
      .returning();
    return result;
  }

  async getRedditPostsByUserId(userId: number): Promise<RedditPost[]> {
    return await db
      .select()
      .from(redditPosts)
      .where(eq(redditPosts.userId, userId))
      .orderBy(desc(redditPosts.createdAt));
  }

  async getRedditPostById(id: number): Promise<RedditPost | undefined> {
    const [result] = await db
      .select()
      .from(redditPosts)
      .where(eq(redditPosts.id, id))
      .limit(1);
    return result;
  }

  async updateRedditPost(id: number, updates: Partial<RedditPost>): Promise<RedditPost | undefined> {
    const [result] = await db
      .update(redditPosts)
      .set({ ...updates, updatedAt: new Date() })
      .where(eq(redditPosts.id, id))
      .returning();
    return result;
  }

  async deleteRedditPost(id: number): Promise<void> {
    await db
      .delete(redditPosts)
      .where(eq(redditPosts.id, id));
  }

  // Subreddit analytics methods
  async createSubredditAnalytics(analytics: InsertSubredditAnalytics): Promise<SubredditAnalytics> {
    const [result] = await db
      .insert(subredditAnalytics)
      .values(analytics)
      .onConflictDoUpdate({
        target: [subredditAnalytics.userId, subredditAnalytics.subredditName],
        set: {
          subscribers: analytics.subscribers,
          activeUsers: analytics.activeUsers,
          growthRate: analytics.growthRate,
          engagementRate: analytics.engagementRate,
          bestPostTimes: analytics.bestPostTimes,
          topKeywords: analytics.topKeywords,
          competitorPosts: analytics.competitorPosts,
          lastAnalyzed: new Date(),
          updatedAt: new Date()
        }
      })
      .returning();
    return result;
  }

  async getSubredditAnalyticsByUserId(userId: number): Promise<SubredditAnalytics[]> {
    return await db
      .select()
      .from(subredditAnalytics)
      .where(eq(subredditAnalytics.userId, userId))
      .orderBy(desc(subredditAnalytics.lastAnalyzed));
  }

  async getSubredditAnalytics(userId: number, subredditName: string): Promise<SubredditAnalytics | undefined> {
    const [result] = await db
      .select()
      .from(subredditAnalytics)
      .where(
        and(
          eq(subredditAnalytics.userId, userId),
          eq(subredditAnalytics.subredditName, subredditName)
        )
      )
      .limit(1);
    return result;
  }

  // Moderation system storage methods
  async createModqueueItem(item: InsertModerationQueueItem): Promise<ModerationQueueItem> {
    const [result] = await db
      .insert(moderationQueueItems)
      .values(item)
      .returning();
    return result;
  }

  async getModqueueItem(redditId: string): Promise<ModerationQueueItem | undefined> {
    const [result] = await db
      .select()
      .from(moderationQueueItems)
      .where(eq(moderationQueueItems.redditId, redditId))
      .limit(1);
    return result;
  }

  async createAiAnalysisResult(analysis: InsertAiAnalysisResult): Promise<AiAnalysisResult> {
    const [result] = await db
      .insert(aiAnalysisResults)
      .values(analysis)
      .returning();
    return result;
  }

  async getAiAnalysisResult(id: number): Promise<AiAnalysisResult | undefined> {
    const [result] = await db
      .select()
      .from(aiAnalysisResults)
      .where(eq(aiAnalysisResults.id, id))
      .limit(1);
    return result;
  }

  async createModerationAction(action: InsertModerationAction): Promise<ModerationAction> {
    const [result] = await db
      .insert(moderationActions)
      .values(action)
      .returning();
    return result;
  }

  async getModerationHistory(subredditName: string, limit: number = 50): Promise<any[]> {
    return await db
      .select({
        id: moderationActions.id,
        actionType: moderationActions.actionType,
        actionReason: moderationActions.actionReason,
        wasAiSuggested: moderationActions.wasAiSuggested,
        aiConfidenceScore: moderationActions.aiConfidenceScore,
        moderatorNote: moderationActions.moderatorNote,
        actionTimestamp: moderationActions.actionTimestamp,
        itemType: moderationQueueItems.itemType,
        author: moderationQueueItems.author,
        title: moderationQueueItems.title,
        content: moderationQueueItems.content
      })
      .from(moderationActions)
      .leftJoin(moderationQueueItems, eq(moderationActions.modqueueItemId, moderationQueueItems.id))
      .where(eq(moderationQueueItems.subredditName, subredditName))
      .orderBy(desc(moderationActions.actionTimestamp))
      .limit(limit);
  }

  async upsertSubredditRules(rules: InsertSubredditRulesCache): Promise<SubredditRulesCache> {
    const [result] = await db
      .insert(subredditRulesCache)
      .values(rules)
      .onConflictDoUpdate({
        target: subredditRulesCache.subredditName,
        set: {
          rulesJson: rules.rulesJson,
          description: rules.description,
          sidebarHtml: rules.sidebarHtml,
          subredditType: rules.subredditType,
          submissionType: rules.submissionType,
          lastUpdated: new Date()
        }
      })
      .returning();
    return result;
  }

  async getSubredditRules(subredditName: string): Promise<SubredditRulesCache | undefined> {
    const [result] = await db
      .select()
      .from(subredditRulesCache)
      .where(eq(subredditRulesCache.subredditName, subredditName))
      .limit(1);
    return result;
  }
}

export const storage = new DatabaseStorage();
