import type { Express } from "express";
import Stripe from "stripe";
import { storage } from "./storage";

if (!process.env.STRIPE_SECRET_KEY) {
  throw new Error('Missing required Stripe secret: STRIPE_SECRET_KEY');
}

const stripe = new Stripe(process.env.STRIPE_SECRET_KEY, {
  apiVersion: "2024-11-20.acacia",
});

// Point pricing tiers
const POINT_PACKAGES = {
  starter: { points: 1500, priceUSD: 5.00 },
  value: { points: 2500, priceUSD: 7.50 },
  popular: { points: 5000, priceUSD: 10.00 },
  pro: { points: 10000, priceUSD: 20.00 }
};

export function registerStripeRoutes(app: Express) {
  // Create payment intent for point purchase
  app.post("/api/create-payment-intent", async (req, res) => {
    if (!req.isAuthenticated()) {
      return res.status(401).json({ message: "Authentication required" });
    }

    try {
      const { packageType } = req.body;
      
      if (!packageType || !POINT_PACKAGES[packageType]) {
        return res.status(400).json({ message: "Invalid package type" });
      }

      const pkg = POINT_PACKAGES[packageType];
      const amountCents = Math.round(pkg.priceUSD * 100);

      // Create Stripe payment intent
      const paymentIntent = await stripe.paymentIntents.create({
        amount: amountCents,
        currency: "usd",
        automatic_payment_methods: {
          enabled: true,
        },
        metadata: {
          userId: req.user.id.toString(),
          packageType,
          points: pkg.points.toString(),
        },
      });

      // Create point purchase record in pending state
      await storage.createPointPurchase({
        userId: req.user.id,
        stripePaymentIntentId: paymentIntent.id,
        points: pkg.points,
        amountCents,
        currency: "USD",
        status: "pending",
      });

      console.log(`💰 [STRIPE] Created payment intent ${paymentIntent.id} for user ${req.user.id}: ${pkg.points} points for $${pkg.priceUSD}`);

      res.json({ 
        clientSecret: paymentIntent.client_secret,
        packageType,
        points: pkg.points,
        amount: pkg.priceUSD
      });
    } catch (error: any) {
      console.error('❌ [STRIPE] Error creating payment intent:', error);
      res.status(500).json({ 
        message: "Error creating payment intent: " + error.message 
      });
    }
  });

  // Handle successful payment completion
  app.post("/api/complete-payment", async (req, res) => {
    if (!req.isAuthenticated()) {
      return res.status(401).json({ message: "Authentication required" });
    }

    try {
      const { paymentIntentId } = req.body;

      if (!paymentIntentId) {
        return res.status(400).json({ message: "Payment intent ID required" });
      }

      // Retrieve payment intent from Stripe
      const paymentIntent = await stripe.paymentIntents.retrieve(paymentIntentId);

      if (paymentIntent.status !== 'succeeded') {
        return res.status(400).json({ message: "Payment not completed" });
      }

      // Get point purchase record
      const purchase = await storage.getPointPurchaseByPaymentIntentId(paymentIntentId);
      if (!purchase) {
        return res.status(404).json({ message: "Purchase record not found" });
      }

      // Verify user owns this purchase
      if (purchase.userId !== req.user.id) {
        return res.status(403).json({ message: "Access denied" });
      }

      // If already completed, return success
      if (purchase.status === 'completed') {
        return res.json({ success: true, points: purchase.points });
      }

      // Complete the purchase
      await storage.updatePointPurchase(purchase.id, {
        status: 'completed',
        completedAt: new Date(),
      });

      // Add purchased points to user account
      await storage.addPurchasedPoints(req.user.id, purchase.points);

      console.log(`✅ [STRIPE] Completed point purchase for user ${req.user.id}: ${purchase.points} points added`);

      res.json({ 
        success: true, 
        points: purchase.points,
        message: `Successfully added ${purchase.points} points to your account!`
      });
    } catch (error: any) {
      console.error('❌ [STRIPE] Error completing payment:', error);
      res.status(500).json({ 
        message: "Error completing payment: " + error.message 
      });
    }
  });

  // Get point packages for frontend
  app.get("/api/point-packages", (req, res) => {
    res.json(POINT_PACKAGES);
  });

  // Get user's purchase history
  app.get("/api/purchase-history", async (req, res) => {
    if (!req.isAuthenticated()) {
      return res.status(401).json({ message: "Authentication required" });
    }

    try {
      const purchases = await storage.getPointPurchasesByUserId(req.user.id);
      res.json(purchases);
    } catch (error: any) {
      console.error('❌ [STRIPE] Error fetching purchase history:', error);
      res.status(500).json({ message: "Error fetching purchase history" });
    }
  });
}