import { z } from "zod";

// YouTube API Response Types
export interface YouTubeChannelResponse {
  items: Array<{
    id: string;
    snippet: {
      title: string;
      description: string;
      customUrl?: string;
      publishedAt: string;
      thumbnails: {
        default: { url: string };
        medium: { url: string };
        high: { url: string };
      };
    };
    statistics: {
      viewCount: string;
      subscriberCount: string;
      hiddenSubscriberCount: boolean;
      videoCount: string;
    };
  }>;
}

export interface YouTubeVideoResponse {
  items: Array<{
    id: string;
    snippet: {
      title: string;
      description: string;
      publishedAt: string;
      thumbnails: {
        default: { url: string };
        medium: { url: string };
        high: { url: string };
        standard?: { url: string };
        maxres?: { url: string };
      };
      channelId: string;
      channelTitle: string;
    };
    statistics: {
      viewCount: string;
      likeCount?: string;
      commentCount?: string;
    };
    contentDetails: {
      duration: string;
    };
  }>;
}

export interface YouTubeSearchResponse {
  items: Array<{
    id: {
      kind: string;
      videoId?: string;
      channelId?: string;
    };
    snippet: {
      title: string;
      description: string;
      publishedAt: string;
      thumbnails: {
        default: { url: string };
        medium: { url: string };
        high: { url: string };
      };
      channelId: string;
      channelTitle: string;
    };
  }>;
}

class YouTubeAPI {
  private apiKey: string;
  private baseUrl = 'https://www.googleapis.com/youtube/v3';

  constructor() {
    this.apiKey = process.env.YOUTUBE_API_KEY!;
    if (!this.apiKey) {
      throw new Error('YOUTUBE_API_KEY environment variable is required');
    }
  }

  // Parse various YouTube URL formats to extract channel identifier
  parseChannelUrl(url: string): { type: string; id: string } | null {
    const patterns = [
      { type: 'channelId', regex: /youtube\.com\/channel\/([a-zA-Z0-9_-]+)/ },
      { type: 'handle', regex: /youtube\.com\/@([a-zA-Z0-9_.-]+)/ },
      { type: 'customUrl', regex: /youtube\.com\/c\/([a-zA-Z0-9_.-]+)/ },
      { type: 'username', regex: /youtube\.com\/user\/([a-zA-Z0-9_.-]+)/ },
      { type: 'customUrl', regex: /youtube\.com\/([a-zA-Z0-9_.-]+)$/ }
    ];

    for (const pattern of patterns) {
      const match = url.match(pattern.regex);
      if (match) {
        return { type: pattern.type, id: match[1] };
      }
    }
    return null;
  }

  // Get channel by any URL format
  async getChannelByUrl(url: string): Promise<YouTubeChannelResponse | null> {
    const parsed = this.parseChannelUrl(url);
    if (!parsed) {
      throw new Error('Invalid YouTube channel URL format');
    }

    let channel = null;
    
    try {
      switch (parsed.type) {
        case 'channelId':
          channel = await this.getChannelById(parsed.id);
          break;
        case 'handle':
          channel = await this.getChannelByHandle(parsed.id);
          break;
        case 'username':
          channel = await this.getChannelByUsername(parsed.id);
          break;
        case 'customUrl':
          // Try custom URL first, then search
          const searchResult = await this.searchChannels(parsed.id);
          if (searchResult?.items?.length && searchResult.items.length > 0) {
            const channelId = searchResult.items[0].id.channelId;
            if (channelId) {
              channel = await this.getChannelById(channelId);
            }
          }
          break;
      }
    } catch (error) {
      console.error(`Failed to get channel with ${parsed.type}: ${parsed.id}`, error);
    }

    return channel;
  }

  async getChannelByUsername(username: string): Promise<YouTubeChannelResponse | null> {
    try {
      // Remove @ symbol if present
      const cleanUsername = username.replace('@', '');
      
      const response = await fetch(
        `${this.baseUrl}/channels?part=snippet,statistics&forUsername=${cleanUsername}&key=${this.apiKey}`
      );
      
      if (!response.ok) {
        console.error('YouTube API error:', response.statusText);
        return null;
      }
      
      const data = await response.json();
      return data.items?.length > 0 ? data : null;
    } catch (error) {
      console.error('Error fetching channel by username:', error);
      return null;
    }
  }

  async getChannelByHandle(handle: string): Promise<YouTubeChannelResponse | null> {
    try {
      // Remove @ symbol if present
      const cleanHandle = handle.replace('@', '');
      
      const response = await fetch(
        `${this.baseUrl}/channels?part=snippet,statistics&forHandle=${cleanHandle}&key=${this.apiKey}`
      );
      
      if (!response.ok) {
        console.error('YouTube API error:', response.statusText);
        return null;
      }
      
      const data = await response.json();
      return data.items?.length > 0 ? data : null;
    } catch (error) {
      console.error('Error fetching channel by handle:', error);
      return null;
    }
  }

  async searchChannels(query: string, maxResults: number = 10): Promise<YouTubeSearchResponse | null> {
    try {
      const response = await fetch(
        `${this.baseUrl}/search?part=snippet&q=${encodeURIComponent(query)}&type=channel&maxResults=${maxResults}&key=${this.apiKey}`
      );
      
      if (!response.ok) {
        console.error('YouTube API error:', response.statusText);
        return null;
      }
      
      return await response.json();
    } catch (error) {
      console.error('Error searching channels:', error);
      return null;
    }
  }

  async getChannelById(channelId: string): Promise<YouTubeChannelResponse | null> {
    try {
      const response = await fetch(
        `${this.baseUrl}/channels?part=snippet,statistics&id=${channelId}&key=${this.apiKey}`
      );
      
      if (!response.ok) {
        console.error('YouTube API error:', response.statusText);
        return null;
      }
      
      const data = await response.json();
      return data.items?.length > 0 ? data : null;
    } catch (error) {
      console.error('Error fetching channel by ID:', error);
      return null;
    }
  }

  // Search for videos by title within a specific channel
  async searchChannelVideos(channelId: string, query: string, maxResults: number = 10): Promise<YouTubeVideoResponse | null> {
    try {
      const searchResponse = await fetch(
        `${this.baseUrl}/search?part=snippet&channelId=${channelId}&q=${encodeURIComponent(query)}&type=video&maxResults=${maxResults}&key=${this.apiKey}`
      );
      
      if (!searchResponse.ok) {
        console.error('YouTube API error searching videos:', searchResponse.statusText);
        return null;
      }
      
      const searchData = await searchResponse.json();
      const videoIds = searchData.items?.map((item: any) => item.id.videoId) || [];
      
      if (videoIds.length === 0) {
        return { items: [] };
      }
      
      // Get detailed video information
      const videosResponse = await fetch(
        `${this.baseUrl}/videos?part=snippet,statistics,contentDetails&id=${videoIds.join(',')}&key=${this.apiKey}`
      );
      
      if (!videosResponse.ok) {
        console.error('YouTube API error fetching video details:', videosResponse.statusText);
        return null;
      }
      
      return await videosResponse.json();
    } catch (error) {
      console.error('Error searching channel videos:', error);
      return null;
    }
  }

  // Extract the proper channel URL from YouTube channel data
  getChannelUrl(channelData: any): string {
    // Priority: customUrl > handle (if available) > fallback to channel ID
    if (channelData.snippet?.customUrl) {
      return `https://youtube.com/${channelData.snippet.customUrl}`;
    }
    
    // Try to extract handle from customUrl or use a basic format
    const channelName = channelData.snippet?.title || '';
    const cleanName = channelName.toLowerCase()
      .replace(/[^a-z0-9]/g, '')
      .substring(0, 20);
    
    return `https://youtube.com/@${cleanName}`;
  }

  async getChannelVideos(channelId: string, maxResults: number = 30): Promise<YouTubeVideoResponse | null> {
    try {
      // First get the uploads playlist ID
      const channelResponse = await fetch(
        `${this.baseUrl}/channels?part=contentDetails&id=${channelId}&key=${this.apiKey}`
      );
      
      if (!channelResponse.ok) {
        console.error('YouTube API error fetching channel:', channelResponse.statusText);
        return null;
      }
      
      const channelData = await channelResponse.json();
      const uploadsPlaylistId = channelData.items?.[0]?.contentDetails?.relatedPlaylists?.uploads;
      
      if (!uploadsPlaylistId) {
        console.error('No uploads playlist found for channel');
        return null;
      }
      
      // Get video IDs from uploads playlist (newest videos are first by default)
      const playlistResponse = await fetch(
        `${this.baseUrl}/playlistItems?part=snippet&playlistId=${uploadsPlaylistId}&maxResults=${maxResults}&key=${this.apiKey}`
      );
      
      if (!playlistResponse.ok) {
        console.error('YouTube API error fetching playlist:', playlistResponse.statusText);
        return null;
      }
      
      const playlistData = await playlistResponse.json();
      const videoIds = playlistData.items?.map((item: any) => item.snippet.resourceId.videoId) || [];
      
      if (videoIds.length === 0) {
        return { items: [] };
      }
      
      // Get detailed video information
      const videosResponse = await fetch(
        `${this.baseUrl}/videos?part=snippet,statistics,contentDetails&id=${videoIds.join(',')}&key=${this.apiKey}`
      );
      
      if (!videosResponse.ok) {
        console.error('YouTube API error fetching videos:', videosResponse.statusText);
        return null;
      }
      
      return await videosResponse.json();
    } catch (error) {
      console.error('Error fetching channel videos:', error);
      return null;
    }
  }

  // Parse YouTube duration format (PT15M33S) to seconds
  parseDuration(duration: string): number {
    const match = duration.match(/PT(?:(\d+)H)?(?:(\d+)M)?(?:(\d+)S)?/);
    if (!match) return 0;
    
    const hours = parseInt(match[1] || '0');
    const minutes = parseInt(match[2] || '0');
    const seconds = parseInt(match[3] || '0');
    
    return hours * 3600 + minutes * 60 + seconds;
  }

  // Convert duration in seconds to human readable format
  formatDuration(seconds: number): string {
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = seconds % 60;
    
    if (hours > 0) {
      return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    }
    return `${minutes}:${secs.toString().padStart(2, '0')}`;
  }

  // Calculate estimated watch time hours from total views and average duration
  calculateWatchTimeHours(totalViews: number, averageDurationSeconds: number): number {
    // Estimate based on average view duration (typically 40-60% of video length)
    const averageViewDuration = averageDurationSeconds * 0.5; // 50% retention rate estimate
    const totalWatchTimeSeconds = totalViews * averageViewDuration;
    return Math.floor(totalWatchTimeSeconds / 3600);
  }
}

export const youtubeAPI = new YouTubeAPI();