import { pgTable, text, serial, integer, boolean, timestamp, real, unique, jsonb, decimal } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";
import { users } from "./schema";

// Reddit authentication connections
export const redditConnections = pgTable("reddit_connections", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  redditUserId: text("reddit_user_id").notNull(),
  redditUsername: text("reddit_username").notNull(),
  accessToken: text("access_token").notNull(),
  refreshToken: text("refresh_token").notNull(),
  expiresAt: timestamp("expires_at").notNull(),
  scope: text("scope").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  userIdUnique: unique("reddit_user_id_unique").on(table.userId),
}));

// Managed subreddits - subreddits users want to track/manage
export const managedSubreddits = pgTable("managed_subreddits", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  subredditName: text("subreddit_name").notNull(),
  subredditId: text("subreddit_id").notNull(),
  displayName: text("display_name").notNull(),
  title: text("title"),
  description: text("description"),
  publicDescription: text("public_description"),
  subscribers: integer("subscribers").default(0),
  activeUsers: integer("active_users").default(0),
  isNsfw: boolean("is_nsfw").default(false),
  subredditType: text("subreddit_type").default("public"),
  isModerator: boolean("is_moderator").default(false),
  isActive: boolean("is_active").default(true),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
}, (table) => ({
  userSubredditUnique: unique("user_subreddit_unique").on(table.userId, table.subredditName),
}));

// Reddit posts tracking
export const redditPosts = pgTable("reddit_posts", {
  id: serial("id").primaryKey(),
  userId: integer("user_id").notNull().references(() => users.id, { onDelete: "cascade" }),
  subredditId: integer("subreddit_id").references(() => managedSubreddits.id, { onDelete: "cascade" }),
  postId: text("post_id").notNull().unique(),
  title: text("title").notNull(),
  content: text("content"),
  url: text("url"),
  permalink: text("permalink"),
  score: integer("score").default(0),
  upvoteRatio: real("upvote_ratio").default(0),
  numComments: integer("num_comments").default(0),
  isNsfw: boolean("is_nsfw").default(false),
  isStickied: boolean("is_stickied").default(false),
  postType: text("post_type").default("text"), // text, link, image, video
  scheduledFor: timestamp("scheduled_for"),
  status: text("status").default("draft"), // draft, scheduled, posted, failed
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Subreddit analytics tracking
export const subredditAnalytics = pgTable("subreddit_analytics", {
  id: serial("id").primaryKey(),
  subredditId: integer("subreddit_id").notNull().references(() => managedSubreddits.id, { onDelete: "cascade" }),
  date: text("date").notNull(), // YYYY-MM-DD format
  subscribers: integer("subscribers").default(0),
  activeUsers: integer("active_users").default(0),
  avgPosts: integer("avg_posts").default(0),
  avgComments: integer("avg_comments").default(0),
  topPostScore: integer("top_post_score").default(0),
  engagementRate: real("engagement_rate").default(0),
  createdAt: timestamp("created_at").defaultNow(),
}, (table) => ({
  subredditDateUnique: unique("subreddit_date_unique").on(table.subredditId, table.date),
}));

// Schema definitions
export const insertRedditConnectionSchema = createInsertSchema(redditConnections).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertManagedSubredditSchema = createInsertSchema(managedSubreddits).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertRedditPostSchema = createInsertSchema(redditPosts).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export const insertSubredditAnalyticsSchema = createInsertSchema(subredditAnalytics).omit({
  id: true,
  createdAt: true,
});

// Subreddit rules cache
export const subredditRulesCache = pgTable("subreddit_rules_cache", {
  id: serial("id").primaryKey(),
  subredditName: text("subreddit_name").notNull().unique(),
  rulesJson: jsonb("rules_json").notNull(),
  description: text("description"),
  sidebarHtml: text("sidebar_html"),
  subredditType: text("subreddit_type"),
  submissionType: text("submission_type"),
  lastUpdated: timestamp("last_updated").defaultNow(),
  createdAt: timestamp("created_at").defaultNow(),
});

// Moderation queue items cache
export const moderationQueueItems = pgTable("moderation_queue_items", {
  id: serial("id").primaryKey(),
  subredditName: text("subreddit_name").notNull(),
  redditId: text("reddit_id").notNull().unique(),
  itemType: text("item_type").notNull(), // 'submission' or 'comment'
  author: text("author"),
  title: text("title"),
  content: text("content"),
  url: text("url"),
  permalink: text("permalink"),
  score: integer("score").default(0),
  numReports: integer("num_reports").default(0),
  userReports: jsonb("user_reports").default([]),
  modReports: jsonb("mod_reports").default([]),
  createdUtc: integer("created_utc").notNull()
});

// AI analysis results
export const aiAnalysisResults = pgTable("ai_analysis_results", {
  id: serial("id").primaryKey(),
  modqueueItemId: integer("modqueue_item_id").notNull().references(() => moderationQueueItems.id, { onDelete: "cascade" }),
  violationsFound: jsonb("violations_found").default([]),
  confidenceScore: decimal("confidence_score", { precision: 3, scale: 2 }),
  suggestedAction: text("suggested_action"), // 'approve', 'remove', 'spam', 'manual_review'
  reasoning: text("reasoning"),
  severity: text("severity"), // 'low', 'medium', 'high'
  removalReason: text("removal_reason"),
  requiresHumanReview: boolean("requires_human_review").default(false),
  additionalActions: jsonb("additional_actions").default([]),
  createdAt: timestamp("created_at").defaultNow(),
});

// Moderation actions taken
export const moderationActions = pgTable("moderation_actions", {
  id: serial("id").primaryKey(),
  modqueueItemId: integer("modqueue_item_id").references(() => moderationQueueItems.id),
  userId: integer("user_id").notNull().references(() => users.id),
  actionType: text("action_type").notNull(), // 'approve', 'remove', 'spam', 'manual_review'
  actionReason: text("action_reason"),
  wasAiSuggested: boolean("was_ai_suggested").default(false),
  aiConfidenceScore: decimal("ai_confidence_score", { precision: 3, scale: 2 }),
  moderatorNote: text("moderator_note"),
  humanResponse: text("human_response"), // Human-like response for modmail
  actionTimestamp: timestamp("action_timestamp").defaultNow(),
  redditResponse: jsonb("reddit_response"),
});

// Schema definitions for moderation tables
export const insertSubredditRulesCacheSchema = createInsertSchema(subredditRulesCache).omit({
  id: true,
  lastUpdated: true,
  createdAt: true,
});

export const insertModerationQueueItemSchema = createInsertSchema(moderationQueueItems).omit({
  id: true,
});

export const insertAiAnalysisResultSchema = createInsertSchema(aiAnalysisResults).omit({
  id: true,
  createdAt: true,
});

export const insertModerationActionSchema = createInsertSchema(moderationActions).omit({
  id: true,
  actionTimestamp: true,
}).extend({
  modqueueItemId: z.number().nullable()
});

// Type definitions
export type InsertRedditConnection = z.infer<typeof insertRedditConnectionSchema>;
export type RedditConnection = typeof redditConnections.$inferSelect;
export type InsertManagedSubreddit = z.infer<typeof insertManagedSubredditSchema>;
export type ManagedSubreddit = typeof managedSubreddits.$inferSelect;
export type InsertRedditPost = z.infer<typeof insertRedditPostSchema>;
export type RedditPost = typeof redditPosts.$inferSelect;
export type InsertSubredditAnalytics = z.infer<typeof insertSubredditAnalyticsSchema>;
export type SubredditAnalytics = typeof subredditAnalytics.$inferSelect;

// Moderation types
export type InsertSubredditRulesCache = z.infer<typeof insertSubredditRulesCacheSchema>;
export type SubredditRulesCache = typeof subredditRulesCache.$inferSelect;
export type InsertModerationQueueItem = z.infer<typeof insertModerationQueueItemSchema>;
export type ModerationQueueItem = typeof moderationQueueItems.$inferSelect;
export type InsertAiAnalysisResult = z.infer<typeof insertAiAnalysisResultSchema>;
export type AiAnalysisResult = typeof aiAnalysisResults.$inferSelect;
export type InsertModerationAction = z.infer<typeof insertModerationActionSchema>;
export type ModerationAction = typeof moderationActions.$inferSelect;